/* XMLTablePanel.java
 *
 * Authors:
 * Stefanovic Nenad  chupo@iis.ns.ac.yu
 * Bojanic Sasa      sasaboy@neobee.net
 * Puskas Vladimir   vpuskas@eunet.yu
 * Pilipovic Goran   zboniek@uns.ac.yu
 *
 */

package com.ds.bpm.bpd.xml.panels;

import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.Collection;
import java.util.Iterator;
import java.util.Vector;

import javax.swing.AbstractAction;
import javax.swing.Box;
import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JViewport;
import javax.swing.KeyStroke;
import javax.swing.ListSelectionModel;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import com.ds.bpm.bpd.xml.XMLCollection;
import com.ds.bpm.bpd.xml.XMLComplexElement;
import com.ds.bpm.bpd.xml.XMLElement;
import com.ds.bpm.bpd.xml.XMLElementDialog;
import com.ds.bpm.bpd.xml.XMLUtil;

/**
 * Creates a table panel. 表格Panel
 */
public class XMLTablePanel extends XMLPanel {

	protected static Dimension miniTableDimension = new Dimension(300, 75);

	protected static Dimension smallTableDimension = new Dimension(300, 150);

	protected static Dimension mediumTableDimension = new Dimension(600, 200);

	protected static Dimension largeTableDimension = new Dimension(800, 200);

	public XMLTablePanel(XMLCollection myOwner) {
		this(myOwner, "");

	}

	public XMLTablePanel(XMLCollection myOwner, String title) {
		this(myOwner, title, true, false);
	}

	public XMLTablePanel(XMLCollection myOwner, String title,
			boolean hasBorder, boolean automaticWidth) {
		this(myOwner, title, hasBorder, automaticWidth, false);
	}

	public XMLTablePanel(XMLElement myOwner, int noOfElements, String title,
			int layout, boolean isVertical, boolean hasBorder) {
		super(myOwner, 2, title, XMLPanel.BOX_LAYOUT, true, hasBorder);
	}

	public XMLTablePanel(XMLCollection myOwner, String title,
			boolean hasBorder, boolean automaticWidth, boolean miniDimension) {

		super(myOwner, 2, title, XMLPanel.BOX_LAYOUT, true, hasBorder);

		// creating a table which do not allow cell editing
		Vector columnNames = new Vector();
		columnNames.add("Object");
		int noOfVisibleColumns = 0;

		Collection c = myOwner.getElementStructure();
		Iterator it = c.iterator();
		while (it.hasNext()) {
			XMLElement el = (XMLElement) it.next();
			columnNames.add(el.toLabel());
			noOfVisibleColumns++;
		}

		JTable allItems = new JTable(new Vector(), columnNames) {
			public boolean isCellEditable(int row, int col) {
				return false;
			}
		};

		int[] invisibleFieldOrdinals = myOwner.getInvisibleTableFieldOrdinals();

		TableColumnModel tcm = allItems.getColumnModel();
		TableColumn column;
		// setting the first column (object column) to be invisible
		column = allItems.getColumnModel().getColumn(0);
		column.setMinWidth(0);
		column.setMaxWidth(0);
		column.setPreferredWidth(0);
		column.setResizable(false);
		// setting fields that will not be displayed within the table
		if (invisibleFieldOrdinals != null && invisibleFieldOrdinals.length > 0) {
			noOfVisibleColumns -= invisibleFieldOrdinals.length;
			for (int i = 0; i < invisibleFieldOrdinals.length; i++) {
				column = allItems.getColumnModel().getColumn(
						invisibleFieldOrdinals[i] + 1);
				column.setMinWidth(0);
				column.setMaxWidth(0);
				column.setPreferredWidth(0);
				column.setResizable(false);
			}
		}

		// setting some table properties
		allItems.setColumnSelectionAllowed(false);
		allItems.setRowSelectionAllowed(true);
		allItems.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		allItems.getTableHeader().setReorderingAllowed(false);

		// fills Table
		DefaultTableModel dtm = (DefaultTableModel) allItems.getModel();
		it = myOwner.getTableElements().iterator();
		while (it.hasNext()) {
			XMLComplexElement elem = (XMLComplexElement) it.next();
			Vector v = new Vector(elem.toComplexTypeValues());
			v.add(0, elem);
			dtm.addRow(v);
		}

		// mouse listener for editing on double-click
		allItems.addMouseListener(new MouseAdapter() {
			public void mouseClicked(MouseEvent me) {
				if (me.getClickCount() > 1) {
					editElementDialog();
				}
			}
		});

		allItems.getInputMap(JComponent.WHEN_FOCUSED).put(
				KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0, false), "edit");
		allItems.getActionMap().put("edit", new AbstractAction() {
			public void actionPerformed(ActionEvent e) {
				editElementDialog();
			}
		});

		allItems.getInputMap(JComponent.WHEN_FOCUSED).put(
				KeyStroke.getKeyStroke(KeyEvent.VK_DELETE, 0, false), "delete");
		allItems.getActionMap().put("delete", new AbstractAction() {
			public void actionPerformed(ActionEvent e) {
				deleteElementDialog();
			}
		});

		allItems.getInputMap(JComponent.WHEN_FOCUSED).put(
				KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0, false), "cancel");
		allItems.getActionMap().put("cancel", new AbstractAction() {
			public void actionPerformed(ActionEvent e) {
				XMLElementDialog d = ((XMLElementDialog) getDialog());
				d.setCanceled(true);
				d.dispose();
			}
		});

		// creates panel
		JScrollPane allItemsPane = new JScrollPane();
		allItemsPane.setViewportView(allItems);
		Dimension tDim;
		if (miniDimension) {
			tDim = new Dimension(miniTableDimension);
		} else if (noOfVisibleColumns <= 3) {
			tDim = new Dimension(smallTableDimension);
		} else if (noOfVisibleColumns <= 5) {
			tDim = new Dimension(mediumTableDimension);
		} else {
			tDim = new Dimension(largeTableDimension);
		}

		if (automaticWidth) {
			tDim.width = allItems.getPreferredScrollableViewportSize().width;
		}
		allItems.setPreferredScrollableViewportSize(new Dimension(tDim));

		add(allItemsPane);
		add(Box.createVerticalGlue());
	}

	public JTable getTable() {
		JScrollPane jsp = (JScrollPane) getComponent(0);
		JViewport jvp = (JViewport) jsp.getComponent(0);
		JTable jt = (JTable) jvp.getComponent(0);
		return jt;
	}

	public XMLComplexElement getSelectedElement() {
		JTable t = getTable();
		int row = t.getSelectedRow();
		if (row >= 0) {
			return (XMLComplexElement) t.getValueAt(row, 0);
		}
		return null;

	}

	protected void modifyElement(XMLComplexElement e) {
		JTable t = getTable();
		int row = t.getSelectedRow();
		DefaultTableModel dtm = (DefaultTableModel) t.getModel();
		Vector v = new Vector(((XMLComplexElement) e).toComplexTypeValues());
		// add object itself to the first column (it is invisible column)
		v.add(0, e);
		dtm.removeRow(row);
		dtm.insertRow(row, v);
		try {
			t.setRowSelectionInterval(row, row);
		} catch (Exception ex) {
		}
	}

	public void removeElement(XMLComplexElement e) {
		((XMLCollection) getOwner()).remove(getSelectedElement());
		JTable t = getTable();
		int row = t.getSelectedRow();
		DefaultTableModel dtm = (DefaultTableModel) t.getModel();
		dtm.removeRow(row);
		try {
			if (row == 0) {
				row++;
			}
			t.setRowSelectionInterval(row - 1, row - 1);
		} catch (Exception ex) {
		}
		t.requestFocus();
	}

	protected void editElementDialog() {
		XMLComplexElement editElement = getSelectedElement();
		if (editElement != null) {
			XMLElementDialog de = new XMLElementDialog(getDialog(), editElement
					.toLabel()
					+ " '"
					+ editElement.toString()
					+ "' - "
					+ XMLUtil.getLanguageDependentString("EditingKey"));
			de.editXMLElement(editElement.getPanel(), true, false);
			if (!de.isCanceled()) {
				modifyElement(editElement);
				((XMLCollection) getOwner()).onElementModified(editElement);
			}
		} else {
			JOptionPane
					.showMessageDialog(
							getDialog(),
							XMLUtil
									.getLanguageDependentString("WarningEmptySelectionToEditOrDelete"),
									XMLUtil.getLanguageDependentString("Title"),
							JOptionPane.WARNING_MESSAGE);
		}
		getTable().requestFocus();
	}

	protected void deleteElementDialog() {
		XMLComplexElement deleteElement = getSelectedElement();
		if (deleteElement == null) {
			JOptionPane
					.showMessageDialog(
							getDialog(),
							XMLUtil
									.getLanguageDependentString("WarningEmptySelectionToEditOrDelete"),
									XMLUtil.getLanguageDependentString("Title"),
							JOptionPane.WARNING_MESSAGE);
		} else if (deleteElement.isReadOnly()) {
			JOptionPane.showMessageDialog(getDialog(), XMLUtil
					.getLanguageDependentString(((XMLCollection) getOwner())
							.getReadOnlyMessageName(deleteElement)), XMLUtil
					.getLanguageDependentString("Title"),
					JOptionPane.WARNING_MESSAGE);
		} else if (!((XMLCollection) getOwner())
				.canRemoveElement(deleteElement)) {
			JOptionPane.showMessageDialog(getDialog(), XMLUtil
					.getLanguageDependentString(((XMLCollection) getOwner())
							.getInUseMessageName(deleteElement)), XMLUtil
					.getLanguageDependentString("Title"),
					JOptionPane.WARNING_MESSAGE);
		} else {
			int yn = JOptionPane
					.showConfirmDialog(
							getDialog(),
							XMLUtil
									.getLanguageDependentString("MessageDoYouReallyWantToRemoveSelectedItem"),
							deleteElement.toLabel()
									+ " - "
									+ XMLUtil
											.getLanguageDependentString("DeletingKey"),
							JOptionPane.YES_NO_OPTION);
			if (yn == JOptionPane.YES_OPTION) {
				removeElement(deleteElement);
				((XMLCollection) getOwner()).onElementDeleted(deleteElement);
				XMLElementDialog.notifyListeners(new XMLElement());
			}
		}

	}

}
