/* FormalParameters.java
 *
 * Authors:
 * Stefanovic Nenad  chupo@iis.ns.ac.yu
 * Bojanic Sasa      sasaboy@neobee.net
 * Puskas Vladimir   vpuskas@eunet.yu
 * Pilipovic Goran   zboniek@uns.ac.yu
 *
 */

package com.ds.bpm.bpd.xml.elements.formula;

import java.util.Collection;
import java.util.Iterator;

import com.ds.bpm.bpd.xml.XML;
import com.ds.bpm.bpd.xml.XMLCollection;
import com.ds.bpm.bpd.xml.XMLComplexElement;
import com.ds.bpm.bpd.xml.XMLElement;
import com.ds.bpm.bpd.xml.activity.Activities;
import com.ds.bpm.bpd.xml.elements.ActivitySets;
import com.ds.bpm.bpd.xml.elements.Transitions;
import com.ds.bpm.bpd.xml.elements.WorkflowProcess;

/**
 * Represents a WfMC DTD element that has the similar name. This class is a
 * collection of class <b>FormalParameter</b> instances.
 * 
 * @see XML
 */
public class FormalParameters extends XMLCollection {
	/** The package which elements holds instance of this class. */
	private transient com.ds.bpm.bpd.xml.elements.Package myPackage = null;

	/**
	 * Creates a new instance of the class.
	 * 
	 * @param myPackage
	 *            The package which elements holds instance of this class.
	 */
	public FormalParameters(XMLComplexElement myOwner, com.ds.bpm.bpd.xml.elements.Package myPackage) {
		super(myOwner);
		this.myPackage = myPackage;
	}

	/**
	 * Generates a new element of the class which instances are members of
	 * collection of this class. return The generated instance of class that
	 * makes collection.
	 */
	public XMLElement generateNewElement() {
		FormalParameter fp = new FormalParameter(this, myPackage);
		fp.setRequired(true);
		return fp;
	}

	/**
	 * It is overriden to set the read-only property of Id attribute to the
	 * proper value in the case of formal parameters of workflow process: if
	 * FormalParameter instance is used in transition conditions, it is set to
	 * be read only, otherwise, it is set to be editable.
	 */
	public Collection getTableElements() {
		if (myOwner instanceof WorkflowProcess) {
			if (!((WorkflowProcess) myOwner).getPackage().isReadOnly()) {
				Iterator i = refCollectionElements.iterator();
				while (i.hasNext()) {
					FormalParameter fp = (FormalParameter) i.next();
					// Checks for formal parameter if it is used inside some
					// transition condition expression, and if it isn't, set
					// it's ID attribute not to be read only.
					if (!isUsedWithinTransitionConditions(fp)) {
						fp.get("Id").setReadOnly(false);
					} else {
						fp.get("Id").setReadOnly(true);
					}
				}
			}
		}
		return super.getTableElements();
	}

	/**
	 * Returns a formal parameter with specified ID.
	 * 
	 * @param ID
	 *            The 'Id' attribute of wanted instance of FormalParameter
	 *            element defined within collection.
	 * @return The instance of FormalParameter element that have the specified
	 *         ID as an 'Id' attribute if it exist among all defined formal
	 *         parameters within collection, if it doesn't exist, <tt>null</tt>
	 *         is returned.
	 */
	public FormalParameter getFormalParameter(String ID) {
		return (FormalParameter) super.getCollectionElement(ID);
	}

	/**
	 * Checks if specified formal parameter can be removed. FormalParameter can
	 * be removed only if it is not used anywhere.
	 */
	public boolean canRemoveElement(XMLElement el) {
		boolean remove = true;
		FormalParameter toRemove = (FormalParameter) el;
		// if this is FormalParameter from workflow process, check if it is
		// used by any activity of subflow or generic type, or if it
		// is used within transition conditions within workflow process
		if (myOwner instanceof WorkflowProcess) {
			Activities acts = (Activities) myOwner.get("Activities");
			remove = acts.canRemoveDataFieldOrFormalParameter(toRemove);
			// if it claims to be removed, check transitions
			if (remove == true) {
				Transitions ts = (Transitions) myOwner.get("Transitions");
				remove = ts.canRemoveDataFieldOrFormalParameter(toRemove);
			}
			// if it claims to be removed, check activity sets
			if (remove == true) {
				ActivitySets actsts = (ActivitySets) myOwner
						.get("ActivitySets");
				remove = actsts.canRemoveDataFieldOrFormalParameter(toRemove);
			}
		}
		return remove;
	}

	protected boolean isUsedWithinTransitionConditions(FormalParameter fp) {
		boolean inUse = false;
		// if this is formal parameter from workflow process, check all
		// transitions
		// within this workflow process
		if (myOwner instanceof WorkflowProcess) {
			Transitions ts = (Transitions) myOwner.get("Transitions");
			inUse = !ts.canRemoveDataFieldOrFormalParameter(fp);
			// if it claims not to be used, check activity sets
			if (inUse == false) {
				ActivitySets actsts = (ActivitySets) myOwner
						.get("ActivitySets");
				inUse = actsts
						.isDataFieldOrFormalParameterUsedWithinTransitionConditions(fp);
			}
		}
		return inUse;
	}

	public int[] getInvisibleTableFieldOrdinals() {
		if (!(myOwner instanceof WorkflowProcess)) {
			int[] itfo = new int[2];
			itfo[0] = 0;
			itfo[1] = 4;
			return itfo;
		} else {
			int[] itfo = new int[1];
			itfo[0] = 4;
			return itfo;
		}
	}

}
