/* UserProperties.java
 *
 * Authors:
 * Stefanovic Nenad  chupo@iis.ns.ac.yu
 * Bojanic Sasa      sasaboy@neobee.net
 * Puskas Vladimir   vpuskas@eunet.yu
 * Pilipovic Goran   zboniek@uns.ac.yu
 *
 */

package com.ds.bpm.bpd.xml.elements;

import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import com.ds.bpm.bpd.xml.XML;
import com.ds.bpm.bpd.xml.XMLCollection;
import com.ds.bpm.bpd.xml.XMLElement;

/**
 * This is not a real WfMC DTD element, but it is introduced to help. This class
 * is a collection of class <b>UserProperty</b> instances.
 * 
 * @see XML
 */
public class UserProperties extends XMLCollection {
	/**
	 * Creates a new instance of the class.
	 * 
	 * @param myPackage
	 *            The reference to package that defines properties.
	 */
	public UserProperties(Package myPackage) {
		super(myPackage);
	}

	// min=0, max=unbounded
	/**
	 * Generates a new element of the class which instances are members of
	 * collection of this class. return The generated instance of class that
	 * makes collection.
	 */
	public XMLElement generateNewElement() {
		UserProperty up = new UserProperty(this);
		up.setRequired(true);
		return up;
	}

	public int[] getInvisibleTableFieldOrdinals() {
		int[] itfo = new int[1];
		itfo[0] = 0;
		return itfo;
	}

	/**
	 * Overrides super-class method to return only optional properties to be
	 * choosable.
	 * 
	 * @return Collection of choosable properties.
	 */
	public Collection getChoosable() {
		Set optionalProps = new HashSet();

		Iterator it = refCollectionElements.iterator();
		while (it.hasNext()) {
			UserProperty up = (UserProperty) it.next();
			if (up.isOptional()) {
				optionalProps.add(up);
			}
		}

		return optionalProps;
	}

	/**
	 * Returns set of mandatory and 'mandatory from it's creation' properties
	 * defined in package.
	 * 
	 * @return The set of all mandatory-like properties.
	 */
	public Set getMandatoryProperties() {
		Set mandatoryProps = new HashSet();

		Iterator it = refCollectionElements.iterator();
		while (it.hasNext()) {
			UserProperty up = (UserProperty) it.next();
			if (!up.isOptional()) {
				mandatoryProps.add(up);
			}
		}

		return mandatoryProps;
	}

	/**
	 * Returns a property specified by ID.
	 * 
	 * @param ID
	 *            An 'Id' attribut of desired property.
	 * @return The property specified by ID parameter. If such doesn't exist,
	 *         returns <tt>null</tt>.
	 */
	public UserProperty getProperty(String ID) {
		return (UserProperty) super.getCollectionElement(ID);
	}

	/**
	 * Overrides super-class method to implement class specific actions when
	 * element is created. In this case, user property, which is created, is
	 * dded to all activities if it is a mandatory property.
	 * 
	 * @param el
	 *            Created element - in this case it must be instance of
	 *            UserProperty class.
	 */
	public void onElementCreated(XMLElement el) {
		UserProperty up = (UserProperty) el;
		if (up.isMandatory()) {
			((Package) myOwner).addPropertyToAllActivities(up);
		}
	}

	/**
	 * Overrides super-class method to implement class specific actions when
	 * element is modified. In this case, user property, which is modified, is
	 * added to all activities if it become mandatory property, otherwise, if it
	 * stoped to be mandatory, sets some flags to allow it's removal from
	 * activities collection of properties.
	 * 
	 * @param el
	 *            Modified element - in this case it must be instance of
	 *            UserProperty class.
	 */
	public void onElementModified(XMLElement el) {
		UserProperty up = (UserProperty) el;
		if (up.isMandatory()) {
			((Package) myOwner).addPropertyToAllActivities(up);
		} else {
			((Package) myOwner).refreshMandatories(up);
		}
	}

	/**
	 * Overrides super-class method to implement class specific actions when
	 * element is deleted. In this case, user property, which is deleted, is
	 * removed from all activities.
	 * 
	 * @param el
	 *            Deleted element - in this case it must be instance of
	 *            UserProperty class.
	 */
	public void onElementDeleted(XMLElement el) {
		UserProperty up = (UserProperty) el;
		((Package) myOwner).removePropertyFromAllActivities(up);
	}

	/**
	 * Overrides parent method to be accessible within package.
	 */
	protected void updateID(String someID) {
		super.updateID(someID);
	}
}
