/* Participants.java
 *
 * Authors:
 * Stefanovic Nenad  chupo@iis.ns.ac.yu
 * Bojanic Sasa      sasaboy@neobee.net
 * Puskas Vladimir   vpuskas@eunet.yu
 * Pilipovic Goran   zboniek@uns.ac.yu
 *
 */

package com.ds.bpm.bpd.xml.elements;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import com.ds.bpm.bpd.xml.XMLCollection;
import com.ds.bpm.bpd.xml.XMLComplexElement;
import com.ds.bpm.bpd.xml.XMLElement;
import com.ds.bpm.bpd.xml.activity.Activities;

/**
 * Represents a WfMC DTD element that has the similar name. Participants are
 * collection of class Participant instances.
 * 
 * @see Participant
 */
public class Participants extends XMLCollection {
	private ArrayList externalParticipants = new ArrayList();

	/**
	 * Creates a new instance of the class. Constructor must know who the senior
	 * element is, because behaviour depends entirely on this information.
	 * Package contained Participants are visible throughout package, whilst
	 * those created whitin workflow process can be used only locally.
	 * 
	 * @param senior
	 *            DTDComplexElement
	 */
	public Participants(XMLComplexElement senior) {
		super(senior);
	}

	/**
	 * Checks if specified participant can be removed. Participant can be
	 * removed only if it is not used anywhere.
	 */
	public boolean canRemoveElement(XMLElement el) {
		Participant toRemove = (Participant) el;
		// if this participant doesn't belong to collection, return false
		if (!isMine(toRemove))
			return false;

		return canRemoveParticipant(toRemove);
	}

	public boolean canRemoveParticipant(Participant toRemove) {
		// check if it is somewhere in the graph of processes
		if (toRemove.isGraphContained()) {
			// System.out.println("Participant "+toRemove+" has
			// "+toRemove.getNoOfGraphReferences()+" graph references");
			return false;
		}

		// List of workflow processes to check. If this is
		// participants instance at workflow process level, the list
		// will contain only the owner of this instance, and
		// if this is participants instance at package level,
		// it will contain all processes within package
		ArrayList wpsToCheck = new ArrayList();

		if (myOwner instanceof Package) {
			// check if participant is responsible for the package
			Responsibles rs = (Responsibles) ((RedefinableHeader) myOwner
					.get("RedefinableHeader")).get("Responsibles");
			if (!rs.canRemoveParticipant(toRemove)) {
				return false;
			}
			// add all process to check
			WorkflowProcesses wps = (WorkflowProcesses) myOwner
					.get("WorkflowProcesses");
			wpsToCheck.addAll(wps.toCollection());
		} else {
			wpsToCheck.add(myOwner);
		}

		// check all processes for removal
		Iterator it = wpsToCheck.iterator();
		while (it.hasNext()) {
			WorkflowProcess wp = (WorkflowProcess) it.next();
			// check if it is a performer of some activity
			Activities acts = (Activities) wp.get("Activities");
			if (!acts.canRemoveParticipant(toRemove)) {
				return false;
			}
			ActivitySets actsts = (ActivitySets) wp.get("ActivitySets");
			if (!actsts.canRemoveParticipant(toRemove)) {
				return false;
			}

			// check if it has referenced as responsible for a process
			Responsibles rs = (Responsibles) ((RedefinableHeader) wp
					.get("RedefinableHeader")).get("Responsibles");
			if (!rs.canRemoveParticipant(toRemove)) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Oddly named method tells if this instance of Participants is workflow
	 * process contained, thus is able to show or hide participants. This
	 * feature is meaningless for package Participants.
	 * 
	 * @return true - for workflow process contained Participants
	 */
	public boolean canHideOrShow() {
		return (myOwner instanceof WorkflowProcess);
	}

	/**
	 * Generates a new element of the class which instances are members of
	 * collection of this class. return The generated instance of class that
	 * makes collection.
	 */
	public XMLElement generateNewElement() {
		Participant p = new Participant(this);
		p.setRequired(true);
		// p.get("ExternalReference").setRequired(false);
		return p;
	}

	/**
	 * It is overriden to give the caller all Participants:
	 * <ul>
	 * <li> If Participants instance is defined at the the workflow process
	 * level, it gives all of it's participants plus the ones defined at the
	 * package level (including all participants from externaly referenced
	 * packages).
	 * <li> If Participants instance is defined at the the package level, it
	 * gives all of it's participants plus the ones from externaly referenced
	 * packages).
	 * </ul>
	 */
	public Collection getTableElements() {
		ArrayList mine = new ArrayList();
		if (myOwner instanceof Package) {
			mine.add(Participant.getFreeTextExpressionParticipant());
			mine.addAll(refCollectionElements);
			mine.addAll(externalParticipants);
		} else {
			Participants ps = (Participants) (((WorkflowProcess) myOwner)
					.getPackage()).get("Participants");
			mine.add(Participant.getFreeTextExpressionParticipant());
			mine.addAll(refCollectionElements);
			mine.addAll(ps.refCollectionElements);
			mine.addAll(ps.externalParticipants);
		}

		// if this is collection within the package that is not external,
		// remove read-only attribute for all participants that belong
		// to the collection, and set to others
		if (!getPackage().isReadOnly()) {
			Iterator i = mine.iterator();
			while (i.hasNext()) {
				Participant p = (Participant) i.next();
				p.setReadOnly(!isMine(p));
				// set Id attribute of participant to be read-only
				// p.get("Id").setReadOnly(true);
			}
		}
		return mine;
	}

	/**
	 * Returns collection of all participants from the level where this
	 * collection belongs (package or process level) and all levels above, but
	 * leaving out the participants from higher level that were redefined at
	 * lower level.
	 * 
	 * @see #getTableElements
	 */
	public Collection getChoosable() {
		ArrayList chos = new ArrayList();
		if (myOwner instanceof Package) {
			chos.addAll(refCollectionElements);
			Iterator i = externalParticipants.iterator();
			while (i.hasNext()) {
				Participant p = (Participant) i.next();
				if (getCollectionElement(p.getID()) == null) {
					chos.add(p);
				}
			}
		} else {
			Participants ps = (Participants) (((WorkflowProcess) myOwner)
					.getPackage()).get("Participants");
			chos.addAll(refCollectionElements);
			Collection eps = ps.getChoosable();
			Iterator i = eps.iterator();
			while (i.hasNext()) {
				Participant p = (Participant) i.next();
				if (getCollectionElement(p.getID()) == null) {
					chos.add(p);
				}
			}
		}

		// if this is collection within the package that is not external,
		// remove read-only attribute for all participants that belong
		// to the collection, and set to others
		if (!getPackage().isReadOnly()) {
			Iterator i = chos.iterator();
			while (i.hasNext()) {
				Participant p = (Participant) i.next();
				p.setReadOnly(!isMine(p));
			}
		}
		return chos;
	}

	public boolean isMine(Participant p) {
		return refCollectionElements.contains(p);
	}

	public boolean isExternal(Participant p) {
		return (p.getPackage() != getPackage());
	}

	public Package getPackage() {
		Package pack = (myOwner instanceof WorkflowProcess) ? ((WorkflowProcess) myOwner)
				.getPackage()
				: ((Package) myOwner);
		return pack;
	}

	/**
	 *
	 */
	protected void removeFromExternal(Package ep) {
		Participants externalPs = (Participants) ep.get("Participants");
		Iterator it = externalPs.toCollection().iterator();
		while (it.hasNext()) {
			Participant p = (Participant) it.next();
			// removes only participants that directly belong to specified
			// package
			externalParticipants.remove(p);
		}
	}

	/**
	 *
	 */
	protected void insertFromExternal(Package ep) {
		Participants externalPs = (Participants) ep.get("Participants");
		// insert only the ps that owns package 'ep'
		// and are not referenced from some ext. pack. of 'ep'
		Iterator it = externalPs.refCollectionElements.iterator();
		while (it.hasNext()) {
			Participant p = (Participant) it.next();
			if (!externalParticipants.contains(p)) {
				externalParticipants.add(p);
			}
		}
	}

	public Participant getParticipant(String ID) {
		Participant toReturn = (Participant) super.getCollectionElement(ID);

		// if the participant haven't been found, and this is an instance of
		// participants at workflow process level, search the package level
		if ((toReturn == null) && (myOwner instanceof WorkflowProcess)) {
			toReturn = ((Participants) getPackage().get("Participants"))
					.getParticipant(ID);
		}

		// if the participant haven't been found, search external participants
		if (toReturn == null) {
			String extID;
			Iterator it = externalParticipants.iterator();
			while (it.hasNext()) {
				Participant ptmp = (Participant) it.next();
				extID = ptmp.getID();
				if (extID.equals(ID)) {
					toReturn = ptmp;
					break;
				}
			}
		}

		return toReturn;
	}

	public String getReadOnlyMessageName(XMLComplexElement el) {
		if (!refCollectionElements.contains(el)) {
			if (el == Participant.getFreeTextExpressionParticipant()) {
				return "WarningCannotDeleteFreeTextExpressionParticipant";
			} else if (myOwner instanceof Package) {
				return "WarningCannotDeleteExternalParticipant";
			} else {
				return "WarningParticipantDefinedAtPackageLevelCannotBeDeletedFromProcessLevel";
			}
		}
		return "";
	}

	public String getInUseMessageName(XMLComplexElement el) {
		return "WarningCannotDeleteParticipantThatIsInUse";
	}

	public int[] getInvisibleTableFieldOrdinals() {
		int[] itfo = new int[3];
		itfo[0] = 3;
		itfo[1] = 4;
		itfo[2] = 5;
		return itfo;
	}

	// 实现clone方法
	public Object clone() {
		Participants ps = (Participants) super.clone();
		return ps;
	}

}
