/* ExtendedAttribute.java
 *
 * Authors:
 * Stefanovic Nenad  chupo@iis.ns.ac.yu
 * Bojanic Sasa      sasaboy@neobee.net
 * Puskas Vladimir   vpuskas@eunet.yu
 * Pilipovic Goran   zboniek@uns.ac.yu
 *
 */

package com.ds.bpm.bpd.xml.elements;

import java.io.StringReader;
import java.util.Set;

import javax.swing.JTextField;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.xml.sax.InputSource;

import com.ds.bpm.bpd.xml.ParsingErrors;
import com.ds.bpm.bpd.xml.XML;
import com.ds.bpm.bpd.xml.XMLAttribute;
import com.ds.bpm.bpd.xml.XMLCollection;
import com.ds.bpm.bpd.xml.XMLCollectionElement;
import com.ds.bpm.bpd.xml.XMLComplexElement;
import com.ds.bpm.bpd.xml.XMLElement;
import com.ds.bpm.bpd.xml.XMLUtil;
import com.ds.bpm.bpd.xml.activity.Activity;
import com.ds.bpm.bpd.xml.elements.specialpanels.XMLComplexContentPanel;
import com.ds.bpm.bpd.xml.panels.XMLGroupPanel;
import com.ds.bpm.bpd.xml.panels.XMLPanel;
import com.ds.bpm.bpd.xml.panels.XMLTextPanel;
import com.ds.common.util.XMLUtility;

/**
 * Represents a WfMC DTD element that has the similar name. 扩展属性类
 * 
 * @see XML
 */
public class ExtendedAttribute extends XMLComplexElement {
	private XMLAttribute attrName = new XMLAttribute("Name"); // required

	private XMLAttribute attrValue = new XMLAttribute("Value");

	private XMLAttribute attrType = new XMLAttribute("Type");

	private XMLAttribute attrId = new XMLAttribute("Id");

	private XMLAttribute attrDesc = new XMLAttribute("Description");

	private XMLCollection refAny = new XMLCollection(null, "Any");

	private ExtendedAttributes myOwner = null;

	/**
	 * Creates a new instance of the class.
	 */
	public ExtendedAttribute(ExtendedAttributes owner) {
		super();
		attrValue.setRequired(true);
		this.myOwner = owner;
		fillStructure();
	}

	public ExtendedAttribute() {
		super();
		fillStructure();
	}

	/**
	 * Returns the package where ext. attrib. belongs to.
	 */
	public Package getPackage() {
		XMLElement own = myOwner.getOwner();
		if (own instanceof Package) {
			return (Package) own;
		} else if (own instanceof Participant) {
			return ((Participant) own).getPackage();
		} else if (own instanceof XMLCollectionElement) {
			XMLComplexElement own2 = ((XMLCollectionElement) own)
					.getCollection().getOwner();
			if (own2 instanceof Package) {
				return (Package) own2;
			} else if (own2 instanceof WorkflowProcess) {
				return ((WorkflowProcess) own2).getPackage();
			} else if (own2 instanceof Activity) {
				return ((Activity) own2).getOwnerProcess().getPackage();
			} else if (own2 instanceof ActivitySet) {
				return ((ActivitySet) own2).getOwnerProcess().getPackage();
			} else {
				return null;
			}
		} else if (own instanceof ExternalPackage) {
			return ((ExternalPackage) own).getMyPackage();
		} else {
			return null;
		}
	}

	/**
	 * Defines the super-class method. Read the explanation for this method
	 * within XMLComplexElement class.
	 */
	protected void fillStructure() {
		attrName.setRequired(true);
		complexStructure.add(attrName);
		attributes.add(attrName);
		complexStructure.add(attrValue);
		attributes.add(attrValue);
		complexStructure.add(attrType);
		attributes.add(attrType);
		complexStructure.add(attrId);
		attributes.add(attrId);
		complexStructure.add(attrDesc);
		attributes.add(attrDesc);
		complexStructure.add(refAny);
	}

	/**
	 * Overrides super-class method to realize this class specific writting to
	 * XML file.
	 */
	public void toXML(Node parent) {
		Node node = (parent.getOwnerDocument()).createElement(name);
		for (int i = 0; i < attributes.size(); i++) {
			XMLAttribute attr = (XMLAttribute) attributes.get(i);
			if (!attr.isEmpty() || attr.isRequired()) {
				attr.toXML(node);
			}
		}
		if (refAny.size() != 0) {
			refAny.toXML(node);
		}
		parent.appendChild(node);
	}

	/**
	 * Prepares the panel for showing extended attribute of some XML element.
	 * 
	 * @return XMLPanel to be shown.
	 */
	public XMLPanel getPanel() {
		return new XMLGroupPanel(this,
				new XMLElement[] { attrName, attrValue }, toLabel());
	}

	public String toString() {
		String toRet = XMLUtil.getLanguageDependentString("NameKey") + "="
				+ attrName.toString() + ", "
				+ XMLUtil.getLanguageDependentString("ValueKey") + "="
				+ attrValue.toString();
		if (toRet == null) {
			return "";
		} else {
			return toRet;
		}
	}

	/**
	 * Used to create exact copy of instance of this class. The newly created
	 * instance will have all the properties same as the copied one.
	 * 
	 * @return The newly created instance of this class.
	 */
	public Object clone() {
		ExtendedAttribute ea = (ExtendedAttribute) super.clone();
		ea.attrName = (XMLAttribute) this.attrName.clone();
		ea.attrValue = (XMLAttribute) this.attrValue.clone();
		ea.attrType = (XMLAttribute) this.attrType.clone();
		ea.attrId = (XMLAttribute) this.attrId.clone();
		ea.attrDesc = (XMLAttribute) this.attrDesc.clone();
		ea.fillStructure();
		return ea;
	}

	/**
	 * Parses the String entered within content box, and produces node from it.
	 */
	private Node parseContent(String toParse) {
		Document document = null;

		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		factory.setValidating(false);

		try {
			ParsingErrors pErrors = new ParsingErrors();

			DocumentBuilder parser = factory.newDocumentBuilder();
			parser.setErrorHandler(pErrors);
			// adding helper tag, so after parsing, all its children
			// will be taken into account
			toParse = "<ExtAttribsAddition>" + toParse
					+ "</ExtAttribsAddition>";
			document = parser.parse(new InputSource(new StringReader(toParse)));
			Set errorMessages = pErrors.getErrorMessages();
			if (errorMessages.size() > 0) {
				System.err.println("Errors in schema type");
			}
		} catch (Exception ex) {
			System.err
					.println("Fatal error while parsing ext. attributes complex content");
			return null;
		}
		if (document != null) {
			return document.getDocumentElement();
		} else {
			return null;
		}

	}

	/**
	 * This method checks if the name of the attribute is valid, and if complex
	 * part of extended attribute is valid. It is called only if user presses OK
	 * button within the editing dialog.
	 * 
	 * @param p
	 *            The panel for editing extended attribute.
	 * @return If complex content is parsed properly, returns <tt>true</tt>,
	 *         otherwise returns <tt>false</tt>.
	 */
	public boolean isValidEnter(XMLPanel p) {
		XMLGroupPanel gp = (XMLGroupPanel) p;
		XMLComplexContentPanel ccp = (XMLComplexContentPanel) gp.getPanel(2);
		try {
			XMLTextPanel tp = (XMLTextPanel) gp.getPanel(0);
			String nameToCheck = tp.getText();
			if (!XMLCollection.isIdValid(nameToCheck)) {
				String message = XMLUtil
						.getLanguageDependentString("ErrorValueIsNotValid");
				String dialogTitle = XMLUtil.getLanguageDependentString("Title");
				XMLPanel.errorMessage(p.getDialog(), dialogTitle, attrName
						.toLabel()
						+ " ", message);
				((JTextField) tp.getComponent(2)).requestFocus();
				return false;
			}
		} catch (Exception ex) {
		}
		
		//如果页面元素不对应则不进行效验
		if (ccp==null){
			return true;
		}

		String cmplxCnt = ccp.getText().trim();
		if (cmplxCnt.length() > 0) {
			Object ret = parseContent(cmplxCnt);
			if (ret == null) {
				XMLUtil
						.message(
								XMLUtil
										.getLanguageDependentString("ErrorComplexContentOfExtendedAttributeIsNotValid"),
								javax.swing.JOptionPane.ERROR_MESSAGE);
				return false;
			}
		}

		return true;
	}

	public void setAnyXMLCollection(XMLCollection collection) {
		refAny = collection;
	}

	public XMLCollection getAnyXMLCollection() {
		return refAny;
	}

	// 解析XPDL文件保存到扩展属性类
	public void fromXML(Node node) {
		super.fromXML(node);
		if (refAny instanceof RightGroups || refAny instanceof Listeners|| refAny instanceof Listeners) {
			refAny.fromXML(XMLUtility.getFirstChild(node, Node.ELEMENT_NODE));
		}
	}
}
