/* DataField.java
 *
 * Authors:
 * Stefanovic Nenad  chupo@iis.ns.ac.yu
 * Bojanic Sasa      sasaboy@neobee.net
 * Puskas Vladimir   vpuskas@eunet.yu
 * Pilipovic Goran   zboniek@uns.ac.yu
 *
 */

package com.ds.bpm.bpd.xml.elements;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import javax.swing.JTextField;

import com.ds.bpm.bpd.xml.XML;
import com.ds.bpm.bpd.xml.XMLAttribute;
import com.ds.bpm.bpd.xml.XMLCollection;
import com.ds.bpm.bpd.xml.XMLCollectionElement;
import com.ds.bpm.bpd.xml.XMLComplexChoice;
import com.ds.bpm.bpd.xml.XMLComplexElement;
import com.ds.bpm.bpd.xml.XMLElement;
import com.ds.bpm.bpd.xml.XMLUtil;
import com.ds.bpm.bpd.xml.panels.XMLGroupPanel;
import com.ds.bpm.bpd.xml.panels.XMLPanel;
import com.ds.bpm.bpd.xml.panels.XMLTextPanel;

import org.w3c.dom.Node;

/**
* Represents a WfMC DTD element that has the similar name.
*
* @see XML
*/
public class DataField extends XMLCollectionElement {
   private transient Package myPackage;

   private DataType refDataType;
   private InitialValue refInitialValue=new InitialValue(); // min=0
   private Length refLength=new Length(); // min=0
   private Description refDescription=new Description(); // min=0
   private ExtendedAttributes refExtendedAttributes=new ExtendedAttributes(this); // min=0

   private XMLAttribute attrName=new XMLAttribute("Name");
   // default="FALSE"
   private XMLAttribute attrIsArray=new XMLAttribute("IsArray",
      new String[] {
         "",
         "TRUE",
         "FALSE"
      },2);

   /** Enables canceling of changes to the extended attributes collection. */
   private ExtendedAttributes clonedEAs;

   /**
   * Creates a new instance of the class.
   *
   * @param dfs  The reference to collection of  data fields where
   *             this instance will be put into.
   * @param porw The class instance which member is this class instance.
   */
   public DataField (DataFields dfs,XMLComplexElement porw) {
      super(dfs);

      try {
         if (porw instanceof Package) {
            this.myPackage=(Package)porw;
            refDataType=new DataType((Package)porw); // must be defined
         }
         else {
            this.myPackage=((WorkflowProcess)porw).getPackage();
            refDataType=new DataType(((WorkflowProcess)porw).getPackage()); // must be defined
         }
      } catch (Exception ex) {}

      fillStructure();
   }

   /**
   * Defines the super-class method. Read the explanation for
   * this method within XMLComplexElement class.
   */
   protected void fillStructure () {
      super.fillStructure();
      // allow ID to be modified
      attrId.setValue("");
      //attrName.setRequired(true);
      complexStructure.add(attrName);
      attributes.add(attrName);
      complexStructure.add(attrIsArray);
      attributes.add(attrIsArray);
      refDataType.setRequired(true);
      complexStructure.add(refDataType);
      complexStructure.add(refInitialValue);
      complexStructure.add(refLength);
      complexStructure.add(refDescription);
      complexStructure.add(refExtendedAttributes);
   }

   public XMLPanel getPanel () {
      clonedEAs=(ExtendedAttributes)refExtendedAttributes.clone();
      XMLPanel p=new XMLGroupPanel(this,
             new XMLElement[] {
                attrId,
                attrName,
                attrIsArray,
                refDataType,
                refInitialValue,
                refLength,
                refDescription,
                clonedEAs
             },toLabel());

      return p;
   }

   /**
   * Overrides super-class method to return an attribute
   * specific XML input.
   */
   public void fromXML(Node node) {
      attrIsArray.setValue("");
      super.fromXML(node);
   }

   /**
   * Overrides the super class method to return the value
   * of it's "Name" attribute value.
   *
   * @return The value of this class "Name" attribute.
   */
   public Object toValue () {
      return attrName.toValue();
   }

   /**
   * Checks if an ID entered by the user is unique.
   */
   public boolean isIDUniqueAndValid (XMLPanel groupPanel) {
      XMLTextPanel tp=(XMLTextPanel)((XMLGroupPanel)groupPanel).getPanel(0);
      String IDToCheck=tp.getText();
      // if there is an element with given ID, return false
      XMLComplexElement df=((DataFields)getCollection()).getCollectionElement(IDToCheck);
      boolean isOK=true;
      String message=null;
      String dialogTitle=null;
      if (df!=null && df!=this) {
         message=XMLUtil.getLanguageDependentString("ErrorIDMustBeUnique");
         dialogTitle=XMLUtil.getLanguageDependentString("DialogIDIsNotUnique");
         isOK=false;
      } else if (!XMLCollection.isIdValid(IDToCheck)) {
         message=XMLUtil.getLanguageDependentString("ErrorIDMustBeValid");
         dialogTitle=XMLUtil.getLanguageDependentString("DialogIDIsNotValid");
         isOK=false;
      }
      if (!isOK) {
         XMLPanel.errorMessage(groupPanel.getDialog(),dialogTitle,"",message);
         ((JTextField)tp.getComponent(2)).requestFocus();
      }
      return isOK;
   }

   /**
   * Overrides super-class method to retreive the value
   * of this class "Id" attribute.
   * This is used when displaying instance of this class
   * within dialog.
   *
   * @return The "Id" attribute value of this class.
   */
   public String toString () {
      return attrId.toString();
   }

   /**
     * This method is called only if user doesn't press Cancel button
     * within the dialog for editing DataField properties, so
     * the changes to the real collection of DataField are applied here.
     * @param p The panel for editing parameters.
     * @return always returns <tt>true</tt>.
     */
    public boolean isValidEnter (XMLPanel p) {
       if (clonedEAs!=null) {
          complexStructure.remove(refExtendedAttributes);
          refExtendedAttributes=clonedEAs;
          complexStructure.add(7,refExtendedAttributes);
       }
       return true;
    }

   /**
   * Overrides parent method to display data type details for BasicType and
   * for DeclaredType.
   */
   public Collection toComplexTypeValues () {
      java.util.List l=new ArrayList();
      Iterator it=complexStructure.iterator();
      while (it.hasNext()) {
         XMLElement el=(XMLElement)it.next();
         if (el instanceof XMLAttribute) {
            l.add(el.toString());
         } else if (el instanceof DataType) {
            DataType dt=(DataType)el;
            DataTypes dtt=(DataTypes)dt.get("Type");
            Object type=dtt.toValue();
            if (type instanceof BasicType) {
               l.add(el.toValue()+" - "+((XMLAttribute)((BasicType)type).get("Type")).getChoosen());
            } else if (type instanceof DeclaredType) {
               l.add(el.toValue()+" - "+((XMLComplexChoice)((DeclaredType)type).get("SubType")).getChoosen());
            } else {
               l.add(el.toValue());
            }
         } else {
            l.add(el.toValue());
         }
      }
      return l;
   }

   public Package getPackage () {
      return myPackage;
   }

}
