/* ActualParameter.java
 *
 * Authors:
 * Stefanovic Nenad  chupo@iis.ns.ac.yu
 * Bojanic Sasa      sasaboy@neobee.net
 * Puskas Vladimir   vpuskas@eunet.yu
 * Pilipovic Goran   zboniek@uns.ac.yu
 *
 */

package com.ds.bpm.bpd.xml.elements;

import java.util.Iterator;

import org.w3c.dom.DOMException;
import org.w3c.dom.Node;

import com.ds.bpm.bpd.xml.XML;
import com.ds.bpm.bpd.xml.XMLCollectionElement;
import com.ds.bpm.bpd.xml.XMLSimpleElement;
import com.ds.bpm.bpd.xml.elements.formula.FormalParameter;
import com.ds.bpm.bpd.xml.panels.XMLComboButtonPanel;
import com.ds.bpm.bpd.xml.panels.XMLPanel;

/**
 * Represents a WfMC DTD element that has the similar name.
 * 
 * @see XML
 */
public class ActualParameter extends XMLSimpleElement {
	/**
	 * Reference to all defined DataFields, within workflow process that element
	 * which member is instance of this element belongs to, plus all defined
	 * DataFields within Package. These DataFields are used as ActualParameters.
	 */
	private transient DataFields definedDataFields = null;

	/**
	 * Creates a new instance of the class.
	 * 
	 * @param wp
	 *            workflow process that defines DataField objects which are used
	 *            as actual parameters of an application or of a process itself.
	 */
	public ActualParameter(WorkflowProcess wp) {
		super();
		try {
			this.definedDataFields = (DataFields) wp.get("DataFields");
		} catch (NullPointerException npe) {
			//
		}
	}

	/**
	 * Prepares the panel with a combo box to choose the one of defined
	 * DataFields that will be used as an actual parameter.
	 * 
	 * @return XMLPanel to be shown.
	 */
	public XMLPanel getPanel() {
		Object[] choices = null;
		if (definedDataFields != null) {
			choices = definedDataFields.getChoosable().toArray();
		} else {
			choices = null;
		}
		return new XMLComboButtonPanel(this, choices, definedDataFields, true);
	}

	/**
	 * Overrides super-class method to realize this class specific writting to
	 * XML file.
	 */
	public void toXML(Node parent) throws DOMException {
		// the DataField or FormalParameter object is held in
		// it's value member - if it isn't, there is no actual param.
		// or there is one manually defined as expression
		if (value == null)
			return;
		if ((value instanceof DataField) || (value instanceof FormalParameter)) {
			// prepearing DataField ID (or workflow process FormalParameter ID)
			Object realValue = value;
			// value=((DataField)value).getID();
			value = ((XMLCollectionElement) value).getID();
			super.toXML(parent);
			value = realValue;
		} else {
			super.toXML(parent);
		}

	}

	/**
	 * Used to create exact copy of instance of this class. The newly created
	 * instance will have all the properties same as the copied one.
	 * 
	 * @return The newly created instance of this class.
	 */
	public Object clone() {
		ActualParameter ap = (ActualParameter) super.clone();
		// set the same datafield - (it was cloned to)
		ap.setValue(this.toValue());
		ap.definedDataFields = this.definedDataFields;
		return ap;
	}

	/**
	 * Called by Tool or SubFlow object after importing of XML file to set the
	 * proper DataField objects for ActualParameter that correspodents to
	 * Application or WorkflowProcess formal parameter, depending on that who
	 * owns ActualParameter (Tool or Subflow). Data fields are set according to
	 * it's ID's read from XML.
	 */
	public void afterImporting() {
		if (value == null)
			return;
		Iterator it = definedDataFields.getChoosable().iterator();
		while (it.hasNext()) {
			XMLCollectionElement dfOrFp = (XMLCollectionElement) it.next();
			// DataField df=(DataField)it.next();
			if (dfOrFp.getID().equals(value.toString())) {
				value = dfOrFp;
				break;
			}
		}
	}

}
