/* XMLUtil.java
 *
 * Authors:
 * Stefanovic Nenad  chupo@iis.ns.ac.yu
 * Bojanic Sasa      sasaboy@neobee.net
 * Puskas Vladimir   vpuskas@eunet.yu
 * Pilipovic Goran   zboniek@uns.ac.yu
 *
 */

package com.ds.bpm.bpd.xml;

import com.ds.bpm.bpd.BPD;
import com.ds.bpm.bpd.MainLeftDownPanel;
import com.ds.bpm.bpd.ResourceManager;
import com.ds.bpm.bpd.TextPreview;
import com.ds.bpm.bpd.graph.End;
import com.ds.bpm.bpd.graph.Start;
import com.ds.bpm.bpd.misc.TextTreePanel;
import com.ds.bpm.bpd.xml.activity.*;
import com.ds.bpm.bpd.xml.activity.Event;
import com.ds.bpm.bpd.xml.elements.*;
import com.ds.bpm.bpd.xml.elements.Package;
import com.ds.bpm.bpd.xml.elements.formula.FormalParameter;
import com.ds.bpm.bpd.xml.elements.formula.FormalParameters;
import org.htmlparser.Attribute;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import javax.swing.*;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import java.awt.*;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.net.URL;
import java.util.*;

/**
 * Class which purpose is to provide static methods which are used by classes that represents program apstraction of XML
 * elements. These methods offers support for reading or writting an XML document and for generating the tooltips for
 * for the classes that needs it.
 */
public class XMLUtil {

    /**
     * Used for tooltips
     */
    private static final String EMPTY_STRING = "";
    /**
     * Used for tooltips
     */
    private static final String HTML_OPEN = "<html>";
    /**
     * Used for tooltips
     */
    private static final String HTML_CLOSE = "</html>";
    /**
     * Used for tooltips
     */
    private static final String STRONG_OPEN = "<strong>";
    /**
     * Used for tooltips
     */
    private static final String STRONG_CLOSE = "</strong>";
    /**
     * Used for tooltips
     */
    private static final String LINE_BREAK = "<br>";
    /**
     * Used for tooltips
     */
    private static final String COLON_SPACE = " ";

    private static Window myWindow;

    private static ResourceBundle defaultResources;
    private static ResourceBundle choosenResources;
    private static Properties properties;

    // filter for XML files
    private static XMLFilter allFilter = new XMLFilter(null);
    private static XMLFilter xmlFilter = new XMLFilter("xml");
    private static XMLFilter xpdlFilter = new XMLFilter("xpdl");
    private static XMLFilter jpgFilter = new XMLFilter("jpg");
    private static XMLFilter svgFilter = new XMLFilter("svg");
    private static XMLFilter txtFilter = new XMLFilter("txt");
    private static XMLFilter htmlFilter = new XMLFilter("html");
    private static XMLFilter jspFilter = new XMLFilter("jsp");
    private static XMLFilter htmFilter = new XMLFilter("htm");

    private static javax.swing.filechooser.FileFilter[] lastChoosenFilter = new javax.swing.filechooser.FileFilter[8];

    static {
        lastChoosenFilter[0] = xpdlFilter;
        lastChoosenFilter[1] = jpgFilter;
        lastChoosenFilter[2] = svgFilter;
        lastChoosenFilter[3] = txtFilter;
        lastChoosenFilter[4] = allFilter;
        lastChoosenFilter[5] = htmlFilter;
        lastChoosenFilter[6] = jspFilter;
        lastChoosenFilter[7] = htmFilter;
    }

    private static JFileChooser chooser;

    static {
        String userDir = System.getProperty("user.dir");
        chooser = new JFileChooser(userDir);
    }

    public static final int INFORMATION_MESSAGE = JOptionPane.INFORMATION_MESSAGE;
    public static final int ERROR_MESSAGE = JOptionPane.ERROR_MESSAGE;

    // ************************* XMLFILTER CLASS ***********************************

    /**
     * Used to filter XML files during opening or saving.
     */
    private static final class XMLFilter extends javax.swing.filechooser.FileFilter {
        private String myExtension = null;

        XMLFilter(String extension) {
            this.myExtension = extension;
        }

        public String getExtension() {
            return myExtension;
        }

        public boolean accept(File f) {
            if (f != null) {
                if (f.isDirectory()) {
                    return true;
                }
                if (myExtension != null) {
                    String extension = null;
                    String fName = f.getName();
                    int i = fName.lastIndexOf('.');
                    if (i > 0 && i < fName.length() - 1) {
                        extension = fName.substring(i + 1).toLowerCase();
                    }

                    if (extension != null && extension.equalsIgnoreCase(myExtension)) {
                        return true;
                    }
                } else {
                    return true;
                }
            }
            return false;
        }

        public String getDescription() {
            if (myExtension != null) {
                return XMLUtil.getLanguageDependentString(myExtension.toUpperCase() + "Description");
            } else {
                return XMLUtil.getLanguageDependentString("ALLDescription");
            }
        }
    }
    // ********************** END OF XMLFILTER CLASS ******************************

    public static void setApplicationWindow(Window w) {
        myWindow = w;
    }

    public static void setDefaultResources(ResourceBundle defaultR) {
        defaultResources = defaultR;
    }

    public static void setChoosenResources(ResourceBundle choosenR) {
        choosenResources = choosenR;
    }

    public static void setProperties(Properties props) {
        properties = props;
    }

    /* Show a dialog with the given error message. */
    public static void message(String message, int type) {
        JOptionPane.showMessageDialog(myWindow, message, XMLUtil.getLanguageDependentString("Title"), type);
    }

    /*
     * Open a dialog and return the filename. Returns null if canceled.
     *
     * @param parent The parent component of dialog.
     *
     * @param message The message to write in the title of dialog
     *
     * @param mode 0 - open dialog, 1 - save dialog
     *
     * @param filteringMode if 0 - displays .xml and .xpdl files, if 1 - displays .jpg files, 2 - displays SVG files, 3
     * - displays .xml and .xpdl files, otherwise displays all files
     *
     * @param initialName The initial name of the file to be saved or opened
     *
     * @return The filename of opened/saved file.
     */
    public static String dialog(Component parent, String message, int mode, int filteringMode, String initialName) {
        chooser.updateUI();
        chooser.setDialogTitle(message);
        int setFilterIndex = 0;
        switch (filteringMode) {
            case 0:
                chooser.setAcceptAllFileFilterUsed(false);
                chooser.setFileFilter(allFilter);
                chooser.setFileFilter(xmlFilter);
                chooser.setFileFilter(xpdlFilter);
                chooser.removeChoosableFileFilter(jpgFilter);
                chooser.removeChoosableFileFilter(svgFilter);
                chooser.removeChoosableFileFilter(txtFilter);

                setFilterIndex = 0;
                break;
            case 1:
                chooser.setAcceptAllFileFilterUsed(false);
                chooser.setFileFilter(allFilter);
                chooser.setFileFilter(jpgFilter);
                chooser.removeChoosableFileFilter(xmlFilter);
                chooser.removeChoosableFileFilter(xpdlFilter);
                chooser.removeChoosableFileFilter(svgFilter);
                chooser.removeChoosableFileFilter(txtFilter);
                chooser.removeChoosableFileFilter(jspFilter);
                chooser.removeChoosableFileFilter(htmFilter);
                chooser.removeChoosableFileFilter(htmlFilter);
                setFilterIndex = 1;
                break;
            case 2:
                chooser.setAcceptAllFileFilterUsed(false);
                chooser.setFileFilter(allFilter);
                chooser.setFileFilter(svgFilter);
                chooser.removeChoosableFileFilter(xmlFilter);
                chooser.removeChoosableFileFilter(xpdlFilter);
                chooser.removeChoosableFileFilter(jpgFilter);
                chooser.removeChoosableFileFilter(txtFilter);
                setFilterIndex = 2;
                break;
            case 3:
                chooser.setAcceptAllFileFilterUsed(false);
                chooser.setFileFilter(allFilter);
                chooser.setFileFilter(txtFilter);
                chooser.removeChoosableFileFilter(xmlFilter);
                chooser.removeChoosableFileFilter(xpdlFilter);
                chooser.removeChoosableFileFilter(jpgFilter);
                chooser.removeChoosableFileFilter(svgFilter);
                setFilterIndex = 3;
                break;
            case 4:
                chooser.setAcceptAllFileFilterUsed(false);
                chooser.setFileFilter(allFilter);
                chooser.setFileFilter(jspFilter);
                chooser.setFileFilter(htmFilter);
                chooser.setFileFilter(htmlFilter);
                chooser.removeChoosableFileFilter(txtFilter);
                chooser.removeChoosableFileFilter(xmlFilter);
                chooser.removeChoosableFileFilter(xpdlFilter);
                chooser.removeChoosableFileFilter(jpgFilter);
                chooser.removeChoosableFileFilter(svgFilter);
                setFilterIndex = 5;
                break;

            default:
                chooser.setAcceptAllFileFilterUsed(false);
                chooser.setFileFilter(allFilter);
                chooser.removeChoosableFileFilter(xmlFilter);
                chooser.removeChoosableFileFilter(xpdlFilter);
                chooser.removeChoosableFileFilter(jpgFilter);
                chooser.removeChoosableFileFilter(svgFilter);
                chooser.removeChoosableFileFilter(txtFilter);
                setFilterIndex = 5;
                break;
        }
        try {
            chooser.setFileFilter(lastChoosenFilter[setFilterIndex]);
        } catch (Exception ex) {
        }

        if (initialName != null && initialName.length() > 0) {
            chooser.setSelectedFile(new File(initialName));
        }

        int returnVal = -1;
        String fileName = null;

        while (true) {
            if (mode == 0) {
                returnVal = chooser.showOpenDialog(parent);
            } else {
                returnVal = chooser.showSaveDialog(parent);
            }

            if (returnVal == JFileChooser.APPROVE_OPTION) {
                File f = chooser.getSelectedFile();
                fileName = f.getAbsolutePath();
                // SAVING:
                // - if extension isn't specified, tries to save the file with default extension
                // - it will not save file with extension if the file with extension already exists
                String extension = ((XMLFilter) chooser.getFileFilter()).getExtension();
                int dotIndex = f.getName().lastIndexOf(".");

                String oldFilename = fileName;
                if (mode == 1) {
                    if ((filteringMode >= 0 && filteringMode <= 2) && extension != null && dotIndex == -1) {
                        fileName += "." + extension;
                    }
                    // check if user have choosed an existing filename
                    if (new File(fileName).exists()) {
                        int r = JOptionPane.showConfirmDialog(myWindow, XMLUtil.getLanguageDependentString("WarningFileAlreadyExistsDoYouWantToReplaceIt"), XMLUtil.getLanguageDependentString("Title"), JOptionPane.YES_NO_OPTION);
                        if (r == JOptionPane.NO_OPTION) {
                            fileName = oldFilename;
                            continue;
                        }
                    }

                    // OPENING:
                    // - if extension isn't specified, and given file doesn't exist, tries
                    // to open the .xml file with the same name
                } else {
                    if (mode == 0 && !f.exists()) {
                        if ((filteringMode >= 0 && filteringMode <= 2) && dotIndex == -1) {
                            fileName += "." + extension;
                            if (!new File(fileName).exists()) {
                                fileName = null;
                            }
                        } else {
                            fileName = null;
                        }
                    }
                    if (fileName == null) {
                        XMLUtil.message(XMLUtil.getLanguageDependentString("WarningFileDoesNotExistPleaseSelectExistingFileToOpen"), JOptionPane.WARNING_MESSAGE);
                        fileName = oldFilename;
                        continue;
                    }
                }
            }
            break;
        }

        try {
            lastChoosenFilter[setFilterIndex] = chooser.getFileFilter();
        } catch (Exception ex) {
        }

        return fileName;
    }

    /**
     * Gets a resource string from the resource bundle.
     * <p>
     * Resource bundle represents the <i>property file</i>. For example, if property file contains something like this:
     * <BR>
     * <CENTER>menubar=file edit help</CENTER> method call getLanguageDependentString("menubar") will give the string
     * <i>file edit help</i> as a result. <BR>
     * This method reads information from property file. If can't find desired resource, returns <b>null</b>.
     *
     * @param nm name of the resource to fetch.
     * @return String value of named resource.
     */
    public static String getLanguageDependentString(String nm) {
        String str;
        try {
            str = choosenResources.getString(nm);
        } catch (MissingResourceException mre) {
            try {
                str = defaultResources.getString(nm);
            } catch (MissingResourceException mre1) {
                str = null;
            } catch (NullPointerException npe) {
                str = null;
            }
        } catch (NullPointerException npe) {
            ResourceBundle orig = ResourceBundle.getBundle("com.ds.bpm.bpd.resources.BPD", new Locale(""));
            try {
                str = orig.getString(nm);
            } catch (Exception ex) {
                str = null;
            }
        }
        if (str != null) {
            str = ResourceManager.getEncodeString(str, true);
        }
        return str;
    }

    /**
     * Gets the url from a resource string.
     *
     * @param key the string key to the url in the properties.
     * @return the resource location.
     * @see Class#getResource
     */
    public static URL getResource(String key) {
        try {
            String name = properties.getProperty(key);
            if (name != null) {
                URL url = XMLUtil.class.getClassLoader().getResource(name);
                return url;
            }
        } catch (Exception ex) {
        }
        return null;
    }

    /**
     * Take the given string and chop it up into a series of strings on given boundries. This is useful for trying to
     * get an array of strings out of the resource file.
     */
    public static String[] tokenize(String input, String boundary) {
        if (input == null)
            input = "";
        Vector v = new Vector();
        StringTokenizer t = new StringTokenizer(input, boundary);
        String cmd[];

        while (t.hasMoreTokens())
            v.addElement(t.nextToken());
        cmd = new String[v.size()];
        for (int i = 0; i < cmd.length; i++)
            cmd[i] = (String) v.elementAt(i);

        return cmd;
    }

    /**
     * Determines the number of string toFind within string toSearch.
     */
    public static int howManyStringsWithinString(String toSearch, String toFind) {
        try {
            int startAt = 0;
            int howMany = 0;

            int fnd;
            while ((fnd = toSearch.indexOf(toFind, startAt)) != -1) {
                howMany++;
                startAt = (fnd + toFind.length());
            }
            return howMany;
        } catch (Exception ex) {
            return -1;
        }
    }

    /**
     * Neat little thing. Makes HTML formated string for tooltip (made of property names and coresponding values).
     */
    public static String makeTooltip(XMLElement[] elements) {
        if (elements == null)
            return "";
        String s = HTML_OPEN;
        for (int i = 0; i < elements.length; i++) {
            s += makeAnotherHtmlLine(elements[i]);
        }
        s = s.substring(0, s.length() - LINE_BREAK.length());
        s += HTML_CLOSE;
        return s;
    }

    /**
     * Helps when generating tooltip for some element.
     */
    private static String makeAnotherHtmlLine(XMLElement el) {
        int MAX_LENGTH = 100;
        int MAX_LINES_PER_TEXT = 15;
        String textToAppend = "";
        textToAppend += STRONG_OPEN;
        textToAppend += el.toLabel() + COLON_SPACE;
        textToAppend += STRONG_CLOSE;
        String val = el.toString();
        val = val.replaceAll("<", "&lt;");
        val = val.replaceAll(">", "&gt;");
        int vl = val.length();
        if (vl > MAX_LENGTH) {
            String newVal = "";
            int hm = vl / MAX_LENGTH;
            for (int i = 0; i <= hm; i++) {
                int startI = i * MAX_LENGTH;
                int endI = (i + 1) * MAX_LENGTH;
                if (endI > vl) {
                    endI = vl;
                }
                newVal = newVal + val.substring(startI, endI);
                if (i == MAX_LINES_PER_TEXT) {
                    newVal = newVal + " ...";
                    break;
                }
                if (i < hm) {
                    newVal += LINE_BREAK;
                    newVal += XMLUtil.makeEmptyHTMLText((el.toLabel() + COLON_SPACE).length());
                }
            }
            val = newVal;
        }
        textToAppend += val;
        textToAppend += LINE_BREAK;

        return textToAppend;
    }

    /**
     * Neat little thing. Makes HTML formated string for tooltip (made of property names and coresponding values).
     */
    public static String makeTooltip(Vector tagArr) {
        if (tagArr == null)
            return "";
        String s = HTML_OPEN;

        for (int j = 0; tagArr.size() > j; j++) {
            Attribute tagArrj = (Attribute) tagArr.elementAt(j);
            String key = tagArrj.getName();
            if (key == null || tagArrj.getValue() == null) {
                continue;
            }
            if (key.equals("$<TAGNAME>$")) {
                continue;
            }
            s += makeTagAnotherHtmlLine(key, (String) tagArrj.getValue());
        }
        s = s.substring(0, s.length() - LINE_BREAK.length());
        s += HTML_CLOSE;
        return s;
    }

    /**
     * Helps when generating tooltip for some element.
     */
    public static String makeTagAnotherHtmlLine(String key, String value) {
        int MAX_LENGTH = 100;
        int MAX_LINES_PER_TEXT = 15;
        String textToAppend = "";
        textToAppend += STRONG_OPEN;
        textToAppend += key + COLON_SPACE;
        textToAppend += STRONG_CLOSE;
        String val = value;
        val = val.replaceAll("<", "&lt;");
        val = val.replaceAll(">", "&gt;");
        int vl = val.length();
        if (vl > MAX_LENGTH) {
            String newVal = "";
            int hm = vl / MAX_LENGTH;
            for (int i = 0; i <= hm; i++) {
                int startI = i * MAX_LENGTH;
                int endI = (i + 1) * MAX_LENGTH;
                if (endI > vl) {
                    endI = vl;
                }
                newVal = newVal + val.substring(startI, endI);
                if (i == MAX_LINES_PER_TEXT) {
                    newVal = newVal + " ...";
                    break;
                }
                if (i < hm) {
                    newVal += LINE_BREAK;
                    newVal += XMLUtil.makeEmptyHTMLText((key + COLON_SPACE).length());
                }
            }
            val = newVal;
        }
        textToAppend += val;
        textToAppend += LINE_BREAK;

        return textToAppend;
    }

    public static String getCanonicalPath(String path, boolean canBeDirectory) {
        File f = new File(path);
        if (!f.isAbsolute()) {
            f = new File(System.getProperty("user.dir") + File.separator + path);
        }
        if (!f.exists() || (f.isDirectory() && !canBeDirectory)) {
            System.err.println("The file " + f.getAbsolutePath() + " does not exist");
            return null;
        } else {
            return getCanonicalPath(f);
        }
    }

    private static String getCanonicalPath(File f) {
        try {
            return f.getCanonicalPath();
        } catch (Exception ex) {
            return f.getAbsolutePath();
        }
    }

    /**
     * Used to parse all elements from the given collection according to the given tagContent. The method is iterating
     * through elements and searching it's name within tagContent. If the name is found, the method calls fromXML method
     * of element. NOTE: The tagContent has to be already parsed to contain only element's part of XML tag without
     * attributes.
     */
    public static void parseElements(Node node, Collection complexStructure) {
        String nameSpacePrefix = node.getPrefix();
        if (nameSpacePrefix != null) {
            nameSpacePrefix += ":";
        } else {
            nameSpacePrefix = "";
        }
        // getting elements
        Iterator it = complexStructure.iterator();
        while (it.hasNext()) {
            XMLElement el = (XMLElement) it.next();
            if (!(el instanceof XMLAttribute) && !(el instanceof XMLComplexChoice) && node != null) {
                // System.out.println("Elem="+el.name);
                // external parser
                Node child = getChildByName(node, nameSpacePrefix + el.name);
                if (child != null) {
                    el.fromXML(child);
                }
            } else {
                if (el instanceof XMLComplexChoice) {
                    Object[] ch = ((XMLComplexChoice) el).getChoices();
                    if (ch != null && node != null) {
                        for (int i = 0; i < ch.length; i++) {
                            XMLElement chc = (XMLElement) ch[i];
                            // external parser
                            Node child = getChildByName(node, nameSpacePrefix + chc.name);
                            if (child != null) {
                                ((XMLComplexChoice) el).fromXML(chc.name, child);
                            }
                        }
                    }
                }
            }
        }
    }

    public static Node getChildByName(Node parent, String childName) {
        NodeList children = parent.getChildNodes();
        for (int i = 0; i < children.getLength(); ++i) {
            Node child = (Node) children.item(i);
            if (child.getNodeName().equals(childName)) {
                return child;
            }
        }
        return null;
    }

    public static String getID(Node node) {
        try {
            NamedNodeMap nnm = node.getAttributes();
            Node attrib = nnm.getNamedItem("Id");
            Object ID;
            if (attrib.hasChildNodes()) {
                ID = attrib.getChildNodes().item(0).getNodeValue();
            } else {
                ID = attrib.getNodeValue();
            }
            return ID.toString();
        } catch (Exception ex) {
            return "";
        }
    }

    public static String getContent(Node node, boolean omitXMLDeclaration) {
        try {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();

            // Use a Transformer for output
            TransformerFactory tFactory = TransformerFactory.newInstance();
            Transformer transformer = tFactory.newTransformer();
            transformer.setOutputProperty("indent", "yes");
            transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "4");
            transformer.setOutputProperty("encoding", "UTF-8");
            if (omitXMLDeclaration) {
                transformer.setOutputProperty("omit-xml-declaration", "yes");
            }

            DOMSource source = new DOMSource(node);
            StreamResult result = new StreamResult(baos);
            transformer.transform(source, result);

            String cont = baos.toString("UTF8");

            baos.close();
            return cont;
        } catch (Exception ex) {
            return "";
        }
    }

    public static String getChildNodesContent(Node node) {
        String txt = "";
        if (node != null) {
            if (node.hasChildNodes()) {
                txt = XMLUtil.getContent(node, true);
                try {
                    Node fc = node.getFirstChild();
                    String fcnc = XMLUtil.getContent(fc, true);
                    String closedTag = "</" + node.getNodeName() + ">";
                    if (fcnc.trim().length() > 0) {
                        fcnc = fcnc.trim();
                    }

                    int i1, i2;
                    i1 = txt.indexOf(fcnc);
                    i2 = txt.lastIndexOf(closedTag);
                    txt = txt.substring(i1, i2).trim();
                } catch (Exception ex) {
                    NodeList nl = node.getChildNodes();
                    txt = "";
                    try {
                        for (int i = 0; i < nl.getLength(); i++) {
                            Node sn = nl.item(i);
                            if (sn instanceof Element) {
                                txt += XMLUtil.getContent(sn, true);
                            } else {
                                String nv = sn.getNodeValue();
                                // trim only the begining of the string
                                if (i > 0) {
                                    txt += nv.substring(1);
                                } else if (i == 0 && nv.trim().length() == 0) {
                                    continue;
                                } else {
                                    txt += nv;
                                }
                            }
                        }
                    } catch (Exception ex2) {
                    }
                }
            }
        }
        return txt;
    }

    /**
     * Returns if given activity has AND type split or join.
     *
     * @param act  The activity that is checked if it has AND type split or join, depending on the second parameter.
     * @param sOrJ if 0, activity is checked for AND type split, otherwise it is checked for AND type join
     * @return true if activity has AND type of split or join
     */
    public static boolean isANDTypeSplitOrJoin(Activity act, int sOrJ) {

        String splitOrJoin = "Split";
        String sjType = "XOR";
        if (sOrJ != 0) {
            splitOrJoin = "Join";
            sjType = act.getJoin().getType();
            if (sjType.equals("AND")) {
                return true;
            } else {
                return false;
            }
        } else {
            sjType = act.getSplit().getType();
            if (sjType.equals("AND")) {
                return true;
            } else {
                return false;
            }
        }
        /*
         * // default type is XOR // loop through the transitions restrictions specified for the // activity to find the
         * type of the activity split/join
         *
         * if ((TransitionRestrictions)act.get("TransitionRestrictions")==null){ return true; } Collection
         * transitionRestrictions = ((TransitionRestrictions)act.get("TransitionRestrictions")).toCollection(); Iterator
         * iter = transitionRestrictions.iterator(); while(iter.hasNext()){ TransitionRestriction
         * restriction=(TransitionRestriction)iter.next(); XMLComplexElement
         * sj=(XMLComplexElement)restriction.get(splitOrJoin); if(sj==null){ // continue until we find it continue; }
         * sjType = sj.get("Type").toValue().toString(); }
         *
         * if (sjType.equals("AND")) { return true; } else { return false; }
         */
        // return false;
    }

    /**
     * Returns the set of (XML) activities that have split or join.
     *
     * @param acts The activities graph objects that are checked if their XML object have split or join, depending on the
     *             second parameter.
     * @param sOrJ if 0, activity is checked for split, otherwise it is checked for join
     */
    public static Set getSplitOrJoinActivities(Collection acts, int sOrJ) {
        Set sOrJactivities = new HashSet();
        if (acts == null)
            return sOrJactivities;
        Transitions ts = null;
        Iterator it = acts.iterator();
        while (it.hasNext()) {
            Activity act = (Activity) it.next();
            if (ts == null) {
                ts = (Transitions) act.getCollection().getOwner().get("Transitions");
            }
            Iterator iter;
            if (sOrJ == 0) {
                iter = ts.getTransitions(act.getID(), -1).iterator();
            } else {
                iter = ts.getTransitions(act.getID(), 1).iterator();
            }
            int noOfTrans = 0;
            while (iter.hasNext()) {
                Transition t = (Transition) iter.next();
                if (t != null && t.getFrom() != null && t.getTo() != null) {
                    noOfTrans++;
                }
            }
            if (noOfTrans > 1) {
                sOrJactivities.add(act);
            }
        }

        return sOrJactivities;
    }

    /**
     * Returns the set of BlockActivity objects contained within given process or block activity. If the BlockActivity
     * objects contains other BlockActivity objects, and the second parameter is set to true, these are also returned,
     * and so on - which means that implementation is recursive.
     */
    public static Set getBlockActivities(XMLComplexElement wpOrAs, boolean recursivly) {
        Collection allActs = ((Activities) wpOrAs.get("Activities")).toCollection();
        Set bas = new HashSet();
        Iterator it = allActs.iterator();
        Activity act;
        while (it.hasNext()) {
            act = (Activity) it.next();
            BlockActivity ba = act.getBlockActivity();
            if (ba != null) {
                bas.add(act);
                if (!recursivly)
                    continue;
                ActivitySets ass = (ActivitySets) act.getOwnerProcess().get("ActivitySets");
                String asId = ba.get("Id").toString();
                ActivitySet as = ass.getActivitySet(asId);
                if (as != null) {
                    bas.addAll(getBlockActivities(as, true));
                }
            }
        }
        return bas;
    }

    /**
     * Returns predefined conformanceClass number.
     *
     * @param conformanceClass The conformance class we are looking for number
     * @return 0 if conformance class is NON_BLOCKED, 1 if conformance class is LOOP_BLOCKED, 2 if conformance class is
     * FULL_BLOCKED, and -1 otherwise
     */
    public static int getConformanceClassNo(String conformanceClass) {
        if (conformanceClass.equals("NON_BLOCKED")) {
            return 0;
        } else if (conformanceClass.equals("LOOP_BLOCKED")) {
            return 1;
        } else if (conformanceClass.equals("FULL_BLOCKED")) {
            return 2;
        } else {
            return -1;
        }
    }

    /**
     * Converts a file specified by the path, to the String.
     */
    public static String fileToString(String fileName) {
        if (fileName != null) {
            // String sLine;
            byte[] utf8Bytes;
            String sFile = new String();
            // Reading input by lines:
            try {
                FileInputStream fis = new FileInputStream(fileName);
                int noOfBytes = fis.available();
                if (noOfBytes > 0) {
                    utf8Bytes = new byte[noOfBytes];
                    fis.read(utf8Bytes);
                    sFile = new String(utf8Bytes, "UTF8");
                }
                fis.close();
            } catch (Exception ex) {
                return null;
            }
            return sFile;
        }
        return null;
    }
    // ******** END OF CREATING SCROLLPANE AND EDITOR COMPONENT(PEJGRAPH) **********

    /**
     * Gets the current date and time string in ISO-8601 format.
     */
    public static String getCurrentDateAndTime() {
        String dateSeparator = "-";
        String timeSeparator = ":";
        Calendar cal = new GregorianCalendar();
        String dateTime = "";
        dateTime = dateTime + String.valueOf(cal.get(Calendar.YEAR)) + dateSeparator;
        int mnth = cal.get(Calendar.MONTH) + 1;
        if (mnth < 10) {
            dateTime = dateTime + "0";
        }
        dateTime = dateTime + String.valueOf(mnth) + dateSeparator;
        int dayOfMnth = cal.get(Calendar.DAY_OF_MONTH);
        if (dayOfMnth < 10) {
            dateTime = dateTime + "0";
        }
        dateTime = dateTime + String.valueOf(dayOfMnth) + " ";
        int hr = cal.get(Calendar.HOUR_OF_DAY);
        int ampm = cal.get(Calendar.AM_PM);
        if (ampm == Calendar.PM && hr < 12) {
            hr += 12;
        }
        if (hr < 10) {
            dateTime = dateTime + "0";
        }
        dateTime = dateTime + String.valueOf(hr) + timeSeparator;
        int min = cal.get(Calendar.MINUTE);
        if (min < 10) {
            dateTime = dateTime + "0";
        }
        dateTime = dateTime + String.valueOf(min) + timeSeparator;
        int sec = cal.get(Calendar.SECOND);
        if (sec < 10) {
            dateTime = dateTime + "0";
        }
        dateTime = dateTime + String.valueOf(sec);

        return dateTime;
    }

    /**
     * Checks if formal and actual parameters are matched by number and by type.
     *
     * @return 0 - if parameters are matched, 1 - if numbere of formal and actual parameters is not the same, 2 - if
     * types of parameters are not matched
     */
    public static int checkParameterMatching(FormalParameters fps, ActualParameters aps) {
        if (fps == null || aps == null || fps.size() != aps.size()) {
            return 1;
        }

        for (int i = 0; i < fps.size(); i++) {
            FormalParameter fp = (FormalParameter) fps.get(i);
            ActualParameter ap = (ActualParameter) aps.get(i);

            String fpMode = fp.get("Mode").toValue().toString();

            // if the formal parameter mode is IN, do not check validity because
            // that could be expression written in any scripting language
            if (fpMode.equals("IN"))
                continue;

            // find the type of formal param.
            DataType fpdt = (DataType) fp.get("DataType");
            XMLComplexChoice fpdtt = (XMLComplexChoice) fpdt.get("Type");
            XMLElement fpType = (XMLElement) fpdtt.getChoosen();

            // find the type of actual param.
            Object apWRD = ap.toValue();
            XMLElement apType = null;
            if ((apWRD instanceof DataField) || (apWRD instanceof FormalParameter)) {
                DataType apdt = (DataType) ((XMLCollectionElement) apWRD).get("DataType");
                XMLComplexChoice apdtt = (XMLComplexChoice) apdt.get("Type");
                apType = (XMLElement) apdtt.getChoosen();
            }

            // if the actual parameter is an expression, and the mode is not
            // IN, return 2, which signals that parameter types don't match
            if (apType == null) {
                return 2;
            }

            if (fpType.getClass().equals(apType.getClass())) {
                // if this is BasicType check for subtype matching
                if (fpType instanceof BasicType) {
                    XMLAttribute fpAT = (XMLAttribute) ((BasicType) fpType).get("Type");
                    XMLAttribute apAT = (XMLAttribute) ((BasicType) apType).get("Type");
                    if (!fpAT.toValue().toString().equals(apAT.toValue().toString())) {
                        return 2;
                    }
                }
                // if this is EnumerationType check for Enumeration values matching
                if (fpType instanceof EnumerationType) {
                    // first check the size of enums
                    if (((EnumerationType) fpType).size() != ((EnumerationType) apType).size()) {
                        return 2;
                    }
                    // check the enum elements values
                    for (int j = 0; j < ((EnumerationType) fpType).size(); j++) {
                        if (!((EnumerationType) fpType).get(j).toString().equals(((EnumerationType) apType).get(j).toString())) {
                            return 2;
                        }
                    }
                }
                // if this is DeclaredType check if their IDs are the same
                if (fpType instanceof DeclaredType) {
                    if (!((DeclaredType) fpType).get("Id").toString().equals(((DeclaredType) apType).get("Id").toString())) {
                        return 2;
                    }
                }

            } else {
                return 2;
            }
        }
        return 0;
    }

    public static String replaceBackslashesWithSlashes(String repBS) {
        if (repBS != null) {
            int ind = -1;
            while ((ind = repBS.indexOf("\\")) != -1) {
                repBS = repBS.substring(0, ind) + "/" + repBS.substring(ind + 1);
            }
        }
        return repBS;
    }

    /**
     * If expand is true, expands all nodes in the tree. Otherwise, collapses all nodes in the tree.
     *
     * @param tree   jtree to expand or collapse
     * @param expand if true expand, otherwise collapses
     */
    public static void expandAll(JTree tree, boolean expand) {
        TreeNode root = (TreeNode) tree.getModel().getRoot();

        // Traverse tree from root
        expandAll(tree, new TreePath(root), expand);
    }

    /**
     * Collapse or expand specified three path.
     *
     * @param tree   jtree to expand or collapse
     * @param parent path to collapse or expand
     * @param expand if true expand, otherwise collapses
     */
    public static void expandAll(JTree tree, TreePath parent, boolean expand) {
        try {
            // Traverse children
            TreeNode node = (TreeNode) parent.getLastPathComponent();
            if (node.getChildCount() >= 0) {
                for (Enumeration e = node.children(); e.hasMoreElements(); ) {
                    TreeNode n = (TreeNode) e.nextElement();
                    TreePath path = parent.pathByAddingChild(n);
                    expandAll(tree, path, expand);
                }
            }
            // Expansion or collapse must be done bottom-up
            if (expand) {
                tree.expandPath(parent);
            } else {
                tree.collapsePath(parent);
            }
        } catch (Exception e) {
            // do nothing, because if somebody click somewhere where node do not exist
        }
    }

    /**
     * Collapse or expand specified three path, depends of state of flag isCollapsed in user object.
     *
     * @param tree   jtree to expand or collapse
     * @param parent path to collapse or expand
     */
    public static void initExpand(JTree tree, TreePath parent) {
        try {
            // Traverse children
            TreeNode node = (TreeNode) parent.getLastPathComponent();
            ArrayList expandedChilds = new ArrayList();
            if (node.getChildCount() >= 0) {
                for (Enumeration e = node.children(); e.hasMoreElements(); ) {
                    TreeNode n = (TreeNode) e.nextElement();
                    if (!((XMLElement) ((DefaultMutableTreeNode) n).getUserObject()).isCollapsed())
                        expandedChilds.add(n);
                }
                for (int i = 0; i < expandedChilds.size(); i++) {
                    TreePath path = parent.pathByAddingChild((TreeNode) expandedChilds.get(i));
                    initExpand(tree, path);
                }

            }
            tree.expandPath(parent);
        } catch (Exception e) {
            // do nothing
        }
    }

    public static Set getStartingActivities(XMLCollectionElement procOrASDef) {
        Activities acts = ((Activities) procOrASDef.get("Activities"));
        Set starts = new HashSet();
        Iterator it = acts.toCollection().iterator();
        Transitions ts = (Transitions) procOrASDef.get("Transitions");
        while (it.hasNext()) {
            Activity act = (Activity) it.next();
            Set trs = act.getIncomingTransitions();
            // the activity is starting one if it has no input transitions ...
            if (trs.size() == 0) {
                starts.add(act);
                // or there is a one input transition, but it is a selfreference
            } else if (trs.size() == 1) {
                Transition t = (Transition) trs.toArray()[0];
                if (t.getFrom() != null && t.getFrom().equals(t.getTo())) {
                    starts.add(act);
                }
            }
        }
        return starts;
    }

    public static Set getEndingActivities(XMLCollectionElement procOrASDef) {
        Activities acts = ((Activities) procOrASDef.get("Activities"));
        Set ends = new HashSet();
        Iterator it = acts.toCollection().iterator();
        Transitions ts = (Transitions) procOrASDef.get("Transitions");
        while (it.hasNext()) {
            Activity act = (Activity) it.next();
            Set trs = act.getNonExceptionalOutgoingTransitions();
            // the activity is ending one if it has no output transitions ...
            if (trs.size() == 0) {
                ends.add(act);
                // or there is a one output transition, but it is a selfreference
            } else if (trs.size() == 1) {
                Transition t = (Transition) trs.toArray()[0];
                if (t.getFrom() != null && t.getFrom().equals(t.getTo())) {
                    ends.add(act);
                }
            }
        }

        return ends;
    }

    public static Set getAllExtendedAttributeNames(XMLComplexElement cel) {
        Set extAttribNames = new HashSet();
        XMLInterface xmlInterface = null;

        if (cel instanceof Activity) {
            xmlInterface = ((Activity) cel).getOwnerProcess().getPackage().getXMLInterface();
        } else if (cel instanceof Application) {
            xmlInterface = ((Application) cel).getPackage().getXMLInterface();
        } else if (cel instanceof DataField) {
            xmlInterface = ((DataField) cel).getPackage().getXMLInterface();
        } else if (cel instanceof ExternalPackage) {
            xmlInterface = ((ExternalPackage) cel).getMyPackage().getXMLInterface();
        } else if (cel instanceof Package) {
            xmlInterface = ((Package) cel).getXMLInterface();
        } else if (cel instanceof Participant) {
            xmlInterface = ((Participant) cel).getPackage().getXMLInterface();
        } else if (cel instanceof Tool) {
            xmlInterface = ((Tool) cel).getPackage().getXMLInterface();
        } else if (cel instanceof Service) {
            xmlInterface = ((Service) cel).getPackage().getXMLInterface();
        } else if (cel instanceof Device) {
            xmlInterface = ((Device) cel).getPackage().getXMLInterface();
        } else if (cel instanceof Event) {
            xmlInterface = ((Event) cel).getPackage().getXMLInterface();
        } else if (cel instanceof Transition) {
            xmlInterface = ((Transition) cel).getPackage().getXMLInterface();
        } else if (cel instanceof TypeDeclaration) {
            xmlInterface = ((TypeDeclaration) cel).getPackage().getXMLInterface();
        } else if (cel instanceof WorkflowProcess) {
            xmlInterface = ((WorkflowProcess) cel).getPackage().getXMLInterface();
        }

        if (xmlInterface != null) {
            Iterator it = xmlInterface.getAllPackages().iterator();
            while (it.hasNext()) {
                extAttribNames.addAll(XMLUtil.getAllExtendedAttributeNames((Package) it.next(), cel));
            }
        }
        return extAttribNames;
    }

    private static Set getAllExtendedAttributeNames(Package pkg, XMLComplexElement cel) {
        Set extAttribNames = new HashSet();
        if (cel instanceof Activity) {
            Iterator it = ((WorkflowProcesses) pkg.get("WorkflowProcesses")).toCollection().iterator();
            while (it.hasNext()) {
                WorkflowProcess wp = (WorkflowProcess) it.next();
                Activities acts = (Activities) wp.get("Activities");
                Iterator acti = acts.toCollection().iterator();
                while (acti.hasNext()) {
                    Activity act = (Activity) acti.next();
                    Collection extAttribs = ((ExtendedAttributes) act.get("ExtendedAttributes")).toCollection();
                    extAttribNames.addAll(XMLUtil.getAllExtendedAttributeNames(extAttribs));
                }
                Iterator asets = ((ActivitySets) wp.get("ActivitySets")).toCollection().iterator();
                while (asets.hasNext()) {
                    ActivitySet as = (ActivitySet) asets.next();
                    acts = (Activities) as.get("Activities");
                    acti = acts.toCollection().iterator();
                    while (acti.hasNext()) {
                        Activity act = (Activity) acti.next();
                        Collection extAttribs = ((ExtendedAttributes) act.get("ExtendedAttributes")).toCollection();
                        extAttribNames.addAll(XMLUtil.getAllExtendedAttributeNames(extAttribs));
                    }
                }
            }
        } else if (cel instanceof Application) {
            Applications aps = (Applications) pkg.get("Applications");
            Iterator api = aps.toCollection().iterator();
            while (api.hasNext()) {
                Application ap = (Application) api.next();
                Collection extAttribs = ((ExtendedAttributes) ap.get("ExtendedAttributes")).toCollection();
                extAttribNames.addAll(XMLUtil.getAllExtendedAttributeNames(extAttribs));
            }
            Iterator it = ((WorkflowProcesses) pkg.get("WorkflowProcesses")).toCollection().iterator();
            while (it.hasNext()) {
                WorkflowProcess wp = (WorkflowProcess) it.next();
                aps = (Applications) wp.get("Applications");
                api = aps.toCollection().iterator();
                while (api.hasNext()) {
                    Application ap = (Application) api.next();
                    Collection extAttribs = ((ExtendedAttributes) ap.get("ExtendedAttributes")).toCollection();
                    extAttribNames.addAll(XMLUtil.getAllExtendedAttributeNames(extAttribs));
                }
            }
        } else if (cel instanceof DataField) {
            DataFields dfs = (DataFields) pkg.get("DataFields");
            Iterator dfi = dfs.toCollection().iterator();
            while (dfi.hasNext()) {
                DataField df = (DataField) dfi.next();
                Collection extAttribs = ((ExtendedAttributes) df.get("ExtendedAttributes")).toCollection();
                extAttribNames.addAll(XMLUtil.getAllExtendedAttributeNames(extAttribs));
            }
            Iterator it = ((WorkflowProcesses) pkg.get("WorkflowProcesses")).toCollection().iterator();
            while (it.hasNext()) {
                WorkflowProcess wp = (WorkflowProcess) it.next();
                dfs = (DataFields) wp.get("DataFields");
                dfi = dfs.toCollection().iterator();
                while (dfi.hasNext()) {
                    DataField df = (DataField) dfi.next();
                    Collection extAttribs = ((ExtendedAttributes) df.get("ExtendedAttributes")).toCollection();
                    extAttribNames.addAll(XMLUtil.getAllExtendedAttributeNames(extAttribs));
                }
            }
        } else if (cel instanceof ExternalPackage) {
            ExternalPackages eps = (ExternalPackages) pkg.get("ExternalPackages");
            Iterator it = eps.toCollection().iterator();
            while (it.hasNext()) {
                ExternalPackage ep = (ExternalPackage) it.next();
                Collection extAttribs = ((ExtendedAttributes) ep.get("ExtendedAttributes")).toCollection();
                extAttribNames.addAll(XMLUtil.getAllExtendedAttributeNames(extAttribs));
            }
        } else if (cel instanceof Package) {
            Collection extAttribs = ((ExtendedAttributes) pkg.get("ExtendedAttributes")).toCollection();
            extAttribNames.addAll(XMLUtil.getAllExtendedAttributeNames(extAttribs));
        } else if (cel instanceof Participant) {
            Participants ps = (Participants) pkg.get("Participants");
            Iterator psi = ps.toCollection().iterator();
            while (psi.hasNext()) {
                Participant p = (Participant) psi.next();
                Collection extAttribs = ((ExtendedAttributes) p.get("ExtendedAttributes")).toCollection();
                extAttribNames.addAll(XMLUtil.getAllExtendedAttributeNames(extAttribs));
            }
            Iterator it = ((WorkflowProcesses) pkg.get("WorkflowProcesses")).toCollection().iterator();
            while (it.hasNext()) {
                WorkflowProcess wp = (WorkflowProcess) it.next();
                ps = (Participants) wp.get("Participants");
                psi = ps.toCollection().iterator();
                while (psi.hasNext()) {
                    Participant p = (Participant) psi.next();
                    Collection extAttribs = ((ExtendedAttributes) p.get("ExtendedAttributes")).toCollection();
                    extAttribNames.addAll(XMLUtil.getAllExtendedAttributeNames(extAttribs));
                }
            }
        } else if (cel instanceof Tool) {
            Iterator it = ((WorkflowProcesses) pkg.get("WorkflowProcesses")).toCollection().iterator();
            while (it.hasNext()) {
                WorkflowProcess wp = (WorkflowProcess) it.next();
                Activities acts = (Activities) wp.get("Activities");
                Iterator acti = acts.toCollection().iterator();
                while (acti.hasNext()) {
                    Activity act = (Activity) acti.next();
                    Tools ts = act.getTools();
                    if (ts != null) {
                        Iterator ti = ts.toCollection().iterator();
                        while (ti.hasNext()) {
                            Tool t = (Tool) ti.next();
                            Collection extAttribs = ((ExtendedAttributes) t.get("ExtendedAttributes")).toCollection();
                            extAttribNames.addAll(XMLUtil.getAllExtendedAttributeNames(extAttribs));
                        }
                    }
                }
                Iterator asets = ((ActivitySets) wp.get("ActivitySets")).toCollection().iterator();
                while (asets.hasNext()) {
                    ActivitySet as = (ActivitySet) asets.next();
                    acts = (Activities) as.get("Activities");
                    acti = acts.toCollection().iterator();
                    while (acti.hasNext()) {
                        Activity act = (Activity) acti.next();
                        Tools ts = act.getTools();
                        if (ts != null) {
                            Iterator ti = ts.toCollection().iterator();
                            while (ti.hasNext()) {
                                Tool t = (Tool) ti.next();
                                Collection extAttribs = ((ExtendedAttributes) t.get("ExtendedAttributes")).toCollection();
                                extAttribNames.addAll(XMLUtil.getAllExtendedAttributeNames(extAttribs));
                            }
                        }
                    }
                }
            }
        } else if (cel instanceof Service) {
            Iterator it = ((WorkflowProcesses) pkg.get("WorkflowProcesses")).toCollection().iterator();
            while (it.hasNext()) {
                WorkflowProcess wp = (WorkflowProcess) it.next();
                Activities acts = (Activities) wp.get("Activities");
                Iterator acti = acts.toCollection().iterator();
                while (acti.hasNext()) {
                    Activity act = (Activity) acti.next();
                    Services ts = act.getServices();
                    if (ts != null) {
                        Iterator ti = ts.toCollection().iterator();
                        while (ti.hasNext()) {
                            Service t = (Service) ti.next();
                            Collection extAttribs = ((ExtendedAttributes) t.get("ExtendedAttributes")).toCollection();
                            extAttribNames.addAll(XMLUtil.getAllExtendedAttributeNames(extAttribs));
                        }
                    }
                }
                Iterator asets = ((ActivitySets) wp.get("ActivitySets")).toCollection().iterator();
                while (asets.hasNext()) {
                    ActivitySet as = (ActivitySet) asets.next();
                    acts = (Activities) as.get("Activities");
                    acti = acts.toCollection().iterator();
                    while (acti.hasNext()) {
                        Activity act = (Activity) acti.next();
                        Tools ts = act.getTools();
                        if (ts != null) {
                            Iterator ti = ts.toCollection().iterator();
                            while (ti.hasNext()) {
                                Service t = (Service) ti.next();
                                Collection extAttribs = ((ExtendedAttributes) t.get("ExtendedAttributes")).toCollection();
                                extAttribNames.addAll(XMLUtil.getAllExtendedAttributeNames(extAttribs));
                            }
                        }
                    }
                }
            }
        } else if (cel instanceof Device) {
            Iterator it = ((WorkflowProcesses) pkg.get("WorkflowProcesses")).toCollection().iterator();
            while (it.hasNext()) {
                WorkflowProcess wp = (WorkflowProcess) it.next();
                Activities acts = (Activities) wp.get("Activities");
                Iterator acti = acts.toCollection().iterator();
                while (acti.hasNext()) {
                    Activity act = (Activity) acti.next();
                    Tools ts = act.getTools();
                    if (ts != null) {
                        Iterator ti = ts.toCollection().iterator();
                        while (ti.hasNext()) {
                            Device t = (Device) ti.next();
                            Collection extAttribs = ((ExtendedAttributes) t.get("ExtendedAttributes")).toCollection();
                            extAttribNames.addAll(XMLUtil.getAllExtendedAttributeNames(extAttribs));
                        }
                    }
                }
                Iterator asets = ((ActivitySets) wp.get("ActivitySets")).toCollection().iterator();
                while (asets.hasNext()) {
                    ActivitySet as = (ActivitySet) asets.next();
                    acts = (Activities) as.get("Activities");
                    acti = acts.toCollection().iterator();
                    while (acti.hasNext()) {
                        Activity act = (Activity) acti.next();
                        Devices ts = act.getDevices();
                        if (ts != null) {
                            Iterator ti = ts.toCollection().iterator();
                            while (ti.hasNext()) {
                                Device t = (Device) ti.next();
                                Collection extAttribs = ((ExtendedAttributes) t.get("ExtendedAttributes")).toCollection();
                                extAttribNames.addAll(XMLUtil.getAllExtendedAttributeNames(extAttribs));
                            }
                        }
                    }
                }
            }
        } else if (cel instanceof Event) {
            Iterator it = ((WorkflowProcesses) pkg.get("WorkflowProcesses")).toCollection().iterator();
            while (it.hasNext()) {
                WorkflowProcess wp = (WorkflowProcess) it.next();
                Activities acts = (Activities) wp.get("Activities");
                Iterator acti = acts.toCollection().iterator();
                while (acti.hasNext()) {
                    Activity act = (Activity) acti.next();
                    Tools ts = act.getTools();
                    if (ts != null) {
                        Iterator ti = ts.toCollection().iterator();
                        while (ti.hasNext()) {
                            Event t = (Event) ti.next();
                            Collection extAttribs = ((ExtendedAttributes) t.get("ExtendedAttributes")).toCollection();
                            extAttribNames.addAll(XMLUtil.getAllExtendedAttributeNames(extAttribs));
                        }
                    }
                }
                Iterator asets = ((ActivitySets) wp.get("ActivitySets")).toCollection().iterator();
                while (asets.hasNext()) {
                    ActivitySet as = (ActivitySet) asets.next();
                    acts = (Activities) as.get("Activities");
                    acti = acts.toCollection().iterator();
                    while (acti.hasNext()) {
                        Activity act = (Activity) acti.next();
                        Devices ts = act.getDevices();
                        if (ts != null) {
                            Iterator ti = ts.toCollection().iterator();
                            while (ti.hasNext()) {
                                Event t = (Event) ti.next();
                                Collection extAttribs = ((ExtendedAttributes) t.get("ExtendedAttributes")).toCollection();
                                extAttribNames.addAll(XMLUtil.getAllExtendedAttributeNames(extAttribs));
                            }
                        }
                    }
                }
            }
        } else if (cel instanceof Transition) {
            Iterator it = ((WorkflowProcesses) pkg.get("WorkflowProcesses")).toCollection().iterator();
            while (it.hasNext()) {
                WorkflowProcess wp = (WorkflowProcess) it.next();
                Transitions ts = (Transitions) wp.get("Transitions");
                Iterator tsi = ts.toCollection().iterator();
                while (tsi.hasNext()) {
                    Transition t = (Transition) tsi.next();
                    Collection extAttribs = ((ExtendedAttributes) t.get("ExtendedAttributes")).toCollection();
                    extAttribNames.addAll(XMLUtil.getAllExtendedAttributeNames(extAttribs));
                }
                Iterator asets = ((ActivitySets) wp.get("ActivitySets")).toCollection().iterator();
                while (asets.hasNext()) {
                    ActivitySet as = (ActivitySet) asets.next();
                    ts = (Transitions) as.get("Transitions");
                    tsi = ts.toCollection().iterator();
                    while (tsi.hasNext()) {
                        Transition t = (Transition) tsi.next();
                        Collection extAttribs = ((ExtendedAttributes) t.get("ExtendedAttributes")).toCollection();
                        extAttribNames.addAll(XMLUtil.getAllExtendedAttributeNames(extAttribs));
                    }
                }
            }
        } else if (cel instanceof TypeDeclaration) {
            Iterator tds = ((TypeDeclarations) pkg.get("TypeDeclarations")).toCollection().iterator();
            while (tds.hasNext()) {
                TypeDeclaration td = (TypeDeclaration) tds.next();
                Collection extAttribs = ((ExtendedAttributes) td.get("ExtendedAttributes")).toCollection();
                extAttribNames.addAll(XMLUtil.getAllExtendedAttributeNames(extAttribs));
            }
        } else if (cel instanceof WorkflowProcess) {
            Iterator it = ((WorkflowProcesses) pkg.get("WorkflowProcesses")).toCollection().iterator();
            while (it.hasNext()) {
                WorkflowProcess wp = (WorkflowProcess) it.next();
                Collection extAttribs = ((ExtendedAttributes) wp.get("ExtendedAttributes")).toCollection();
                extAttribNames.addAll(XMLUtil.getAllExtendedAttributeNames(extAttribs));
            }
        }
        return extAttribNames;
    }

    private static Set getAllExtendedAttributeNames(Collection extAttribs) {
        Set eaNames = new HashSet();
        Iterator it = extAttribs.iterator();
        while (it.hasNext()) {
            ExtendedAttribute ea = (ExtendedAttribute) it.next();
            eaNames.add(ea.get("Name").toString());
        }
        return eaNames;
    }

    public static void setSelectCell(Object object) {
        TextTreePanel jtreeTable = MainLeftDownPanel.getInstance(BPD.getInstance().getPackageEditor()).getTextTreePanel();
        // jtreeTable.refreshView(BPD.getInstance().getPackageEditor().getXMLPackage());

        JTree jtree = jtreeTable.getTree();
        // jtreeTable.refreshView(BPD.getInstance().getPackageEditor().getXMLPackage());
        if (object instanceof com.ds.bpm.bpd.graph.Activity && !(object instanceof Start) && !(object instanceof End)) {
            Activity activity = (Activity) ((com.ds.bpm.bpd.graph.Activity) object).getUserObject();
            TextPreview.getInstance(BPD.getInstance().getPackageEditor()).refreshView(activity);

            int j = jtree.getRowCount();
            for (int i = 0; i < j; i++) {
                TreePath ptp = jtree.getPathForRow(i);
                DefaultMutableTreeNode tn = (DefaultMutableTreeNode) ptp.getLastPathComponent();
                if (tn.getUserObject().equals(activity.getOwnerProcess())) {
                    jtree.expandPath(ptp);
                    for (int h = i + 1; h < jtree.getRowCount(); h++) {
                        TreePath ptpc = jtree.getPathForRow(h);
                        DefaultMutableTreeNode tnc = (DefaultMutableTreeNode) ptpc.getLastPathComponent();
                        if (tnc.getUserObject() instanceof Activities) {
                            Activities activites = (Activities) tnc.getUserObject();
                            jtree.expandPath(ptpc);

                            for (int m = h + 1; m < jtree.getRowCount(); m++) {
                                TreePath ptpcc = jtree.getPathForRow(m);
                                DefaultMutableTreeNode tncc = (DefaultMutableTreeNode) ptpcc.getLastPathComponent();
                                if (tncc.getUserObject().equals(activity)) {
                                    jtree.expandPath(ptpcc);
                                    jtree.setSelectionPath(ptpcc);

                                    jtreeTable.getJScrollPane().getVerticalScrollBar().getModel().setValue(m * 18);
                                    jtreeTable.getJScrollPane().updateUI();
                                    return;
                                }
                            }

                        }

                    }

                }

            }
        }
        if (object instanceof com.ds.bpm.bpd.graph.Transition) {
            Transition transition = (Transition) ((com.ds.bpm.bpd.graph.Transition) object).getUserObject();
            TextPreview.getInstance(BPD.getInstance().getPackageEditor()).refreshView(transition);
            int j = jtree.getRowCount();
            for (int i = 0; i < j; i++) {
                TreePath ptp = jtree.getPathForRow(i);
                DefaultMutableTreeNode tn = (DefaultMutableTreeNode) ptp.getLastPathComponent();
                if (tn.getUserObject().equals(transition.getOwnerProcess())) {
                    jtree.expandPath(ptp);
                    for (int h = i + 1; h < jtree.getRowCount(); h++) {
                        TreePath ptpc = jtree.getPathForRow(h);
                        DefaultMutableTreeNode tnc = (DefaultMutableTreeNode) ptpc.getLastPathComponent();
                        if (tnc.getUserObject() instanceof Transitions) {
                            Transitions transitions = (Transitions) tnc.getUserObject();
                            jtree.expandPath(ptpc);
                            for (int m = h + 1; m < jtree.getRowCount(); m++) {
                                TreePath ptpcc = jtree.getPathForRow(m);
                                DefaultMutableTreeNode tncc = (DefaultMutableTreeNode) ptpcc.getLastPathComponent();
                                if (tncc.getUserObject().equals(transition)) {
                                    jtree.expandPath(ptpcc);
                                    jtree.setSelectionPath(ptpcc);
                                    jtreeTable.getJScrollPane().getVerticalScrollBar().getModel().setValue(m * 18);
                                    jtreeTable.getJScrollPane().updateUI();
                                    return;
                                }
                            }

                        }

                    }

                }

            }
        }

    }

    public static String makeEmptyHTMLText(int length) {
        if (length < 0)
            return null;
        String es = "";
        for (int i = 0; i < length; i++) {
            es += "&nbsp;";
        }
        return es;
    }

    /*
     * public static XMLElement makeCopyOfCollectionElement (XMLCollectionElement ce) { if (ce==null) return null;
     *
     * Document document=null; try { DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance(); DocumentBuilder
     * dbuilder = dbf.newDocumentBuilder(); document = dbuilder.newDocument(); } catch (Exception ex) {}
     *
     * Node node = document.createElement(ce.toName()); ce.toXML(node); document.appendChild(node);
     *
     * XMLCollectionElement copyOfEl=(XMLCollectionElement)ce.getCollection().generateNewElement(); String
     * ID=copyOfEl.getID();
     *
     * NodeList nl=document.getElementsByTagName(ce.toName()); Node wantedNode=null; if (nl.getLength()>0) { String
     * extID=ce.getID(); for (int i=0; i<nl.getLength(); i++) { Node n=nl.item(i); String IDOfNode=XMLUtil.getID(n); if
     * (IDOfNode.equals(extID)) { wantedNode=n; break; } } } else { return null; }
     *
     * if (wantedNode==null) return null; //String XML=wp.toXML(null,0).trim(); // trimming is neccessary for our parser
     * copyOfEl.fromXML(wantedNode);
     *
     * // Setting back newly generated ID copyOfEl.set("Id",ID); if (copyOfEl.get("Name")!=null) { copyOfEl.set("Name",
     * XMLUtil.getLanguageDependentString("CopyOfKey")+ " "+copyOfEl.get("Name").toString()); }
     *
     * return copyOfEl; }
     */
}
