/* XMLComplexChoice.java
 *
 * Authors:
 * Stefanovic Nenad  chupo@iis.ns.ac.yu
 * Bojanic Sasa      sasaboy@neobee.net
 * Puskas Vladimir   vpuskas@eunet.yu
 * Pilipovic Goran   zboniek@uns.ac.yu
 *
 */

package com.ds.bpm.bpd.xml;

import com.ds.bpm.bpd.xml.panels.XMLComboChoicePanel;
import com.ds.bpm.bpd.xml.panels.XMLPanel;
import com.ds.bpm.bpd.xml.panels.XMLRadioChoicePanel;

import org.w3c.dom.DOMException;
import org.w3c.dom.Node;

/**
 * Represents complex type choice of some XML XML.
 */
public class XMLComplexChoice extends XMLChoice {
	/** Attribute for visual presentation of an object within the panel. */
	private boolean isComboPanel;

	/** Attribute for visual presentation of an object within the panel. */
	private boolean isVertical;

	/** Attribute for visual presentation of an object within the panel. */
	private boolean isChoiceVertical;

	/** Attribute for visual presentation of an object within the panel. */
	private boolean hasBorder;

	/** Attribute for visual presentation of an object within the panel. */
	private boolean hasTitle;

	/**
	 * Creates a new instance of element with the specified name, specified
	 * choices for the element, and with <b>chooseIndex</b>'th element choosen.
	 * All visual attributes are default, and the 'shortcut' to the dialog for
	 * defining the new choice does not exist.
	 * 
	 * @param name
	 *            The name of the 'choice' XML element that this class
	 *            represents.
	 * @param choices
	 *            The possible choices for this element.
	 * @param choosenIndex
	 *            The index of element to be choosen right after creation.
	 */
	public XMLComplexChoice(String name, XMLElement[] choices, int choosenIndex) {
		this(name, choices, choosenIndex, true);
	}

	/**
	 * Creates a new instance of element with the specified name, specified
	 * choices for the element, and with <b>chooseIndex</b>'th element choosen,
	 * and with specified panel type (combo or radio). All other visual
	 * attributes are default, and the 'shortcut' to the dialog for defining the
	 * new choice does not exist.
	 * 
	 * @param name
	 *            The name of the 'choice' XML element that this class
	 *            represents.
	 * @param choices
	 *            The possible choices for this element.
	 * @param choosenIndex
	 *            The index of element to be choosen right after creation.
	 * @param isComboPanel
	 *            Indicates if the choices are displayed within the combo panel,
	 *            or the radio panel.
	 */
	public XMLComplexChoice(String name, XMLElement[] choices,
			int choosenIndex, boolean isComboPanel) {

		this(name, choices, choosenIndex, isComboPanel, true, false, true, true);
	}

	/**
	 * Creates a new instance of element with the specified name, specified
	 * choices for the element, and with <b>chooseIndex</b>'th element choosen,
	 * and with specified all visual attributes. The 'shortcut' to the dialog
	 * for defining the new choice might exist.
	 * 
	 * @param name
	 *            The name of the 'choice' XML element that this class
	 *            represents.
	 * @param choices
	 *            The possible choices for this element.
	 * @param choosenIndex
	 *            The index of element to be choosen right after creation.
	 * @param isComboPanel
	 *            Indicates if the choices are displayed within the combo panel,
	 *            or the radio panel.
	 * @param isVertical
	 *            Indicates if label will stand left to the combo panel, or
	 *            above it.
	 * @param isChoiceVertical
	 *            Indicates if the panel of choosen element will be vertical or
	 *            not.
	 * @param hasBorder
	 *            Indicates if the panel will have the border around it's area.
	 * @param hasTitle
	 *            Indicates if the panel will have the title for it's area.
	 */
	public XMLComplexChoice(String name, XMLElement[] choices,
			int choosenIndex, boolean isComboPanel, boolean isVertical,
			boolean isChoiceVertical, boolean hasBorder, boolean hasTitle) {

		super(name, choices, choosenIndex);
		this.isComboPanel = isComboPanel;
		this.isVertical = isVertical;
		this.isChoiceVertical = isChoiceVertical;
		this.hasBorder = hasBorder;
		this.hasTitle = hasTitle;
	}

	/**
	 * Indicates if element is empty.
	 * 
	 * @return <tt>true</tt> if user haven't defined the element value within
	 *         dialog, <tt>false</tt> otherwise.
	 */
	public boolean isEmpty() {
		if (choices == null) {
			return true;
		} else {
			if (choosen != null) {
				return false;
			} else {
				return true;
			}
		}
	}

	public void toXML(Node parent) throws DOMException {
		// System.out.println("Choosen="+choosen);
		// Calls the toXML of the choosen element
		if (choosen != null) {
			((XMLElement) choosen).toXML(parent);
		}
	}

	public void fromXML(String choosenName, Node node) {
		// System.out.println("Processing cmplx choice "+getClass()+" for
		// choosen name="+choosenName);
		if (choices != null) {
			// Search for the element wich name is choosen,
			// and calls it's fromXML with the 'tagContent'
			for (int i = 0; i < choices.length; i++) {
				if (((XMLElement) choices[i]).name.equals(choosenName)) {
					setValue(choices[i]);
					((XMLElement) choices[i]).fromXML(node);
					break;
				}
			}
		}
	}

	public XMLPanel getPanel() {
		if (isComboPanel) {
			return new XMLComboChoicePanel(this, (hasTitle) ? toLabel() : "",
					XMLPanel.BOX_LAYOUT, isVertical, isChoiceVertical,
					hasBorder);
		} else {
			return new XMLRadioChoicePanel(this, (hasTitle) ? toLabel() : "",
					XMLPanel.BOX_LAYOUT, isVertical, isChoiceVertical,
					hasBorder);
		}
	}

	public Object clone() {
		XMLComplexChoice d = (XMLComplexChoice) super.clone();
		d.isComboPanel = this.isComboPanel;
		d.isVertical = this.isVertical;
		d.isChoiceVertical = this.isChoiceVertical;
		d.hasBorder = this.hasBorder;
		d.hasTitle = this.hasTitle;
		return d;
	}
}
/* End of XMLComplexChoice.java */
