/* XML.java
 *
 * Authors:
 * Stefanovic Nenad  chupo@iis.ns.ac.yu
 * Bojanic Sasa      sasaboy@neobee.net
 * Puskas Vladimir   vpuskas@eunet.yu
 * Pilipovic Goran   zboniek@uns.ac.yu
 *
 */

package com.ds.bpm.bpd.xml;


import com.ds.bpm.bpd.xml.elements.ExternalPackage;
import com.ds.bpm.bpd.xml.elements.ExternalPackages;
import com.ds.bpm.bpd.xml.elements.Package;
import com.sun.org.apache.xerces.internal.parsers.DOMParser;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import java.io.*;
import java.nio.channels.FileLock;
import java.util.*;

/**
 * Class which purpose is to provide methods which are
 * used by classes that represents program apstraction of
 * XML elements. These methods offers support for reading or
 * writting an XML document and for generating the tooltips for
 * for the classes that needs it.
 */
public class XML implements XMLInterface {

   private LinkedHashMap idToPackage=new LinkedHashMap();
   private LinkedHashMap xmlFileToPackage=new LinkedHashMap();
   private LinkedHashMap pkgIdToFileContent=new LinkedHashMap();
   private LinkedHashMap packageToParentDirectory=new LinkedHashMap();

   private Map fileLocks=new HashMap();
   private Map rndAccessFiles=new HashMap();

   private static Map workflowVersionMap = new HashMap();
   
   private String mainPackageReference;

   private boolean fileLocking=true;

   private Map parsingErrorMessages=new HashMap();
   private boolean isValidationON=true;

   public void setValidation (boolean isActive) {
      isValidationON=isActive;
   }

   public void setFileLockingStatus (boolean status) {
      fileLocking=status;
   }

   public void clearParserErrorMessages () {
      parsingErrorMessages.clear();
   }

   /**
    * This method has to be called from the newly created package after its
    * Id is entered.
    * @param pkg
    */
   public void register (Package pkg) {
      idToPackage.put(pkg.get("Id").toString(),pkg);
   }

   /**
    * This method is called when first saving new package, or when saving it with
    * a different name.
    * @param filename
    * @param pkg
    */
   public void registerPackageFilename (String filename,Package pkg) {
      String pkgId=pkg.get("Id").toString();
      Iterator it=xmlFileToPackage.entrySet().iterator();
      String uToRem=getAbsoluteFilePath(pkg);
      if (uToRem!=null) {
         xmlFileToPackage.remove(uToRem);
      }
      // release lock
      try {
         FileLock fl=(FileLock)fileLocks.remove(pkg);
         fl.release();
      } catch (Exception ex) {}
      // close the file stream
      try {
         RandomAccessFile raf=(RandomAccessFile)rndAccessFiles.remove(pkg);
         raf.close();
      } catch (Exception ex) {}

      String cp=XMLUtil.getCanonicalPath(filename, false);
      xmlFileToPackage.put(cp,pkg);

      File f=new File(cp);

      try {
         RandomAccessFile raf=new RandomAccessFile(f,"rw");
         rndAccessFiles.put(pkg,raf);
         if (fileLocking) {
            FileLock fl=raf.getChannel().tryLock();
            if (fl!=null) {
               fileLocks.put(pkg,fl);
               // this should never happend
            } else {
               System.out.println("Can't lock");
            }
         }
         // this happens when using jdk1.4.0 under Linux, where
         // the locking is not supported
      } catch (IOException ex) {
      } catch (Exception ex) {}

      // register parent directory with the package
      try {
         packageToParentDirectory.put(pkg,f.getParentFile().getCanonicalPath());
      } catch (Exception ex) {
         packageToParentDirectory.put(pkg,f.getParentFile().getAbsolutePath());
      }
   }

   public boolean isPackageOpened (String pkgId) {
      return idToPackage.containsKey(pkgId);
   }

   public void putPkgIdToFileContentMapping (String pkgId,
                                             String fileContent) {
      //System.out.println("Adding mapping for file with id="+pkgId);
      pkgIdToFileContent.put(pkgId,fileContent);
   }

   public String getPackageFileContent (String pkgId) {
      return (String)pkgIdToFileContent.get(pkgId);
   }

   public Package getPackageById (String pkgId) {
      return (Package)idToPackage.get(pkgId);
   }

   public Package getPackageByFilename (String filename) {
      filename=XMLUtil.getCanonicalPath(filename, false);
      return (Package)xmlFileToPackage.get(filename);
   }

   public RandomAccessFile getRaf (Package pkg){
      return (RandomAccessFile)rndAccessFiles.get(pkg);
   }

   public Package getExternalPackageByRelativeFilePath (
                                                        String relativePathToExtPkg,Package rootPkg) {

      File f=new File(relativePathToExtPkg);
      if (!f.isAbsolute()) {
         f=new File(getParentDirectory(rootPkg)+File.separator+relativePathToExtPkg);
      }
      if (f.exists()) {
         //System.out.println("Pkg for "+relativePathToExtPkg+"->"+f.getAbsolutePath()+" is found");
         return getPackageByFilename(f.getAbsolutePath());
      } else {
         //System.out.println("Pkg for "+relativePathToExtPkg+"->"+f.getAbsolutePath()+" is not found");
         return null;
      }
   }

   public String getAbsoluteFilePath (Package pkg) {
      Iterator it=xmlFileToPackage.entrySet().iterator();
      String fullPath=null;
      while (it.hasNext()) {
         Map.Entry me=(Map.Entry)it.next();
         String u=(String)me.getKey();
         Package p=(Package)me.getValue();
         if (p.equals(pkg)) {
            fullPath=u;
            break;
         }
      }
      return fullPath;
   }

   public Collection getAllPackages () {
      return idToPackage.values();
   }

   public Collection getAllPackageIds () {
      return idToPackage.keySet();
   }

   public Collection getAllPackageFilenames () {
      return xmlFileToPackage.keySet();
   }

   public boolean doesPackageFileExists (String xmlFile) {
      if (new File(xmlFile).exists() || getPackageFileContent(xmlFile)!=null) {
         return true;
      } else {
         return false;
      }
   }

   public String getParentDirectory (Package pkg) {
      return (String)packageToParentDirectory.get(pkg);
   }

   public Package openPackage (String pkgReference,boolean openFromStream) {
      parsingErrorMessages.clear();
      if (!openFromStream) {
         mainPackageReference=pkgReference;
      }
      // remember the all packages before opening the given package
      Set pre=new HashSet(getAllPackages());
      // this method opens the package and all of it's external packages
      Package pkg=openDocument(pkgReference,openFromStream);
      // get all packages after opening the given package
      Set post=new HashSet(getAllPackages());
      // extract newly opened packages
      post.removeAll(pre);
      //System.out.println("Newly opened packages are: "+post);
      // append all external packages of each newly added package (this means
      // the packages referenced by the package's external packages, because
      // till now, only the first level external packages are added).
      // (there can be cross-reference, so it must be done this way).
      Iterator newPackages=post.iterator();
      while (newPackages.hasNext()) {
         Package p=(Package)newPackages.next();
         appendAllExternalPackagesForPackage(p);
         //System.out.println("Eps for p "+p+" are "+p.getAllExternalPackages());
      }

      // calling afterImporting method on package and all ext. packages
      // NOTE: this can't be done within the previous loop
      newPackages=post.iterator();
      while (newPackages.hasNext()) {
         Package p=(Package)newPackages.next();
         p.afterImporting();
      }
      if (pkg!=null && !openFromStream) {
         System.setProperty("user.dir",getParentDirectory(pkg));
      }
      //printDebug();
      return pkg;
   }

   public void printDebug () {
      System.out.println("idToPackage="+idToPackage);
      System.out.println("xmlFileToPackage="+xmlFileToPackage);
      System.out.println("pkgIdToFileContent="+pkgIdToFileContent);
      System.out.println("packageToWorkingDirectory="+packageToParentDirectory);
      System.out.println("fileLocks="+fileLocks);
      System.out.println("rndAccessFiles="+rndAccessFiles);
   }

   // Recursive implementation
   private Package openDocument (String pkgReference,boolean openFromStream) {

      Package pkg=null;
      File f=null;
      String oldP=pkgReference;

      if (!openFromStream) {
         pkgReference=XMLUtil.getCanonicalPath(pkgReference, false);
         if (pkgReference==null) {
            Set fem=new HashSet();
            fem.add("File does not exist");
            parsingErrorMessages.put(oldP,fem);
            return null;
         } else {

            f=new File(pkgReference);
            // set the proper user dir
            try {
               System.setProperty("user.dir",f.getParentFile().getCanonicalPath());
            } catch (Exception ex) {
               System.setProperty("user.dir",f.getParentFile().getAbsolutePath());
            }
         }
      }

      if (xmlFileToPackage.containsKey(pkgReference)) {
         return getPackageByFilename(pkgReference);
      }

      if (openFromStream && idToPackage.containsKey(pkgReference)) {
         return getPackageById(pkgReference);
      }

      if (!openFromStream) {
         pkg=parseDocument(pkgReference,true);
         try {
            // trying to open main package file as 'rw'
            // and to lock it exclusivly
            if (oldP.equals(mainPackageReference)) {
               RandomAccessFile raf=new RandomAccessFile(f,"rw");
               rndAccessFiles.put(pkg,raf);
               if (fileLocking) {
                  FileLock fl=raf.getChannel().tryLock();
                  // this happens if the main package is not already locked
                  if (fl!=null) {
                     fileLocks.put(pkg,fl);
                     // this happens if the file is already opened as 'rw' and locked
                     // exclusivly, or if it is opened as 'r' and locked as shared
                  } else {
                     Set errorMessages = new HashSet();
                     errorMessages.add(XMLUtil.getLanguageDependentString("ErrorTheFileIsLocked"));
                     parsingErrorMessages.put(pkgReference,errorMessages);
                     return null;
                  }
               }
               // trying to open external package file as 'rw'
               // and to lock it exclusivly
            } else {
               RandomAccessFile raf=new RandomAccessFile(f,"r");
               rndAccessFiles.put(pkg,raf);
               if (fileLocking) {
                  FileLock fl=raf.getChannel().tryLock(0L,Long.MAX_VALUE,true);
                  // this happens if the file isn't already opened as
                  // 'rw' and locked exclusivly
                  if (fl!=null) {
                     fileLocks.put(pkg,fl);
                     // this happens if the file is opened as 'rw' and locked exclusivly
                  } else {
                     Set errorMessages = new HashSet();
                     errorMessages.add(XMLUtil.getLanguageDependentString("ErrorTheFileIsLocked"));
                     parsingErrorMessages.put(pkgReference,errorMessages);
                     return null;
                  }
               }
            }
            // this exception happens if using jdk1.4.0 under Linux
         } catch (Exception ex) {
            //ex.printStackTrace();
         }
      } else {
		  pkg = parseDocument(pkgReference, false);
         //pkg=parseDocument(getPackageFileContent(pkgReference),false);
      }

      if (pkg!=null) {
         String pkgId=pkg.get("Id").toString();
         // check if package is already imported
         if (idToPackage.containsKey(pkgId)) {
            // check if this is the same package, or just the one with the same id
            if ((!openFromStream && xmlFileToPackage.containsKey(pkgReference)) ||
                   (openFromStream && getPackageFileContent(pkgReference)!=null)) {
               return getPackageById(pkgId);
            } else {
               return null;
            }
         }
         idToPackage.put(pkgId,pkg);
         if (!openFromStream) {
            xmlFileToPackage.put(pkgReference,pkg);
            try {
               packageToParentDirectory.put(pkg,f.getParentFile().getCanonicalPath());
            } catch (Exception ex) {
               packageToParentDirectory.put(pkg,f.getParentFile().getAbsolutePath());
            }
         }
		 if ((ExternalPackages)pkg.get("ExternalPackages")!=null){
         // open all external packages
         Iterator eps=((ExternalPackages)pkg.get("ExternalPackages")).
        
         toCollection().iterator();
		
         while (eps.hasNext()) {
            String pathToExtPackage=((ExternalPackage)eps.next()).get("href").toString();
            // setting working dir to be the one of the current package
            if (!openFromStream) {
               System.setProperty("user.dir",packageToParentDirectory.get(pkg).toString());
            }
            Package extPkg=openDocument(pathToExtPackage,openFromStream);
            if (extPkg!=null) {
               pkg.addExternalPackage(extPkg);
            }
         }
         }
      } else {
         System.err.println("Problems with opening file "+pkgReference);
      }
      return pkg;
   }

   private void appendAllExternalPackagesForPackage(Package m) {
      Stack s = new Stack();
      s.addAll(m.getAllExternalPackages());
      Set result=new HashSet();
      while (!s.isEmpty()) {
         Package tmp=(Package)s.pop();
         Iterator extPkgs=tmp.getAllExternalPackages().iterator();
         while (extPkgs.hasNext()) {
            Object nextP=extPkgs.next();
            if (!m.getAllExternalPackages().contains(nextP) && !s.contains(nextP) && nextP!=m) {
               s.add(nextP);
               m.addExternalPackage((Package)nextP);
            }
         }
      }
   }

   public Package parseDocument (String toParse,boolean isFile) {
      Package pkg=null;
      //  Create a Xerces DOM Parser
      DOMParser parser = new DOMParser();
  

      //  Parse the Document and traverse the DOM
      try {
         parser.setFeature("http://apache.org/xml/features/continue-after-fatal-error",true);
         ParsingErrors pErrors=new ParsingErrors();
         parser.setErrorHandler(pErrors);
         if (isValidationON) {
            parser.setEntityResolver(new XPDLEntityResolver());
            parser.setFeature("http://xml.org/sax/features/validation",true);
            parser.setFeature("http://apache.org/xml/features/validation/schema",true);
            //parser.setFeature("http://apache.org/xml/features/validation/schema-full-checking",true);
         }
         if (isFile) {
            //System.out.println("Parsing from file");
            File f=new File(toParse);
            if (!f.exists()) {
               f=new File(f.getCanonicalPath());
            }
            //parser.parse(xmlFile);
            parser.parse(new InputSource(new FileInputStream(f))); // Fixed by Harald Meister
         } else {
            //System.out.println("Parsing from stream");
			 System.out.print(toParse);
            parser.parse(new InputSource(new StringReader(toParse)));
         }
      
         Document document = parser.getDocument();
         Set errorMessages = pErrors.getErrorMessages();
         if (errorMessages.size()>0) {
            //System.err.println("Errors during document parsing");
            if (isFile) {
               parsingErrorMessages.put(toParse,errorMessages);
            } else {
               parsingErrorMessages.put("",errorMessages);
            }
         }
         if (document!=null) {
            pkg=new Package(this);
            pkg.fromXML(document.getDocumentElement());
            //System.out.println("package "+pkg+" imported");
         }
      } catch (Exception ex) {
         ex.printStackTrace();
         System.err.println("Fatal error while parsing document");
         Set fem=new HashSet();
         fem.add("Fatal error while parsing document");
         if (isFile) {
            parsingErrorMessages.put(toParse,fem);
         } else {
            parsingErrorMessages.put("",fem);
         }
         return null;
      }
      return pkg;
   }

   /**
    * This method should be called immediatelly after opening a document,
    * otherwise, messages could be invalid.
    * @return The map which keys are opened packages, and values are the sets
    * of errors for corresponding package.
    */
   public Map getParsingErrorMessages () {
      return parsingErrorMessages;
   }

   public void closePackage (String pkgId) {
      Package toRemove=(Package)idToPackage.remove(pkgId);
      if (toRemove!=null) {
         // removing file to package mapping
         Iterator it=xmlFileToPackage.entrySet().iterator();
         Object keyToRemove=null;
         while (it.hasNext()) {
            Map.Entry me=(Map.Entry)it.next();
            Object key=me.getKey();
            Object val=me.getValue();
            if (val.equals(toRemove)) {
               keyToRemove=key;
               break;
            }
         }
         if (keyToRemove!=null) {
            xmlFileToPackage.remove(keyToRemove);
         }

         packageToParentDirectory.remove(toRemove);
         // close file
         RandomAccessFile raf=(RandomAccessFile)rndAccessFiles.remove(toRemove);
         try {
            raf.close();
         } catch (Exception ex) {}
         // unlock file
         FileLock fl=(FileLock)fileLocks.remove(toRemove);
         try {
            fl.release();
         } catch (Exception ex) {}
      }

      pkgIdToFileContent.remove(pkgId);
   }

   public void closeAllPackages () {
      idToPackage.clear();
      xmlFileToPackage.clear();
      packageToParentDirectory.clear();
      pkgIdToFileContent.clear();
      // close all files
      Iterator it=rndAccessFiles.values().iterator();
      while (it.hasNext()) {
         RandomAccessFile raf=(RandomAccessFile)it.next();
         try {
            raf.close();
         } catch (Exception ex) {}
      }
      rndAccessFiles.clear();
      unlockAllFiles();
   }

   public void unlockAllFiles () {
      // unlock all files
      Iterator it=fileLocks.values().iterator();
      while (it.hasNext()) {
         FileLock fl=(FileLock)it.next();
         try {
            fl.release();
         } catch (Exception ex) {}
      }
      fileLocks.clear();
   }

   public void lockAllFiles () {
      if (!fileLocking) return;
      Iterator it=rndAccessFiles.entrySet().iterator();
      while (it.hasNext()) {
         Map.Entry me=(Map.Entry)it.next();
         Package pkg=(Package)me.getKey();
         RandomAccessFile raf=(RandomAccessFile)me.getValue();

         try {
            // lock main package exclusivly
            if (pkg.equals(mainPackageReference)) {
               FileLock fl=raf.getChannel().tryLock();
               // this happens if the main package is not already locked
               if (fl!=null) {
                  fileLocks.put(pkg,fl);
                  // this happens if the file is already opened as 'rw' and locked
                  // exclusivly, or if it is opened as 'r' and locked as shared
               } else {
                  continue;
               }
               // lock external package exclusivly
            } else {
               FileLock fl=raf.getChannel().tryLock(0L,Long.MAX_VALUE,true);
               // this happens if the file isn't already opened as
               // 'rw' and locked exclusivly
               if (fl!=null) {
                  fileLocks.put(pkg,fl);
                  // this happens if the file is opened as 'rw' and locked exclusivly
               } else {
                  continue;
               }
            }
            // this exception happens if using jdk1.4.0 under Linux
         } catch (Exception ex) {
            //ex.printStackTrace();
         }
      }
   }

   public String getIDFromFile (String xmlFile) {
      try {
         //  Create parser
         DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
         factory.setValidating(false);
         DocumentBuilder parser = factory.newDocumentBuilder();
         Document document=null;

         //  Parse the Document
         try {
            File f=new File(xmlFile);
            if (!f.exists()) {
               f=new File(f.getCanonicalPath());
            }
            //document=parser.parse(xmlFile);
            document=parser.parse(new InputSource(new FileInputStream(f))); // Fixed by Harald Meister
         } catch (Exception ex) {
            document=parser.parse(new InputSource(new StringReader(getPackageFileContent(xmlFile))));
         }
         return XMLUtil.getID(document.getDocumentElement());
      } catch (Exception ex) {
         return "";
      }
   }

	// 注册workflow的version，并返回version
	public static int registerWorkflowVersion(String workflowId) {
		Integer version = new Integer("1");
		if (workflowVersionMap.containsKey(workflowId)) {
			version = (Integer) workflowVersionMap.get(workflowId);
			version = new Integer(version.intValue() + 1);
		}
		workflowVersionMap.put(workflowId, version);
		return version.intValue();
	}

}

