package com.ds.bpm.bpd.xml;

import java.io.ByteArrayOutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.Document;

import com.ds.bpm.bpd.Utils;
import com.ds.bpm.bpd.enums.ActivityDefTypeEnums;
import com.ds.bpm.bpd.xml.activity.Activities;
import com.ds.bpm.bpd.xml.activity.Activity;
import com.ds.bpm.bpd.xml.activity.Device;
import com.ds.bpm.bpd.xml.activity.Devices;
import com.ds.bpm.bpd.xml.activity.Event;
import com.ds.bpm.bpd.xml.activity.Events;
import com.ds.bpm.bpd.xml.activity.Service;
import com.ds.bpm.bpd.xml.activity.Services;
import com.ds.bpm.bpd.xml.activity.SubFlow;
import com.ds.bpm.bpd.xml.activity.Tool;
import com.ds.bpm.bpd.xml.activity.Tools;
import com.ds.bpm.bpd.xml.elements.ActivitySet;
import com.ds.bpm.bpd.xml.elements.ActivitySets;
import com.ds.bpm.bpd.xml.elements.ActualParameters;
import com.ds.bpm.bpd.xml.elements.Application;
import com.ds.bpm.bpd.xml.elements.BlockActivity;
import com.ds.bpm.bpd.xml.elements.ConformanceClass;
import com.ds.bpm.bpd.xml.elements.DataField;
import com.ds.bpm.bpd.xml.elements.DataType;
import com.ds.bpm.bpd.xml.elements.Deadline;
import com.ds.bpm.bpd.xml.elements.Deadlines;
import com.ds.bpm.bpd.xml.elements.DeclaredType;
import com.ds.bpm.bpd.xml.elements.Join;
import com.ds.bpm.bpd.xml.elements.Package;
import com.ds.bpm.bpd.xml.elements.PackageHeader;
import com.ds.bpm.bpd.xml.elements.Participant;
import com.ds.bpm.bpd.xml.elements.RedefinableHeader;
import com.ds.bpm.bpd.xml.elements.Responsible;
import com.ds.bpm.bpd.xml.elements.Responsibles;
import com.ds.bpm.bpd.xml.elements.Split;
import com.ds.bpm.bpd.xml.elements.Transition;
import com.ds.bpm.bpd.xml.elements.TransitionRefs;
import com.ds.bpm.bpd.xml.elements.Transitions;
import com.ds.bpm.bpd.xml.elements.TypeDeclaration;
import com.ds.bpm.bpd.xml.elements.TypeDeclarations;
import com.ds.bpm.bpd.xml.elements.WorkflowProcess;
import com.ds.bpm.bpd.xml.elements.WorkflowProcesses;
import com.ds.bpm.bpd.xml.elements.formula.FormalParameter;
import com.ds.bpm.bpd.xml.elements.formula.FormalParameters;

/**
 * Validates an XPDL package.
 */
public class PackageValidator {

    protected static final String CURRENT_XPDL_VERSION = "1.0";

    protected Package pkg;

    protected boolean checkingBybpd;

    protected boolean checkingForbpd;

    protected boolean getExistingSchemaValidationErrors;

    protected boolean checkExternalPackages;

    protected Map xpdlSchemaValidationErrors = new HashMap();

    protected Map graphsConnectionErrors = new HashMap();

    protected Map basicGraphConnectionErrors = new HashMap();

    protected Map graphsConformanceErrors = new HashMap();

    protected Map basicGraphsConformanceErrors = new HashMap();

    protected Map logicErrors = new HashMap();

    protected Map basicLogicErrors = new HashMap();

    public Map getXPDLSchemaValidationErrors() {
        return xpdlSchemaValidationErrors;
    }

    public Map getGraphsConnectionErrors(XMLComplexElement pkgOrWpOrAs) {
        return (Map) graphsConnectionErrors.get(pkgOrWpOrAs);
    }

    public String getBasicGraphConnectionError(XMLComplexElement pkgOrWpOrAs) {
        return (String) basicGraphConnectionErrors.get(pkgOrWpOrAs);
    }

    public Map getGraphConformanceErrors(XMLComplexElement pkgOrWpOrAs) {
        return (Map) graphsConformanceErrors.get(pkgOrWpOrAs);
    }

    public List getBasicGraphConformanceErrors(XMLComplexElement pkgOrWpOrAs) {
        return (List) basicGraphsConformanceErrors.get(pkgOrWpOrAs);
    }

    public Map getLogicErrors(XMLComplexElement pkgOrWpOrAs) {
        return (Map) logicErrors.get(pkgOrWpOrAs);
    }

    public String getBasicLogicError(XMLComplexElement pkgOrWpOrAs) {
        return (String) basicLogicErrors.get(pkgOrWpOrAs);
    }

    public PackageValidator(Package pkg, boolean checkingBybpd,
                            boolean checkingForbpd, boolean getExistingSchemaValidationErrors,
                            boolean checkExternalPackages) {
        this.pkg = pkg;
        this.checkingBybpd = checkingBybpd;
        this.checkingForbpd = checkingForbpd;
        this.getExistingSchemaValidationErrors = getExistingSchemaValidationErrors;
        this.checkExternalPackages = checkExternalPackages;
    }

    public boolean validateAll(boolean fullCheck) {
        try {
            boolean isValid = validateAgainstXPDLSchema();
            if (fullCheck || isValid) {
                isValid = checkPackage(fullCheck) && isValid;
            }
            if (fullCheck || isValid) {
                isValid = checkGraphConnections(fullCheck) && isValid;
            }
            if (fullCheck || isValid) {
                isValid = checkGraphConformance(fullCheck) && isValid;
            }

            return isValid;
        } catch (Exception ex) {
            ex.printStackTrace();
            return false;
        }
    }

    // ********************* validation against XPDL schema
    // *********************
    public boolean validateAgainstXPDLSchema() {
        if (getExistingSchemaValidationErrors) {
            xpdlSchemaValidationErrors = pkg.getXMLInterface().getParsingErrorMessages();
            if (xpdlSchemaValidationErrors.size() > 0) {
                return false;
            } else {
                return true;
            }
        }
        try {
            Document document = null;

            DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
            DocumentBuilder dbuilder = dbf.newDocumentBuilder();
            document = dbuilder.newDocument();
            ByteArrayOutputStream baos = new ByteArrayOutputStream();

            // The extended attributes for all package elements must
            // be updated if current package is not externally
            // referenced package.
            // Save.updateExtendedAttributesForWorkflowProcesses();

            // Here we get all document elements
            pkg.toXML(document);
            // Use a Transformer for output
            TransformerFactory tFactory = TransformerFactory.newInstance();
            Transformer transformer = tFactory.newTransformer();
            transformer.setOutputProperty("indent", "yes");
            transformer.setOutputProperty(
                    "{http://pkg.getXMLInterface().apache.org/xslt}indent-amount", "4");
            transformer.setOutputProperty("encoding", "UTF-8");
            DOMSource source = new DOMSource(document);
            StreamResult result = new StreamResult(baos);
            transformer.transform(source, result);

            pkg.getXMLInterface().clearParserErrorMessages();
            pkg.getXMLInterface().parseDocument(baos.toString("UTF8"), false);
            baos.close();
            xpdlSchemaValidationErrors = pkg.getXMLInterface().getParsingErrorMessages();
            if (xpdlSchemaValidationErrors.size() > 0) {
                return false;
            }
        } catch (Exception ex) {
            return false;
        }
        return true;

    }

    // ********************* Logic checking
    // **************************************
    public boolean checkPackage(boolean fullCheck) {
        Map les = new LinkedHashMap();
        logicErrors.put(pkg, les);
        basicLogicErrors.remove(pkg);

        boolean isPackageValid = true;
        boolean invalidId = false;
        if (!isIdValid(pkg.get("Id").toString())) {
            isPackageValid = false;
            invalidId = true;
            les.put(pkg, XMLUtil
                    .getLanguageDependentString("ErrorPackageIdIsNotValid"));
        }
        if (fullCheck || isPackageValid) {
            checkPackageHeader(fullCheck);
        }
        /*
         * if (fullCheck || isPackageValid) { isPackageValid =
         * checkRedefinableHeader(pkg, fullCheck) && isPackageValid; ; }
         */
        if (fullCheck || isPackageValid) {
            isPackageValid = checkConformanceClass(fullCheck) && isPackageValid;
            ;
        }
        /*
         * if (fullCheck || isPackageValid) { isPackageValid =
         * checkScript(fullCheck) && isPackageValid; ; }
         */
        /*
         * if ((fullCheck || isPackageValid) && checkExternalPackages) {
         * isPackageValid = checkExternalPackages(fullCheck) && isPackageValid; ; }
         */
        /*
         * if (fullCheck || isPackageValid) { isPackageValid =
         * checkCollection("TypeDeclarations", pkg, fullCheck) &&
         * isPackageValid; ; }
         */
        if (fullCheck || isPackageValid) {
            isPackageValid = checkCollection("Participants", pkg, fullCheck)
                    && isPackageValid;
            ;
        }
        /*
         * if (fullCheck || isPackageValid) { isPackageValid =
         * checkCollection("Applications", pkg, fullCheck) && isPackageValid; ; }
         * if (fullCheck || isPackageValid) { isPackageValid =
         * checkCollection("DataFields", pkg, fullCheck) && isPackageValid; ; }
         */
        boolean areProcessesValid = true;
        if (fullCheck || isPackageValid) {
            areProcessesValid = checkCollection("WorkflowProcesses", pkg,
                    fullCheck);
            isPackageValid = areProcessesValid && isPackageValid;
        }
        if (!isPackageValid) {
            if (invalidId) {
                basicLogicErrors
                        .put(
                                pkg,
                                XMLUtil
                                        .getLanguageDependentString("ErrorPackageIdIsNotValid"));
            } else if (!areProcessesValid) {
                basicLogicErrors
                        .put(
                                pkg,
                                XMLUtil
                                        .getLanguageDependentString("ErrorOneOrMoreProcessesHaveLogicErrors"));
            } else {
                basicLogicErrors.put(pkg, les.values().toArray()[0]);
            }
        }
        return isPackageValid;
    }

    public boolean checkPackageHeader(boolean fullCheck) {
        PackageHeader phdr = (PackageHeader) pkg.get("PackageHeader");
        String xpdlv = "XPDLVersion";
        if (!phdr.get(xpdlv).toString().trim().equals(CURRENT_XPDL_VERSION)) {
            Map les = getLogicErrors(pkg);
            les.put(phdr, XMLUtil
                    .getLanguageDependentString("ErrorInvalidXPDLVersion"));
            return false;
        } else {
            return true;
        }
    }

    public boolean checkRedefinableHeader(XMLComplexElement pkgOrWp,
                                          boolean fullCheck) {
        boolean isValid = true;
        RedefinableHeader rh = (RedefinableHeader) pkg.get("RedefinableHeader");
        Iterator rspns = ((Responsibles) rh.get("Responsibles")).toCollection()
                .iterator();
        while (rspns.hasNext()) {
            Responsible r = (Responsible) rspns.next();
            if (!(r.toValue() instanceof Participant)) {
                isValid = false;
                Map les = getLogicErrors(pkgOrWp);
                les
                        .put(
                                rh,
                                XMLUtil
                                        .getLanguageDependentString("ErrorOneOrMoreResponsibleParticipantsDoNotExist"));
                break;
            }
        }
        return isValid;
    }

    public boolean checkConformanceClass(boolean fullCheck) {
        // ConformanceClass cc=(ConformanceClass)pkg.get("ConformanceClass");
        return true;
    }

    public boolean checkScript(boolean fullCheck) {
        //
        return true;
    }

    public boolean checkExternalPackages(boolean fullCheck) {
        boolean isValid = true;
        Map les = getLogicErrors(pkg);
        Iterator it = pkg.getAllExternalPackages().iterator();
        while (it.hasNext() && (fullCheck || isValid)) {
            Package p = (Package) it.next();
            PackageValidator pv = new PackageValidator(p, false, true,
                    getExistingSchemaValidationErrors, false);
            if (!pv.validateAll(false)) {
                isValid = false;
                if (les != null) {
                    les
                            .put(
                                    p,
                                    XMLUtil
                                            .getLanguageDependentString("ErrorInvalidExternalPackage"));
                }
            }
        }
        return isValid;
    }

    public boolean checkCollection(String colName, XMLComplexElement cOwner,
                                   boolean fullCheck) {
        boolean isValid = true;
        Iterator it = ((XMLCollection) cOwner.get(colName)).toCollection()
                .iterator();
        while (it.hasNext() && (fullCheck || isValid)) {
            isValid = checkCollectionElement((XMLCollectionElement) it.next(),
                    fullCheck)
                    && isValid;
        }
        return isValid;
    }

    public boolean checkCollectionElement(XMLCollectionElement ce,
                                          boolean fullCheck) {
        boolean isValid = true;
        if (!isIdValid(ce.getID())) {
            isValid = false;
            XMLComplexElement firstOwner = ce.getCollection().getOwner();
            Map les;
            if (firstOwner instanceof Application) {
                les = getLogicErrors(((Application) firstOwner).getCollection()
                        .getOwner());
            } else {
                les = getLogicErrors(firstOwner);
            }
            les.put(ce, XMLUtil.getLanguageDependentString("ErrorIdIsNotValid"));
        }
        if (fullCheck || isValid) {
            if (!isUniqueId(ce.getCollection(), ce.getID())) {
                isValid = false;
                XMLComplexElement firstOwner = ce.getCollection().getOwner();
                Map les;
                if (firstOwner instanceof Application) {
                    les = getLogicErrors(((Application) firstOwner)
                            .getCollection().getOwner());
                } else {
                    les = getLogicErrors(firstOwner);
                }
                String msg = (String) les.get(ce);
                msg = prepeareMessageString(msg);
                msg = msg
                        + XMLUtil.getLanguageDependentString("ErrorIdIsNotUnique");
                les.put(ce, msg);
            }
        }
        if (fullCheck || isValid) {
            if (ce instanceof TypeDeclaration) {
                isValid = checkTypeDeclaration((TypeDeclaration) ce, fullCheck)
                        && isValid;
            } else if (ce instanceof Participant) {
                isValid = checkParticipant((Participant) ce, fullCheck)
                        && isValid;
            } else if (ce instanceof Application) {
                isValid = checkApplication((Application) ce, fullCheck)
                        && isValid;
            } else if (ce instanceof DataField) {
                isValid = checkDataField((DataField) ce, fullCheck) && isValid;
            } else if (ce instanceof FormalParameter) {
                isValid = checkFormalParameter((FormalParameter) ce, fullCheck)
                        && isValid;
            } else if (ce instanceof WorkflowProcess) {
                isValid = checkWorkflowProcess((WorkflowProcess) ce,
                        !checkingForbpd)
                        && isValid;
            } else if (ce instanceof ActivitySet) {
                isValid = checkActivitySet((ActivitySet) ce, fullCheck)
                        && isValid;
            } else if (ce instanceof Activity) {
                isValid = checkActivity((Activity) ce, fullCheck) && isValid;
            } else if (ce instanceof Transition) {
                isValid = checkTransition((Transition) ce, fullCheck)
                        && isValid;
            }
        }
        return isValid;
    }

    public boolean checkTypeDeclaration(TypeDeclaration td, boolean fullCheck) {
        //
        return true;
    }

    public boolean checkParticipant(Participant p, boolean fullCheck) {
        //
        return true;
    }

    public boolean checkApplication(Application app, boolean fullCheck) {
        boolean isValid = true;
        if (((XMLComplexChoice) app.get("Choice")).getChoosen() instanceof FormalParameters) {
            FormalParameters fps = (FormalParameters) ((XMLComplexChoice) app
                    .get("Choice")).getChoices()[0];
            Iterator it = fps.toCollection().iterator();
            while (it.hasNext() && (fullCheck || isValid)) {
                isValid = checkCollectionElement((XMLCollectionElement) it
                        .next(), fullCheck)
                        && isValid;
            }
        }
        return isValid;
    }

    public boolean checkDataField(DataField df, boolean fullCheck) {
        //
        return checkDataType(df, fullCheck);
    }

    public boolean checkFormalParameter(FormalParameter fp, boolean fullCheck) {
        //
        return checkDataType(fp, fullCheck);
    }

    public boolean checkDataType(XMLCollectionElement dfOrFp, boolean fullCheck) {
        boolean isValid = true;

        DataType xpdlType = (DataType) dfOrFp.get("DataType");
        Object type = xpdlType.get("Type").toValue();
        if (type instanceof DeclaredType) {
            TypeDeclarations tds = (TypeDeclarations) pkg
                    .get("TypeDeclarations");
            TypeDeclaration td = (TypeDeclaration) tds
                    .getDeclaredType(((DeclaredType) type).get("Id").toString());
            if (td == null) {
                isValid = false;
                XMLComplexElement firstOwner = dfOrFp.getCollection()
                        .getOwner();
                Map les;
                if (dfOrFp instanceof DataField) {
                    les = getLogicErrors(firstOwner);
                } else {
                    if (firstOwner instanceof Application) {
                        les = getLogicErrors(((Application) firstOwner)
                                .getCollection().getOwner());
                    } else {
                        les = getLogicErrors(firstOwner);
                    }
                }
                String msg = (String) les.get(dfOrFp);
                msg = prepeareMessageString(msg);
                msg = msg
                        + XMLUtil
                        .getLanguageDependentString("ErrorNonExistingDeclaredTypeReference");
                les.put(dfOrFp, msg);
            }
        }
        return isValid;
    }

    public boolean checkWorkflowProcess(WorkflowProcess wp, boolean fullCheck) {
        Map les = new LinkedHashMap();
        logicErrors.put(wp, les);
        basicLogicErrors.remove(wp);

        boolean notDefined = false;
        boolean isValid = checkProcessHeader(wp, fullCheck);

        //boolean isValid = true;
        /*
         * if (fullCheck || isValid) { isValid = checkRedefinableHeader(wp,
         * fullCheck) && isValid; }
         */
        /*
         * if (fullCheck || isValid) {
         * isValid=checkCollection("Participants",wp,fullCheck) && isValid; }
         */
        if (fullCheck || isValid) {
            isValid = checkCollection("ActivitySets", wp, fullCheck) && isValid;
        }
        if (fullCheck || isValid) {
            if (((Activities) wp.get("Activities")).toCollection().size() == 0) {
                isValid = false;
                notDefined = true;
                les
                        .put(
                                wp,
                                XMLUtil
                                        .getLanguageDependentString("ErrorProcessIsNotDefined"));
            } else {
                isValid = checkCollection("Activities", wp, fullCheck)
                        && isValid;
            }
        }
        if (fullCheck || isValid) {
            isValid = checkCollection("Transitions", wp, fullCheck) && isValid;
        }
        if (!isValid) {
            basicLogicErrors.put(wp, les.values().toArray()[0]);
            Map pkgles = getLogicErrors(pkg);
            if (pkgles != null) {
                if (notDefined) {
                    pkgles
                            .put(
                                    wp,
                                    XMLUtil
                                            .getLanguageDependentString("ErrorProcessIsNotDefined"));
                } else {
                    pkgles
                            .put(
                                    wp,
                                    XMLUtil
                                            .getLanguageDependentString("ErrorProcessContainsOneOrMoreLogicErrors"));
                }
            }
        }
        return isValid;
    }

    public boolean checkProcessHeader(WorkflowProcess wp, boolean fullCheck) {
        //
        return true;
    }

    public boolean checkActivitySet(ActivitySet as, boolean fullCheck) {
        Map les = new LinkedHashMap();
        logicErrors.put(as, les);
        basicLogicErrors.remove(as);
        boolean isValid = true;
        boolean notDefined = false;
        if (((Activities) as.get("Activities")).toCollection().size() == 0) {
            isValid = false;
            notDefined = true;
            les
                    .put(
                            as,
                            XMLUtil
                                    .getLanguageDependentString("ErrorBlockActivityIsNotDefined"));
        } else {
            isValid = checkCollection("Activities", as, fullCheck);
        }
        if (fullCheck || isValid) {
            isValid = checkCollection("Transitions", as, fullCheck) && isValid;
        }
        if (!isValid) {
            basicLogicErrors.put(as, getLogicErrors(as).values().toArray()[0]);
            Map wples = getLogicErrors(as.getOwnerProcess());
            Activity blockActivity = findBlockActivity(as);
            if (!(wples == null || blockActivity == null)) {
                if (notDefined) {
                    wples
                            .put(
                                    blockActivity,
                                    XMLUtil
                                            .getLanguageDependentString("ErrorBlockActivityIsNotDefined"));
                } else {
                    wples
                            .put(
                                    blockActivity,
                                    XMLUtil
                                            .getLanguageDependentString("ErrorInnerLogicError"));
                }
            } else if (wples != null) {
                wples
                        .put(
                                as,
                                XMLUtil
                                        .getLanguageDependentString("ErrorBlockActivityIsNotDefined"));
            }
        }
        return isValid;
    }

    public boolean checkActivity(Activity act, boolean fullCheck) {
        // check performer
        boolean isValid = checkActivityPerformer(act, fullCheck);

        if (!(fullCheck || isValid)) {
            return false;
        }

        // if this is a block activity
        ActivityDefTypeEnums type = act.getType();
        switch (type) {
            case Tool: // AutoActivity type
                isValid = checkActivityTools(act, fullCheck) && isValid;
            case No: // No type
                break;
            case SubFlow: // SubFlow type
                isValid = checkActivitySubFlow(act, fullCheck) && isValid;
                break;
            case Block: // Block type
                isValid = checkActivityBlock(act, fullCheck) && isValid;
                break;
            case Service: // Block type
                isValid = checkActivityServices(act, fullCheck) && isValid;
                break;
            case Device: // Block type
                isValid = checkActivityDevices(act, fullCheck) && isValid;
                break;
            case Event: // Block type
                isValid = checkActivityEvents(act, fullCheck) && isValid;
                break;
        }


        if (!(fullCheck || isValid)) {
            return false;
        }

        Transitions trans = (Transitions) act.getCollection().getOwner().get(
                "Transitions");
        Set outTrans = trans.getTransitions(act.getID(), -1);
        Set inTrans = trans.getTransitions(act.getID(), 1);

        // check deadlines
        // isValid = checkActivityDeadlines(act, fullCheck) && isValid;
        if (!(fullCheck || isValid)) {
            return false;
        }

        Map les = getLogicErrors(act.getCollection().getOwner());
        String msg = (String) les.get(act);
        // Split type and no. of outgoing transitions
        Split split = act.getSplit();
        Join join = act.getJoin();
        if (!(fullCheck || isValid)) {
            return false;
        }

        if (checkingBybpd) {
            // TransitionRefs size must be the same as the one of outgoing
            // transitions
            TransitionRefs splittRfs = (TransitionRefs) split.getRefTransitionRefs();
            if (splittRfs.size() < 2
                    && split.getType().equals("AND")) {
                isValid = false;
                msg = prepeareMessageString(msg);
                msg = msg
                        + XMLUtil
                        .getLanguageDependentString("ErrorNumberOfActivitiesIngoingTransitionsOnlyOneAndTransitionRefsIsSame");
                les.put(act, msg);
            }
            if (!(fullCheck || isValid)) {
                return false;
            }
            // TransitionRefs must refer to valid transitions.
            Iterator tRefs = splittRfs.toCollection().iterator();
            boolean invalidTref = false;
            while (tRefs.hasNext()) {
                String transitionId = tRefs.next().toString();
                Transition t = trans.getTransition(transitionId);
                if (t == null || !outTrans.contains(t)) {
                    isValid = false;
                    invalidTref = true;
                }
            }

            if (invalidTref) {
                msg = prepeareMessageString(msg);
                msg = msg
                        + XMLUtil
                        .getLanguageDependentString("ErrorTransitionRefIsNotValid");
                les.put(act, msg);
            }
            //TransitionRefs jointRfs = (TransitionRefs) join.get("TransitionRefs");

            if (inTrans.size() < 2
                    && join.getType().equals("AND")) {

                isValid = false;
                msg = prepeareMessageString(msg);
                msg = msg
                        + XMLUtil
                        .getLanguageDependentString("ErrorNumberOfActivitiesOutGoingTransitionsOnlyOneAndTransitionRefsIsSame");
                les.put(act, msg);

            }


            if (!(fullCheck || isValid)) {
                return false;
            }

        }

        // Join type and no. of incoming transitions


        if (!(fullCheck || isValid)) {
            return false;
        }

        isValid = checkMultipleOtherwiseOrDefaultExceptionTransitions(act,
                fullCheck)
                && isValid;

        return isValid;
    }

    public boolean checkActivityPerformer(Activity act, boolean fullCheck) {
        boolean isValid = true;

        // check performer
        if (act.getSubflow() == null && act.getBlockActivity() == null && act.getOutflow() == null) {
            Object performer = act.get("Performer").toValue();
            // if this is a block activity
            ActivityDefTypeEnums type = act.getType();
            if (!type.equals(ActivityDefTypeEnums.SubFlow)
                    && !type.equals(ActivityDefTypeEnums.Block)
                    && !type.equals(ActivityDefTypeEnums.No)
                    && !type.equals(ActivityDefTypeEnums.OutFlow)
                    && ((performer instanceof Participant) || performer.toString()
                    .trim().length() > 0)) {
                isValid = false;
                Map les = getLogicErrors(act.getCollection().getOwner());
                String msg = (String) les.get(act);
                msg = prepeareMessageString(msg);
                msg = msg
                        + XMLUtil
                        .getLanguageDependentString("ErrorActivityCannotHavePerformer");
                les.put(act, msg);
            }
        }
        return isValid;

    }

    public boolean checkActivityTools(Activity act, boolean fullCheck) {
        boolean isValid = true;
        boolean nonExistingToolReference = false;

        Tools tools = act.getTools();
        if (tools != null) {
            Iterator it = tools.toCollection().iterator();
            while (it.hasNext() && (fullCheck || isValid)) {
                Tool tool = (Tool) it.next();
//				XMLComplexChoice apps = (XMLComplexChoice) tool
//						.get("Application");
//				Object choosenApp = apps.getChoosen();
//				String toolID = null;
//				if (choosenApp != null && choosenApp instanceof Application) {
//					toolID = ((Application) choosenApp).getID();
//				}
//				if (toolID == null) {
//					isValid = false;
//					nonExistingToolReference = true;
//				}
//				if (!(isValid || fullCheck))
//					break;
//				try {
//					isValid = checkParameterMappings(tool,
//							(Application) choosenApp, fullCheck)
//							&& isValid;
//				} catch (Exception ex) {
//				}
            }
        }

        if (!isValid) {
            Map les = getLogicErrors(act.getCollection().getOwner());
            String msg = (String) les.get(act);
            msg = prepeareMessageString(msg);
            if (nonExistingToolReference) {
                msg += XMLUtil
                        .getLanguageDependentString("ErrorNonExistingToolReference");
            } else {
                msg += XMLUtil
                        .getLanguageDependentString("ErrorToolsFormalAndActualParametersDoNotMatch");
            }
            les.put(act, msg);
        }

        return isValid;
    }


    public boolean checkActivityServices(Activity act, boolean fullCheck) {
        boolean isValid = true;
        boolean nonExistingToolReference = false;

        Services services = act.getServices();
        if (services != null) {
            Iterator it = services.toCollection().iterator();
            while (it.hasNext() && (fullCheck || isValid)) {
                Service tool = (Service) it.next();
//				XMLComplexChoice apps = (XMLComplexChoice) tool
//						.get("Application");
//				Object choosenApp = apps.getChoosen();
//				String toolID = null;
//				if (choosenApp != null && choosenApp instanceof Application) {
//					toolID = ((Application) choosenApp).getID();
//				}
//				if (toolID == null) {
//					isValid = false;
//					nonExistingToolReference = true;
//				}
//				if (!(isValid || fullCheck))
//					break;
//				try {
//					isValid = checkParameterMappings(tool,
//							(Application) choosenApp, fullCheck)
//							&& isValid;
//				} catch (Exception ex) {
//				}
            }
        }

        if (!isValid) {
            Map les = getLogicErrors(act.getCollection().getOwner());
            String msg = (String) les.get(act);
            msg = prepeareMessageString(msg);
            if (nonExistingToolReference) {
                msg += XMLUtil
                        .getLanguageDependentString("ErrorNonExistingToolReference");
            } else {
                msg += XMLUtil
                        .getLanguageDependentString("ErrorToolsFormalAndActualParametersDoNotMatch");
            }
            les.put(act, msg);
        }

        return isValid;
    }

    public boolean checkActivityDevices(Activity act, boolean fullCheck) {
        boolean isValid = true;
        boolean nonExistingToolReference = false;

        Devices devices = act.getDevices();
        if (devices != null) {
            Iterator it = devices.toCollection().iterator();
            while (it.hasNext() && (fullCheck || isValid)) {
                Device device = (Device) it.next();
//				XMLComplexChoice apps = (XMLComplexChoice) device
//						.get("Application");
//				Object choosenApp = apps.getChoosen();
//				String toolID = null;
//				if (choosenApp != null && choosenApp instanceof Application) {
//					toolID = ((Application) choosenApp).getID();
//				}
//				if (toolID == null) {
//					isValid = false;
//					nonExistingToolReference = true;
//				}
//				if (!(isValid || fullCheck))
//					break;
//				try {
//					isValid = checkParameterMappings(device,
//							(Application) choosenApp, fullCheck)
//							&& isValid;
//				} catch (Exception ex) {
//				}
            }
        }

        if (!isValid) {
            Map les = getLogicErrors(act.getCollection().getOwner());
            String msg = (String) les.get(act);
            msg = prepeareMessageString(msg);
            if (nonExistingToolReference) {
                msg += XMLUtil
                        .getLanguageDependentString("ErrorNonExistingToolReference");
            } else {
                msg += XMLUtil
                        .getLanguageDependentString("ErrorToolsFormalAndActualParametersDoNotMatch");
            }
            les.put(act, msg);
        }

        return isValid;
    }


    public boolean checkActivityEvents(Activity act, boolean fullCheck) {
        boolean isValid = true;
        boolean nonExistingToolReference = false;

        Events events = act.getEvents();
        if (events != null) {
            Iterator it = events.toCollection().iterator();
            while (it.hasNext() && (fullCheck || isValid)) {
                Event event = (Event) it.next();
//				XMLComplexChoice apps = (XMLComplexChoice) device
//						.get("Application");
//				Object choosenApp = apps.getChoosen();
//				String toolID = null;
//				if (choosenApp != null && choosenApp instanceof Application) {
//					toolID = ((Application) choosenApp).getID();
//				}
//				if (toolID == null) {
//					isValid = false;
//					nonExistingToolReference = true;
//				}
//				if (!(isValid || fullCheck))
//					break;
//				try {
//					isValid = checkParameterMappings(device,
//							(Application) choosenApp, fullCheck)
//							&& isValid;
//				} catch (Exception ex) {
//				}
            }
        }

        if (!isValid) {
            Map les = getLogicErrors(act.getCollection().getOwner());
            String msg = (String) les.get(act);
            msg = prepeareMessageString(msg);
            if (nonExistingToolReference) {
                msg += XMLUtil
                        .getLanguageDependentString("ErrorNonExistingToolReference");
            } else {
                msg += XMLUtil
                        .getLanguageDependentString("ErrorToolsFormalAndActualParametersDoNotMatch");
            }
            les.put(act, msg);
        }

        return isValid;
    }


    public boolean checkActivitySubFlow(Activity act, boolean fullCheck) {
        boolean isValid = true;
        boolean nonExistingProcessReference = false;
        boolean notAllowedProcessReference = false;
        boolean processReferenceInnerErr = false;
        SubFlow s = act.getSubflow();
        if (s != null) {
            XMLComplexChoice wp = (XMLComplexChoice) s.get("WorkflowProcess");

            String subflowID = s.getAttrId();
            //null;
//			Object choosenWorkflow = wp.getChoosen();
//			if (choosenWorkflow != null
//					&& choosenWorkflow instanceof WorkflowProcess) {
//				WorkflowProcess subflowWorkflow=(WorkflowProcess) choosenWorkflow;
//				subflowID = subflowWorkflow.getID();
//				
//				if (!this.checkWorkflowProcess(subflowWorkflow,true)
//						||!this.checkGraphConnections(subflowWorkflow,false)){
//					//subflowWorkflow.getActivities().
//					processReferenceInnerErr=true;
//					isValid = false;
//				}
//				
//			}
            if (subflowID == null) {
                isValid = false;
                nonExistingProcessReference = true;
            } else if (subflowID == act.getOwnerProcess().getID()) {
                isValid = false;
                notAllowedProcessReference = true;
            }
			
			/*
			if (fullCheck || isValid) {
				try {
					isValid = checkParameterMappings(s,
							(WorkflowProcess) choosenWorkflow, fullCheck)
							&& isValid;
				} catch (Exception ex) {
				}
			}*/

        }
        if (!isValid) {
            Map les = getLogicErrors(act.getCollection().getOwner());
            String msg = (String) les.get(act);
            msg = prepeareMessageString(msg);
            if (nonExistingProcessReference) {
                msg = msg
                        + XMLUtil
                        .getLanguageDependentString("ErrorNonExistingProcessReference");
            } else if (nonExistingProcessReference) {
                msg = msg
                        + XMLUtil
                        .getLanguageDependentString("ErrorNotAllowedProcessReference");
            } else if (processReferenceInnerErr) {
                msg = msg
                        + XMLUtil
                        .getLanguageDependentString("ErrorProcessReferenceInnerErr");
            } else {
                msg = msg
                        + XMLUtil
                        .getLanguageDependentString("ErrorSubFlowFormalAndActualParametersDoNotMatch");
            }
            les.put(act, msg);
        }
        return isValid;
    }

    public boolean checkActivityBlock(Activity act, boolean fullCheck) {
        boolean isValid = true;
        BlockActivity blk = act.getBlockActivity();
        String blockId = blk.get("Id").toString();
        // check if the activity set exists
        ActivitySets ass = (ActivitySets) act.getOwnerProcess().get(
                "ActivitySets");
        ActivitySet as = ass.getActivitySet(blockId);
        // check if there is activity set with the referenced id and if
        // block activity is inside other block activity, and references it's
        // owner
        if (as == null || act.getCollection().getOwner().equals(as)) {
            isValid = false;
            Map les = getLogicErrors(act.getCollection().getOwner());
            String msg = (String) les.get(act);
            msg = prepeareMessageString(msg);
            if (as == null) {
                msg = msg
                        + XMLUtil
                        .getLanguageDependentString("ErrorNonExistingActivitySetReference");
            } else {
                msg = msg
                        + XMLUtil
                        .getLanguageDependentString("ErrorNotAllowedActivitySetReference");
            }
            les.put(act, msg);
        }
        return isValid;
    }

    public boolean checkActivityDeadlines(Activity act, boolean fullCheck) {
        boolean isValid = true;

        Transitions trans = (Transitions) act.getCollection().getOwner().get(
                "Transitions");
        Set outTrans = trans.getTransitions(act.getID(), -1);

        if (act.get("Deadlines") == null) {

        }

        Iterator dls = ((Deadlines) act.get("Deadlines")).toCollection()
                .iterator();
        Set deadlineExceptions = new HashSet();
        int syncCount = 0;
        while (dls.hasNext()) {
            Deadline dl = (Deadline) dls.next();
            XMLElement dc = dl.get("DeadlineCondition");
            XMLElement en = dl.get("ExceptionName");
            deadlineExceptions.add(en.toString().trim());
            if (dl.get("Execution").toValue().toString().equals("SYNCHR")) {
                syncCount++;
            }
        }

        Map les = getLogicErrors(act.getCollection().getOwner());
        String msg = (String) les.get(act);
        if (syncCount > 1) {
            isValid = false;
            msg = prepeareMessageString(msg);
            msg += XMLUtil
                    .getLanguageDependentString("ErrorActivityCanHaveOnlyOneSynchronousDeadline");
            les.put(act, msg);
        }
        if (!(fullCheck || isValid)) {
            return false;
        }
        /*
         * Iterator it=outTrans.iterator(); while (it.hasNext()) { Transition
         * t=(Transition)it.next(); Condition c=(Condition)t.get("Condition");
         * String ct=((Condition)t.get("Condition")).get("Type").
         * toValue().toString(); if (ct.equals("DEFAULTEXCEPTION")) {
         * deadlineExceptions.clear(); break; } else if (ct.equals("EXCEPTION")) {
         * deadlineExceptions.remove(c.toString().trim()); } }
         */
        if (deadlineExceptions.size() > 0) {
            isValid = false;
            msg = prepeareMessageString(msg);
            msg += XMLUtil
                    .getLanguageDependentString("ErrorDeadlineExceptionIsNotHandledByAnyOutgoingTransitionWithExceptionOrDefaultExceptionConditionType");
            les.put(act, msg);
        }
        return isValid;
    }

    public boolean checkMultipleOtherwiseOrDefaultExceptionTransitions(
            Activity act, boolean fullCheck) {
        Transitions trans = (Transitions) act.getCollection().getOwner().get(
                "Transitions");
        Set outTrans = trans.getTransitions(act.getID(), -1);
        // Check outgoing transitions
        // do not allow more then 1 transitions of type otherwise or
        // default_exception
        boolean foundOtherwise = false;
        boolean foundMultipleOtherwise = false;
        boolean foundDefaultException = false;
        boolean foundMultipleDefaultException = false;
        /*
         * Iterator ts=outTrans.iterator(); while (ts.hasNext()) { Transition
         * t=(Transition)ts.next(); String
         * ct=((Condition)t.get("Condition")).get("Type"). toValue().toString();
         * if (ct.equals("OTHERWISE")) { if (foundOtherwise) {
         * foundMultipleOtherwise=true; if (foundMultipleDefaultException ||
         * !fullCheck) break; } else { foundOtherwise=true; } } else if
         * (ct.equals("DEFAULTEXCEPTION")) { if (foundDefaultException) {
         * foundMultipleDefaultException=true; if (foundMultipleOtherwise ||
         * !fullCheck) break; } else { foundDefaultException=true; } } }
         */

        if (foundMultipleOtherwise || foundMultipleDefaultException) {
            Map les = getLogicErrors(act.getCollection().getOwner());
            String msg = (String) les.get(act);
            msg = prepeareMessageString(msg);
            if (foundMultipleDefaultException && foundMultipleOtherwise) {
                msg = msg
                        + XMLUtil
                        .getLanguageDependentString("ErrorMoreThenOneOTHERWISEAndDEFAULTEXCEPTIONTypeOutgoingTransition");
            } else if (foundMultipleOtherwise) {
                msg = msg
                        + XMLUtil
                        .getLanguageDependentString("ErrorMoreThenOneOTHERWISETypeOutgoingTransition");
            } else if (foundMultipleDefaultException) {
                msg = msg
                        + XMLUtil
                        .getLanguageDependentString("ErrorMoreThenOneDEFAULTEXCEPTIONTypeOutgoingTransition");
            }
            les.put(act, msg);
            return false;
        } else {
            return true;
        }
    }

    public boolean checkParameterMappings(XMLComplexElement toolOrSbflw,
                                          XMLComplexElement appOrWp, boolean fullCheck) {
        FormalParameters fps;
        if (appOrWp instanceof WorkflowProcess) {
            fps = (FormalParameters) appOrWp.get("FormalParameters");
        } else {
            if (((XMLComplexChoice) appOrWp.get("Choice")).getChoosen() instanceof FormalParameters) {
                fps = (FormalParameters) ((XMLComplexChoice) appOrWp
                        .get("Choice")).getChoices()[0];
                // do not check if application is externally defined
            } else {
                return true;
            }
        }
        ActualParameters aps = (ActualParameters) toolOrSbflw
                .get("ActualParameters");
        int pm = XMLUtil.checkParameterMatching(fps, aps);
        if (pm != 0) {
            return false;
        } else {
            return true;
        }
    }

    public boolean checkTransition(Transition transition, boolean fullCheck) {
        boolean isValid = true;
        Map les = getLogicErrors(transition.getCollection().getOwner());
        if (les == null) {
            return true;
        }
        String msg = (String) les.get(transition);

        if (transition.getFrom() == null) {
            isValid = false;
            msg = prepeareMessageString(msg);
            msg += XMLUtil
                    .getLanguageDependentString("ErrorNonExistingFromActivityReference");
        }
        if (transition.getTo() == null) {
            isValid = false;
            msg = prepeareMessageString(msg);
            msg += XMLUtil
                    .getLanguageDependentString("ErrorNonExistingToActivityReference");
        }
        if (!isValid) {
            les.put(transition, msg);
        }
        return isValid;
    }

    public boolean isIdValid(String id) {
        return XMLCollection.isIdValid(id);
    }

    public static boolean isEmpty(String str) {
        if (str == null || str.trim().length() == 0) {
            return true;
        } else {
            return false;
        }
    }

    public static boolean isUniqueId(XMLCollection xmlCol, String id) {
        int idCnt = 0;
        Iterator it = xmlCol.toCollection().iterator();
        while (it.hasNext()) {
            try {
                XMLCollectionElement xmlce = (XMLCollectionElement) it.next();
                String cId = xmlce.getID();
                if (cId.equals(id)) {
                    idCnt++;
                    if (idCnt > 1) {
                        return false;
                    }
                }
            } catch (ClassCastException cce) {
                return true;
            }
        }
        return true;
    }

    public static void main(String[] args) {
		/*
		try {
			Package pkg = pkg1.getXMLInterface().parseDocument(args[0], true);
			PackageValidator validator = new PackageValidator(pkg, false, true,
					false, true);
			if (validator.validateAll(true)) {
				System.out.println(args[0] + " is a valid XPDL package");
			} else {
				System.out.println(args[0] + " is not a valid XPDL package");
			}
		} catch (Exception ex) {
			ex.printStackTrace();
			System.exit(1);
		}
		*/
    }

    /**
     * Used for debug only
     */
    public static void printIM(boolean[][] im, List acts) {
        if (im != null) {
            for (int i = 0; i < im.length; i++) {
                for (int j = 0; j < im[i].length; j++) {
                    System.out.print(acts.get(i) + "->" + acts.get(j) + "="
                            + im[i][j] + " ");
                }
                System.out.println();
            }
        } else {
            System.out.println("Passed array is null !!!");
        }
    }

    /**
     * Used for debug only
     */
    public static void printIM2(boolean[][] im, List acts) {
        System.out.println("Activities are" + acts);
        if (im != null) {
            for (int i = 0; i < im.length; i++) {
                for (int j = 0; j < im[i].length; j++) {
                    System.out.print(((im[i][j]) ? "1" : "0") + " ");
                }
                System.out.println();
            }
        } else {
            System.out.println("Passed array is null !!!");
        }
    }

    // ************************** GRAPH CONFORMANCE CHECKING
    // ****************************
    public boolean checkGraphConformance(boolean fullCheck) {
        boolean areGraphsConformant = true;

        Map graphConformanceErrors = new LinkedHashMap();
        List basicGraphConformanceErrors = new ArrayList();

        Iterator procs = ((WorkflowProcesses) pkg.get("WorkflowProcesses"))
                .toCollection().iterator();
        while (procs.hasNext()) {
            WorkflowProcess wp = (WorkflowProcess) procs.next();
            if (!checkGraphConformance(wp, fullCheck)) {
                areGraphsConformant = false;
                if (!fullCheck) {
                    break;
                }
                String msg = "";
                Iterator bces = getBasicGraphConformanceErrors(wp).iterator();
                while (bces.hasNext()) {
                    msg = msg + bces.next().toString() + "<br>";
                }
                graphConformanceErrors.put(wp, msg);
            }
        }
        if (!areGraphsConformant) {
            basicGraphConformanceErrors
                    .add(XMLUtil
                            .getLanguageDependentString("ErrorOneOrMoreProcessesDoNotSatisfyGraphConformance"));
        }
        basicGraphsConformanceErrors.put(pkg, basicGraphConformanceErrors);
        graphsConformanceErrors.put(pkg, graphConformanceErrors);
        return areGraphsConformant;
    }

    /**
     * Checks if graph conforms to the given conformance class.
     *
     * @return true if graph is conformant, false otherwise
     */
    public boolean checkGraphConformance(XMLCollectionElement wpOrAs,
                                         boolean fullCheck) {
        Map graphConformanceErrors = new LinkedHashMap();
        List basicGraphConformanceErrors = new ArrayList();

        Collection allActs = ((XMLCollection) wpOrAs.get("Activities"))
                .toCollection();

        Package pkg;
        if (wpOrAs instanceof WorkflowProcess) {
            pkg = ((WorkflowProcess) wpOrAs).getPackage();
        } else {
            pkg = ((WorkflowProcess) wpOrAs.getCollection().getOwner())
                    .getPackage();
        }
        String conformanceClass = ((ConformanceClass) pkg
                .get("ConformanceClass")).get("GraphConformance").toValue()
                .toString();
        // ct=0->NON_BLOCKED, ct=1->LOOP_BLOCKED, ct=2->FULL_BLOCKED,
        // ct=-1->default NON_BLOCKED
        int ct = XMLUtil.getConformanceClassNo(conformanceClass);

        Activities acts = (Activities) wpOrAs.get("Activities");
        List activities = (List) acts.toCollection();

        if (activities.size() == 0) {
            graphsConformanceErrors.put(wpOrAs, graphConformanceErrors);
            basicGraphsConformanceErrors.put(wpOrAs,
                    basicGraphConformanceErrors);
            return true;
        }

        boolean isGraphConformant = true;

        Set splitActs = XMLUtil.getSplitOrJoinActivities(activities, 0);
        Set joinActs = XMLUtil.getSplitOrJoinActivities(activities, 1);

        Set noSplitActs = new HashSet(activities);
        noSplitActs.removeAll(splitActs);

        GraphChecker gc = null;
        if (ct > 0 && (isGraphConformant || fullCheck)) {
            boolean[][] incidenceMatrix = createIncidenceMatrix(activities);
            if (incidenceMatrix == null) {
                basicGraphConformanceErrors.add("Unexpected error");
                graphsConformanceErrors.put(wpOrAs, graphConformanceErrors);
                basicGraphsConformanceErrors.put(wpOrAs,
                        basicGraphConformanceErrors);
                return false;
            }

            gc = new GraphChecker(incidenceMatrix);

            // call method to check loop cycling
            /*
             * boolean loopError = false; if (fullCheck) { int[] loopNodes =
             * gc.getCyclicNodes(); if (loopNodes != null) { isGraphConformant =
             * false; loopError = true; for (int i = 0; i < loopNodes.length;
             * i++) { Activity act = (Activity) activities.get(loopNodes[i]);
             * graphConformanceErrors.put( act, XMLUtil.getLanguageDependentString(
             * "ErrorLoopContainedActivity")); } } } else { loopError =
             * gc.isGraphCyclic(); if (loopError) { isGraphConformant = false; } }
             * if (loopError) { basicGraphConformanceErrors.add(
             * XMLUtil.getLanguageDependentString("ErrorTheGraphIsCyclic")); }
             */
        }
        // Here we check FULL_BLOCK conformance
        if (ct == 2 && (isGraphConformant || fullCheck)) {
            // check if there is more then one starting activity
            if (getStartActivities(wpOrAs).size() > 1) {
                isGraphConformant = false;
                basicGraphConformanceErrors
                        .add(XMLUtil
                                .getLanguageDependentString("ErrorMultipleStartsAreNotAllowedInTheFullBlockedMode"));
            }
            /*
             * // check if there is more then one ending activity if
             * ((isGraphConformant || fullCheck) &&
             * getEndActivities(wpOrAs).size() > 1) { isGraphConformant = false;
             * basicGraphConformanceErrors.add( XMLUtil.getLanguageDependentString(
             * "ErrorMultipleEndsAreNotAllowedInTheFullBlockedMode")); }
             */
            // check if the number of splits and joins matches
            boolean smerr = false;
            if ((isGraphConformant || fullCheck)
                    && splitActs.size() != joinActs.size()) {
                if (splitActs.size() > joinActs.size()) {
                    basicGraphConformanceErrors
                            .add(XMLUtil
                                    .getLanguageDependentString("ErrorTheNumberOfSplitsAndJoinsIsNotTheSame-MoreSplits"));
                } else {
                    basicGraphConformanceErrors
                            .add(XMLUtil
                                    .getLanguageDependentString("ErrorTheNumberOfSplitsAndJoinsIsNotTheSame-MoreJoins"));
                }
                isGraphConformant = false;
                smerr = true;
            }

            // check for split/join type mismatch
            if ((isGraphConformant || fullCheck) && !smerr) {
                if (getNoOfANDSplitsOrJoins(splitActs, 0) != getNoOfANDSplitsOrJoins(
                        joinActs, 1)) {
                    basicGraphConformanceErrors
                            .add(XMLUtil
                                    .getLanguageDependentString("ErrorOneOrMoreSplitsDoNotHaveCorrespondingJoinBecauseOfTypeMismatch"));
                    isGraphConformant = false;
                }
            }
            // first check for correct outgoing transitions
            if (isGraphConformant || fullCheck) {
                Iterator it = splitActs.iterator();
                boolean andSplitError = false;
                boolean xorSplitError = false;
                while (it.hasNext()) {
                    Activity act = (Activity) it.next();
                    if (XMLUtil.isANDTypeSplitOrJoin(act, 0)) {
                        if (!checkANDSplit(act)) {
                            isGraphConformant = false;
                            andSplitError = true;
                            String msg = (String) graphConformanceErrors
                                    .get(act);
                            msg = prepeareMessageString(msg);
                            msg = msg
                                    + XMLUtil
                                    .getLanguageDependentString("ErrorOneOrMoreConditionalOutgoingTransitions");
                            graphConformanceErrors.put(act, msg);
                            if (!fullCheck) {
                                break;
                            }
                        }
                    } else {
                        if (!checkXORSplit(act)) {
                            isGraphConformant = false;
                            xorSplitError = true;
                            String msg = (String) graphConformanceErrors
                                    .get(act);
                            msg = prepeareMessageString(msg);
                            msg = msg
                                    + XMLUtil
                                    .getLanguageDependentString("ErrorMissingOTHERWISETypeOutgoingTransition");
                            graphConformanceErrors.put(act, msg);
                            if (!fullCheck) {
                                break;
                            }
                        }
                    }
                }

                // check activities that has only one outgoing transition, if
                // there is condition on it -> report XOR split with conditional
                // transition error
                it = noSplitActs.iterator();
                while (it.hasNext()) {
                    Activity act = (Activity) it.next();
                    if (!checkXORSplit(act)) {
                        isGraphConformant = false;
                        xorSplitError = true;
                        String msg = (String) graphConformanceErrors.get(act);
                        msg = prepeareMessageString(msg);
                        msg = msg
                                + XMLUtil
                                .getLanguageDependentString("ErrorMissingOTHERWISETypeOutgoingTransition");
                        graphConformanceErrors.put(act, msg);
                        if (!fullCheck) {
                            break;
                        }
                    }
                }

                if (andSplitError) {
                    basicGraphConformanceErrors
                            .add(XMLUtil
                                    .getLanguageDependentString("ErrorOneOrMoreANDSplitsHaveConditionalOutgoingTransitions"));
                }
                if (xorSplitError) {
                    basicGraphConformanceErrors
                            .add(XMLUtil
                                    .getLanguageDependentString("ErrorOneOrMoreXORSplitsWithConditionalTransitionsDoNotHaveOTHERWISETransition"));
                }
            }

            // now perform search on every split activity for corresponding join
            // activity
            if (isGraphConformant || fullCheck) {
                boolean noCorrespondingJoinError = false;
                Iterator it = splitActs.iterator();
                while (it.hasNext()) {
                    Activity act = (Activity) it.next();
                    int splitIndex = activities.indexOf(act);
                    if (splitIndex == -1) {
                        basicGraphConformanceErrors.add("Unexpected error");
                        isGraphConformant = false;
                        if (!fullCheck) {
                            break;
                        } else {
                            continue;
                        }
                    }
                    int ji = gc.getJoinIndex(splitIndex);
                    // The correspondin join can't be found
                    if (ji < 0) {
                        isGraphConformant = false;
                        noCorrespondingJoinError = true;
                        String msg = (String) graphConformanceErrors.get(act);
                        msg = prepeareMessageString(msg);
                        msg = msg
                                + XMLUtil
                                .getLanguageDependentString("ErrorThereIsNoCorrespondingJoinActivity");
                        graphConformanceErrors.put(act, msg);
                        if (!fullCheck) {
                            break;
                        }
                        // if the join is found and their types are different
                        // the graph is not conformant
                    } else {
                        if (XMLUtil.isANDTypeSplitOrJoin(act, 0) != XMLUtil
                                .isANDTypeSplitOrJoin((Activity) activities
                                        .get(ji), 1)) {
                            isGraphConformant = false;
                            noCorrespondingJoinError = true;
                            String msg = (String) graphConformanceErrors
                                    .get(act);
                            msg = prepeareMessageString(msg);
                            if (XMLUtil.isANDTypeSplitOrJoin((Activity) act, 0)) {
                                msg = msg
                                        + XMLUtil
                                        .getLanguageDependentString("ErrorTheCorrespondingJoinActivityDoesNotHaveTheSameType-ANDXOR");
                            } else {
                                msg = msg
                                        + XMLUtil
                                        .getLanguageDependentString("ErrorTheCorrespondingJoinActivityDoesNotHaveTheSameType-XORAND");
                            }
                            graphConformanceErrors.put(act, msg);
                            if (!fullCheck) {
                                break;
                            }
                        }
                    }
                }
                if (noCorrespondingJoinError) {
                    basicGraphConformanceErrors
                            .add(XMLUtil
                                    .getLanguageDependentString("ErrorOneOrMoreSplitsDoNotHaveCorrespondingJoin"));
                }
            }
        }

        // if so far the graph is conformant, or the full check is required,
        // check the graphs block activities
        if (isGraphConformant || fullCheck) {
            Set blockActivities = XMLUtil.getBlockActivities(wpOrAs, false);
            boolean innerConformanceError = false;
            Iterator it = blockActivities.iterator();
            while (it.hasNext()) {
                Activity act = (Activity) it.next();
                BlockActivity ba = act.getBlockActivity();
                ActivitySets ass = (ActivitySets) act.getOwnerProcess().get(
                        "ActivitySets");
                String asId = ba.get("Id").toString();
                ActivitySet as = ass.getActivitySet(asId);
                if (as != null && !checkGraphConformance(as, false)) {
                    isGraphConformant = false;
                    innerConformanceError = true;
                    String msg = (String) graphConformanceErrors.get(ba);
                    msg = prepeareMessageString(msg);
                    msg = msg
                            + XMLUtil
                            .getLanguageDependentString("ErrorInnerGraphConformanceError");
                    graphConformanceErrors.put(act, msg);
                    graphConformanceErrors.put(act, msg);
                    if (!fullCheck) {
                        break;
                    }
                }
            }
            if (innerConformanceError) {
                basicGraphConformanceErrors
                        .add(XMLUtil
                                .getLanguageDependentString("ErrorOneOrMoreBlockActivitiesAreNotValid"));
            }
        }

        graphsConformanceErrors.put(wpOrAs, graphConformanceErrors);
        basicGraphsConformanceErrors.put(wpOrAs, basicGraphConformanceErrors);
        return isGraphConformant;
    }

    protected boolean[][] createIncidenceMatrix(List activities) {
        int size = activities.size();
        boolean[][] incidenceMatrix = new boolean[size][size];
        Transitions ts = null;
        for (int indAct = 0; indAct < size; indAct++) {
            Activity a = (Activity) activities.get(indAct);
            if (ts == null) {
                ts = (Transitions) a.getCollection().getOwner().get(
                        "Transitions");
            }
            Set oas = new HashSet();
            Iterator trs = ts.getTransitions(a.getID(), -1).iterator();
            while (trs.hasNext()) {
                Transition t = (Transition) trs.next();
                if (t != null && t.getTo() != null) {
                    Object aOut = t.getTo();
                    if (aOut instanceof Activity) {
                        int indOut = activities.indexOf(aOut);
                        if (indOut == -1)
                            return null;
                        incidenceMatrix[indAct][indOut] = true;
                    }
                }
            }
        }
        return incidenceMatrix;
    }

    /**
     * Returns the number of activities in the given set that have split or
     * join, depending on second parameter.
     *
     * @param acts The set of activities that are searched for split or join
     * @param sOrJ 0 -> searching for split, otherwise, searching for join
     */
    protected int getNoOfANDSplitsOrJoins(Set acts, int sOrJ) {
        int no = 0;
        Iterator it = acts.iterator();
        while (it.hasNext()) {
            Activity act = (Activity) it.next();
            if (XMLUtil.isANDTypeSplitOrJoin(act, sOrJ)) {
                no++;
            }
        }
        return no;
    }

    protected boolean checkANDSplit(Activity act) {
        return !hasAnyPostcondition(act);
    }

    protected boolean checkXORSplit(Activity act) {
        // if activity has any postcondition, it must have an otherwise
        // transition
        if (hasAnyPostcondition(act)) {
            Transitions ts = (Transitions) act.getCollection().getOwner().get(
                    "Transitions");
            Set ots = ts.getTransitions(act.getID(), -1);
            Iterator trs = ots.iterator();
            while (trs.hasNext()) {
                Transition t = (Transition) trs.next();
                if (t.getTo() != null) {
                    /*
                     * if (((XMLComplexElement) t.get("Condition")) .get("Type")
                     * .toValue() .toString() .equals("OTHERWISE")) { return
                     * true; }
                     */
                }
            }
            return false;
        } else {
            return true;
        }
    }

    protected boolean hasAnyPostcondition(Activity act) {
        Transitions ts = (Transitions) act.getCollection().getOwner().get(
                "Transitions");
        Set outL = ts.getTransitions(act.getID(), -1);
        Iterator it = outL.iterator();
        while (it.hasNext()) {
            if (((Transition) it.next()).hasCondition()) {
                return true;
            }
        }
        return false;
    }

    // ************************** GRAPH CONNECTIONS CHECKING
    // ****************************

    public boolean checkGraphConnections(boolean fullCheck) {
        basicGraphConnectionErrors.remove(pkg);
        graphsConnectionErrors.remove(pkg);

        boolean areWellConnected = true;
        String basicGraphConnectionError;
        Map connectionErrorMessages = new LinkedHashMap();

        Iterator procs = ((WorkflowProcesses) pkg.get("WorkflowProcesses"))
                .toCollection().iterator();
        while (procs.hasNext()) {
            WorkflowProcess wp = (WorkflowProcess) procs.next();
            if (!checkGraphConnections(wp, false)) {
                areWellConnected = false;
                if (!fullCheck) {
                    break;
                }
                String msg = getBasicGraphConnectionError(wp);
                if (msg == null) {
                    msg = "";
                }
                connectionErrorMessages.put(wp, msg);
            }
        }
        if (!areWellConnected) {
            basicGraphConnectionError = XMLUtil
                    .getLanguageDependentString("InformationOneOrMoreProcessesHaveImproperlyConnectedElements");
            basicGraphConnectionErrors.put(pkg, basicGraphConnectionError);
        }
        graphsConnectionErrors.put(pkg, connectionErrorMessages);
        return areWellConnected;
    }

    public boolean checkGraphConnections(XMLCollectionElement wpOrAs,
                                         boolean fullCheck) {
        if (wpOrAs == null)
            return false;
        basicGraphConnectionErrors.remove(wpOrAs);
        graphsConnectionErrors.remove(wpOrAs);

        boolean isWellConnected = true;
        boolean basicError = false;
        Map connectionErrorMessages = new LinkedHashMap();

        Transitions ts = (Transitions) wpOrAs.get("Transitions");
        Collection acts = ((Activities) wpOrAs.get("Activities"))
                .toCollection();
        if (acts == null || acts.size() == 0) {
            graphsConnectionErrors.put(wpOrAs, connectionErrorMessages);
            return true;
        }

        Set startActs = null;
        Set endActs = null;
        if (fullCheck || isWellConnected) {
            startActs = getStartActivities(wpOrAs);
            if (startActs.size() == 0) {
                isWellConnected = false;
                basicError = true;
                String msg = (String) connectionErrorMessages.get(wpOrAs);
                msg = prepeareMessageString(msg);
                msg += XMLUtil
                        .getLanguageDependentString("ErrorStartingActivityDoesNotExist");
                connectionErrorMessages.put(wpOrAs, msg);
            }
        }
        if (fullCheck || isWellConnected) {
            endActs = getEndActivities(wpOrAs);
            if (endActs.size() == 0) {
                isWellConnected = false;
                basicError = true;
                String msg = (String) connectionErrorMessages.get(wpOrAs);
                msg = prepeareMessageString(msg);
                msg += XMLUtil
                        .getLanguageDependentString("ErrorEndingActivityDoesNotExist");
                connectionErrorMessages.put(wpOrAs, msg);
            }
        }
        if (fullCheck || isWellConnected) {
            Iterator it = acts.iterator();
            while (it.hasNext()) {
                Activity act = (Activity) it.next();
                String cem = checkActivityConnection(act, ts, startActs,
                        endActs, fullCheck);
                if (cem != null) {
                    connectionErrorMessages.put(act, cem);
                    isWellConnected = false;
                    if (!fullCheck) {
                        break;
                    }
                }

            }
        }

        if (!isWellConnected) {
            if (basicError) {
                basicGraphConnectionErrors.put(wpOrAs, connectionErrorMessages
                        .get(wpOrAs));
            } else {
                basicGraphConnectionErrors
                        .put(
                                wpOrAs,
                                XMLUtil
                                        .getLanguageDependentString("InformationOneOrMoreElementsAreNotProperlyConnected"));
            }
        }
        graphsConnectionErrors.put(wpOrAs, connectionErrorMessages);
        return isWellConnected;
    }

    /**
     * Checks if given activity is well connected.
     *
     * @return <ul>
     * <li> 0 if activity is well connected.
     * <li> 1 if activity does not have incoming transition and it is not a
     * start activity
     * <li> -1 if activity does not have outgoing transition and it is not an
     * end activity
     * <li> 2 if activity does not have any transitions, and it is not the only
     * activity in the graph
     */
    public String checkActivityConnection(Activity act, Transitions ts,
                                          Set startActs, Set endActs, boolean fullCheck) {
        Set incomingTransitions = ts.getTransitions(act.getID(), 1);
        Set outgoingTransitions = ts.getTransitions(act.getID(), -1);
        int noOfIncoming = incomingTransitions.size();
        int noOfOutgoing = outgoingTransitions.size();
        act.setPosition("NORMAL");
        // if type is normal, there should be at least one incoming and
        // one outgoing transition to this activity
        String connectionErrorMsg = "";
        if (noOfIncoming == 0 && !startActs.contains(act)
                && !checkConnectToStart(act, startActs)) {
            connectionErrorMsg += XMLUtil
                    .getLanguageDependentString("ErrorIncomingTransitionIsMissing")
                    + "; ";
            if (!fullCheck)
                return connectionErrorMsg;
        }
        if (noOfOutgoing == 0 && !endActs.contains(act)
                && !checkConnectToEnd(act, endActs)) {
            connectionErrorMsg += XMLUtil
                    .getLanguageDependentString("ErrorOutgoingTransitionIsMissing")
                    + "; ";
            if (!fullCheck)
                return connectionErrorMsg;
        }

        if (noOfIncoming > 0 && startActs.contains(act)) {
            connectionErrorMsg += XMLUtil
                    .getLanguageDependentString("ErrorCannotHaveIncomingTransitions")
                    + "; ";
            if (!fullCheck)
                return connectionErrorMsg;
        }

        if (noOfOutgoing > 0 && endActs.contains(act)) {
            connectionErrorMsg += XMLUtil
                    .getLanguageDependentString("ErrorCannotHaveOutgoingTransitions")
                    + "; ";
            if (!fullCheck)
                return connectionErrorMsg;
        }

        // if this is a block activity, check inner transitions
        BlockActivity ba = act.getBlockActivity();
        if (ba != null) {
            String asId = ba.get("Id").toString();
            ActivitySets ass = (ActivitySets) act.getOwnerProcess().get(
                    "ActivitySets");
            ActivitySet as = ass.getActivitySet(asId);
            if (as != null) {
                if (!checkGraphConnections(as, false)) {
                    connectionErrorMsg += XMLUtil
                            .getLanguageDependentString("ErrorInnerTransitionError")
                            + "; ";
                }
            }
        }

        if (connectionErrorMsg.length() == 0)
            connectionErrorMsg = null;

        return connectionErrorMsg;
    }

    // 获取流程下所有跟开始节点相连的活动
    protected Set getStartActivities(XMLCollectionElement wpOrAs) {
        Activities acts = ((Activities) wpOrAs.get("Activities"));

        Set starts = new HashSet();
        if (checkingBybpd) {
            Set sds = null;
            if (wpOrAs instanceof WorkflowProcess) {
                sds = ((WorkflowProcess) wpOrAs).getStartDescriptions();
            } else {
                Activity bact = findBlockActivity((ActivitySet) wpOrAs);
                if (sds != null) {
                    sds = bact.getStartDescriptions();
                }
            }
            /*
             * Iterator it = sds.iterator(); while (it.hasNext()) { String sd =
             * (String) it.next(); String[] startD = pkg.getXMLInterface().tokenize(sd, ";"); try {
             * String id = startD[1]; Activity sAct = acts.getActivity(id); if
             * (sAct != null) { starts.add(sAct); } } catch (Exception ex) { } }
             */
            if (sds != null) {
                starts = new HashSet(sds);
            } else {
                starts = new HashSet();
            }

        } else {
            Iterator it = acts.toCollection().iterator();
            Transitions ts = (Transitions) wpOrAs.get("Transitions");
            while (it.hasNext()) {
                Activity act = (Activity) it.next();
                if (ts.getTransitions(act.getID(), 1).size() == 0) {
                    starts.add(act);
                }
            }
        }
        return starts;
    }

    // 获取流程下所有跟结束节点相连的活动
    protected Set getEndActivities(XMLCollectionElement wpOrAs) {
        Activities acts = (Activities) wpOrAs.get("Activities");

        Set ends = new HashSet();
        if (checkingBybpd) {
            Set eds;
            if (wpOrAs instanceof WorkflowProcess) {
                eds = ((WorkflowProcess) wpOrAs).getEndDescriptions();
            } else {
                Activity bact = findBlockActivity((ActivitySet) wpOrAs);
                eds = bact.getEndDescriptions();
            }
            /*
             * Iterator it = eds.iterator(); while (it.hasNext()) { String ed =
             * (String) it.next(); String[] endD = pkg.getXMLInterface().tokenize(ed, ";"); try {
             * String id = endD[1]; Activity eAct = acts.getActivity(id); if
             * (eAct != null) { ends.add(eAct); } } catch (Exception ex) { } }
             */
            ends = new HashSet(eds);
        } else {
            Iterator it = acts.toCollection().iterator();
            Transitions ts = (Transitions) wpOrAs.get("Transitions");
            while (it.hasNext()) {
                Activity act = (Activity) it.next();
                if (ts.getTransitions(act.getID(), -1).size() == 0) {
                    ends.add(act);
                }
            }
        }
        return ends;
    }

    protected Activity findBlockActivity(ActivitySet as) {
        String asId = as.getID();
        WorkflowProcess wp = as.getOwnerProcess();
        Set bas = XMLUtil.getBlockActivities(wp, true);
        Iterator it = bas.iterator();
        while (it.hasNext()) {
            Activity a = (Activity) it.next();
            String baId = a.getBlockActivity().get("Id").toString();
            if (baId.equals(asId)) {
                return a;
            }
        }
        return null;
    }

    protected String prepeareMessageString(String msg) {
        if (msg != null) {
            msg = msg + "; ";
        } else {
            msg = "";
        }
        return msg;
    }

    /**
     * 检查活动是否连接开始节点
     *
     * @param act       被检查的活动对象
     * @param startActs 开始节点集合
     * @return
     */
    private boolean checkConnectToStart(Activity act, Set startActs) {
        if (act == null || startActs == null || startActs.size() == 0) {
            return false;
        }
        boolean ret = false;
        String actID = act.getID();
        Iterator it = startActs.iterator();
        while (it.hasNext()) {
            String sd = (String) it.next();
            String[] startD = Utils.tokenize(sd, ";");
            String id = startD[1];
            if (actID.equals(id)) {
                // 设置活动位置
                act.setPosition("START");
                ret = true;
                break;
            }
        }
        return ret;
    }

    /**
     * 检查活动是否连接结束节点
     *
     * @param act     被检查的活动对象
     * @param endActs 结束节点集合
     * @return
     */
    private boolean checkConnectToEnd(Activity act, Set endActs) {
        if (act == null || endActs == null || endActs.size() == 0) {
            return false;
        }
        boolean ret = false;
        String actID = act.getID();
        Iterator it = endActs.iterator();
        while (it.hasNext()) {
            String ed = (String) it.next();
            String[] endD = XMLUtil.tokenize(ed, ";");
            String id = endD[1];
            String[] ids = Utils.tokenize(id,
                    WorkflowProcess.ENDOFWORKFLOWACT_SEPARATE);
            if (Utils.isExistedInArray(ids, actID)) {
                // 设置活动位置
                act.setPosition("END");
                ret = true;
                break;
            }
        }
        return ret;
    }
}
