/* XMLElementDialog.java
 *
 * Authors:
 * Stefanovic Nenad  chupo@iis.ns.ac.yu
 * Bojanic Sasa      sasaboy@neobee.net
 * Puskas Vladimir   vpuskas@eunet.yu
 * Pilipovic Goran   zboniek@uns.ac.yu
 * Harald Meister    harald.meister@abacus.ch
 *
 */


package com.ds.bpm.bpd.xml;

import java.awt.Component;
import java.awt.Container;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;

import javax.swing.AbstractAction;
import javax.swing.BoxLayout;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.KeyStroke;

import com.ds.bpm.bpd.xml.panels.NewXMLGroupPanel;
import com.ds.bpm.bpd.xml.panels.NewXMLPanel;
import com.ds.bpm.bpd.xml.panels.XMLPanel;

/**
 * The dialog for showing objects derived from {@link XMLPanel} classes. The
 * given XMLPanel object must have it's owner, which type is a class
 * derived from the {@link XMLElement} class, and serves as a representation
 * of an XML tag.
 * <p> The dialog enables editing of all editable fields contained
 * within given panel and after user presses OK button, the new values
 * contained within edited fields are set to corresponding members of
 * panel's owner.
 */
public class NewXMLElementDialog extends JDialog {
   private NewXMLPanel panelToEdit=new NewXMLPanel();
   private JButton buttonOK;
   private JButton buttonCancel;

   private WindowListener wl=new WindowAdapter() {
      public void windowClosing(WindowEvent e) {
         isCanceled=true;
         dispose();
      }
   };

   /** Set to true if cancel hasn't been pressed */
   private boolean isCanceled=false;

   /** The list of XML change listeners. */
   private static ArrayList xmlChangeListeners = new ArrayList();

   /**
    * Constructs a new modal unparented instance that will define new element.
    */
   public NewXMLElementDialog () {
      this((JFrame)null,"");
   }

   /**
    * Constructs a new modal instance which parent is <code>parent</code> and
    * proceeds a title of dialog.
    *
    * @param parent Parent frame.
    * @param title  Dialog title.
    */
   public NewXMLElementDialog (JFrame parent,String title) {
      this(parent,title,true);
   }

   /**
    * Constructs a new modal instance which parent is <code>parent</code> and
    * proceeds a title of dialog.
    *
    * @param parent Parent dialog.
    * @param title  Dialog title.
    */
   public NewXMLElementDialog (JDialog parent,String title) {
      this(parent,title,true);
   }

   /**
    * Constructs a new modal instance which parent is <code>parent</code> and
    * proceeds a title of dialog.
    *
    * @param parent  Parent frame.
    * @param title   Dialog title.
    * @param isModal true if dialog is modal.
    */
   public NewXMLElementDialog (JFrame parent,String title,boolean isModal) {
      super(parent,title,isModal);
      initDialog();
   }

   /**
    * Constructs a new modal instance which parent is <code>parent</code> and
    * proceeds a title of dialog.
    *
    * @param parent Parent dialog.
    * @param title  Dialog title.
    * @param isModal true if dialog is modal.
    */
   public NewXMLElementDialog (JDialog parent,String title,boolean isModal) {
      super(parent,title,isModal);
      initDialog();
   }

   public static void addXMLElementChangedListener (XMLElementChangeListener ecl) {
      xmlChangeListeners.add(ecl);
   }

   public static void removeXMLElementChangedListener (XMLElementChangeListener ecl) {
      xmlChangeListeners.remove(ecl);
   }

   /**
    * @return <tt>true</tt> if cancel hasn't been pressed,
    *         <tt>false</tt> otherwise.
    */
   public boolean isCanceled () {
      return isCanceled;
   }

   public void setCanceled (boolean isCanceled) {
      this.isCanceled=isCanceled;
   }

   /** Returns the panel that is currently beeing edited. */
   public NewXMLPanel getEditingPanel () {
      return panelToEdit;
   }

   /**
    * Displays dialog and adds specified panel to it, along
    * with OK and/or Cancel button. The dialog allows user to edit
    * the panel field(s) that represents corresponding member of
    * it's owner class. After user presses OK, the content of
    * panel field(s) is transfered to the corresponding owner
    * class member. The member(s) of owner class represent a
    * program equivalent for real XML entity.
    *
    * @param elementPanel The panel with visual presentation of it's
    *                     owner members.
    */
   public void editXMLElement (NewXMLPanel elementPanel,
                               boolean hasCancelButton,boolean doNotClose) {

      Container cp=getContentPane();
      cp.remove(panelToEdit);
      panelToEdit=elementPanel;
      cp.add(panelToEdit,0);
      pack();

      // doNotClose is true only for dialog for entering
      // unique package Id - when we start to edit new package
      if (doNotClose) {
         removeWindowListener(wl);
         getRootPane().getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).
            remove(KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE,0,false));
      }

      if (hasCancelButton) {
         buttonCancel.setVisible(true);
      } else {
         buttonCancel.setVisible(false);
         if (!doNotClose) {
            buttonOK.requestFocus();
         }
      }

      setLocationRelativeTo(getParent());
      pack();
      show();
      requestFocus();
   }

   private void initDialog () {
      try {
         JPanel buttonPanel=new JPanel();

         buttonPanel.setAlignmentX(Component.LEFT_ALIGNMENT);
         buttonPanel.setAlignmentY(Component.TOP_ALIGNMENT);

         buttonOK=new JButton(XMLUtil.getLanguageDependentString("OKKey"));
         URL u = XMLUtil.getResource("OKImage");
         if (u!=null) {
            buttonOK.setIcon(new ImageIcon(u));
         }

         buttonCancel=new JButton(XMLUtil.getLanguageDependentString("CancelKey"));
         u = XMLUtil.getResource("CancelImage");
         if (u!=null) {
            buttonCancel.setIcon(new ImageIcon(u));
         }

         buttonPanel.add(buttonOK);
         buttonPanel.add(buttonCancel);

         Container cp=getContentPane();
         cp.setLayout(new BoxLayout(cp,BoxLayout.Y_AXIS));

         cp.add(panelToEdit);
         cp.add(buttonPanel);

         // action listener for confirming
         buttonOK.addActionListener( new ActionListener(){
                  public void actionPerformed( ActionEvent ae ){
                     if (panelToEdit.getOwner().isReadOnly()) {
                        dispose();
                     } else if (canApplyChanges()) {
                        applyChanges();
                        dispose();
                        if (getParent() != null){
                           getParent().repaint();//do repaint
                        }
                     }
                  }
               });

         // action listener for canceling
         buttonCancel.addActionListener( new ActionListener(){
                  public void actionPerformed( ActionEvent ae ){
                     isCanceled=true;
                     dispose();
                  }
               });

         addWindowListener(wl);

         getRootPane().getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW)
            .put(KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE,0,false),"Cancel");
         getRootPane().getActionMap().put("Cancel", new AbstractAction() {
                  public void actionPerformed(ActionEvent e) {
                     isCanceled=true;
                     dispose();
                  }
               });

      } catch (Exception e) {
         e.printStackTrace();
      }
      setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);
      setResizable(true);
      setLocationRelativeTo(getParent());
      buttonOK.setDefaultCapable(true);
      getRootPane().setDefaultButton(buttonOK);

   }


   public void switchPanel (NewXMLPanel newPanel,String newTitle,boolean hasCancelButton) {
      if (isModal()) return;

      if (canApplyChanges()) {
         applyChanges();
      } else {
         return;
      }

      setTitle(newTitle);

      Container cp=getContentPane();
      cp.remove(panelToEdit);
      panelToEdit=newPanel;
      cp.add(panelToEdit,0);
      pack();

      try {
         panelToEdit.getComponent(0).requestFocus();
      } catch (Exception ex) {
         panelToEdit.requestFocus();
      }

      if (hasCancelButton) {
         buttonCancel.setVisible(true);
      } else {
         buttonCancel.setVisible(false);
         buttonOK.requestFocus();
      }

      pack();
      show();

   }

   public boolean canApplyChanges () {
      if (panelToEdit.checkRequired()) {
         if (!panelToEdit.getOwner().isValidEnter(panelToEdit)) {
            return false;
         }
         if (panelToEdit.getOwner() instanceof NewXMLCollectionElement) {
            if (((NewXMLCollectionElement)panelToEdit.getOwner()).
                isIDUniqueAndValid(panelToEdit)) {
               return true;
            } else {
               return false;
            }
         } else {
            return true;
         }
      } else {
         return false;
      }
   }

   public void applyChanges () {
      panelToEdit.setElements();
      notifyListeners(panelToEdit.getOwner());
   }

   public static void notifyListeners (NewXMLElement changedEl) {
      Iterator it=xmlChangeListeners.iterator();
      while (it.hasNext()) {
        NewXMLElementChangeListener cl=(NewXMLElementChangeListener)it.next();
         cl.xmlElementChanged(changedEl);
      }
   }

   public void requestFocus() {
      try {
         if (panelToEdit instanceof NewXMLGroupPanel) {
            if (panelToEdit.getComponent(0).isEnabled()) {
               panelToEdit.getComponent(0).requestFocus();
            } else {
               panelToEdit.getComponent(1).requestFocus();
            }
         }
      } catch (Exception ex) {
         panelToEdit.requestFocus();
      }
   }

}


/* End of XMLElementDialog.java */
