/* XMLElement.java
 *
 * Authors:
 * Stefanovic Nenad  chupo@iis.ns.ac.yu
 * Bojanic Sasa      sasaboy@neobee.net
 * Puskas Vladimir   vpuskas@eunet.yu
 * Pilipovic Goran   zboniek@uns.ac.yu
 *
 */


package com.ds.bpm.bpd.xml;

import java.io.Serializable;

import javax.swing.text.AttributeSet;
import javax.swing.text.SimpleAttributeSet;
import javax.swing.text.StyledDocument;
import javax.swing.tree.DefaultMutableTreeNode;

import com.ds.bpm.bpd.xml.panels.NewXMLPanel;

import org.w3c.dom.DOMException;
import org.w3c.dom.Node;

/**
* XMLElement class instance represents program equivalence to
* the element defined in some XML XML. It contains main
* characteristics common for all XML elements, and is intended to
* be uses as the base class for classes that will represent concrete
* XML element such as "simple" element, "complex" element,
* "collection" of elements, "choice" of elements, "attribute" ...
* <p>When classes that extends this class are created, they
* can call this class constructor without arguments, in which
* case the member of this class that corresponds to the XML element
* name is filled with the name of extended class, or with the
* argument that represents the name of element, where such name
* directly sets the previously mentioned member (This is the case
* when creating 'attribute' or 'choice' XML elements). The label to
* be written in a dialog for editing value of element, is set
* to the language specific value, which has to be defined in
* the property file under the key: "<b>ElementName</b>Key".
* <p> This class contains method #getPanel, which
* should be overriden by classes that extends this class, and
* which is used to show the panel with the field that represent
* a value of XML element (In the case of complex elements, the
* panel will show fields of all elements that complex element
* is made of). This panel is shown in a dialog (see {@link
* XMLElementDialog}) that allows user to edit this field, and
* when user presses OK, the new value is put in the value member
* of this class.
* <p>When program decides to write an XML file according to
* it's XMLElement derived class members, it calls #toXML
* method of those classes, which is defined by this class, but
* has no implementation. This method has to write the proper
* element opening and closing tag (defined by element name), and the
* previously defined value of element to put inside tags.
* <p>When there is a need of reading an XML file, the values that
* corresponds to particular element (determined by element name
* and it's position within other elements) are put into this
* element value member by calling #fromXML method, and passing it
* a tag to process.
* <p> This class also defines methods that are performing a check
* for validity, emptiness or mandatory status of the element, and
* are also supposed to be redefined by classes that extends it.
* <p><p>NOTE: Although this class is not declared abstract, it is
*             uselles without redefining it's methods.
*/
public class NewXMLElement implements Serializable, Cloneable {

   /**
   * Equivalent for XML element name. Used when writting
   * instance of this class to XML file.
   */
   protected String name;
   /**
   * Language specific name of XML element that is shown in
   * editing dialog.
   */
   protected String labelName;
   /**
   * Supposed to contain the value for XML element.
   * This is true for simple elements and attributes,
   * more complex elements uses it as they need.
   */
   protected Object value;
   /**
   * Indicates if an element is required - corresponds
   * to the same XML element definition.
   */
   protected boolean isRequired=false;
   /** Used in dialogs to forbid editing of an element value. */
   protected boolean isReadOnly=false;

   /** Flag that say is node is collapsed */
   protected boolean isCollapsed = true;

//private static int genEl=0;
   /**
   * Creates a new instance of element: sets the <code>name</code> member
   * to the name of inner class that extends this element, and the
   * <code>labelName</code> member to the value read from property file
   * under the key of previously set <code>name</code> member.
   */
   public NewXMLElement () {
      name=getClass().getName();
      int lastDot=name.lastIndexOf(".");
      if (lastDot>=0) {
         name=name.substring(lastDot+1,name.length());
      }

      labelName=XMLUtil.getLanguageDependentString(name+"Key");
      value=new String();
//System.out.println("Generated element no. "+(++genEl));
   }

   /**
   * Creates a new instance of element: sets the <code>name</code> member
   * to the specified one, and the <code>labelName</code> member to the
   * value read from property file under the key of previously set
   * <code>name</code> member.
   *
   * @param name The name of XML element that this class represents.
   */
   public NewXMLElement (String name) {
      this.name=name;
      this.labelName=XMLUtil.getLanguageDependentString(name+"Key");
      value=new String();
//System.out.println("Generated element no. "+(++genEl));
   }

   /**
   * Sets the 'read only' property of element to the specified one.
   * This enables/disables editing of the element within dialog.
   * Some derived classes (especially class that represents complex
   * element) need to redefine this method.
   *
   * @param ro <tt>true</tt> if element will be 'read only',
   *           <tt>false</tt> otherwise.
   */
   public void setReadOnly (boolean ro) {
      isReadOnly=ro;
   }

   /**
   * Returns the 'read only' status of element.
   * This indicates if the editing of element within dialog
   * is enabled or disabled.
   *
   * @return <tt>true</tt> if element is 'read only',
   *         <tt>false</tt> otherwise.
   */
   public boolean isReadOnly () {
      return isReadOnly;
   }

   /**
   * Sets the 'required' property of the element.
   * If element is 'required', the dialog for editing
   * it's value can't be closed until the value is defined.
   *
   * @param r <tt>true</tt> if element will be 'required',
   *          <tt>false</tt> otherwise.
   */
   public void setRequired (boolean r) {
      isRequired=r;
   }

   /**
   * Returns the 'required' property of the element.
   * If element is 'required', the dialog for editing
   * it's value can't be closed until the value is defined.
   *
   * @return <tt>true</tt> if element is 'required',
   *         <tt>false</tt> otherwise.
   */public boolean isRequired () {
      return isRequired;
   }

   /**
   * Indicates if element is empty. Some derived classes
   * (especially class that represents complex element) need to
   * redefine this method.
   *
   * @return <tt>true</tt> if user haven't defined the element
   *         value within dialog, <tt>false</tt> otherwise.
   */
   public boolean isEmpty () {
      return !(value!=null && value.toString().trim().length()>0);
   }

   /**
   * Indicates if element is valid in XML XML sense.
   * This implementation claims that element is always valid. The derived
   * classes has to implement it's own definition of validity.
   *
   * @return <tt>true</tt> if element is valid, <tt>false</tt> otherwise.
   */
   public boolean isValid () {
      return true;
   }

   /**
   * Indicates if dialog data enter for element is valid.
   * This implementation claims that element is always valid. The derived
   * classes has to implement it's own definition of validity.
   *
   * @return <tt>true</tt> if dialog data for the element are valid,
   * <tt>false</tt> otherwise.
   */
   public boolean isValidEnter (NewXMLPanel panelToEdit) {
      return true;
   }

   /**
   * Supposed to be overriden by derived classes to implement
   * element specific writting to an XML file.
   *
   * @param parent parent node
   * @return         The text for a corresponding WfMC XML element tag.
   */
   public void toXML(Node parent) throws DOMException {
      // external parser
      if (parent!=null) {
         if (null != value) {
            Node node = (parent.getOwnerDocument()).createElement(name);
            node.setNodeValue(toString());
            parent.appendChild(node);
         }
      }
   }

   /**
   * Supposed to be overriden by derived classes to implement
   * element specific reading from an XML file. This implementation
   * satisfies only "simple" elements.
   *
   * @param node root node
   */
   public void fromXML(Node node) {
      if (node!=null) {
//System.out.println("Node is "+node.getNodeName()+", value is="+node.getNodeValue());
         Object newVal;
         if (node.hasChildNodes()) {
            newVal=node.getChildNodes().item(0).getNodeValue();
         // should never happen
         } else {
            newVal=node.getNodeValue();
         }
         if (newVal!=null) {
            value=newVal;
         }
      }
   }

   /**
   * Supposed to be used (and overriden) by class that
   * represent XML choice of complex elements.
   *
   * @param name       The element name.
   * @param node       root node
   */
   public void fromXML(String name,Node node) {
      try {
         this.name=name;
         this.labelName=XMLUtil.getLanguageDependentString(name+"Key");
      } catch (Exception ex) {}
      fromXML(node);
   }

   /**
   * Returns the panel for editing the element. This implementation
   * returns an empty basic panel, the derived classes should implement
   * this method to suite their needs.
   *
   * @return A panel to put into dialog for editing element.
   */
   public NewXMLPanel getPanel () {
      return new NewXMLPanel();
   }

   public void setLabelName (String lName) {
      labelName=lName;
   }

   /**
   * Sets the element value (in the sense of XML element).
   *
   * @param v The value to set.
   */
   public void setValue(Object v) {
      value=v;
   }

   /**
   * Gets the element value. The value represents an XML
   * tag content for this element.
   *
   * @return The value of this element.
   */
   public Object toValue() {
      return value;
   }

   /**
   * Returns the text for element 'name' presentation within
   * a dialog for editing element. This text is language
   * specific and is set reading a property file.
   *
   * @return The language specific presentation of element 'name'.
   */
   public String toLabel () {
      return labelName;
   }

   /**
   * Returns the name of element.
   *
   * @return The 'name' attribute of element.
   */
   public String toName () {
      return name;
   }

   /**
   * Returns the text representation of element's 'value'.
   * This method only calls <code>toString</code> method
   * of it's value member, if value is <code>null</code>,
   * returns labelName member.
   * @return The value's string representation.
   */
   public String toString () {
      if (value!=null) {
         return value.toString().trim();
      } else if (labelName!=null) {
         return labelName;
      } else {
         return "";
      }
   }

   /**
   * Used to create exact copy of instance of this class.
   * The newly created instance will have the same name,
   * label name and value as the copied instance. Also it
   * will have the same 'read only' and 'required' properties
   * as the copied one.
   *
   * @return The newly created instance of XMLElement class.
   */
   public Object clone () {
      NewXMLElement d=null;
      try {
         //System.out.println("Cloning XMLELement "+name);
         d=(NewXMLElement)super.clone();
         d.name=new String(this.name);
         if (this.labelName!=null) {
            d.labelName=new String(this.labelName);
         } else {
            d.labelName=null;
         }
         if (value instanceof NewXMLElement) {
            d.value=((NewXMLElement)value).clone();
         } else {
            if (this.value!=null) {
               d.value=new String(this.value.toString());
            } else {
               this.value=null;
            }
         }
         d.isRequired=this.isRequired;
         d.isReadOnly=this.isReadOnly;
      } catch (CloneNotSupportedException e) {
         // Won't happen because we implement Cloneable
         throw new Error(e.toString());
      }
      return d;
   }

   public void refreshLabelName() {
      labelName=XMLUtil.getLanguageDependentString(name+"Key");
   }

   /**
    * Make tree node for this element and all subelements.
    * @return tree node for this element.
    */
   public DefaultMutableTreeNode getNode() {
      return new ToNameMutableTreeNode(this);
   }

   protected String OFFSET = "   ";
   protected String NEWLINE = "\n";
   protected AttributeSet atts = new SimpleAttributeSet();

   /**
    * This method will call getPrintDescription(String indent) with parameter OFFSET.
    * This method is called on root node from which description will be maked.
    * @return Description of this element and all subelements.
    */
   public String getPrintDescription(StyledDocument doc) {
      return this.getPrintDescription(OFFSET, doc);
   }

   /**
    * Description of this element and all subelements.
    * @return return print description of element.
    */
   public String getPrintDescription(String indent, StyledDocument doc) {
      String retVal = "";
      try {
         StringBuffer all = new StringBuffer();
         all.append( toName() + " : " + value.toString().replaceAll( "\n", " " ) );
         retVal = all.toString();

         AttributeSet atts = new SimpleAttributeSet();
         doc.insertString( doc.getLength(), retVal, atts );

      }catch(Exception e) {
         e.printStackTrace();
      }

      return retVal;
   }

   public void setCollapsed(boolean isCollapsed) {
      this.isCollapsed = isCollapsed;
   }

   public boolean isCollapsed() {
      return this.isCollapsed;
   }


}

/* End of XMLElement.java */
