/* XMLChoice.java
 *
 * Authors:
 * Stefanovic Nenad  chupo@iis.ns.ac.yu
 * Bojanic Sasa      sasaboy@neobee.net
 * Puskas Vladimir   vpuskas@eunet.yu
 * Pilipovic Goran   zboniek@uns.ac.yu
 *
 */


package com.ds.bpm.bpd.xml;

import com.ds.bpm.bpd.xml.panels.NewXMLComboPanel;
import com.ds.bpm.bpd.xml.panels.NewXMLPanel;
import com.ds.bpm.bpd.xml.panels.XMLComboPanel;

/**
 * XMLChoice class instance represents program equivalence to
 * the 'choice' element defined in some XML XML. It is used
 * to enable user to visually choose the one of the choices,
 * and to logically support the choice, which means to
 * properly hold it, and to properly write it into XML document
 * (or read it from XML document when the time comes).
 * <p> For example if XML schema defines some choice as follows:
 * <pre>
 *  &lt;xsd:choice>
 *    &lt;xsd:element ref="RecordType"/&gt;
 *    &lt;xsd:element ref="UnionType"/&gt;
 *    &lt;xsd:element ref="EnumerationType"/&gt;
 *    &lt;xsd:element ref="ArrayType"/&gt;
 *    &lt;xsd:element ref="ListType"/&gt;
 *    &lt;xsd:element ref="BasicType"/&gt;
 *    &lt;xsd:element ref="PlainType"/&gt;
 *    &lt;xsd:element ref="DeclaredType"/&gt;
 *  &lt;/xsd:choice&gt;
 * </pre>
 * such choice is represented by this class instance (actually by
 * the instance of it's derived class XMLComplexChoice), and choices
 * would be the instances of appropriate classes derived from
 * XMLElement class (actually from it's derived class XMLComplexElement),
 * which names are: 'RecordType', 'UnionType', 'EnumerationType'
 * , ..., 'DeclaredType'. The combo box that will be shown in the panel
 * of this element will offer this choices, and the names of the choices
 * will be shown as language specific presentation of this choices.
 * <p> The another example of using choices is when choices are
 * just some predefined strings. This is the case when some
 * XML 'attribute' has the exact number of possible choices,
 * for e.g.:
 * <pre>
 *  &lt;xsd:attribute name="Type" use="required"&gt;
 *    &lt;xsd:simpleType&gt;
 *       &lt;xsd:restriction base="xsd:NMTOKEN"&gt;
 *          &lt;xsd:enumeration value="STRING"/&gt;
 *          &lt;xsd:enumeration value="FLOAT"/&gt;
 *          &lt;xsd:enumeration value="INTEGER"/&gt;
 *          &lt;xsd:enumeration value="REFERENCE"/&gt;
 *          &lt;xsd:enumeration value="DATETIME"/&gt;
 *       &lt;/xsd:restriction&gt;
 *    &lt;/xsd:simpleType&gt;
 *  &lt;/xsd:attribute&gt;
 * </pre>
 * such choice is actually represented by the instance of derived class
 * XMLAttribute, and choices would be the language specific strings that
 * correspodents to the 'STRING', 'FLOAT',..., 'DATETIME'.
 *
 * <p>NOTE: Although this class is not declared abstract, it is
 *          uselles without redefining it's methods. The classes
 *          that are really used are derived classes {@link
 *          XMLAttribute} and {@link XMLComplexChoice}.
 */
public class NewXMLChoice extends NewXMLElement {
   /** The currently choosen element. */
   protected Object choosen;
   /** The possible choices. */
   protected Object[] choices;

   /**
    * Creates a new instance of element with the specified name, and
    * specifies the values of choices for the element. Assumes that
    * the first choice is choosen when object is created.
    *
    * @param name    The name of the 'choice' XML element
    *                that this class represents.
    * @param choices The possible choices for this element.
    */
   public NewXMLChoice (String name,Object[] choices) {
      this(name,choices,0);
   }

   /**
    * Creates a new instance of element with the specified name,
    * specified choices for the element, and with <b>chooseIndex</b>'th
    * value choosen.
    *
    * @param name         The name of the 'choice' XML element
    *                     that this class represents.
    * @param choices      The possible choices for this element.
    * @param choosenIndex The index of element to be choosen right after creation.
    */
   public NewXMLChoice (String name,Object[] choices,int choosenIndex) {
      super(name);
      try {
         this.choices=choices;
         this.choosen=choices[choosenIndex];
         this.value=choosen;
      }
      catch (Exception e) {
         this.value="";
      }
   }

   /**
    * Overrides super-method to set this element and all it's choice elements
    * (if choices are instances of XMLElement class) read only value to the
    * one specified.
    *
    * @param ro <tt>true</tt> if element should be <i>read only</i>,
    *           <tt>false</tt> otherwise.
    */
   public void setReadOnly (boolean ro) {
      super.setReadOnly(ro);
      if (choices!=null) {
         for (int i=0; i<choices.length; i++) {
            if (choices[i] instanceof XMLElement) {
               ((XMLElement)choices[i]).setReadOnly(ro);
            }
         }
      }
   }

   /**
    * Overrides the super-method to set the value field and the choosen field
    * to the specified value. The value is, in fact, one of possible choices.
    *
    * @param v the choice to be set.
    */
   public void setValue(Object v) {
      super.setValue(v);
      choosen=v;
   }

   /**
    * The possible choices could be instances of XMLElement class, or
    * can be instances of String.
    *
    * @return the possible choices for this element.
    */
   public Object[] getChoices() {
      return choices;
   }

   /**
    * Choosen element is one of the choices.
    *
    * @return the choosen element.
    * @see #getChoices
    */
   public Object getChoosen() {
      return choosen;
   }

   /**
    * The default panel that represents this class visually
    * is instance of {@link XMLComboPanel} class. The panel
    * has the combo box that is filled with content of
    * <code>choices</code> field.
    *
    * @return the XMLComboPanel which owner is this class instance.
    */
   public NewXMLPanel getPanel () {
      return new NewXMLComboPanel(this);
   }

   /**
    * Overrides super-method to apply cloning of <code>choices</code>
    * field content, if it's content are instances of <code>XMLElement
    * </code> class.
    *
    * @return the new instance that has all similar attributes as the
    *         cloned one.
    */
   public Object clone () {
      NewXMLChoice d=(NewXMLChoice)super.clone();

      d.choosen=null;
      d.choices=null;

      if (this.choices!=null && this.choices.length>0) {
         if (this.choices[0] instanceof String) {
            d.choices=this.choices;
            d.choosen=this.choosen;
         }
         else {
            d.choices=new Object[this.choices.length];
            for (int i=0; i<this.choices.length; i++) {
               Object och=this.choices[i];
               if (och instanceof XMLElement) {
                  XMLElement ch=(XMLElement)this.choices[i];
                  d.choices[i]=ch.clone();
                  if (this.choosen==ch) {
                     d.choosen=d.choices[i];
                     //System.out.println("d.choosen="+d.choosen);
                  }
               } else {
                  d.choices[i]=och;
                  if (this.choosen!=null && this.choosen.equals(och)) {
                     d.choosen=d.choices[i];
                  }
               }
            }
         }
      }
      return d;
   }

   public void refreshLabelName() {
      super.refreshLabelName();
      if (choices!=null) {
         for (int i=0; i<choices.length; i++) {
            if (choices[i] instanceof XMLElement) {
               ((XMLElement) choices[i]).refreshLabelName();
            }
         }
      }
   }
}

/* End of XMLChoice.java */
