package com.ds.bpm.bpd.misc;

import com.ds.esd.client.ProjectVersion;
import com.ds.esd.tool.module.EUModule;
import com.ds.esd.tool.ui.component.Component;
import com.ds.esd.tool.ui.component.Properties;
import com.ds.esd.tool.ui.component.container.AbsUIProperties;
import com.ds.esd.tool.ui.component.list.AbsListProperties;
import com.ds.esd.tool.ui.module.ModuleComponent;

import javax.swing.tree.TreeNode;
import java.util.Enumeration;
import java.util.NoSuchElementException;
import java.util.Vector;

public class ComponentTreeNode<T> implements TreeNode {

    private TreeNode parent;
    private Vector<TreeNode> children;
    private boolean leaf;
    private T userObject;

    public static final Enumeration EMPTY_ENUMERATION = new Enumeration() {
        public boolean hasMoreElements() {
            return false;
        }

        public Object nextElement() {
            throw new NoSuchElementException("No more elements");
        }
    };

    public ComponentTreeNode(T userObject) {
        super();
        this.parent = null;
        this.userObject = userObject;
    }

    public ComponentTreeNode() {
        this(null);
    }

    /**
     * @return Returns the children.
     */
    public TreeNode[] getChildren() {
        if (children == null) {
            return new TreeNode[0];
        }
        return (TreeNode[]) children.toArray(new TreeNode[0]);
    }

    /**
     * @return Returns the parent.
     */
    public TreeNode getParent() {
        return parent;
    }

    /**
     * @param parent The parent to set.
     */
    public void setParent(TreeNode parent) {
        this.parent = parent;
    }

    /**
     * @return Returns the leaf.
     */
    public boolean isLeaf() {
        return leaf;
    }

    /**
     * @param leaf The leaf to set.
     */
    public void setLeaf(boolean leaf) {
        this.leaf = leaf;
    }

    public void insert(TreeNode newChild, int index) {
        if (!(newChild instanceof ComponentTreeNode)) {
            throw new IllegalArgumentException("new child is not a OrgTreeNode");
        }
        if (newChild == null) {
            throw new IllegalArgumentException("new child is null");
        }
        ComponentTreeNode oldParent = (ComponentTreeNode) newChild.getParent();

        if (oldParent != null) {
            oldParent.remove(newChild);
        }
        ((ComponentTreeNode) newChild).setParent(this);
        if (children == null) {
            children = new Vector();
        }
        children.add(index, newChild);
    }

    public void insert(TreeNode newChild) {
        ComponentTreeNode oldParent = (ComponentTreeNode) newChild.getParent();

        if (oldParent != null) {
            oldParent.remove(newChild);
        }
        ((ComponentTreeNode) newChild).setParent(this);
        if (children == null) {
            children = new Vector();
        }
        children.add(newChild);
    }

    public void remove(TreeNode aChild) {
        if (aChild == null) {
            throw new IllegalArgumentException("argument is null");
        }

        if (!isNodeChild(aChild)) {
            throw new IllegalArgumentException("argument is not a child");
        }
        remove(getIndex(aChild)); // linear search
    }

    public void remove(int index) {
        ComponentTreeNode child = (ComponentTreeNode) getChildAt(index);
        children.remove(index);
        child.setParent(null);
    }

    public void removeAllChildren() {
        for (int i = getChildCount() - 1; i >= 0; i--) {
            remove(i);
        }
    }

    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }
        if (o instanceof ComponentTreeNode) {
            ComponentTreeNode oo = (ComponentTreeNode) o;
            if (oo.userObject == null) {
                return false;
            }
            if (oo.userObject.equals(this.userObject)) {
                return true;
            }
        }
        return false;
    }

    /**
     * @return Returns the userObject.
     */
    public Object getUserObject() {
        return userObject;
    }

    /**
     * @param userObject The userObject to set.
     */
    public void setUserObject(T userObject) {
        this.userObject = userObject;
    }

    public int getIndex(TreeNode aChild) {
        if (aChild == null) {
            throw new IllegalArgumentException("argument is null");
        }

        if (!isNodeChild(aChild)) {
            return -1;
        }
        return children.indexOf(aChild); // linear search
    }

    public TreeNode getChildAt(int index) {
        return (TreeNode) children.get(index);
    }

    public void removeFromParent() {
        ComponentTreeNode parent = (ComponentTreeNode) getParent();
        if (parent != null) {
            parent.remove(this);
        }
    }

    /**
     * Returns true if <code>aNode</code> is a child of this node. If <code>aNode</code> is null, this method returns
     * false.
     *
     * @return true if <code>aNode</code> is a child of this node; false if <code>aNode</code> is null
     */
    public boolean isNodeChild(TreeNode aNode) {
        boolean retval;

        if (aNode == null) {
            retval = false;
        } else {
            if (getChildCount() == 0) {
                retval = false;
            } else {
                retval = (aNode.getParent() == this);
            }
        }

        return retval;
    }

    /**
     * Returns the number of children of this node.
     *
     * @return an int giving the number of children of this node
     */
    public int getChildCount() {
        if (children == null) {
            return 0;
        } else {
            return children.size();
        }
    }

    public int getLevel() {
        TreeNode ancestor;
        int levels = 0;

        ancestor = this;
        while ((ancestor = ancestor.getParent()) != null) {
            levels++;
        }

        return levels;
    }

    public boolean isRoot() {
        return getParent() == null;
    }

    public Enumeration children() {
        if (children == null) {
            return EMPTY_ENUMERATION;
        } else {
            return children.elements();
        }
    }

    public boolean getAllowsChildren() {
        return true;
    }

    public String toString() {
        if (userObject == null) {
            return "root";
        }


        if (userObject instanceof ProjectVersion) {
            ProjectVersion version = (ProjectVersion) userObject;
            return version.getDesc() == null ? version.getProject().getProjectName() : version.getDesc();
        }


        if (userObject instanceof ModuleComponent) {
            ModuleComponent moduleComponent = (ModuleComponent) userObject;
            Properties properties = moduleComponent.getProperties();
            String desc = "";
            if (properties != null) {
                desc = properties.getDesc();
                if (desc == null && moduleComponent.getTopComponentBox() != null) {
                    desc = moduleComponent.getTopComponentBox().getProperties().getDesc();
                }

                if (desc == null) {
                    desc = moduleComponent.typeKey.getType();
                }

                if (desc != null) {
                    desc = "(" + desc + ")";
                }
            }


            return moduleComponent.getAlias() + desc;

        }
        if (userObject instanceof Component) {
            Component component = (Component) userObject;
            Properties properties = component.getProperties();
            String desc = "";
            if (properties != null) {
                desc = properties.getDesc();

                if (desc == null) {
                    if (properties instanceof AbsUIProperties) {
                        desc = ((AbsUIProperties) properties).getCaption();
                    } else {
                        desc = component.typeKey.getType();
                    }
                }

                if (desc == null) {
                    desc = component.typeKey.getType();
                }

                if (desc != null) {
                    desc = "(" + desc + ")";
                }
            }
            return desc + component.getAlias();
        }
        if (userObject instanceof EUModule) {
            return ((Component) userObject).getAlias();
        }


        return "未知类型";
    }
}
