/*
 * Authors:
 * Stefanovic Nenad  chupo@iis.ns.ac.yu
 * Bojanic Sasa      sasaboy@neobee.net
 * Puskas Vladimir   vpuskas@eunet.yu
 * Pilipovic Goran   zboniek@uns.ac.yu
 *
 */

package com.ds.bpm.bpd.ldap;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;

import javax.naming.Context;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attributes;
import javax.naming.directory.DirContext;
import javax.naming.directory.InitialDirContext;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.swing.AbstractAction;
import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.KeyStroke;
import javax.swing.border.Border;

import com.ds.bpm.bpd.BPD;
import com.ds.bpm.bpd.BPDConfig;
import com.ds.bpm.bpd.BPDConstants;
import com.ds.bpm.bpd.PackageEditor;
import com.ds.bpm.bpd.ResourceManager;
import com.ds.bpm.bpd.actions.ActionBase;
import com.ds.bpm.bpd.xml.XMLAttribute;
import com.ds.bpm.bpd.xml.XMLCollection;
import com.ds.bpm.bpd.xml.XMLElementDialog;
import com.ds.bpm.bpd.xml.elements.ExtendedAttribute;
import com.ds.bpm.bpd.xml.elements.ExtendedAttributes;
import com.ds.bpm.bpd.xml.elements.Participant;
import com.ds.bpm.bpd.xml.elements.ParticipantType;
import com.ds.bpm.bpd.xml.panels.XMLComboPanel;
import com.ds.bpm.bpd.xml.panels.XMLPanel;

/**
 * m * Class that realizes <B>import external process</B> action.
 */
public class ImportExternalParticipants extends ActionBase {

	private ExternalParticipants eps;

	private XMLAttribute idMapping = new XMLAttribute(
			"ChooseAttributeNameForIdMapping", new String[] { "NoMapping",
					"uid", "userId" }, 0) {
		// to enable manual enterance of object class
		public XMLPanel getPanel() {
			return new XMLComboPanel(this, null, XMLPanel.BOX_LAYOUT, false,
					true);
		}
	};

	public ImportExternalParticipants(PackageEditor editor) {
		super(editor);
	}

	public void actionPerformed(ActionEvent e) {
		PackageEditor pe = (PackageEditor) editor;
		if (!pe.isInitialized()) {
			pe.enterPackageID();
		}

		if (eps == null) {
			eps = new ExternalParticipants(pe.getXMLPackage());
		}

		XMLElementDialog d = new XMLElementDialog(
				(JFrame) pe.getWindow(),
				ResourceManager.getLanguageDependentString("PackageKey")
						+ " '"
						+ pe.getXMLPackage().get("Id").toString()
						+ "' - "
						+ ResourceManager
								.getLanguageDependentString("ImportExternalParticipantsLabel"));
		d.editXMLElement(eps.getPanel(), true, false);
		if (!d.isCanceled()) {
			// here is the place where connection to the LDAP server is made
			// to collect informations
			java.util.List allEntries = getEntries(eps);

			java.util.List selectedEntries = showDialog(editor.getWindow(),
					allEntries);
			if (selectedEntries.size() > 0) {
				createParticipants(selectedEntries);
			}
		}
	}

	/**
	 * Connects to the LDAP server and retrieves wanted data. All neccessary
	 * information to establish connection to the server, and to retrieve data
	 * are hold within given parameter ep.
	 */
	private java.util.List getEntries(ExternalParticipants ep) {
		java.util.List entries = new java.util.ArrayList();

		int ldapPort;
		try {
			ldapPort = Integer.valueOf(ep.get("Port").toString()).intValue();
		} catch (Exception ex) {
			System.err.println("Port invalid, using default port 389");
			ldapPort = 389;
		}

		int searchScope;
		if (ep.get("SearchScope").toValue().toString().equals("SCOPE_ONE")) {
			searchScope = SearchControls.ONELEVEL_SCOPE;
		} else {
			searchScope = SearchControls.SUBTREE_SCOPE;
		}

		// boolean attributeOnly = true; // returns attributes
		boolean attributeOnly = false; // returns attributes and their values

		String ldapHost = ep.get("Host").toString();
		String searchBase = ep.get("BaseDN").toString();
		// String searchFilter = "(objectclass=*)";
		String searchFilter;
		if (ep.get("ObjectClassFilter").toValue().toString()
				.equals("AllListed")) {
			searchFilter = "(|(objectClass=organizationalUnit)"
					+ "(objectClass=organizationalRole)(objectClass=person)"
					+ "(objectClass=organizationalPerson)(objectClass=inetOrgPerson))";
		} else {
			searchFilter = "(objectClass="
					+ ep.get("ObjectClassFilter").toValue().toString() + ")";
		}

		try {
			/*
			 * Create an environment for the initial directory context. The
			 * properties specify the LDAP provider, the LDAP server, the LDAP
			 * version, and if needed username and password, or if not, no
			 * security (anonymous bind).
			 */
			java.util.Properties env = new java.util.Properties();
			env.put(Context.INITIAL_CONTEXT_FACTORY,
					"com.sun.jndi.ldap.LdapCtxFactory");
			env.put(Context.REFERRAL, "throw");
			env
					.put(Context.PROVIDER_URL, "ldap://" + ldapHost + ":"
							+ ldapPort);

			Security security = (Security) ep.get("Security");
			if (!security.get("Level").toValue().toString().equals("Anonymous")) {
				String ldapUser = security.get("UserDN").toString();
				String ldapPassword = security.get("Password").toString();
				env.put(Context.SECURITY_PRINCIPAL, ldapUser);
				env.put(Context.SECURITY_CREDENTIALS, ldapPassword);
			}

			/* Create the initial directory context. */
			DirContext ctx = new InitialDirContext(env);
			/*
			 * Set up and perform the search. Find all people in IBM in the
			 * United States whose common name starts with Sue or Johan.
			 */
			SearchControls constraints = new SearchControls();
			constraints.setSearchScope(searchScope);
			try {
				constraints.setCountLimit(BPDConfig.getInstance()
						.getLDAPCountLimit());
			} catch (Exception ex) {
				constraints.setCountLimit(0);
			}
			try {
				constraints.setTimeLimit(BPDConfig.getInstance()
						.getLDAPTimeLimit());
			} catch (Exception ex) {
				constraints.setTimeLimit(0);
			}
			NamingEnumeration results = ctx.search(searchBase, searchFilter,
					constraints);

			/* For each entry found. */
			while (results.hasMore()) {
				SearchResult sr = (SearchResult) results.next();
				// System.out.println(sr.getName());
				entries.add(sr);
			}
			ctx.close();
		} catch (NamingException e) {
			/* Handle any name/directory exceptions. */
			if (e.getMessage().indexOf("Sizelimit Exceeded") != -1) {
				BPD
						.getInstance()
						.message(
								ResourceManager
										.getLanguageDependentString("ErrorLDAPSearchPartiallyFailedSizelimitExceeded"),
								JOptionPane.ERROR_MESSAGE);

			} else if (e.getMessage().indexOf("Timelimit Exceeded") != -1) {
				BPD
						.getInstance()
						.message(
								ResourceManager
										.getLanguageDependentString("ErrorLDAPSearchPartiallyFailedTimelimitExceeded"),
								JOptionPane.ERROR_MESSAGE);

			} else {
				BPD
						.getInstance()
						.message(
								ResourceManager
										.getLanguageDependentString("ErrorLDAPSearchFailed"),
								JOptionPane.ERROR_MESSAGE);
			}
			System.err.println("Search failed: " + e.getMessage());
		} catch (Exception e) {
			/* Handle any other types of exceptions. */
			BPD
					.getInstance()
					.message(
							ResourceManager
									.getLanguageDependentString("ErrorLDAPSearchFailed"),
							JOptionPane.ERROR_MESSAGE);
			System.err.println("Non-naming error: " + e.getMessage());
		}
		return entries;
	}

	/**
	 * Shows the dialog for viewing LDAP search results.
	 * 
	 * @return The list of LDAP entries for which participants are to be
	 *         created.
	 */
	private java.util.List showDialog(Window parent,
			final java.util.List allEntries) {
		String title = ResourceManager
				.getLanguageDependentString("DialogLDAPEntries");
		final JDialog myDialog;
		if (parent instanceof JDialog) {
			myDialog = new JDialog((JDialog) parent, title, true);
		} else {
			myDialog = new JDialog((JFrame) parent, title, true);
		}

		final XMLPanel idp = idMapping.getPanel();

		final java.util.List selectedEntries = new java.util.ArrayList();

		Dimension listFieldDimension = new Dimension(800, 500);
		final JList entries;

		JScrollPane scrollActivities = new JScrollPane();
		entries = new JList(allEntries.toArray());
		scrollActivities.setViewportView(entries);
		scrollActivities.setPreferredSize(new Dimension(listFieldDimension));

		JPanel pm = new JPanel();
		Border emptyb = BorderFactory.createEmptyBorder(10, 10, 10, 10);
		Border inb = BorderFactory
				.createMatteBorder(1, 1, 1, 1, Color.darkGray);
		inb = BorderFactory.createTitledBorder(inb, ResourceManager
				.getLanguageDependentString("SelectActivityKey"));
		pm.setBorder(BorderFactory.createCompoundBorder(emptyb, inb));
		pm.setLayout(new BoxLayout(pm, BoxLayout.Y_AXIS));
		pm.add(idp);
		pm.add(scrollActivities);

		JPanel btnPanel = new JPanel();
		JButton dialogImportAllButton = new JButton(ResourceManager
				.getLanguageDependentString("ImportAllKey"));
		java.net.URL u = ResourceManager.getResource("ImportAll"
				+ BPDConstants.IMAGE_SUFFIX);
		if (u != null) {
			dialogImportAllButton.setIcon(new ImageIcon(u));
		}
		dialogImportAllButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				idp.setElements();
				selectedEntries.addAll(allEntries);
				myDialog.dispose();
			}
		});
		btnPanel.add(dialogImportAllButton);

		JButton dialogImportSelectedButton = new JButton(ResourceManager
				.getLanguageDependentString("ImportSelectedKey"));
		u = ResourceManager.getResource("ImportSelected"
				+ BPDConstants.IMAGE_SUFFIX);
		if (u != null) {
			dialogImportSelectedButton.setIcon(new ImageIcon(u));
		}
		dialogImportSelectedButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				idp.setElements();
				selectedEntries.addAll(java.util.Arrays.asList(entries
						.getSelectedValues()));
				myDialog.dispose();
			}
		});
		btnPanel.add(dialogImportSelectedButton);

		JButton dialogCancelButton = new JButton(ResourceManager
				.getLanguageDependentString("CancelKey"));
		u = ResourceManager.getResource("Cancel" + BPDConstants.IMAGE_SUFFIX);
		if (u != null) {
			dialogCancelButton.setIcon(new ImageIcon(u));
		}
		dialogCancelButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				myDialog.dispose();
			}
		});
		btnPanel.add(dialogCancelButton);

		Container cp = myDialog.getContentPane();
		cp.setLayout(new BorderLayout());
		cp.add(pm, BorderLayout.CENTER);
		cp.add(btnPanel, BorderLayout.SOUTH);

		myDialog.getRootPane().getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW)
				.put(KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0, false),
						"Cancel");
		myDialog.getRootPane().getActionMap().put("Cancel",
				new AbstractAction() {
					public void actionPerformed(ActionEvent e) {
						myDialog.dispose();
					}
				});

		myDialog.pack();

		myDialog.setLocationRelativeTo(parent);
		myDialog.setResizable(false);
		myDialog.show();
		return selectedEntries;
	}

	/**
	 * Creates participants based on LDAP entries.
	 * 
	 * @param allEntries
	 *            All retreived entries from LDAP server
	 */
	private void createParticipants(java.util.List allEntries) {
		PackageEditor pe = (PackageEditor) editor;
		com.ds.bpm.bpd.xml.elements.Participants ps = (com.ds.bpm.bpd.xml.elements.Participants) pe
				.getXMLPackage().get("Participants");

		java.util.Iterator iter = allEntries.iterator();
		String mappingAttribute = idMapping.toValue().toString();
		while (iter.hasNext()) {
			Participant p = (Participant) ps.generateNewElement();

			SearchResult nextEntry = (SearchResult) iter.next();
			Attributes attributes = nextEntry.getAttributes();
			String objectClass = null;

			// Iterate through all object classes for entry, and if
			// one with person in it's name is found, take it,
			java.util.Enumeration allObjClasses = null;
			try {
				allObjClasses = attributes.get("objectClass").getAll();
			} catch (Exception ex) {
			}
			if (allObjClasses != null) {
				while (allObjClasses.hasMoreElements()) {
					String val = (String) allObjClasses.nextElement();
					if (val.toLowerCase().endsWith("person")) {
						objectClass = val;
						break;
					} else if (val.equals("organizationalRole")) {
						objectClass = val;
						break;
					}
				}
			}

			// Iterate through all object classes for entry, and if
			// one if one of it's name is organizationalUnit, take it
			if (objectClass == null) {
				try {
					allObjClasses = attributes.get("objectClass").getAll();
				} catch (Exception ex) {
				}
				if (allObjClasses != null) {
					while (allObjClasses.hasMoreElements()) {
						String val = (String) allObjClasses.nextElement();
						if (val.equals("organizationalUnit")) {
							objectClass = val;
							break;
						}
					}
				}
			}
			// If object class is not found till now, take the first one
			if (objectClass == null) {
				try {
					objectClass = (String) attributes.get("objectClass").get();
				} catch (Exception ex) {
				}
			}
			// if it is still null, take the Role
			if (objectClass == null) {
				objectClass = "organizationalRole";
			}

			String id;
			String name = null;
			String pType;
			String description = null;
			try {
				description = (String) attributes.get("description").get();
			} catch (Exception ex) {
			}

			// if objectClass has person in it's name, make a HUMAN type
			// participant
			if (objectClass.toLowerCase().endsWith("person")) {
				pType = ResourceManager.getLanguageDependentString("HUMANKey");
				try {
					name = (String) attributes.get("cn").get();
				} catch (Exception ex) {
				}
				// make an ROLE or ORGANIZATIONAL_UNIT participant
			} else if (objectClass.equals("organizationalRole")) {
				pType = ResourceManager.getLanguageDependentString("ROLEKey");
			} else if (objectClass.equals("organizationalUnit")) {
				pType = ResourceManager
						.getLanguageDependentString("ORGANIZATIONAL_UNITKey");
			} else {
				pType = ResourceManager.getLanguageDependentString("ROLEKey");
			}

			if (name == null) {
				// set name attribute of participant to be the bottom of the
				// tree-structure of DN
				try {
					int ind1 = nextEntry.getName().indexOf("=") + 1;
					int ind2 = nextEntry.getName().indexOf(",");
					if (ind2 == -1)
						ind2 = nextEntry.getName().length();
					name = nextEntry.getName().substring(ind1, ind2);
				} catch (Exception ex) {
				}
			}

			id = p.getID();
			if (!mappingAttribute.equals("NoMapping")) {
				// set id of participant to the attribute choosen for mapping,
				// if it
				// don't have it, leave the automatically generated id
				String userId = null;
				try {
					userId = (String) attributes.get(mappingAttribute).get();
				} catch (Exception ex) {
				}
				// if id made from entry data is not null, and doesn't exist
				// already
				// make it to be the participant id
				if (userId != null && ps.getCollectionElement(userId) == null
						&& XMLCollection.isIdValid(userId)) {
					id = userId;
				}
			}

			// Set the participant id, name, type and extended attribs
			p.set("Id", id);
			if (name != null) {
				p.set("Name", name);
			}
			((ParticipantType) p.get("ParticipantType")).set("Type", pType);
			if (description != null) {
				p.set("Description", description);
			}

			ExtendedAttributes eas = (ExtendedAttributes) p
					.get("ExtendedAttributes");
			// put the DN of entry into extended attributes
			ExtendedAttribute ea = new ExtendedAttribute(eas);
			ea.set("Name", "DN");
			String eaVal = nextEntry.getName();
			ea.set("Value", eaVal);
			eas.add(ea);

			ps.add(p);
			BPD.getInstance().setModified(true);
		}
	}

	public void refreshLabels() {
		if (eps != null) {
			eps.refreshLabelName();
		}
		idMapping.refreshLabelName();
	}
}
