/* WorkflowManager.java
 *
 * Title : BPM工作流图形定义工具BPD
 * Class Desription：工作流管理类
 * Authors： wenzhang li
 * Company： 基督山BPM
 * CreatedTime：2005-12-6
 *
 */
package com.ds.bpm.bpd;

import com.ds.bpm.bpd.enums.ActivityDefTypeEnums;
import com.ds.bpm.bpd.graph.*;
import com.ds.bpm.bpd.graph.AutoActivity;
import com.ds.bpm.bpd.graph.BlockActivity;
import com.ds.bpm.bpd.graph.DeviceActivity;
import com.ds.bpm.bpd.graph.Participant;
import com.ds.bpm.bpd.graph.Transition;
import com.ds.bpm.bpd.xml.*;
import com.ds.bpm.bpd.xml.activity.Activities;
import com.ds.bpm.bpd.xml.elements.*;
import com.ds.bpm.bpd.xml.elements.Package;
import com.ds.bpm.client.ProcessDef;
import com.ds.bpm.enums.route.RouteDirction;
import org.jgraph.graph.*;

import javax.swing.*;
import javax.swing.tree.DefaultMutableTreeNode;
import java.awt.*;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.io.Serializable;
import java.rmi.RemoteException;
import java.util.*;
import java.util.List;

/**
 * Class intended to serve as a control center for creation, removal, resizing and changing position of Participants as
 * well as for doing the same things with Activity objects and Transitions. Class manages this actions in such a way
 * that undoing of operations are possible. It incorporates multiple view and model changes into one by doing them
 * virtually, and after all changes are done, in interaction with BPDGraphModel class applies this changes so that undo
 * is possible. Instance of this class is joined to to all objects of classes derived from AbstractGraph class
 * (PackageGraph, ProcessGraph and BlockActivityGraph).
 * <p>
 * This class also handles the relationships between visual and logical representation of workflow graph.
 * <p>
 * When reading a package from an XML file, this class creates imported objects and establishes relationship between
 * graph objects (classes within bpd and bpd.graph package) and 'graph logic' objects ( classes within bpd.xml package).
 */
public class WorkflowManager implements Serializable {

    /**
     * Graph reference
     */
    private transient AbstractGraph graph;

    private transient PackageEditor pkgEditor;
    ;

    /**
     * Offset for drawing - no meaning (it was used in previous versions)
     */
    private int horizontalOffset = 0;

    // some default values for setting participants and activity sizes
    /**
     * Variable that holds width of process
     */
    private static final int defProcessWidth = BPDConfig.getInstance().getProcessWidth();

    /**
     * Variable that holds height of process
     */
    private static final int defProcessHeight = BPDConfig.getInstance().getProcessHeight();

    /**
     * Variable that holds minimum width for any participant
     */
    private static final int minParWidth = BPDConfig.getInstance().getMinParticipantWidth();

    /**
     * Variable that holds minimum height for any participant
     */
    private static final int minParHeight = BPDConfig.getInstance().getMinParticipantHeight();

    /**
     * Variable that holds the width for participant name section
     */
    private static final int defParNameWidth = BPDConfig.getInstance().getParticipantNameWidth();

    /**
     * Variable that holds width of activities
     */
    private static final int defActivityWidth = BPDConfig.getInstance().getActivityWidth() + 10;

    /**
     * Variable that holds height of activities
     */
    private static final int defActivityHeight = BPDConfig.getInstance().getActivityHeight();

    private boolean creatingGraph = false;

    /**
     * Creates new workflow manager for given graph.
     *
     * @param g The graph that manager manages.
     */
    public WorkflowManager(AbstractGraph g) {
        this.graph = g;
    }

    public WorkflowManager(AbstractGraph g, PackageEditor pkgEditor) {
        this.graph = g;
        this.pkgEditor = pkgEditor;
    }

    /**
     * Returns the graph which is managed.
     */
    public AbstractGraph getGraph() {
        return graph;
    }

    /**
     * Returns the object representing the main element of XML file that defines the workflow model logic.
     */
    public Package getXMLPackage() {
        if (graph != null) {
            return graph.getXMLPackage();
        } else {
            return pkgEditor.getXMLPackage();
        }
    }

    /**
     * Returns the graph model - the model that represents the graph view. (See JGraph documentation).
     */
    public BPDGraphModel graphModel() {
        return (BPDGraphModel) graph.getModel();
    }

    /**
     * Creates graph representation of given workflow process. It creates a graph entities (participants, activities,
     * transitions) and associates the workflow logic to it. The graph entities are inserted according to the data that
     * <tt>wp</tt> object holds (the data from XML file).
     * <p>
     * This is used when reading a workflow definition from an XML file.
     *
     * @param wp Object that mapps the logic of WorkflowProcess element of XML - defines a particular Workflow process.
     */
    public void createWorkflowGraph(WorkflowProcess wp) {
        creatingGraph = true;
        // checks if it is graph made by BPD
        boolean isMine = true;
        if (!getXMLPackage().isMadeByBPD()) {
            isMine = false;
        }
        // ********* the creation other is very important and shouldn't be
        // changed
        // System.out.println("Creating graph for WP "+wp+" and
        // VO="+getVisualOwner());
        Set entitiesToInsert = wp.getEntitiesToInsert(getVisualOwner());
        // System.out.println("Entities to insert="+entitiesToInsert);
        // show Participants
        createGraphParticipants(wp, entitiesToInsert);
        // showing Activities
        createGraphActivitiesAndBlockActivities(wp, entitiesToInsert, isMine);
        // show transitions
        createGraphTransitions(wp, entitiesToInsert, isMine);
        // show start
        createGraphStarts(wp, isMine);
        // show ends
        createGraphEnds(wp, isMine);
        // if workflow model is not created by BPD, perform some
        // kind of layout, and create start and end objects
        if (!isMine) {
            layoutActivities();
            ProcessEditor pe = (ProcessEditor) graph.getProcessEditor();
            Set startDescs = Utils.getStartDescriptions(pe);
            Set endDescs = Utils.getEndDescriptions(pe);
            XMLComplexElement vo = getVisualOwner();
            if (vo instanceof WorkflowProcess) {
                ((WorkflowProcess) vo).setStartDescriptions(startDescs);
                ((WorkflowProcess) vo).setEndDescriptions(endDescs);
            } else {
                ((com.ds.bpm.bpd.xml.activity.Activity) vo).setStartDescriptions(startDescs);
                ((com.ds.bpm.bpd.xml.activity.Activity) vo).setEndDescriptions(endDescs);
            }
        }
        creatingGraph = false;
        Dimension prefSize = new Dimension(getRootParticipantWidth(null, null) + 50, getNewRootParYPos(null, null) + 50);
        graph.setPreferredSize(prefSize);
    }

    /**
     * Returns the object (part of mapped XML logic) that is represented by the graph managed by WorklowManager. That
     * object can be instance of the Package, WorkflowProcess or ...xml.elements.BlockActivity class, and is held as a
     * property of the manager's graph object.
     *
     * @return The object (representing XML logic) that is represented by this manager's graph.
     */
    public XMLComplexElement getVisualOwner() {
        if (graph instanceof BlockActivityGraph) {
            return (com.ds.bpm.bpd.xml.activity.Activity) ((BlockActivityGraph) graph).getMyBlockActivity().getUserObject();
        } else if (graph instanceof ProcessGraph) {
            return (WorkflowProcess) graph.getPropertyObject();
        } else {
            return (Package) graph.getPropertyObject();
        }
    }

    /**
     * Returns the (XML logic) Activities object, that holds all (XML logic) Activity objects which graphical
     * representation is managed by this workflow manager. That could be activites belonging directly to the
     * WorkflowProcess, or activities belonging to the BlockActivity element. If this is a manager of Package graph,
     * <tt>null</tt> is returned.
     */
    private Activities getActivitiesCollection() {
        Activities acts;
        XMLComplexElement myOwner = getVisualOwner();
        if (myOwner instanceof WorkflowProcess) {
            acts = (Activities) myOwner.get("Activities");
        } else {
            BlockActivity ba = ((BlockActivityGraph) graph).getMyBlockActivity();
            ActivitySet as = getBlockActivitySet(ba);
            acts = (Activities) as.get("Activities");
        }
        return acts;
    }

    /**
     * Returns the (XML logic) Transitions object, that holds all (XML logic) Transition objects which graphical
     * representation is managed by this workflow manager. That could be transitions belonging directly to the
     * WorkflowProcess, or transitions belonging to the BlockActivity element. If this is a manager of Package graph,
     * <tt>null</tt> is returned.
     */
    public Transitions getTransitionsCollection() {
        Transitions trans;
        XMLComplexElement myOwner = getVisualOwner();
        if (myOwner instanceof WorkflowProcess) {
            trans = (Transitions) myOwner.get("Transitions");
        } else {
            BlockActivity ba = ((BlockActivityGraph) graph).getMyBlockActivity();
            ActivitySet as = getBlockActivitySet(ba);
            trans = (Transitions) as.get("Transitions");
        }
        return trans;
    }

    /**
     * Returns the (XML logic) ActivitySet object that is associated to the given graph's object logic.
     */
    private ActivitySet getBlockActivitySet(BlockActivity ba) {
        try {
            String ID = ba.getBlockID();
            ActivitySet as = ((ActivitySets) getWorkflowProcess().get("ActivitySets")).getActivitySet(ID);
            return as;
        } catch (Exception ex) {
            return null;
        }
    }

    /**
     * Returns the (XML logic) WorkflowProcess object that is represented by the manager's graph, or (if the graph
     * represents the (XML logic) BlockActivity content) the WorkflowProcess object that holds BlockActivity represented
     * by manager's graph. If graph represents (XML logic) Package object, <tt>null</tt> is returned.
     */
    private WorkflowProcess getWorkflowProcess() {
        if (graph.getPropertyObject() instanceof WorkflowProcess) {
            return (WorkflowProcess) graph.getPropertyObject();
        } else {
            return null;
        }
    }

    /**
     * Inserts new process cell into (graph) model and creates new (XML logic) WorkflowProcess object. Depending on
     * <tt>updateCollection</tt> parameter, it calls the method that updates the collection of existing WorkflowProcess
     * objects (it is set to <tt>true</tt> when user inserts the new WorkflowProcess into the Package graph, and to
     * <tt>false</tt> during importing from XML). the process cell should be inserted.
     *
     * @param updateCollection <tt>true</tt> if collection of WorkflowProcess objects should be updated (user inserts object), and
     *                         <tt>false</tt> if not (XML file is importing, and objects are constructed depending on data held
     *                         within the file).
     * @return The created graph object.
     */
    public com.ds.bpm.bpd.graph.Process insertProcess(WorkflowProcess workflowProcess, boolean updateCollection) {
        return insertProcess(workflowProcess, updateCollection, null);
    }

    /**
     * WENZHANG 06-03-04 本方法存在缺陷使用时请注意条件 本方法适用的范围：每个包内只能有一个主流程 如果主流程不存在（包内只有一个或多个子流程情况可能存在） 如果存在多个主流程按顺序返回第一个
     *
     * @return
     */
    public WorkflowProcess getMainProcess() {
        WorkflowProcesses wps = (WorkflowProcesses) pkgEditor.getRealXMLPackage().get("WorkflowProcesses");
        Iterator it = wps.getTableElements().iterator();
        for (; it.hasNext(); ) {
            WorkflowProcess wp = (WorkflowProcess) it.next();
            if (!wp.isSubFlow()) {
                return wp;
            }
    }

        return null;
    }

    public com.ds.bpm.bpd.graph.Process newSubProcess(WFApplication app, String processId) {
        com.ds.bpm.bpd.graph.Process pr = new com.ds.bpm.bpd.graph.Process(pkgEditor);

        WorkflowProcess wp = new WorkflowProcess((WorkflowProcesses) pkgEditor.getRealXMLPackage().get("WorkflowProcesses"), pkgEditor.getRealXMLPackage(), app, true, processId);
        pr.setUserObject(wp);
        // 是否添加一个流程
        Object[] insert = new Object[]{pr};
        refreshCollections(insert, true);

        return pr;
    }

    public com.ds.bpm.bpd.graph.Process insertProcess(WorkflowProcess workflowProcess, boolean updateCollection, WFApplication app) {
        com.ds.bpm.bpd.graph.Process pr = new com.ds.bpm.bpd.graph.Process(pkgEditor);
        WorkflowProcess wp;

        if (workflowProcess != null) {
            wp = workflowProcess;
        } else {
            if (app != null) {
                wp = new WorkflowProcess((WorkflowProcesses) pkgEditor.getRealXMLPackage().get("WorkflowProcesses"), pkgEditor.getRealXMLPackage(), app, updateCollection, null);
            } else {
                wp = new WorkflowProcess((WorkflowProcesses) pkgEditor.getRealXMLPackage().get("WorkflowProcesses"), pkgEditor.getRealXMLPackage());

            }
            pr.setUserObject(wp);
        }

        // 是否添加一个流程
        Object[] insert = new Object[]{pr};

        if (updateCollection) {
            refreshCollections(insert, true);
        }

        return pr;
    }

    /**
     * Inserts cells <b>insert</b> into model. First, the parent of cells is searched, and when found, put into
     * ParentMap object (they are not inserted into model at ones). After that model's view is arranged (Participants
     * are moved and translated along with it's children cells) to suite to the new model state - this is done
     * "virtually" which means that changes are not directly applied to view until all changes are made. At the end, all
     * changes are applied to model and view. Such procedure enables compound undo support. <BR>
     * This method is called when pasting cells into model.
     *
     * @see #updateModelAndArrangeParticipants
     */
    public void insertCellsAndArrangeParticipants(Object[] insert, Map viewMap) {
        if (BPDGraphModel.getRootParticipants(graphModel()) != null) {
            updateModelAndArrangeParticipants(insert, null, null, viewMap, ResourceManager.getLanguageDependentString("MessageDuplicatingObject"), true);
        } else {
            JOptionPane.showMessageDialog(graph.getEditor().getWindow(), ResourceManager.getLanguageDependentString("WarningCannotInsertObjectsOutsideParticipant"), BPD.getAppTitle(), JOptionPane.WARNING_MESSAGE);
        }
    }

    // paste cells
    public void insertCellsAndArrangeParticipants(Object[] insert, Map viewMap, ConnectionSet cs) {
        if (BPDGraphModel.getRootParticipants(graphModel()) != null) {
            updateModelAndArrangeParticipants(insert, null, cs, null, viewMap, ResourceManager.getLanguageDependentString("MessageDuplicatingObject"), true);
        } else {
            JOptionPane.showMessageDialog(graph.getEditor().getWindow(), ResourceManager.getLanguageDependentString("WarningCannotInsertObjectsOutsideParticipant"), BPD.getAppTitle(), JOptionPane.WARNING_MESSAGE);
        }
    }

    /**
     * Moves cells given in a <b>propertyMap</b> to new location (also given in <b>propertyMap</b>. First, the parent of
     * cells is searched, and when found, put into ParentMap object (the model is not changed at ones). After that
     * model's view is arranged (Participants are moved and translated along with it's children cells) to suite to the
     * new model state - this is done "virtually" which means that changes are not directly applied to view until all
     * changes are made. At the end, all changes are applied to model and view. Such procedure enables compound undo
     * support. <BR>
     * This method is called when moving cells along participants.
     *
     * @see #updateModelAndArrangeParticipants
     */
    public void moveCellsAndArrangeParticipants(Map propertyMap) {
        updateModelAndArrangeParticipants(null, propertyMap, null, null, ResourceManager.getLanguageDependentString("MessageMovingObjects"), false);
    }

    /**
     * Inserts new Subflow cell at position <b>p</b> into model. First, the parent of new Subflow is searched, and when
     * found, put into ParentMap object (it is not inserted into model at ones). After that model's view is arranged
     * (Participants are moved and translated along with it's children cells) to suite to the new model state - this is
     * done "virtually" which means that changes are not directly applied to view until all changes are made. At the
     * end, all changes are applied to model and view. Such procedure enables compound undo support. <BR>
     * This method is called when user inserts new Subflow into graph.
     *
     * @see #updateModelAndArrangeParticipants
     */
    public void insertSubflowAndArrangeParticipants(Point p) {
        Set rootParticipants = BPDGraphModel.getRootParticipants(graphModel());
        if (rootParticipants != null && rootParticipants.size() > 0) {
            Map viewMap = new Hashtable();
            Subflow sbf = createSubflow(p, null, viewMap);
            // updateModelAndArrangeParticipants(new Object[]
            // {sbf},viewMap,null,null,
            updateModelAndArrangeParticipants(new Object[]{sbf}, null, null, viewMap, ResourceManager.getLanguageDependentString("MessageInsertingSubflowActivity"), true);
        } else {
            JOptionPane.showMessageDialog(graph.getEditor().getWindow(), ResourceManager.getLanguageDependentString("WarningCannotInsertSubflowActivityOutsideParticipant"), BPD.getAppTitle(), JOptionPane.WARNING_MESSAGE);
        }
    }

    /**
     * Inserts new Subflow cell at position <b>p</b> into model. First, the parent of new Subflow is searched, and when
     * found, put into ParentMap object (it is not inserted into model at ones). After that model's view is arranged
     * (Participants are moved and translated along with it's children cells) to suite to the new model state - this is
     * done "virtually" which means that changes are not directly applied to view until all changes are made. At the
     * end, all changes are applied to model and view. Such procedure enables compound undo support. <BR>
     * This method is called when user inserts new Subflow into graph.
     *
     * @see #updateModelAndArrangeParticipants
     */
    public void insertOutflowAndArrangeParticipants(Point p) {
        Set rootParticipants = BPDGraphModel.getRootParticipants(graphModel());
        if (rootParticipants != null && rootParticipants.size() > 0) {
            Map viewMap = new Hashtable();
            Outflow sbf = createOutflow(p, null, viewMap);
            // updateModelAndArrangeParticipants(new Object[]
            // {sbf},viewMap,null,null,
            updateModelAndArrangeParticipants(new Object[]{sbf}, null, null, viewMap, ResourceManager.getLanguageDependentString("MessageInsertingSubflowActivity"), true);
        } else {
            JOptionPane.showMessageDialog(graph.getEditor().getWindow(), ResourceManager.getLanguageDependentString("WarningCannotInsertSubflowActivityOutsideParticipant"), BPD.getAppTitle(), JOptionPane.WARNING_MESSAGE);
        }
    }

    /**
     * Inserts new BlockActivity cell at position <b>p</b> into model. First, the parent of new BlockActivity is
     * searched, and when found, put into ParentMap object (it is not inserted into model at ones). After that model's
     * view is arranged (Participants are moved and translated along with it's children cells) to suite to the new model
     * state - this is done "virtually" which means that changes are not directly applied to view until all changes are
     * made. At the end, all changes are applied to model and view. Such procedure enables compound undo support. <BR>
     * This method is called when user inserts new BlockActivity into graph.
     *
     * @see #updateModelAndArrangeParticipants
     */
    public void insertBlockActivityAndArrangeParticipants(Point p) {
        Set rootParticipants = BPDGraphModel.getRootParticipants(graphModel());
        if (rootParticipants != null && rootParticipants.size() > 0) {
            Map viewMap = new Hashtable();
            BlockActivity bla = createBlockActivity(p, null, viewMap);
            // updateModelAndArrangeParticipants(new Object[]
            // {bla},viewMap,null,null,
            updateModelAndArrangeParticipants(new Object[]{bla}, null, null, viewMap, ResourceManager.getLanguageDependentString("MessageInsertingBlockActivity"), true);
        } else {
            JOptionPane.showMessageDialog(graph.getEditor().getWindow(), ResourceManager.getLanguageDependentString("WarningCannotInsertBlockActivityOutsideParticipant"), BPD.getAppTitle(), JOptionPane.WARNING_MESSAGE);
        }
    }

    /**
     * Inserts new Start activity cell at position <b>p</b> into model. First, the parent of new Activity is searched,
     * and when found, put into ParentMap object (it is not inserted into model at ones). After that model's view is
     * arranged (Participants are moved and translated along with it's children cells) to suite to the new model state -
     * this is done "virtually" which means that changes are not directly applied to view until all changes are made. At
     * the end, all changes are applied to model and view. Such procedure enables compound undo support. <BR>
     * This method is called when user inserts new Start into graph.
     *
     * @see #updateModelAndArrangeParticipants
     */
    public void insertStartAndArrangeParticipants(Point p) {
        // Set rootParticipants=BPDGraphModel.getRootParticipants(graphModel());
        // if(rootParticipants != null && rootParticipants.size()>0) {
        Map viewMap = new Hashtable();
        Start s = createStart(p, viewMap);
        updateModelAndArrangeParticipants(new Object[]{s}, null, null, viewMap, ResourceManager.getLanguageDependentString("MessageInsertingStart"), true);
        /*
         * } else { String msg=ResourceManager.getLanguageDependentString(
         * "WarningCannotInsertStartOutsideParticipant"); JOptionPane.showMessageDialog(graph.getEditor().getWindow(),
         * msg,BPD.getAppTitle(),JOptionPane.WARNING_MESSAGE); }
         */
        if (BPDConfig.getInstance().getTooltipStatus()) {
            ToolTipManager.sharedInstance().setEnabled(true);
        }
    }

    /**
     * Inserts new End activity cell at position <b>p</b> into model. First, the parent of new Activity is searched, and
     * when found, put into ParentMap object (it is not inserted into model at ones). After that model's view is
     * arranged (Participants are moved and translated along with it's children cells) to suite to the new model state -
     * this is done "virtually" which means that changes are not directly applied to view until all changes are made. At
     * the end, all changes are applied to model and view. Such procedure enables compound undo support. <BR>
     * This method is called when user inserts new End into graph.
     *
     * @see #updateModelAndArrangeParticipants
     */
    public void insertEndAndArrangeParticipants(Point p) {
        Set rootParticipants = BPDGraphModel.getRootParticipants(graphModel());
        if (rootParticipants != null && rootParticipants.size() > 0) {
            Map viewMap = new Hashtable();
            Activity e = createEnd(p, viewMap);
            // updateModelAndArrangeParticipants(new Object[]
            // {e},viewMap,null,null,
            updateModelAndArrangeParticipants(new Object[]{e}, null, null, viewMap, ResourceManager.getLanguageDependentString("MessageInsertingEnd"), true);
        } else {
            JOptionPane.showMessageDialog(graph.getEditor().getWindow(), ResourceManager.getLanguageDependentString("WarningCannotInsertEndOutsideParticipant"), BPD.getAppTitle(), JOptionPane.WARNING_MESSAGE);
        }
    }

    /**
     * Inserts new Activity cell at position <b>p</b> into model. First, the parent of new Activity is searched, and
     * when found, put into ParentMap object (it is not inserted into model at ones). After that model's view is
     * arranged (Participants are moved and translated along with it's children cells) to suite to the new model state -
     * this is done "virtually" which means that changes are not directly applied to view until all changes are made. At
     * the end, all changes are applied to model and view. Such procedure enables compound undo support. <BR>
     * This method is called when user inserts new generic activity into graph.
     *
     * @see #updateModelAndArrangeParticipants
     */
    public void insertActivityAndArrangeParticipants(Point p) {
        Set rootParticipants = BPDGraphModel.getRootParticipants(graphModel());
        if (rootParticipants != null && rootParticipants.size() > 0) {
            Map viewMap = new Hashtable();
            Activity act = createActivity(p, null, viewMap);
            // updateModelAndArrangeParticipants(new Object[]
            // {act},viewMap,null,null,
            updateModelAndArrangeParticipants(new Object[]{act}, null, null, viewMap, ResourceManager.getLanguageDependentString("MessageInsertingManualActivity"), true);
        } else {
            JOptionPane.showMessageDialog(graph.getEditor().getWindow(), ResourceManager.getLanguageDependentString("WarningCannotInsertGenericManualOutsideParticipant"), BPD.getAppTitle(), JOptionPane.WARNING_MESSAGE);
        }
    }

    /**
     * Inserts new AutoActivity activity cell at position <b>p</b> into model. First, the parent of new Activity is
     * searched, and when found, put into ParentMap object (it is not inserted into model at ones). After that model's
     * view is arranged (Participants are moved and translated along with it's children cells) to suite to the new model
     * state - this is done "virtually" which means that changes are not directly applied to view until all changes are
     * made. At the end, all changes are applied to model and view. Such procedure enables compound undo support. <BR>
     * This method is called when user inserts new AutoActivity activity into graph.
     *
     * @see #updateModelAndArrangeParticipants
     */
    public void insertAutoActivityAndArrangeParticipants(Point p) {
        Set rootParticipants = BPDGraphModel.getRootParticipants(graphModel());
        if (rootParticipants != null && rootParticipants.size() > 0) {
            Map viewMap = new Hashtable();
            AutoActivity r = createAutoActivity(p, null, viewMap);
            // updateModelAndArrangeParticipants(new Object[]
            // {r},viewMap,null,null,
            updateModelAndArrangeParticipants(new Object[]{r}, null, null, viewMap, ResourceManager.getLanguageDependentString("MessageInsertingAutoActivity"), true);
        } else {
            JOptionPane.showMessageDialog(graph.getEditor().getWindow(), ResourceManager.getLanguageDependentString("WarningCannotInsertAutoActivityOutsideParticipant"), BPD.getAppTitle(), JOptionPane.WARNING_MESSAGE);
        }
    }

    public void insertServiceActivityAndArrangeParticipants(Point p) {
        Set rootParticipants = BPDGraphModel.getRootParticipants(graphModel());
        if (rootParticipants != null && rootParticipants.size() > 0) {
            Map viewMap = new Hashtable();
            ServiceActivity r = createServiceActivity(p, null, viewMap);
            // updateModelAndArrangeParticipants(new Object[]
            // {r},viewMap,null,null,
            updateModelAndArrangeParticipants(new Object[]{r}, null, null, viewMap, ResourceManager.getLanguageDependentString("MessageInsertingServiceActivity"), true);
        } else {
            JOptionPane.showMessageDialog(graph.getEditor().getWindow(), ResourceManager.getLanguageDependentString("WarningCannotInsertServiceActivityOutsideParticipant"), BPD.getAppTitle(), JOptionPane.WARNING_MESSAGE);
        }
    }

    public void insertDeviceActivityAndArrangeParticipants(Point p) {
        Set rootParticipants = BPDGraphModel.getRootParticipants(graphModel());
        if (rootParticipants != null && rootParticipants.size() > 0) {
            Map viewMap = new Hashtable();
            DeviceActivity r = createDeviceActivity(p, null, viewMap);
            // updateModelAndArrangeParticipants(new Object[]
            // {r},viewMap,null,null,
            updateModelAndArrangeParticipants(new Object[]{r}, null, null, viewMap, ResourceManager.getLanguageDependentString("MessageInsertingServiceActivity"), true);
        } else {
            JOptionPane.showMessageDialog(graph.getEditor().getWindow(), ResourceManager.getLanguageDependentString("WarningCannotInsertServiceActivityOutsideParticipant"), BPD.getAppTitle(), JOptionPane.WARNING_MESSAGE);
        }
    }

    public void insertEventActivityAndArrangeParticipants(Point p) {
        Set rootParticipants = BPDGraphModel.getRootParticipants(graphModel());
        if (rootParticipants != null && rootParticipants.size() > 0) {
            Map viewMap = new Hashtable();
            EventActivity r = createEventActivity(p, null, viewMap);
            // updateModelAndArrangeParticipants(new Object[]
            // {r},viewMap,null,null,
            updateModelAndArrangeParticipants(new Object[]{r}, null, null, viewMap, ResourceManager.getLanguageDependentString("MessageInsertingServiceActivity"), true);
        } else {
            JOptionPane.showMessageDialog(graph.getEditor().getWindow(), ResourceManager.getLanguageDependentString("WarningCannotInsertServiceActivityOutsideParticipant"), BPD.getAppTitle(), JOptionPane.WARNING_MESSAGE);
        }
    }

    /**
     * Inserts new Participant cell at position <b>p</b> into model. First, the parent of new Participant is searched,
     * and if found, put into ParentMap (it is not inserted into model at ones). If parent participant isn't found ->
     * root participant will be inserted. After that model's view is arranged (Participants are moved and translated
     * along with it's children cells) to suite to the new model state - this is done "virtually" which means that
     * changes are not directly applied to view until all changes are made. At the end, all changes are applied to model
     * and view. Such procedure enables compound undo support. <BR>
     * This method is called when inserting new Participant into model. <BR>
     * If <tt>parUO</tt> is <tt>null</tt>, the new object is inserted into XML logic, and if not, the predefined XML
     * object is used.
     *
     * @param p     The position of participant within the graph.
     * @param parUO The XML logic object that holds the data for participant to be inserted and that will be held as a
     *              property of the graph object.
     */
    public Participant insertParticipantAndArrangeParticipants(Point p, com.ds.bpm.bpd.xml.elements.Participant parUO) {
        ToolTipManager.sharedInstance().setEnabled(false);
        Participant par = null;
        Map viewMap = new HashMap();
        ParentMap parentMap = null;
        Map propertyMap = null;
        // get the topmost participant at the location where
        // user want to insert participant
        Object cell = ((ProcessGraph) graph).getFirstParticipantForLocation(p.x, p.y);
        // if there is no cell at the mouse position,
        // add new participant below existing ones
        if (cell == null) {
            par = createParticipant(new Rectangle(horizontalOffset, getNewRootParYPos(null, null), getRootParticipantWidth(null, null), minParHeight), viewMap);
            propertyMap = new HashMap(viewMap);
            // otherwise, display an warning message that
            // participant can't be inserted here
        } else {
            JOptionPane.showMessageDialog(graph.getEditor().getWindow(), ResourceManager.getLanguageDependentString("WarningParticipantCannotHaveParticipants"), BPD.getAppTitle(), JOptionPane.WARNING_MESSAGE);
            return null;
        }
        Participants wfp = (Participants) getWorkflowProcess().get("Participants");
        // if user inserts an existing (logical) participant into graph
        if (parUO != null) {
            Object[] insert = new Object[]{par};
            graphModel().insertAndEdit(insert, propertyMap, null, parentMap, null, ResourceManager.getLanguageDependentString("MessageInsertingParticipant"));
            par.setUserObject(parUO);
            // notifies participants user object that one of it's graph
            // presentation is added
            // ADDED 19.02.03 - conditional graph reference adding, only
            // if this is not external model graph
            if (!getXMLPackage().isReadOnly()) {
                parUO.graphReferenceAdded();
            }
            // if user creates new (logical) participant by selecting
            // it from the toolbox toolbar
        } else {
            parUO = (com.ds.bpm.bpd.xml.elements.Participant) wfp.generateNewElement();
            // show dialog for entering initial participant data
            Window w = graph.getEditor().getWindow();
            XMLElementDialog de = new XMLElementDialog((JFrame) w, ResourceManager.getLanguageDependentString("DialogNewParticipant"));
            de.editXMLElement(parUO.getPanel(), true, false, false);
            /*
             * // if cancel hasn't been pressed if (!de.isCanceled()) { Object[] insert = new Object[]{par};
             * //graphModel().insertAndEdit(insert,null,propertyMap,parentMap,viewMap,
             * graphModel().insertAndEdit(insert, propertyMap, null, parentMap, null,
             * ResourceManager.getLanguageDependentString("MessageInsertingParticipant")); par.setUserObject(parUO); //
             * notifies participants user object that one of it's graph // presentation is added // ADDED 19.02.03 -
             * conditional graph reference adding, only // if this is not external model graph if
             * (!getXMLPackage().isReadOnly()) { parUO.graphReferenceAdded(); } // adds object to the collection
             * wfp.add(parUO); }
             */
            Object[] insert = new Object[]{par};
            graphModel().insertAndEdit(insert, propertyMap, null, parentMap, null, ResourceManager.getLanguageDependentString("MessageInsertingParticipant"));
            par.setUserObject(parUO);
            // notifies participants user object that one of it's graph
            // presentation is added
            // ADDED 19.02.03 - conditional graph reference adding, only
            // if this is not external model graph
            if (!getXMLPackage().isReadOnly()) {
                parUO.graphReferenceAdded();
            }
            // adds object to the collection
            wfp.add(parUO);
        }
        if (BPDConfig.getInstance().getTooltipStatus()) {
            ToolTipManager.sharedInstance().setEnabled(true);
        }
        if (!creatingGraph) {
            Dimension prefSize = new Dimension(getRootParticipantWidth(propertyMap, parentMap) + 50, getNewRootParYPos(propertyMap, parentMap) + 50);
            graph.setPreferredSize(prefSize);
        }
        return par;
    }

    /**
     * Called when user wants to insert an existing (logical) participant into graph. Participant is inserted below all
     * already inserted participants.
     *
     * @return <tt>true</tt> if participant is successfully inserted into graph, <tt>false</tt> otherwise.
     */
    public boolean showNode(com.ds.bpm.bpd.xml.elements.Participant parUO) {
        if (parUO == null)
            return false;
        Point ofInsert = new Point(10, Integer.MAX_VALUE);
        Object graphObj = getGraphObjectForXMLParticipant(parUO);
        // if participant is already inserted into graph, just return true
        if (null != graphObj) {
            return true;
        }
        Object ins = insertParticipantAndArrangeParticipants(ofInsert, parUO);
        if (ins == null) {
            return false;
        } else {
            return true;
        }
    }

    /**
     * Called when user wants to remove an existing (logical) participant from the graph. The participant can be removed
     * only if there are no activities that it is responsible for (activity graph objects that lies inside the
     * participant graph object).
     */
    public void hideNode(com.ds.bpm.bpd.xml.elements.Participant parUO) {
        Participant p = getGraphObjectForXMLParticipant(parUO);
        if (p != null && !p.hasAnyActivity()) {
            removeCellsAndArrangeParticipants(new Object[]{p});
        }
    }

    /**
     * Returns the graph object representing given (XML logic) Participant object.
     *
     * @param source The (XML logic) Participant object which graph object should be returned by this method.
     * @return The graph object for given XML logic object if it is inserted into graph, <tt>null</tt> otherwise.
     */
    private Participant getGraphObjectForXMLParticipant(com.ds.bpm.bpd.xml.elements.Participant source) {
        Set graphParts = BPDGraphModel.getAllParticipantsInModel(graphModel());
        if (graphParts != null) {
            Participant graphPart;
            Iterator it = graphParts.iterator();
            while (it.hasNext()) {
                graphPart = (Participant) it.next();
                if (graphPart.getUserObject() == source) {
                    return graphPart;
                }
            }
        }
        return null;
    }

    /**
     * Inserts new Transition cell, between points <b>start</b> and <b>end</b>, connected to the ports <b>source</b> and
     * <b>target</b> into model. <BR>
     * This method is called when inserting new Transition into graph. Depending on <tt>updateCollection</tt> parameter,
     * it calls the method that updates the collection of existing Transition objects (it is set to <tt>true</tt> when
     * user inserts the new Transition into the Process or BlockActivity graph, and to <tt>false</tt> during importing
     * from XML).
     *
     * @param start            The position of start point of transition within the graph.
     * @param end              The position of end point of transition within the graph.
     * @param source           The source port of transition's graph object.
     * @param target           The target port of transition's graph object.
     * @param updateCollection <tt>true</tt> if collection of Transition objects should be updated (user inserts object), and
     *                         <tt>false</tt> if not (XML file is importing, and objects are constructed depending on data held
     *                         within the file).
     * @return The created transition's graph object. 创建一路由
     */
    public Transition insertTransition(Point start, Point end, PortView source, PortView target, boolean isRouting, boolean updateCollection) {
        Point p = (Point) graph.fromScreen(new Point(start));// HM, JGraph3.4.1
        Point p2 = (Point) graph.fromScreen(new Point(end));// HM, JGraph3.4.1
        ArrayList list = new ArrayList();
        list.add(p);
        list.add(p2);
        // creating line style
        Map map = GraphConstants.createMap();
        GraphConstants.setPoints(map, list);
        if (isRouting) {
            GraphConstants.setRouting(map, GraphConstants.ROUTING_SIMPLE);
        }
        GraphConstants.setLineColor(map, Utils.getColor(BPDConfig.getInstance().getTransitionColor()));
        GraphConstants.setLineEnd(map, GraphConstants.ARROW_TECHNICAL);
        GraphConstants.setEndFill(map, true);
        GraphConstants.setEndSize(map, 10);
        GraphConstants.setMoveable(map, true);
        // GraphConstants.setFont(map,Font.getFont("10"));
        // GraphConstants.setFontSize(map, 10);
        Map viewMap = new Hashtable();
        Transition transition = new Transition();
        viewMap.put(transition, map);
        Object[] insert = new Object[]{transition};
        ConnectionSet cs = new ConnectionSet();
        cs.connect(transition, source.getCell(), target.getCell());
        Transitions ts = getTransitionsCollection();
        com.ds.bpm.bpd.xml.elements.Transition uo = new com.ds.bpm.bpd.xml.elements.Transition(ts);
        uo.get("Name").setValue(ResourceManager.getLanguageDependentString("TransitionKey"));

        // 判断目标活动是否与开始活动相连
        Object p1 = ((DefaultPort) target.getCell()).getParent();
        if (p1 instanceof Activity) {
            Activity act = (Activity) p1;
            Set trns = act.getIncomingTransitions();
            if (trns.size() > 0) {
                if (((Transition) trns.toArray()[0]).getSourceActivity() instanceof Start) {
                    // 将此路由的方向设置为后退路由，并且改变路由的颜色
                    uo.setAttributeValue("RouteDirection", RouteDirction.BACK.getType(), true);
                    GraphConstants.setLineColor(map, Utils.getColor(BPDConfig.getInstance().getBackwardTransitionColor()));
                    GraphConstants.setLineEnd(map, GraphConstants.ARROW_CLASSIC);
                    GraphConstants.setEndFill(map, true);
                    GraphConstants.setEndSize(map, 10);
                    // GraphConstants.setFontSize(map, 20);
                }
            }
        }
        transition.setUserObject(uo);
        // cs.connect(transition,source.getCell(),true);
        // cs.connect(transition,target.getCell(),false);
        String undoMsg = ResourceManager.getLanguageDependentString("MessageInsertingTransition");
        // graphModel().insertAndEdit(insert,cs,null,null,viewMap,undoMsg);
        graphModel().insertAndEdit(insert, viewMap, cs, null, null, undoMsg);
        // must set source and target activity objects after inserting into
        // model
        com.ds.bpm.bpd.xml.activity.Activity sourceAct = null;
        try {
            sourceAct = (com.ds.bpm.bpd.xml.activity.Activity) transition.getSourceActivity().getPropertyObject();
            uo.setFrom(sourceAct);
        } catch (Exception e) {
        }
        try {
            Object obj = transition.getTargetActivity();

            uo.setTo(transition.getTargetActivity().getPropertyObject());
        } catch (Exception e) {
            e.printStackTrace();
        }
        // refreshes collection of transitions within workflow process
        if (updateCollection) {
            refreshCollections(insert, true);
        }
        return transition;
    }

    /**
     * Removes cells <b>cellsToDelete</b> from model. This means that given cells and all of their descendants as well
     * as all transitions that connects given cells, will be removed from model. First, all remained participants are
     * moved and resized according to participants that are beeing removed and ParentMap for all removed cells is
     * created (all these things are made "virtually" - not applied to model and view). After that model's new view is
     * arranged (Participants are moved and translated (along with it's children cells) according to the remained
     * children - this is also done "virtually". At the end, all changes are applied to model and view. Such procedure
     * enables compound undo support. <BR>
     * This method is called when deleting or cutting cells from graph.
     */
    public void removeCellsAndArrangeParticipants(Object[] cellsToDelete) {
        Set participantsToArrange = new HashSet();
        Map propertyMap = new HashMap();
        ParentMap parentMap = new BPDParentMap();

        Set ports = new HashSet();
        // begining arrangement of parent of cells that will be deleted
        if (cellsToDelete != null && cellsToDelete.length > 0) {
            for (int i = 0; i < cellsToDelete.length; i++) {
                if (cellsToDelete[i] instanceof Participant) {
                    Participant par = (Participant) cellsToDelete[i];

                    // getting bounds of rectangle
                    Rectangle2D r = getBounds(par, propertyMap);// HM, JGraph3.4.1
                    int yPos = r.getBounds().y + r.getBounds().height - 1;// HM, JGraph3.4.1

                    // resize all parent participants (reduce their sizes) if needed, and
                    // also if needed, translate up all participants under yPos
                    Participant ppar = (Participant) par.getParent();
                    if (ppar != null) {
                        // resizing and translating is needed only if "first" parent
                        // has other children except one that is beeing removed
                        // or if its height can be reduced
                        if (ppar.getChildCount() > 1 || getParticipantHeight(ppar, propertyMap) > minParHeight) {
                            // gets all parents and resizes it
                            Object[] allParents = ppar.getPath();
                            // calculates resize value
                            int resizeValue = r.getBounds().height;// HM, JGraph3.4.1
                            int pHeight = getParticipantHeight(ppar, propertyMap);
                            // do not allow to resize participant under it's minimal height
                            if (pHeight - r.getBounds().height < minParHeight) {// HM, JGraph3.4.1
                                resizeValue = pHeight - minParHeight;
                            }
                            resize(allParents, propertyMap, 0, -resizeValue);
                            // translate up all participants under yPos
                            translateVertically(propertyMap, null, yPos, -resizeValue);
                        }
                    }
                    // if participant is root there is a need of translating participants under it
                    else {
                        // translate up all participants under yPos
                        translateVertically(propertyMap, null, yPos, -r.getBounds().height);// HM, JGraph3.4.1
                    }
                    // if some port is originally set to be deleted it must be removed
                } else if (cellsToDelete[i] instanceof Port) {
                    ports.add(cellsToDelete[i]);
                }
            }

            // removing ports if they were somehow selected
            Set ctd = new HashSet(Arrays.asList(cellsToDelete));
            ctd.removeAll(ports);
            if (ctd.size() == 0)
                return;
            cellsToDelete = ctd.toArray();

            // after previous, participant's are transleted and resized to certain extent,
            // now final resizing and translation takes place (translation and resizing
            // that calculates minimal possible height and width of participants according
            // to the activities that are contained within participant after deletion)

            // All cells in model to be deleted
            Set cellsToDel = BPDGraphModel.getDescendants(graphModel(), cellsToDelete);
            // getting transitions(edges) which are connected to the cellsForDel -> also has to be deleted
            Set edges = graphModel().getEdges(graphModel(), cellsToDel.toArray());

            // putting all items for deletation (edges and cells) together - thats ALL FOR DELETION
            cellsToDel.addAll(edges);

            // Separate cells and edges
            Set allEdgesToDelete = new HashSet();
            Set allCellsToDelete = new HashSet();
            Iterator it = cellsToDel.iterator();
            while (it.hasNext()) {
                Object cell = it.next();
                if (cell instanceof Edge) {
                    allEdgesToDelete.add(cell);
                } else if (!(cell instanceof Port)) {
                    allCellsToDelete.add(cell);
                }
            }

            // working resize and translation only with cells (edges doesn't count)
            cellsToDelete = allCellsToDelete.toArray();

            for (int i = 0; i < cellsToDelete.length; i++) {
                // adding parent of removing Activity cell (if there is one and if
                // it is Participant) into array for arranging participants, and creating
                // entry in parentMap -> this will be of use after basic resizing
                // and translating operations
                Object parent = ((DefaultMutableTreeNode) cellsToDelete[i]).getParent();
                if ((parent != null) && (parent instanceof Participant)) {
                    if (cellsToDelete[i] instanceof Activity) {
                        participantsToArrange.add(parent);
                    }
                }
                parentMap.addEntry(cellsToDelete[i], null);
            }

            // resizing remained participants
            resizeAllParticipantsHorizontally(propertyMap, parentMap);
            arrangeParticipantsVertically(participantsToArrange.toArray(), propertyMap, parentMap);

            // if this is deletion of process cell - arange other processes
            if (cellsToDelete != null && cellsToDelete.length == 1 && (cellsToDelete[0] instanceof com.ds.bpm.bpd.graph.Process)) {
                com.ds.bpm.bpd.graph.Process pr = (com.ds.bpm.bpd.graph.Process) cellsToDelete[0];
                arrangeProcesses(propertyMap, GraphConstants.getBounds(pr.getAttributes()).getBounds());// HM,
                // JGraph3.4.1
            }
            // 处理子流程如果删除的是子流程
            if (cellsToDelete != null && cellsToDelete.length == 1 && (cellsToDelete[0] instanceof Subflow)) {
                Subflow subflow = (Subflow) cellsToDelete[0];
                com.ds.bpm.bpd.xml.activity.Activity xmlSubflow = (com.ds.bpm.bpd.xml.activity.Activity) subflow.getPropertyObject();
                // 得到子流程的Id
                String subflowId = xmlSubflow.getSubflow().getAttrId();
                if (subflowId != null) {
                    int flag = JOptionPane.showConfirmDialog(BPD.getInstance().getPackageEditor().getWindow(), ResourceManager.getLanguageDependentString("DeleteSubFlow"), BPD.getAppTitle(), JOptionPane.YES_NO_CANCEL_OPTION);
                    // 取得当前正在编辑的工作流对象
                    if (flag == JOptionPane.YES_OPTION) {

                        int flagd = JOptionPane.showConfirmDialog(BPD.getInstance().getPackageEditor().getWindow(), ResourceManager.getLanguageDependentString("DeleteSubProcess"), BPD.getAppTitle(), JOptionPane.YES_NO_OPTION);
                        // 取得当前正在编辑的工作流对象
                        if (flagd == JOptionPane.YES_OPTION) {
                            WorkflowProcess xmlprocess = BPD.getInstance().getRealXMLPackage().getWorkflowProcess(subflowId);
                            WorkflowProcesses workflowProcesses = (WorkflowProcesses) BPD.getInstance().getRealXMLPackage().get("WorkflowProcesses");
                            workflowProcesses.remove(xmlprocess);
                        }

                    } else if (flag == JOptionPane.NO_OPTION) {

                    } else {
                        return;
                    }
                }

            }

            // 处理子流程如果删除的是活动块

            if (cellsToDelete != null && cellsToDelete.length == 1 && (cellsToDelete[0] instanceof BlockActivity)) {
                BlockActivity outflow = (BlockActivity) cellsToDelete[0];
                com.ds.bpm.bpd.xml.activity.Activity xmlOutflow = (com.ds.bpm.bpd.xml.activity.Activity) outflow.getPropertyObject();

                // 得到子流程的Id
                String outflowId = xmlOutflow.getBlockActivity().getAttrId();
                if (outflowId != null) {
                    int flag = JOptionPane.showConfirmDialog(BPD.getInstance().getPackageEditor().getWindow(), ResourceManager.getLanguageDependentString("DeleteOutFlow"), BPD.getAppTitle(), JOptionPane.YES_NO_CANCEL_OPTION);
                    // 取得当前正在编辑的工作流对象
                    if (flag == JOptionPane.YES_OPTION) {

                        int flagd = JOptionPane.showConfirmDialog(BPD.getInstance().getPackageEditor().getWindow(), ResourceManager.getLanguageDependentString("DeleteOutProcess"), BPD.getAppTitle(), JOptionPane.YES_NO_OPTION);
                        // 取得当前正在编辑的工作流对象
                        if (flagd == JOptionPane.YES_OPTION) {
                            Object xmlObject = BPD.getInstance().getActivedProcessEditor().getGraph().getXPDLObject();
                            WorkflowProcess xmlprocess = null;
                            if (xmlObject instanceof WorkflowProcess) {
                                xmlprocess = (WorkflowProcess) xmlObject;
                            } else if (xmlObject instanceof ActivitySet) {
                                ActivitySet xmlSet = (ActivitySet) xmlObject;
                                xmlprocess = xmlSet.getOwnerProcess();
                            }
                            xmlprocess.getActivitySets().remove(xmlprocess.getActivitySets().getActivitySet(outflowId));

                        }

                    } else if (flag == JOptionPane.NO_OPTION) {

                    } else {
                        return;
                    }
                }

            }

            graphModel().removeAndEdit(cellsToDel.toArray(), propertyMap, ResourceManager.getLanguageDependentString("MessageRemovingObjects"));

            // refreshes the collection of XML objects
            refreshCollections(cellsToDel.toArray(), false);
        }

        try {
            Dimension prefSize = new Dimension(getRootParticipantWidth(propertyMap, parentMap) + 50, getNewRootParYPos(propertyMap, parentMap) + 50);
            graph.setPreferredSize(prefSize);
        } catch (Exception ex) {
        }
    }

    /**
     * Returns horizontal offset for inserting participants.
     */
    public int getHorizontalOffset() {
        return horizontalOffset;
    }

    /**
     * Returns the Id attribute of (XML logic) Participant object, which graph object visually holds given activity
     * graph object.
     */
    public String getParticipantID(Activity a) {
        if (a != null) {
            Participant par = (Participant) a.getParent();
            if (par != null) {
                return par.get("Id").toString();
            }
        }
        return "";
    }

    /**
     * Returns the point within the graph where the upper-left corner of given graph activity is placed. The point
     * origin is the upper-left corner of participant graph object that holds given activity.
     */
    public Point getOffset(Activity a) {
        if (a != null && a instanceof Activity) {
            Participant par = (Participant) ((Activity) a).getParent();
            if (par != null) {
                Rectangle2D rpar = getBounds(par, null);// HM, JGraph3.4.1
                Rectangle2D ract = getBounds(a, null);// HM, JGraph3.4.1
                int yOff = ract.getBounds().y - rpar.getBounds().y;// HM, JGraph3.4.1
                int xOff = ract.getBounds().x - rpar.getBounds().x;// HM, JGraph3.4.1
                return new Point(xOff, yOff);
            }
        }
        return new Point(0, 0);
    }

    /**
     * Returns the set of BlockActivity graph objects contained within the manager graph. If the graph of found
     * BlockActivity graph objects contains other BlockActivity graph objects, and the second parameter is set to true,
     * these are also returned, and so on - which means that implementation is recursive.
     */
    public Set getBlockActivities(boolean recursivly) {
        Set allActs = BPDGraphModel.getAllActivitiesInModel(graphModel());
        Set bas = new HashSet();
        if (allActs != null) {
            Iterator it = allActs.iterator();
            Object act;
            while (it.hasNext()) {
                act = it.next();
                if (act instanceof BlockActivity) {
                    bas.add(act);
                    if (!recursivly)
                        continue;
                    WorkflowManager wm = ((BlockActivity) act).getWorkflowManager();
                    if (wm != null) {
                        bas.addAll(wm.getBlockActivities(true));
                    }
                }
            }
        }
        return bas;
    }

    /**
     * Creates new graph participant object.
     */
    protected Participant createParticipant(Rectangle bounds, Map viewMap) {
        Map map;
        Participant par = new Participant();
        map = GraphConstants.createMap();
        GraphConstants.setBounds(map, bounds);
        GraphConstants.setOpaque(map, false);
        GraphConstants.setBorderColor(map, Color.white);
        GraphConstants.setMoveable(map, false);
        viewMap.put(par, map);
        return par;
    }

    /**
     * Creates new graph subflow object and associates the new XML logic activity object of SubFlow type to it.
     */
    protected Subflow createSubflow(Point p, com.ds.bpm.bpd.xml.activity.Activity ap, Map viewMap) {
        Point realP = (Point) graph.fromScreen(new Point(p));// HM, JGraph3.4.1
        int ulx = realP.x;
        int uly = realP.y;
        Rectangle bounds = new Rectangle(ulx, uly, defActivityWidth, defActivityHeight);
        Map map;
        Subflow sbf = new Subflow();
        WorkflowProcess wp = getWorkflowProcess();
        Activities acts = getActivitiesCollection();
        if (ap == null) {
            ap = new com.ds.bpm.bpd.xml.activity.Activity(acts, wp, ActivityDefTypeEnums.SubFlow);
            ap.set("Name", ResourceManager.getLanguageDependentString("SubFlowKey"));
        }

        setMandatoryProperties(ap);
        sbf.setUserObject(ap);
        map = GraphConstants.createMap();
        GraphConstants.setBounds(map, bounds);
        GraphConstants.setOpaque(map, true);
        GraphConstants.setBorderColor(map, new Color(15, 0, 115));
        viewMap.put(sbf, map);
        return sbf;
    }

    /**
     * Creates new graph subflow object and associates the new XML logic activity object of SubFlow type to it.
     */
    protected Outflow createOutflow(Point p, com.ds.bpm.bpd.xml.activity.Activity ap, Map viewMap) {
        Point realP = (Point) graph.fromScreen(new Point(p));// HM, JGraph3.4.1
        int ulx = realP.x;
        int uly = realP.y;
        Rectangle bounds = new Rectangle(ulx, uly, defActivityWidth, defActivityHeight);
        Map map;
        Outflow sbf = new Outflow();
        WorkflowProcess wp = getWorkflowProcess();
        Activities acts = getActivitiesCollection();
        if (ap == null) {
            ap = new com.ds.bpm.bpd.xml.activity.Activity(acts, wp, ActivityDefTypeEnums.OutFlow);
            ap.set("Name", ResourceManager.getLanguageDependentString("OutFlowKey"));
        }

        setMandatoryProperties(ap);
        sbf.setUserObject(ap);
        map = GraphConstants.createMap();
        GraphConstants.setBounds(map, bounds);
        GraphConstants.setOpaque(map, true);
        GraphConstants.setBorderColor(map, new Color(15, 0, 115));
        viewMap.put(sbf, map);
        return sbf;
    }

    /**
     * Creates new graph BlockActivity object and associates the new XML logic activity object of BlockActivity type to
     * it.
     */
    protected BlockActivity createBlockActivity(Point p, com.ds.bpm.bpd.xml.activity.Activity ap, Map viewMap) {
        Point realP = (Point) graph.fromScreen(new Point(p));// HM, JGraph3.4.1
        int ulx = realP.x;
        int uly = realP.y;
        Rectangle bounds = new Rectangle(ulx, uly, defActivityWidth, defActivityHeight);
        Map map;
        BlockActivity bla = new BlockActivity((PackageEditor) getGraph().getEditor());

        if (ap == null) {
            ap = new com.ds.bpm.bpd.xml.activity.Activity(getActivitiesCollection(), getWorkflowProcess(), ActivityDefTypeEnums.Block);
            ap.set("Name", ResourceManager.getLanguageDependentString("BlockActivityKey"));
            ap.getBlockActivity().set("Id", ((ActivitySets) getWorkflowProcess().get("ActivitySets")).generateID());
        }
        // set the ID of block

        bla.setUserObject(ap);
        map = GraphConstants.createMap();
        GraphConstants.setBounds(map, bounds);
        GraphConstants.setOpaque(map, true);
        GraphConstants.setBorderColor(map, new Color(15, 0, 115));
        viewMap.put(bla, map);
        return bla;
    }

    /**
     * Creates start of graph object.
     */
    protected Start createStart(Point p, Map viewMap) {
        Point realP = (Point) graph.fromScreen(new Point(p));// HM, JGraph3.4.1
        int ulx = realP.x;
        int uly = realP.y;
        Rectangle bounds = new Rectangle(ulx, uly, defActivityHeight, defActivityHeight);
        Map map;
        Start s = new Start();
        map = GraphConstants.createMap();
        GraphConstants.setBounds(map, bounds);
        GraphConstants.setOpaque(map, true);
        GraphConstants.setBorderColor(map, Color.darkGray);
        viewMap.put(s, map);
        return s;
    }

    /**
     * Creates end of graph object.
     */
    protected End createEnd(Point p, Map viewMap) {
        Point realP = (Point) graph.fromScreen(new Point(p));// HM, JGraph3.4.1
        int ulx = realP.x;
        int uly = realP.y;
        Rectangle bounds = new Rectangle(ulx, uly, defActivityHeight, defActivityHeight);
        Map map;
        End e = new End(new EndActivity());
        map = GraphConstants.createMap();
        GraphConstants.setBounds(map, bounds);
        GraphConstants.setOpaque(map, true);
        GraphConstants.setBorderColor(map, Color.darkGray);
        viewMap.put(e, map);
        return e;
    }

    /**
     * Creates new activity graph object, and associates the new XML logic activity object of Tool type to it.
     */
    protected Activity createActivity(Point p, com.ds.bpm.bpd.xml.activity.Activity ap, Map viewMap) {
        Point realP = (Point) graph.fromScreen(new Point(p));// HM, JGraph3.4.1
        int ulx = realP.x;
        int uly = realP.y;
        Rectangle bounds = new Rectangle(ulx, uly, defActivityWidth, defActivityHeight);
        Map map;

        WorkflowProcess wp = getWorkflowProcess();
        Activities acts = getActivitiesCollection();
        // com.ds.bpm.bpd.xml.activity.Activity ap = new com.ds.bpm.bpd.xml.activity.Activity(acts, wp,
        // ActivityDefTypeEnums.NO);

        if (ap == null) {
            ap = new com.ds.bpm.bpd.xml.activity.Activity(acts, wp, ActivityDefTypeEnums.No);
            ap.set("Name", ResourceManager.getLanguageDependentString("ManualActivityKey"));
        }

        Activity act = new Activity(ap);
        setMandatoryProperties(ap);
        act.setUserObject(ap);
        map = GraphConstants.createMap();
        GraphConstants.setBounds(map, bounds);
        GraphConstants.setOpaque(map, true);
        GraphConstants.setBorderColor(map, Color.darkGray);
        viewMap.put(act, map);
        return act;
    }

    /**
     * Creates new route activity graph object, and associates the new XML logic activity object of AutoActivity type to
     * it.
     */
    protected AutoActivity createAutoActivity(Point p, com.ds.bpm.bpd.xml.activity.Activity ap, Map viewMap) {
        Point realP = (Point) graph.fromScreen(new Point(p));// HM, JGraph3.4.1
        int ulx = realP.x;
        int uly = realP.y;
        Rectangle bounds = new Rectangle(ulx, uly, defActivityWidth, defActivityHeight);
        Map map;

        WorkflowProcess wp = getWorkflowProcess();
        Activities acts = getActivitiesCollection();
        if (ap == null) {
            ap = new com.ds.bpm.bpd.xml.activity.Activity(acts, wp, ActivityDefTypeEnums.Tool);
            ap.set("Name", ResourceManager.getLanguageDependentString("AutoActivityKey"));
        }

        AutoActivity r = new AutoActivity(ap);
        setMandatoryProperties(ap);
        r.setUserObject(ap);
        map = GraphConstants.createMap();
        GraphConstants.setBounds(map, bounds);
        GraphConstants.setOpaque(map, true);
        GraphConstants.setBorderColor(map, Color.darkGray);
        viewMap.put(r, map);
        return r;
    }

    /**
     * Creates new route activity graph object, and associates the new XML logic activity object of AutoActivity type to
     * it.
     */
    protected ServiceActivity createServiceActivity(Point p, com.ds.bpm.bpd.xml.activity.Activity ap, Map viewMap) {
        Point realP = (Point) graph.fromScreen(new Point(p));// HM, JGraph3.4.1
        int ulx = realP.x;
        int uly = realP.y;
        Rectangle bounds = new Rectangle(ulx, uly, defActivityWidth, defActivityHeight);
        Map map;

        WorkflowProcess wp = getWorkflowProcess();
        Activities acts = getActivitiesCollection();
        if (ap == null) {
            ap = new com.ds.bpm.bpd.xml.activity.Activity(acts, wp, ActivityDefTypeEnums.Service);
            ap.set("Name", ResourceManager.getLanguageDependentString("ServiceActivityKey"));
        }
        ServiceActivity r = new ServiceActivity(ap);

        setMandatoryProperties(ap);
        r.setUserObject(ap);
        map = GraphConstants.createMap();
        GraphConstants.setBounds(map, bounds);
        GraphConstants.setOpaque(map, true);
        GraphConstants.setBorderColor(map, Color.darkGray);
        viewMap.put(r, map);
        return r;
    }

    /**
     * Creates new route activity graph object, and associates the new XML logic activity object of AutoActivity type to
     * it.
     */
    protected DeviceActivity createDeviceActivity(Point p, com.ds.bpm.bpd.xml.activity.Activity ap, Map viewMap) {
        Point realP = (Point) graph.fromScreen(new Point(p));// HM, JGraph3.4.1
        int ulx = realP.x;
        int uly = realP.y;
        Rectangle bounds = new Rectangle(ulx, uly, defActivityWidth, defActivityHeight);
        Map map;

        WorkflowProcess wp = getWorkflowProcess();
        Activities acts = getActivitiesCollection();

        if (ap == null) {
            ap = new com.ds.bpm.bpd.xml.activity.Activity(acts, wp, ActivityDefTypeEnums.Device);
            ap.set("Name", ResourceManager.getLanguageDependentString("DeviceActivityKey"));
        }

        DeviceActivity r = new DeviceActivity(ap);

        setMandatoryProperties(ap);
        r.setUserObject(ap);
        map = GraphConstants.createMap();
        GraphConstants.setBounds(map, bounds);
        GraphConstants.setOpaque(map, true);
        GraphConstants.setBorderColor(map, Color.darkGray);
        viewMap.put(r, map);
        return r;
    }

    protected EventActivity createEventActivity(Point p, com.ds.bpm.bpd.xml.activity.Activity ap, Map viewMap) {
        Point realP = (Point) graph.fromScreen(new Point(p));// HM, JGraph3.4.1
        int ulx = realP.x;
        int uly = realP.y;
        Rectangle bounds = new Rectangle(ulx, uly, defActivityWidth, defActivityHeight);
        Map map;

        WorkflowProcess wp = getWorkflowProcess();
        Activities acts = getActivitiesCollection();
        if (ap == null) {
            ap = new com.ds.bpm.bpd.xml.activity.Activity(acts, wp, ActivityDefTypeEnums.Event);
            ap.set("Name", ResourceManager.getLanguageDependentString("EventActivityKey"));
        }
        EventActivity r = new EventActivity(ap);

        setMandatoryProperties(ap);
        r.setUserObject(ap);
        map = GraphConstants.createMap();
        GraphConstants.setBounds(map, bounds);
        GraphConstants.setOpaque(map, true);
        GraphConstants.setBorderColor(map, Color.darkGray);
        viewMap.put(r, map);
        return r;
    }

    /**
     * Inserts new activities into model, changes positions and sizes of participants due to a insertion of new
     * activities or due to a moving of activities. Also, finds and changes parent's of inserted/changed-position
     * activities. This method is called when inserting new activities, when pasting activities, and when moving
     * activities. All changes to the model and to the view are at first made virtually (to the parentMap and to the
     * propertyMap) and when all is done, method insertAndEdit of PEGraphModel class is called to actually make changes.
     * The updateCollection parameter is used when some objects are inserted into model, to update collection of XML
     * elements.
     */
    protected void updateModelAndArrangeParticipants(Object[] insert, Map propertyMap, ParentMap parentMap, Map viewMap, String actionName, boolean updateCollection) {
        updateModelAndArrangeParticipants(insert, propertyMap, null, parentMap, viewMap, actionName, updateCollection);
    }

    // 支持路由paste
    public void updateModelAndArrangeParticipants(Object[] insert, Map propertyMap, ConnectionSet cs, ParentMap parentMap, Map viewMap, String actionName, boolean updateCollection) {

        if (propertyMap == null && viewMap == null)
            return;
        if (propertyMap == null) {
            propertyMap = new HashMap(viewMap);
        }
        if (parentMap == null) {
            parentMap = new BPDParentMap();
        }
        arrangeParticipants(propertyMap, parentMap);
        // extracting viewMap elements out of propertyMap (if any), applying
        // change to viewMap element bounds (this must be done, because user
        // could press mouse when inserting or pasting activities at the
        // forbiden position so position of activity is changed during
        // updateActivityParent method) and applying change to model and view
        /*
         * if (viewMap != null) { Iterator it = viewMap.keySet().iterator(); while (it.hasNext()) { Object
         * cell=it.next(); // removing entry for cell that is contained within viewMap Map
         * mapP=(Map)propertyMap.remove(cell); // apply position changes to corresponding viewMap element Map
         * mapV=(Map)viewMap.get(cell); if (!(cell instanceof Port)) { Rectangle r=GraphConstants.getBounds(mapP);
         * GraphConstants.setBounds(mapV,r); } } }
         */
        int transitionNum = 0;
        if (cs != null) {
            transitionNum = cs.getChangedEdges().size();
        }
        if (insert != null && ((BPDParentMap) parentMap).entryCount() != insert.length - transitionNum) {
            return;
        }
        // makes all changes (if there was any) - to model and to view
        // graphModel().insertAndEdit(insert,null,propertyMap,parentMap,viewMap,actionName);
        graphModel().insertAndEdit(insert, propertyMap, cs, parentMap, null, actionName);
        if (updateCollection) {
            refreshCollections(insert, true);
        }
        if (!creatingGraph) {
            Dimension prefSize = new Dimension(getRootParticipantWidth(propertyMap, parentMap) + 50, getNewRootParYPos(propertyMap, parentMap) + 50);
            graph.setPreferredSize(prefSize);
        }
    }

    /**
     * Arranges participants according to the given property and parent maps.
     */
    protected void arrangeParticipants(Map propertyMap, ParentMap parentMap) {
        Set parsToArrangeVerticaly = new HashSet();
        // going through given propertyMap keys, and if key is
        // activity->updating it's parent if needed.
        // WARNING: must extract keys and put it in a array because
        // propertyMap changes -> ConcurrentModificationException
        // can happend
        Object[] cellsToManage = propertyMap.keySet().toArray();
        for (int i = 0; i < cellsToManage.length; i++) {
            Object cell = cellsToManage[i];
            if (cell instanceof Activity) {
                Set oldAndNewParentPar = updateActivityParent((Activity) cell, propertyMap, parentMap);
                parsToArrangeVerticaly.addAll(oldAndNewParentPar);
            }
        }
        // arrange participants vertically
        arrangeParticipantsVertically(parsToArrangeVerticaly.toArray(), propertyMap, parentMap);
        resizeAllParticipantsHorizontally(propertyMap, parentMap);
    }

    /**
     * Determines old and new participant for activity, adjusts activities position if needed and properly changes
     * parent of activity (adds entry into the parentMap).
     */
    protected Set updateActivityParent(Activity ac, Map propertyMap, ParentMap parentMap) {
        // must return old and new participant to the caller method
        Set oldAndNewPar = new HashSet();
        // old and new parent participant of given activity
        Participant oldPar = (Participant) ac.getParent();
        // adds oldPar to the set to be returned
        if (oldPar != null) {
            oldAndNewPar.add(oldPar);
        }
        Participant newPar = null;
        // taking position elements of Activity
        // taking bounding rectangle
        Rectangle2D acRect = getBounds(ac, propertyMap);// HM, JGraph3.4.1
        // taking upper-left corner, this will be reference for moving activitys
        Point acUpperLeft = acRect.getBounds().getLocation();// HM, JGraph3.4.1
        Point newAcUpperLeft = new Point(acUpperLeft);
        newPar = findParentActivityParticipantForLocation(newAcUpperLeft, propertyMap, parentMap);
        // if previous method changed location of upper-left point,
        // move activity to new location
        if (!newAcUpperLeft.equals(acUpperLeft)) {
            Rectangle r = new Rectangle(acRect.getBounds());// HM, JGraph3.4.1
            r.setLocation(newAcUpperLeft);
            changeBounds(ac, propertyMap, r);
        }
        if (newPar != null) {
            // adds newPar to the set to be returned
            oldAndNewPar.add(newPar);
            // VERY IMPORTANT IS TO CHANGE THE PARENT
            // changing the parent participant of Activity cell if it has
            // changed position
            if (!newPar.equals(oldPar)) {
                parentMap.addEntry(ac, newPar);
            }
            com.ds.bpm.bpd.xml.elements.Participant defaultP = com.ds.bpm.bpd.xml.elements.Participant.getFreeTextExpressionParticipant();
            // Setting the new performer for activity
            if (!(ac instanceof AutoActivity) && !(ac instanceof Start) && !(ac instanceof End) && !(ac instanceof Subflow) && !(ac instanceof BlockActivity) && newPar.getUserObject() != defaultP) {
                ac.set("Performer", newPar.getUserObject());
            }
            if (newPar != oldPar && newPar.getUserObject() == defaultP) {
                ac.set("Performer", "");
            }
        }
        return oldAndNewPar;
    }

    /**
     * Finds new participant for activity after it's position changes. WARNING: this method changes it's argument loc if
     * not appropriate.
     */
    protected Participant findParentActivityParticipantForLocation(Point loc, Map propertyMap, ParentMap parentMap) {
        Participant newPar = null;
        // if user put activity cell somewhere outside visible area, move it
        // back
        if (loc.y <= 0) {
            loc.y = 1;
        }
        if (loc.x <= 0) {
            loc.x = 1;
        }
        // determining the container for new position
        newPar = getLeafParticipantForYPos(loc.y, propertyMap, parentMap);
        // if new container isn't found -> activity is placed under all
        // participants
        // so it's new participant will be the leaf participant with the highest
        // y-coord
        if (newPar == null) {
            newPar = getLeafParticipantForYPos(getNewRootParYPos(propertyMap, null) - 10, propertyMap, parentMap);
        }
        if (newPar != null) {
            // Adjust activities x-pos if needed
            Rectangle newParRect = (Rectangle) getBounds(newPar, propertyMap);// HM, JGraph3.4.1
            // if x-position is not OK, set appropriate position
            if (newParRect.x + defParNameWidth >= loc.x) {
                loc.x = newParRect.x + defParNameWidth + 1;
            }
        }
        // it is activity that belongs to an block activity
        else {
            // if x-position is not OK, set appropriate position
            if (horizontalOffset >= loc.x) {
                loc.x = horizontalOffset;
            }
        }
        return newPar;
    }

    /**
     * Arranging heights and positions of given participants and positions of participants that must be translated due
     * to a change of given participants.
     */
    protected void arrangeParticipantsVertically(Object[] pars, Map propertyMap, ParentMap parentMap) {
        if ((pars == null) || (pars.length == 0))
            return;
        for (int i = 0; i < pars.length; i++) {
            // arrange participants vertically
            arrangeParticipantVertically(pars[i], propertyMap, parentMap);
        }
    }

    /**
     * Resizing participant par and it's parents to appropriate height, and translating other participants accordingly
     * to changes of participant par. The size's and positions are calculated considering propertyMap and parentMap -
     * which means for future children state, and for bounds that are constantly changed during other calculations. If
     * propertyMap and parentMap are null, size's and positions are calculated for current state. Also, if par that is
     * to be arranged has entry in parentMap as removed participant, it will not be arranged.
     */
    protected void arrangeParticipantVertically(Object par, Map propertyMap, ParentMap parentMap) {
        // can't be null, must be instance of Participant,
        // and can't have other participants
        // also it can't be removed
        if (par == null)
            return;
        if (!(par instanceof Participant))
            return;
        Participant p = (Participant) par;
        if (hasAnyParticipant(p, parentMap))
            return;
        ArrayList removedPars = ((BPDParentMap) parentMap).getRemovedNodes();
        if (removedPars.contains(p))
            return;
        // getting optimal and current height for participant
        int optHeight = optimalParticipantHeight(p, propertyMap, parentMap);
        int curHeight = getParticipantHeight(p, propertyMap);
        // calculating value for vertical resizing of new participant
        int dheight = optHeight - curHeight;
        if (dheight != 0) {
            // translating verticaly participants under bottom edge
            // of participant for value of dheight
            translateVertically(propertyMap, parentMap, getBounds(p, propertyMap).getBounds().y + curHeight - 1, dheight);// HM,
            // JGraph3.4.1

            // gets all parents participant (and given participant) into array
            // and resizes them for value of dheight
            Object[] allParentsAndPar = p.getPath();
            resize(allParentsAndPar, propertyMap, 0, dheight);
        }
    }

    /**
     * Method that resizes all participants horizontally to get there minimal needed sizes. The size is calculated
     * considering propertyMap and parentMap - which means for future children state, and for bounds that are constantly
     * changed during other calculations. If propertyMap and parentMap are null, size is calculated for current state.
     */
    protected void resizeAllParticipantsHorizontally(Map propertyMap, ParentMap parentMap) {
        Set participants = BPDGraphModel.getAllParticipantsInModel(graphModel());
        // removing ones which parent in a parentMap is null
        if (parentMap != null && participants != null) {
            participants.removeAll(((BPDParentMap) parentMap).getRemovedNodes());
        }
        // if there is a need for resizing
        int optimalRDW = optimalRootParticipantWidth(participants, propertyMap, parentMap);
        int rootParWidth = getRootParticipantWidth(propertyMap, parentMap);
        if (optimalRDW != rootParWidth) {
            // resize all participants for needed increment
            int dw = optimalRDW - rootParWidth;
            if (participants != null) {
                resize(participants.toArray(), propertyMap, dw, 0);
            }
        }
    }

    /**
     * Calculates the minimal width of root participants. It depends of minimal allowed width of leaf participants,
     * which depends on the position of Activity cells in them. The width is calculated considering propertyMap and
     * parentMap - which means for future children state, and for bounds that are constantly changed during other
     * calculations. If propertyMap and parentMap are null, size is calculated for current state.
     */
    protected int optimalRootParticipantWidth(Set participants, Map propertyMap, ParentMap parentMap) {
        // initial value for width stays the same
        int minWidth = minParWidth;
        // exits if there are no participants created
        if (participants == null)
            return minWidth;
        // finds the leaf participants
        Set leafParticipants = new HashSet();
        Iterator it = participants.iterator();
        while (it.hasNext()) {
            Participant par = (Participant) it.next();
            // if participant doesn't have any other participants,
            // it is a leaf participant and should be added to collection
            if (!hasAnyParticipant(par, parentMap)) {
                leafParticipants.add(par);
            }
        }
        // max. right edge position minus horizontalOffset (of all leafs)
        // becomes minWidth
        it = leafParticipants.iterator();
        int maxRightEdgePosition = 0;
        while (it.hasNext()) {
            Participant lpar = (Participant) it.next();
            int minREdge;
            int minParREdge;
            int minChildrenREdge = 0;
            // getting future participant bounds
            minParREdge = ((Rectangle) getBounds(lpar, propertyMap)).x + minParWidth;// HM, JGraph3.4.1
            // getting the future child views bounding rectangle -> its right
            // edge plus some extra space is min. right edge for that
            // participant
            Rectangle r = getBoundsOfParticipantFutureActivities(lpar, propertyMap, parentMap);
            if (r != null) {
                minChildrenREdge = r.x + r.width + defParNameWidth;
            }
            minREdge = Math.max(minParREdge, minChildrenREdge);
            // if entering first time, set the starting max.
            if (maxRightEdgePosition == 0) {
                maxRightEdgePosition = minREdge;
            } else if (minREdge > maxRightEdgePosition) {
                maxRightEdgePosition = minREdge;
            }
        }
        int minW = maxRightEdgePosition - horizontalOffset;
        // can't allow that the minWidth<minParWidth
        if (minW > minParWidth) {
            minWidth = minW;
        }
        return minWidth;
    }

    /**
     * Calculates minimal participant height, which depends of position of all of its Activity cells. The height is
     * calculated considering propertyMap and parentMap - which means for future children state, and for bounds that are
     * constantly changed during other calculations. If propertyMap and parentMap are null, size is calculated for
     * current state.
     */
    protected int optimalParticipantHeight(Participant par, Map propertyMap, ParentMap parentMap) {
        // initial value for height
        int optHeight = minParHeight;
        // exits returning minParHeight if there are no activity cells within
        // participant
        if (!hasAnyActivity(par, parentMap))
            return optHeight;
        // get bounds of par (either current or future)
        Rectangle rCurrent = (Rectangle) getBounds(par, propertyMap);// HM, JGraph3.4.1
        // get preffered bounding rectangle of participant (according to it's
        // children)
        Rectangle rPreferred = getBoundsOfParticipantFutureActivities(par, propertyMap, parentMap);
        ;
        // difference of these rectangle's bottom positions plus current par
        // height
        // plus some extra space is min. height for given participant
        // calculate difference in bottom edges of these rectangles, and optimal
        // height
        int dBottom = (rPreferred.y + rPreferred.height) - (rCurrent.y + rCurrent.height);
        int optH = rCurrent.height + dBottom + 10;
        // optimal height can't be less then minHeight
        if (optH > optHeight) {
            optHeight = optH;
        }
        return optHeight;
    }

    /**
     * Resizes given participants. The resizing is done to propertyMap which will later (after all needed operation) be
     * applied.
     */
    protected void resize(Object[] cells, Map propertyMap, int dw, int dh) {
        if (cells != null && cells.length > 0) {
            Map map;
            Rectangle r;
            for (int i = 0; i < cells.length; i++) {
                r = new Rectangle(getBounds(cells[i], propertyMap).getBounds());// HM, JGraph3.4.1
                int newWidth = r.width + dw;
                int newHeight = r.height + dh;
                if (newWidth < minParWidth || newHeight < minParHeight) {
                    System.err.println("There was an error in calculating size of participant " + cells[i] + "!!!");
                    System.err.println("New width=" + newWidth + ", new height=" + newHeight);
                }
                r.setSize(newWidth, newHeight);
                changeBounds(cells[i], propertyMap, r);
            }
        }
    }

    /**
     * Translates participants under given position yPos vertically for a value of dv. The translating is done to
     * propertyMap which will later (after all needed operation) be applied.
     *
     * @see #translateParticipants
     */
    protected void translateVertically(Map propertyMap, ParentMap parentMap, int yPos, int dv) {
        Participant[] pars = getParticipantsForYPos(yPos, 0, propertyMap, parentMap);
        translateParticipants(pars, propertyMap, parentMap, 0, dv);
    }

    /**
     * Translates given participants using propertyMap for bounds checking and parentMap for future children checking.
     * The translating is done to propertyMap which will later (after all needed operation) be applied.
     *
     * @see #translateParticipant
     */
    protected void translateParticipants(Participant[] cells, Map propertyMap, ParentMap parentMap, int dx, int dy) {
        if (cells != null && cells.length > 0) {
            for (int i = 0; i < cells.length; i++) {
                translateParticipant(cells[i], propertyMap, parentMap, dx, dy);
            }
        }
    }

    /**
     * Translates single participant and its children.The method checks for bounds of cells within propertyMap and uses
     * parentMap to translate right children (the children that will be it's after applying parentMap). The translating
     * is done to propertyMap which will later (after all needed operation) be applied.
     */
    protected void translateParticipant(Participant par, Map propertyMap, ParentMap parentMap, int dx, int dy) {
        Set participantAndItsFutureActivities = new HashSet();
        participantAndItsFutureActivities.add(par);
        // Get future activities of participant
        Set futureActivities = getParticipantFutureActivities(par, parentMap);
        participantAndItsFutureActivities.addAll(futureActivities);
        // applying translations to the determined cells
        Map map;
        Rectangle r;
        Iterator it = participantAndItsFutureActivities.iterator();
        while (it.hasNext()) {
            Object cell = it.next();
            r = new Rectangle(getBounds(cell, propertyMap).getBounds());// HM, JGraph3.4.1
            r.translate(dx, dy);
            changeBounds(cell, propertyMap, r);
        }
    }

    /**
     * Gets the bounding rectangle of future children activities of given Participant par (that will be participants
     * activities after parent map is applied). Bounding rectangle is union of previous mentioned activities.
     */
    protected Rectangle getBoundsOfParticipantFutureActivities(Participant par, Map propertyMap, ParentMap parentMap) {
        // simulate future state of children and get it's bounds
        Set futureActivities = getParticipantFutureActivities(par, parentMap);
        Set futureActivityBounds = new HashSet();
        Iterator it = futureActivities.iterator();
        while (it.hasNext()) {
            Rectangle actBnd = (Rectangle) getBounds(it.next(), propertyMap);// HM, JGraph3.4.1
            futureActivityBounds.add(actBnd);
        }
        Rectangle[] fab = new Rectangle[futureActivityBounds.size()];
        futureActivityBounds.toArray(fab);
        Rectangle unionBounds = getUnionBounds(fab);
        return unionBounds;
    }

    /**
     * Gets future children activities of given participant par (that will be par's activities after parent map is
     * applied).
     */
    protected Set getParticipantFutureActivities(Participant par, ParentMap parentMap) {
        // if there is no parent map, or there is no entry for participant
        // in it, return current participants activities
        if (parentMap == null || !parentMap.getChangedNodes().contains(par)) {
            return par.getChildActivities();
        }
        Set futureActivities = new HashSet();
        // getting participants which will be empty after applying parentMap
        ArrayList emptyPars = ((BPDParentMap) parentMap).emptyParentList();
        // returns empty set if there will be no activity cells within
        // participant
        if (emptyPars.contains(par))
            return futureActivities;
        // get changed nodes from parent map (nodes which parent changed and
        // nodes which children changed)
        Set changedNodes = parentMap.getChangedNodes();
        // get all (previous) participants activities and make the future look
        // of participant activities
        futureActivities = new HashSet(par.getChildActivities());
        Object[] previousActivities = futureActivities.toArray();
        // iterate through child activities and remove ones which are
        // contained in changed nodes of parent map - they will no longer
        // be the children of current participant
        for (int i = 0; i < previousActivities.length; i++) {
            if (changedNodes.contains(previousActivities[i])) {
                futureActivities.remove(previousActivities[i]);
            }
        }
        // get new children of current participant from parent map
        // and add it to futureActivities set
        ArrayList nc = ((BPDParentMap) parentMap).getNewChildren(par);
        futureActivities.addAll(nc);
        return futureActivities;
    }

    /**
     * Returns leaf participant that bounds given y-coordinate. If y-coordinate is at boundary of two participants,
     * method returns one that is above. The method checks for bounds of cells within propertyMap.
     */
    protected Participant getLeafParticipantForYPos(int yPos, Map propertyMap, ParentMap parentMap) {
        // getting all participants that contains yPos
        Participant[] pars = getParticipantsForYPos(yPos, 2, propertyMap, parentMap);
        // if there is no participants at this location, return null
        if (pars == null)
            return null;
        // getting leaf participant(s)
        Set leafPars = new HashSet();
        for (int i = 0; i < pars.length; i++) {
            if (!hasAnyParticipant(pars[i], parentMap)) {
                leafPars.add(pars[i]);
            }
        }
        // if there is no leaf participant at this location (THIS SHOULD
        // NEVER HAPPEND, BUT ...) return null
        if (leafPars.size() == 0)
            return null;
        // taking first and making it the right one
        Iterator it = leafPars.iterator();
        Object rightPar = it.next();
        // if there is more than one leaf participant, take the one that
        // has minimal y-coord of upper left corner
        if (leafPars.size() > 1) {
            int upperLeftY;
            int minUpperLeftY;
            Rectangle parRect;
            // taking bounding rectangle of first par
            parRect = getBounds(rightPar, propertyMap).getBounds();// HM, JGraph3.4.1
            // taking upper-left corner y-coord
            upperLeftY = parRect.getLocation().y;
            minUpperLeftY = upperLeftY;

            // finding the participant with min. y-coord
            while (it.hasNext()) {
                Object curPar = it.next();
                parRect = getBounds(curPar, propertyMap).getBounds();// HM, JGraph3.4.1
                upperLeftY = parRect.getLocation().y;
                if (upperLeftY < minUpperLeftY) {
                    minUpperLeftY = upperLeftY;
                    rightPar = curPar;
                }
            }
        }
        // return found participant
        return (Participant) rightPar;
    }

    /**
     * Returns participants that are under or above given yPos, or contains that y-position: <BR>
     * direction=0 -> under, <BR>
     * direction=1 -> above, <BR>
     * direction=2 -> contains. The method checks for bounds of cells within propertyMap. If some of model's
     * participants is entered in parentMap as removed participant, this method doesn't consider that participant.
     */
    protected Participant[] getParticipantsForYPos(int yPos, int direction, Map propertyMap, ParentMap parentMap) {
        // getting all participants that are in collection
        Set participants = BPDGraphModel.getAllParticipantsInModel(graphModel());
        // if there are no participants, return null
        if (participants == null)
            return null;
        // removing ones which parent in a parentMap is null
        if (parentMap != null) {
            participants.removeAll(((BPDParentMap) parentMap).getRemovedNodes());
        }
        // making an array of participants
        Participant[] pars = new Participant[participants.size()];
        participants.toArray(pars);
        // Set of participants that will satisfy needs
        Set yPosPars = new HashSet();
        for (int i = 0; i < pars.length; i++) {
            Rectangle r = (Rectangle) getBounds(pars[i], propertyMap);// HM, JGraph3.4.1
            switch (direction) {
                case 0:
                    if (r.y >= yPos) {
                        yPosPars.add(pars[i]);
                    }
                    break;
                case 1:
                    if (r.y < yPos) {
                        yPosPars.add(pars[i]);
                    }
                    break;
                case 2:
                    if ((r.y <= yPos) && (r.y + r.height >= yPos)) {
                        yPosPars.add(pars[i]);
                    }
                    break;
            }
        }
        if (yPosPars.size() > 0) {
            pars = new Participant[yPosPars.size()];
            yPosPars.toArray(pars);
            return pars;
        } else {
            return null;
        }
    }

    /**
     * Gets the insertation point (y-coordinate) of new root participant.
     */
    protected int getNewRootParYPos(Map propertyMap, ParentMap parentMap) {
        int newRootParYPos = 0;
        Set rootPars = BPDGraphModel.getRootParticipants(graphModel());
        // adding to rootPars set a root participants from propertyMap -> that
        // is
        // done because of proper calculation of y-position of in/out ports when
        // new root participant is added to graph
        if (propertyMap != null) {
            Iterator it = propertyMap.keySet().iterator();
            while (it.hasNext()) {
                Object rootPar = it.next();
                if ((rootPar instanceof Participant) && (((DefaultGraphCell) rootPar).getParent() == null)) {
                    rootPars.add(rootPar);
                }
            }
        }
        // removing from rootPars set a root participants from parentMap -> that
        // is
        // done because of proper calculation of y-position of in/out ports when
        // root participant is removed from graph
        if (parentMap != null) {
            rootPars.removeAll(((BPDParentMap) parentMap).getRemovedNodes());
        }
        // if there is no root participants, new Y-position is 0
        if (rootPars == null || rootPars.size() == 0)
            return newRootParYPos;
        newRootParYPos = minParHeight;
        // find the bottom of bottom most root participant
        Iterator it = rootPars.iterator();
        while (it.hasNext()) {
            Rectangle bounds = (Rectangle) getBounds(it.next(), propertyMap);// HM, JGraph3.4.1
            if (bounds.y + bounds.height > newRootParYPos) {
                newRootParYPos = bounds.y + bounds.height;
            }
        }
        return newRootParYPos;
    }

    /**
     * Gets the width of root participants. The method checks for bounds of cells within propertyMap. If some of model's
     * participants has entry within propertyMap as removed, this participant doesn't count.
     */
    protected int getRootParticipantWidth(Map propertyMap, ParentMap parentMap) {
        int rootParWidth = minParWidth;
        Set rootPars = BPDGraphModel.getRootParticipants(graphModel());
        // if there is no root participants, width is equal to minParWidth
        if (rootPars == null)
            return rootParWidth;
        // removing ones which parent in a parentMap is null (this means they
        // will be removed from a graph)
        if (parentMap != null) {
            rootPars.removeAll(((BPDParentMap) parentMap).getRemovedNodes());
        }
        // if there is no root participants, width is equal to minParWidth
        if (rootPars.size() == 0)
            return rootParWidth;
        // all root participants has same width, so take the first
        Iterator it = rootPars.iterator();
        Object firstPar = it.next();
        rootParWidth = getParticipantWidth(firstPar, propertyMap);
        return rootParWidth;
    }

    /**
     * Checks if given participant par has other participants. If parentMap is null, or there is no entry for this this
     * participant within a parentMap, the current state is checked, elsewhere parentMap is checked-in other words
     * future state of participant participants (state after aplying parentMap) is returned.
     * <p>
     * This method has meaning in previous versions of BPD, now it always returns <tt>false</tt>.
     */
    protected boolean hasAnyParticipant(Participant par, ParentMap parentMap) {
        // if there is no parent map, or there is no entry for participant
        // in it, return original participant state
        if (parentMap == null || !parentMap.getChangedNodes().contains(par)) {
            return par.hasAnyParticipant();
        }
        // else, check if participant will be empty after applying parent map
        else {
            return ((BPDParentMap) parentMap).hasAnyParticipant(par);
        }
    }

    /**
     * Checks if given participant par has activities. If parentMap is null, or there is no entry for this this
     * participant within a parentMap, the current state is checked, elsewhere parentMap is checked-in other words
     * future state of participant activities (state after aplying parentMap) is returned.
     */
    protected boolean hasAnyActivity(Participant par, ParentMap parentMap) {
        // if there is no parent map, or there is no entry for participant
        // in it, return original participant state
        if (parentMap == null || !parentMap.getChangedNodes().contains(par)) {
            return par.hasAnyActivity();
        }
        // else, check if participant will be empty after applying parent map
        else {
            // getting participants which will be empty after applying parentMap
            ArrayList emptyPars = ((BPDParentMap) parentMap).emptyParentList();
            // returns empty set if there will be no activity cells within
            // participant
            if (emptyPars.contains(par)) {
                return false;
            } else {
                return true;
            }
        }
    }

    /**
     * Returns starts of process/block.
     */
    protected Set getStarts() {
        Set starts = new HashSet();
        Set allActivities = BPDGraphModel.getAllActivitiesInModel(graphModel());
        if (allActivities != null) {
            Iterator it = allActivities.iterator();
            while (it.hasNext()) {
                Object act = it.next();
                if (act instanceof Start) {
                    starts.add(act);
                }
            }
        }
        return starts;
    }

    /**
     * Returns set of ends of process/block.
     */
    protected Set getEnds() {
        Set ends = new HashSet();
        Set allActivities = BPDGraphModel.getAllActivitiesInModel(graphModel());
        if (allActivities != null) {
            Iterator it = allActivities.iterator();
            while (it.hasNext()) {
                Object act = it.next();
                if (act instanceof End) {
                    ends.add(act);
                }
            }
        }
        return ends;
    }

    /**
     * Gets view of given object.
     */
    public CellView getView(Object cell) {
        // return graph.getView().getMapping(cell,false);
        return graph.getGraphLayoutCache().getMapping(cell, false);
    }

    /**
     * Gets width of given participant (from it's current view or from propertyMap).
     */
    protected int getParticipantWidth(Object par, Map propertyMap) {
        return getBounds(par, propertyMap).getBounds().width;// HM, JGraph3.4.1
    }

    /**
     * Gets height of given participant (from it's current view or from propertyMap).
     */
    protected int getParticipantHeight(Object par, Map propertyMap) {
        return getBounds(par, propertyMap).getBounds().height;// HM, JGraph3.4.1
    }

    /**
     * Replaces bounding rectangle of given cell in propertyMap object with rectangle r if cell is contained within the
     * propertyMap object, otherwise adds new entry to the propertyMap that consists of given cell and a map containing
     * given rectangle r.
     */
    protected void changeBounds(Object cell, Map propertyMap, Rectangle r) {
        Map map;
        if (propertyMap == null || !propertyMap.containsKey(cell)) {
            map = GraphConstants.createMap();
            GraphConstants.setBounds(map, r);
            propertyMap.put(cell, map);
        } else {
            map = (Map) propertyMap.get(cell);
            GraphConstants.setBounds(map, r);
        }
    }

    /**
     * Gets bounding rectangle of given cell. The rectangle is either current rectangle of cellView either from
     * propertyMap where are held bounding rectangles of various cells during multiple resizing and/or translating of
     * cells.
     */
    public Rectangle2D getBounds(Object cell, Map propertyMap) {// HM, JGraph3.4.1
        if (propertyMap != null && propertyMap.containsKey(cell)) {
            Map map = (Map) propertyMap.get(cell);
            return GraphConstants.getBounds(map);
        } else {
            CellView view = getView(cell);

            return view.getBounds();
        }
    }

    /**
     * Gets union of given rectangles.
     */
    protected Rectangle getUnionBounds(Rectangle[] rects) {
        if (rects != null && rects.length > 0) {
            Rectangle unionRect = null;
            for (int i = 0; i < rects.length; i++) {
                if (unionRect == null) {
                    unionRect = new Rectangle(rects[i]);
                } else {
                    SwingUtilities.computeUnion(rects[i].x, rects[i].y, rects[i].width, rects[i].height, unionRect);
                }
            }
            return unionRect;
        }
        return null;
    }

    /**
     * Sets all mandatory user defined properties to the given activity.
     */
    private void setMandatoryProperties(com.ds.bpm.bpd.xml.activity.Activity ap) {
        UserProperties up = getXMLPackage().getUserDefinedActivityProperties();
        Set mandatoryProps = up.getMandatoryProperties();
        Iterator it = mandatoryProps.iterator();
        while (it.hasNext()) {
            ap.addUserProperty((UserProperty) it.next());
        }
    }

    /**
     * Used during import of an XML file to create graph participants for given workflow process, and to associate the
     * XML logic participant objects to it.
     */
    private void createGraphParticipants(WorkflowProcess wp, Set entitiesToInsert) {
        boolean participantInserted = false;
        Iterator it = entitiesToInsert.iterator();
        while (it.hasNext()) {
            Object p = it.next();
            if (p instanceof com.ds.bpm.bpd.xml.elements.Participant) {
                showNode((com.ds.bpm.bpd.xml.elements.Participant) p);
                participantInserted = true;
            }
        }
        if (!participantInserted) {
            Participants pcs = (Participants) wp.get("Participants");
            Activities acts = null;
            if (graph instanceof BlockActivityGraph) {
                ActivitySet as = getBlockActivitySet(((BlockActivityGraph) graph).getMyBlockActivity());
                if (as != null) {
                    acts = (Activities) as.get("Activities");
                }
            } else {
                acts = (Activities) wp.get("Activities");
            }
            if (acts != null && acts.size() > 0) {
                showNode(com.ds.bpm.bpd.xml.elements.Participant.getFreeTextExpressionParticipant());
            }
        }
        // graph.repaint(0);
    }

    /**
     * Used during import of an XML file to create graph activities for the given process, and to associate the XML
     * logic activity objects to it.
     */
    private void createGraphActivitiesAndBlockActivities(WorkflowProcess wp, Set entitiesToInsert, boolean isMine) {
        // System.out.println("Workflow "+wp+", has to insert activities among
        // "+entitiesToInsert.size()+" entities");
        Iterator it = entitiesToInsert.iterator();
        while (it.hasNext()) {
            Object mayBeActivity = it.next();
            // System.out.println("Workflow "+wp+": entity "+aOrIb+" is next");
            if (mayBeActivity instanceof com.ds.bpm.bpd.xml.activity.Activity) {
                // System.out.println("Workflow "+wp+": entity "+aOrIb+" is
                // next");
                insertActivity((com.ds.bpm.bpd.xml.activity.Activity) mayBeActivity, wp, isMine);
            }
        }
    }

    /**
     * Used during import of an XML file to create graph activity object for the given process, and to associate the
     * given XML logic activity object to it. It creates appropriate graph object depending on given logical activity
     * type (AutoActivity, Tool, Subflow or BlockActivity) and determines the participant graph object to put it into.
     */
    private void insertActivity(com.ds.bpm.bpd.xml.activity.Activity a, WorkflowProcess wp, boolean isMine) {
        // System.out.println("Must insert graph object for activity "+a);
        Participants pcs = (Participants) wp.get("Participants");
        Rectangle rPar;
        Point p = null;
        Activity act;
        Participant par;
        Map viewMap;
        Object performer;

        // must save performer to set it later again (because of
        // movement during importing)
        performer = a.get("Performer").toValue();
        String participantID = "";
        if (isMine) {
            participantID = a.getParticipantID();
        }
        if (participantID.length() == 0) {
            if (performer != null && performer instanceof com.ds.bpm.bpd.xml.elements.Participant) {
                participantID = ((com.ds.bpm.bpd.xml.elements.Participant) performer).getID();
            } else {
                participantID = performer.toString();
            }
        }
        par = getGraphObjectForXMLParticipant(pcs.getParticipant(participantID));
        if (par == null) {
            par = getGraphObjectForXMLParticipant(com.ds.bpm.bpd.xml.elements.Participant.getFreeTextExpressionParticipant());
        }
        if (par != null) {
            rPar = (Rectangle) getBounds(par, null);// HM, JGraph3.4.1
            p = new Point(rPar.getLocation());
            p.x += 10;
            p.y += 10;
        } else {
            rPar = null;
            p = new Point(10, 10);
        }
        viewMap = new Hashtable();
        ActivityDefTypeEnums actType = a.getType();
        switch (actType) {
            case Tool:
                act = createAutoActivity(p, a, viewMap);
                break;
            case SubFlow:
                act = createSubflow(p, a, viewMap);
                break;
            case Block:
                act = createBlockActivity(p, a, viewMap);
                ((ActivitySets) wp.get("ActivitySets")).decrementID();
                break;
            case OutFlow:
                act = createOutflow(p, a, viewMap);
                break;
            case Service:
                act = createServiceActivity(p, a, viewMap);
                break;
            case Device:
                act = createDeviceActivity(p, a, viewMap);
                break;
            case Event:
                act = createEventActivity(p, a, viewMap);
                break;
            case No:
                act = createActivity(p, a, viewMap);
                break;
            default:
                act = createActivity(p, a, viewMap);
                break;
        }
        getActivitiesCollection().decrementID();
        // act.setUserObject(a);
        // System.out.println("Graph Activity object for "+act+" is created");
        updateModelAndArrangeParticipants(new Object[]{act}, null, null, viewMap, "", false);
        int xOffset = 0, yOffset = 0;
        if (isMine) {
            xOffset = a.getXOffset();
            yOffset = a.getYOffset();
        }
        adjustActivityPosition(act, xOffset, yOffset, rPar);
        // setting back the activity performer
        act.setUserObject(a);
        if (act.getUserObject() instanceof Activity) {

            ((Activity) act.getUserObject()).set("Performer", performer);

        } else {
            a.set("Performer", performer);
        }

        // if this is a block activity, create it's graph
        if (actType.equals(ActivityDefTypeEnums.Block)) {
            ((BlockActivity) act).createBlockActivityGraph(graph.getEditor().getWindow());
        }
    }

    /**
     * Used during import of an XML file to create graph transitions for the given process, and to associate the XML
     * logic transition objects to it.
     */
    private void createGraphTransitions(WorkflowProcess wp, Set entitiesToInsert, boolean isMine) {
        String IDFrom, IDTo;
        Activity source, target;
        com.ds.bpm.bpd.xml.elements.Transition trans;
        Transition t;
        Iterator it = entitiesToInsert.iterator();
        while (it.hasNext()) {
            Object tr = it.next();
            if (tr instanceof com.ds.bpm.bpd.xml.elements.Transition) {
                trans = (com.ds.bpm.bpd.xml.elements.Transition) tr;
                IDFrom = trans.get("From").toValue().toString();
                IDTo = trans.get("To").toValue().toString();
                source = getActivity(IDFrom);
                target = getActivity(IDTo);
                String rt = trans.getRoutingType();
                boolean isRouting = (rt != null && !rt.equals(com.ds.bpm.bpd.xml.elements.Transition.NO_ROUTING));
                t = connectActivities(source, target, isRouting);
                if (t != null) {
                    if (!isRouting) {
                        TransitionView tv = (TransitionView) getView(t);
                        Map ordNoToPoint = trans.getBreakPoints();
                        // now, inserting takes place - must be inserted in the
                        // right order
                        for (int i = 1; i <= ordNoToPoint.size(); i++) {
                            tv.addPointProgramatically((Point2D) ordNoToPoint.get(new Integer(i)), i);
                        }
                    }
                    t.setUserObject(trans);
                }
                // to see changes
                // graph.paintImmediately(graph.getBounds());
            }
        }
    }

    /**
     * Used during import of an XML file to create graph start.
     */
    // 创建开始节点
    private void createGraphStarts(WorkflowProcess wp, boolean isMine) {
        Activity source = null;
        Activity target = null;
        Hashtable viewMap = new Hashtable();
        com.ds.bpm.bpd.xml.activity.Activity a = null;
        // first, the case of package made by BPD
        if (isMine) {
            Set startDescriptions = new HashSet();
            if (graph instanceof BlockActivityGraph) {
                a = (com.ds.bpm.bpd.xml.activity.Activity) getVisualOwner();
                startDescriptions = a.getStartDescriptions();
            } else {
                startDescriptions = wp.getStartDescriptions();
            }
            if (startDescriptions.size() == 0)
                return;
            Participants pcs = (Participants) wp.get("Participants");
            String ID;
            Participant par;
            Rectangle rPar;
            Point p;
            Iterator it = startDescriptions.iterator();
            String startDesc;
            String[] startD;
            while (it.hasNext()) {
                startDesc = (String) it.next();
                startD = Utils.tokenize(startDesc, ";");
                ID = "";
                try {
                    ID = startD[0];
                } catch (Exception ex) {
                }
                par = getGraphObjectForXMLParticipant(pcs.getParticipant(ID));
                if (par == null) {
                    par = getGraphObjectForXMLParticipant(com.ds.bpm.bpd.xml.elements.Participant.getFreeTextExpressionParticipant());
                }
                if (par != null) {
                    rPar = (Rectangle) getBounds(par, null);// HM, JGraph3.4.1
                    p = new Point(rPar.getLocation());
                    p.x += 10;
                    p.y += 10;
                } else {
                    rPar = null;
                    p = new Point(10, 10);
                }
                viewMap = new Hashtable();
                source = createStart(p, viewMap);
                updateModelAndArrangeParticipants(new Object[]{source}, null, null, viewMap, "", false);
                // adjusting activities position
                try {
                    adjustActivityPosition(source, Integer.parseInt(startD[2]), Integer.parseInt(startD[3]), rPar);
                } catch (Exception ex) {
                }
                try {
                    target = getActivity(startD[1]);
                } catch (Exception ex) {
                }
                boolean isTransitionRouted = false;
                try {
                    String itr = startD[4];
                    if (itr.equals(com.ds.bpm.bpd.xml.elements.Transition.NO_ROUTING)) {
                        isTransitionRouted = false;
                    } else {
                        isTransitionRouted = true;
                    }
                } catch (Exception ex) {
                    isTransitionRouted = false;
                }
                if (source != null && target != null) {
                    connectActivities(source, target, isTransitionRouted);
                }
            }
            // if the package is from other vendor
        } else {
            Set acs = BPDGraphModel.getAllActivitiesInModel(graphModel());
            if (acs != null) {
                int translY = 10;
                Iterator it = acs.iterator();
                while (it.hasNext()) {
                    Activity act = (Activity) it.next();
                    Set trns = act.getIncomingTransitions();
                    if (trns.size() == 0) {
                        target = act;
                        viewMap = new Hashtable();
                        source = createStart(new Point(10, translY), viewMap);
                        translY += 30;
                        updateModelAndArrangeParticipants(new Object[]{source}, null, null, viewMap, "", false);
                        if (source != null && target != null) {
                            connectActivities(source, target, false);
                        }
                    }
                }
            }
        }
    }

    /**
     * Used during import of an XML file to create graph ends.
     */
    // 创建结束节点
    private void createGraphEnds(WorkflowProcess wp, boolean isMine) {
        Activity source = null;
        Activity target = null;
        Hashtable viewMap;
        com.ds.bpm.bpd.xml.activity.Activity a = null;
        // first, the case of our package
        if (isMine) {
            Set endDescriptions = new HashSet();
            if (graph instanceof BlockActivityGraph) {
                a = (com.ds.bpm.bpd.xml.activity.Activity) getVisualOwner();
                endDescriptions = a.getEndDescriptions();
            } else {
                endDescriptions = wp.getEndDescriptions();
            }
            if (endDescriptions.size() == 0)
                return;
            Participants pcs = (Participants) wp.get("Participants");
            String ID;
            Participant par;
            Rectangle rPar;
            Point p;
            Iterator it = endDescriptions.iterator();
            String endDesc;
            String[] endD;
            while (it.hasNext()) {
                endDesc = (String) it.next();
                endD = Utils.tokenize(endDesc, ";");
                ID = "";
                try {
                    ID = endD[0];
                } catch (Exception ex) {
                }
                par = getGraphObjectForXMLParticipant(pcs.getParticipant(ID));
                if (par == null) {
                    par = getGraphObjectForXMLParticipant(com.ds.bpm.bpd.xml.elements.Participant.getFreeTextExpressionParticipant());
                }
                if (par != null) {
                    rPar = (Rectangle) getBounds(par, null);// HM, JGraph3.4.1
                    p = new Point(rPar.getLocation());
                    p.x += 10;
                    p.y += 10;
                } else {
                    rPar = null;
                    p = new Point(10, 10);
                }
                viewMap = new Hashtable();
                target = getEnd(Integer.parseInt(endD[2]), Integer.parseInt(endD[3]));
                if (target == null) {
                    target = createEnd(p, viewMap);
                }
                updateModelAndArrangeParticipants(new Object[]{target}, null, null, viewMap, "", false);
                // adjusting activities position
                try {
                    adjustActivityPosition(target, Integer.parseInt(endD[2]), Integer.parseInt(endD[3]), rPar);
                } catch (Exception ex) {
                    ex.printStackTrace();
                }
                // connecting end of graph and last activity
                try {
                    source = getActivity(endD[1]);
                } catch (Exception ex) {
                }
                boolean isTransitionRouted = false;
                try {
                    String itr = endD[4];
                    if (itr.equals(com.ds.bpm.bpd.xml.elements.Transition.NO_ROUTING)) {
                        isTransitionRouted = false;
                    } else {
                        isTransitionRouted = true;
                    }
                } catch (Exception ex) {
                    isTransitionRouted = false;
                }
                if (source != null && target != null) {
                    connectActivities(source, target, isTransitionRouted);
                }
            }
            // if the package is from other vendor
        } else {
            Set acs = BPDGraphModel.getAllActivitiesInModel(graphModel());
            if (acs != null) {
                int translY = 10;
                Iterator it = acs.iterator();
                while (it.hasNext()) {
                    Activity act = (Activity) it.next();
                    Set trns = act.getOutgoingTransitions();
                    if (trns.size() == 0) {
                        source = act;
                        viewMap = new Hashtable();
                        target = createEnd(new Point(600, translY), viewMap);
                        translY += 10;
                        updateModelAndArrangeParticipants(new Object[]{target}, null, null, viewMap, "", false);
                        if (source != null && target != null) {
                            connectActivities(source, target, false);
                        }
                    }
                }
            }
        }
    }

    /**
     * Used during import of an XML file to adjust position of activities.
     */
    private void adjustActivityPosition(Activity act, int xOff, int yOff, Rectangle rPar) {
        if (rPar == null || (xOff == 0 && yOff == 0))
            return;
        act.setXOff(xOff);
        act.setYOff(yOff);
        int yPar = rPar.y;
        int xPar = rPar.x;
        Map attrib = GraphConstants.cloneMap(act.getAttributes());
        Map propertyMap = new Hashtable();
        ParentMap parentMap = new BPDParentMap();
        propertyMap.put(act, attrib);
        Rectangle bounds = (Rectangle) GraphConstants.getBounds(attrib);// HM, JGraph3.4.1
        int inc = rPar.height - 5;
        int nextYOff = inc;
        if (yOff < inc) {
            inc = yOff;
            nextYOff = yOff;
        }
        bounds.setLocation(new Point(xPar + xOff, yPar + inc));
        arrangeParticipants(propertyMap, parentMap);
        if (yOff > nextYOff) {
            inc = defActivityHeight;
            do {
                nextYOff += inc;
                if (nextYOff > yOff) {
                    nextYOff -= inc;
                    inc = yOff - nextYOff;
                    nextYOff += inc;
                }
                bounds.translate(0, inc);
                arrangeParticipants(propertyMap, parentMap);
            } while (nextYOff < yOff);
        }
        // makes all changes (if there was any) - to model and to view
        graphModel().edit(propertyMap, null, parentMap, null);
        // to see changes
        graph.paintImmediately(graph.getBounds());
    }

    /**
     * Creates and returns the graph Transition object that connects given graph activities. If something goes wrong,
     * returns null.
     */
    private Transition connectActivities(Activity source, Activity target, boolean isRouting) {
        // System.out.println("Connecting activities "+source+" and "+target);
        Point startP = getCenter(source);
        Point endP = getCenter(target);
        try {
            Transition t = insertTransition(startP, endP, (PortView) getView(source.getPort()), (PortView) getView(target.getPort()), isRouting, false);
            return t;
        } catch (Exception ex) {

            System.err.println("Error while connecting activities " + source + " and " + target + " !!!");
            return null;
        }
    }

    /**
     * Used during import of an XML file from other vendor to layout graph activities.
     */
    private void layoutActivities() {
        Set participants = BPDGraphModel.getAllParticipantsInModel(graphModel());
        if (participants != null) {
            Iterator it = participants.iterator();
            while (it.hasNext()) {
                Participant par = (Participant) it.next();
                Set activities = par.getChildActivities();
                Iterator itAct = activities.iterator();
                int cnt = 0;
                double chngDir = (int) Math.sqrt(activities.size());
                int incX = 2 * defActivityWidth;
                int incY = defActivityHeight + 5;
                int translateX = 0;
                int translateY = 0;
                while (itAct.hasNext()) {
                    Activity act = (Activity) itAct.next();
                    if (!(act instanceof Start) && !(act instanceof End)) {
                        cnt++;
                        if ((cnt / chngDir) == ((int) (cnt / chngDir))) {
                            // System.out.println("Changing dir, cnt="+cnt);
                            incX = -incX;
                            translateY += incY;
                        } else {
                            translateX += incX;
                        }
                        Map attrib = GraphConstants.cloneMap(act.getAttributes());
                        Map propertyMap = new Hashtable();
                        ParentMap parentMap = new BPDParentMap();
                        propertyMap.put(act, attrib);
                        Rectangle bounds = (Rectangle) GraphConstants.getBounds(attrib);// HM, JGraph3.4.1
                        if (act instanceof End) {
                            bounds.translate(0, translateY);
                        } else {
                            bounds.translate(translateX, translateY);
                        }
                        arrangeParticipants(propertyMap, parentMap);
                        // makes all changes
                        // graphModel().edit(null,propertyMap,parentMap,null);
                        graphModel().edit(propertyMap, null, parentMap, null);
                        // to see changes
                        // graph.paintImmediately(graph.getBounds());
                    }
                }
            }
        }
    }

    /**
     * Returns graph Activity object which represents the logical activity with given Id attribute.
     */
    public Activity getActivity(String ID) {
        Set allActs = BPDGraphModel.getAllActivitiesInModel(graphModel());
        if (allActs != null) {
            Iterator it = allActs.iterator();
            Activity act;
            String curID;
            while (it.hasNext()) {
                act = (Activity) it.next();
                if (!(act instanceof Start) && !(act instanceof End)) {
                    curID = act.get("Id").toValue().toString();
                    if (curID != null && curID.equals(ID)) {
                        return act;
                    }
                }
            }
        }
        return null;
    }

    /**
     * Returns graph Transition object which represents the logical transition with given Id attribute.
     */
    public Transition getTransition(String ID) {
        Set allTrans = BPDGraphModel.getAllTransitionsInModel(graphModel());
        if (allTrans != null) {
            Iterator it = allTrans.iterator();
            Transition tr;
            String curID;
            while (it.hasNext()) {
                tr = (Transition) it.next();
                com.ds.bpm.bpd.xml.elements.Transition xmlT = (com.ds.bpm.bpd.xml.elements.Transition) tr.getUserObject();
                if (xmlT.getFrom() != null && xmlT.getTo() != null) {
                    curID = tr.get("Id").toValue().toString();
                    if (curID != null && curID.equals(ID)) {
                        return tr;
                    }
                }
            }
        }
        return null;
    }

    /**
     * Returns graph Participant object which represents the logical participant with given Id attribute.
     */
    public Participant getParticipant(String ID) {
        Set allPartic = BPDGraphModel.getAllParticipantsInModel(graphModel());
        if (allPartic != null) {
            Iterator it = allPartic.iterator();
            Participant p;
            String curID;
            while (it.hasNext()) {
                p = (Participant) it.next();
                if (p.get("Id").toString().equals(ID)) {
                    return p;
                }
            }
        }
        return null;
    }

    /**
     * Returns the central point of given graph object.
     */
    private Point getCenter(Object go) {
        if (go == null)
            return null;
        Rectangle r = (Rectangle) getBounds(go, null);// HM, JGraph3.4.1
        if (!(go instanceof Participant)) {
            return new Point(r.x + (int) (r.width / 2), r.y + (int) (r.height / 2));
        } else {
            return new Point(r.x + defParNameWidth / 2, r.y + (int) (r.height / 2));
        }
    }

    /**
     * Refreshes the collections (adds or removes it's elements) of workflow logic objects (activities, transitions,
     * workflow processes, block activities) corresponding to the graph objects that are beeing added to or removed from
     * graph.
     *
     * @param cellsToAddOrRemove The graph objects that are beeing added to or removed from graph.
     * @param toAdd              <tt>true</tt> if graph objects are beeing added, <tt>false</tt> otherwise.
     */
    public void refreshCollections(Object[] cellsToAddOrRemove, boolean toAdd) {
        // the set of activities user objects to add or remove
        Set xmlaAR = new HashSet();
        // the set of transitions user objects to add or remove
        Set xmltAR = new HashSet();
        // the set of workflow processes user objects to add or remove
        Set xmlwAR = new HashSet();
        // the set of block activities to add or remove
        Set baAR = new HashSet();
        if (cellsToAddOrRemove != null && cellsToAddOrRemove.length > 0) {
            for (int i = 0; i < cellsToAddOrRemove.length; i++) {
                Object cell = cellsToAddOrRemove[i];
                // managing the participant references if cell is
                // process, blockActivity or participant
                manageParticipantReferences(cell, toAdd);
                if (cell instanceof Transition) {
                    Transition tr = (Transition) cell;
                    com.ds.bpm.bpd.xml.elements.Transition xmlT = (com.ds.bpm.bpd.xml.elements.Transition) tr.getUserObject();
                    Activity s = null;
                    try {
                        s = tr.getSourceActivity();
                    } catch (Exception ex) {
                        s = getActivity(xmlT.get("From").toString());
                    }
                    Activity t = null;
                    try {
                        t = tr.getTargetActivity();
                    } catch (Exception ex) {
                        t = getActivity(xmlT.get("To").toString());
                    }
                    if (toAdd) {
                        if (s != null && s.getUserObject() instanceof XMLElement && t != null && t.getUserObject() instanceof XMLElement) {
                            xmltAR.add(xmlT);
                        }
                    } else {
                        xmltAR.add(xmlT);
                    }
                    // update split of source activity
                    if (s != null && s.getUserObject() instanceof XMLElement) {
                        com.ds.bpm.bpd.xml.activity.Activity a = (com.ds.bpm.bpd.xml.activity.Activity) s.getUserObject();
                        int noft = s.getOutgoingTransitions().size();
                        String type = a.getSplit().get("Type").toString().trim();
                        if (type.length() == 0) {
                            if (noft > 1) {
                                a.setSplitType(ResourceManager.getLanguageDependentString("XORKey"));
                            }
                        } else {
                            if (noft <= 1) {
                                a.setSplitType("");
                            }
                        }
                    }
                    // update join of target activity
                    if (t != null && !(t.getUserObject() instanceof EndActivity) && t.getUserObject() instanceof XMLElement) {
                        com.ds.bpm.bpd.xml.activity.Activity a = (com.ds.bpm.bpd.xml.activity.Activity) t.getUserObject();
                        int noft = t.getIncomingTransitions().size();
                        String type = a.getJoin().get("Type").toString().trim();
                        if (type.length() == 0) {
                            if (noft > 1) {
                                a.setJoinType(ResourceManager.getLanguageDependentString("XORKey"));
                            }
                        } else {
                            if (noft <= 1) {
                                a.setJoinType("");
                            }
                        }
                    }
                } else if (cell instanceof com.ds.bpm.bpd.graph.Process) {
                    com.ds.bpm.bpd.graph.Process proc = (com.ds.bpm.bpd.graph.Process) cell;
                    WorkflowProcess wp = (WorkflowProcess) proc.getUserObject();
                    xmlwAR.add(wp);
                    // add or remove process to process graph mappings
                    if (toAdd) {
                        BPD.getInstance().getPackageEditor().putProcessObjectMapping(wp, proc);
                    } else {
                        BPD.getInstance().getPackageEditor().removeProcessObjectMapping(wp);
                    }
                } else if (cell instanceof BlockActivity) {
                    xmlaAR.add(((Activity) cell).getUserObject());
                    baAR.add(cell);
                } else if (cell instanceof DeviceActivity) {
                    xmlaAR.add(((Activity) cell).getUserObject());
                    // baAR.add(cell);
                } else if (cell instanceof ServiceActivity) {
                    xmlaAR.add(((Activity) cell).getUserObject());
                } else if (cell instanceof EventActivity) {
                    xmlaAR.add(((Activity) cell).getUserObject());
                    // baAR.add(cell);
                } else if (cell instanceof Activity && !(cell instanceof Start) && !(cell instanceof End)) {
                    xmlaAR.add(((Activity) cell).getUserObject());
                }

            }
        }

        // 2004-2-27修改
        // Package pkg = getXMLPackage();
        Package pkg = BPD.getInstance().getRealXMLPackage();
        WorkflowProcesses wps = (WorkflowProcesses) pkg.get("WorkflowProcesses");
        wps.refreshCollection(xmlwAR, toAdd);
        if (graph != null) {
            XMLElement el = graph.getPropertyObject();
            if (el instanceof WorkflowProcess) {
                WorkflowProcess wp = (WorkflowProcess) el;
                ActivitySets xmlactsts = (ActivitySets) wp.get("ActivitySets");
                Activities xmlacts = getActivitiesCollection();
                Transitions xmltrans = getTransitionsCollection();
                xmlacts.refreshCollection(xmlaAR, toAdd);
                xmltrans.refreshCollection(xmltAR, toAdd);
                // generating or removing activity set
                Iterator it = baAR.iterator();
                while (it.hasNext()) {
                    BlockActivity ba = (BlockActivity) it.next();
                    ActivitySet as = null;
                    String ID = ba.getBlockID();
                    if (toAdd) {
                        as = new ActivitySet(xmlactsts, wp);
                        as.set("Id", ID);
                        as.setIDPrefixForCollections();
                        xmlactsts.add(as);
                    } else {
                        as = xmlactsts.getActivitySet(ID);
                        xmlactsts.remove(as);
                    }
                }
            }
        }
        // updating statusbar
        if (cellsToAddOrRemove != null && cellsToAddOrRemove.length > 0) {
            if (graph != null) {
                getGraph().getEditor().getStatusBar().updateMessage();
                // JPanel jpanel=BPD.getInstance().getErrTablePanel().getTablePanel();
                // getGraph().getEditor().errPanel=jpanel;
                // getGraph().getEditor().graphMainsplitPane.setBottomComponent(jpanel);
            } else {

                pkgEditor.getStatusBar().updateMessage();
            }
        }
    }

    private void manageParticipantReferences(Set cellsToDelete, boolean toAdd) {
        Iterator it = cellsToDelete.iterator();
        while (it.hasNext()) {
            manageParticipantReferences(it.next(), toAdd);
        }
    }

    private void manageParticipantReferences(Object cellToDelete, boolean toAdd) {
        if (cellToDelete instanceof com.ds.bpm.bpd.graph.Process) {
            com.ds.bpm.bpd.graph.Process pr = (com.ds.bpm.bpd.graph.Process) cellToDelete;
            try {
                Set s = BPDGraphModel.getAllParticipantsInModel(pr.getImplementationEditor().getGraph().getModel());
                if (s != null) {
                    Iterator iter = s.iterator();
                    while (iter.hasNext()) {
                        // recursivly calling method but this time passing
                        // participant graph object
                        manageParticipantReferences(iter.next(), toAdd);
                    }
                }
                manageParticipantReferences(pr.getImplementationEditor().getGraph().getWorkflowManager().getBlockActivities(true), toAdd);
            } catch (Exception ex) {
            }
        } else if (cellToDelete instanceof BlockActivity) {
            BlockActivity b = (BlockActivity) cellToDelete;
            try {
                // add or remove all graph references of participants that are
                // visible in a process
                Set s = BPDGraphModel.getAllParticipantsInModel(b.getImplementationEditor().getGraph().getModel());
                if (s != null) {
                    Iterator iter = s.iterator();
                    while (iter.hasNext()) {
                        // recursivly calling method but this time passing
                        // participant graph object
                        manageParticipantReferences(iter.next(), toAdd);
                    }
                }
            } catch (Exception ex) {
            }
        } else if (cellToDelete instanceof Participant) {
            Participant par = (Participant) cellToDelete;
            try {
                com.ds.bpm.bpd.xml.elements.Participant puo = (com.ds.bpm.bpd.xml.elements.Participant) par.getUserObject();
                if (toAdd) {
                    puo.graphReferenceAdded();
                } else {
                    puo.graphReferenceRemoved();
                }
            } catch (Exception ex) {
            }
        }
    }

    /**
     * Finds the nearest cell to the given cell in given direction.
     *
     * @param selectedCell first currently selected cell
     * @param direction    0-Up, 1-Down, 2-Left, 3-Right (if given some other no, the Up is assumed)
     * @return The nearest cell to the given one in given direction, or null if no such cell.
     */
    public Object findNearestCell(Object selectedCell, int direction) {
        // retrieve all cells from model
        Set cellSet = BPDGraphModel.getAllCellsInModel(graphModel());
        // Set cellSet=PEGraphModel.getAllActivitiesInModel(graphModel());
        // if there is no any return null
        if (cellSet == null || cellSet.size() == 0)
            return null;
        Object[] cells = cellSet.toArray();
        // if current cell is null (nothing is selected), return
        // the first cell from array
        if (selectedCell == null)
            return cells[0];
        // get the center point of currently selected cell
        Point centerOfSelectedCell = getCenter(selectedCell);
        // if something went wrong, return the first cell from array
        if (centerOfSelectedCell == null)
            return cells[0];
        // initialy set the nearest cell to null, and min distance
        // to max. double value
        Object nearestCell = null;
        double minDistance = Double.MAX_VALUE;
        // search for the nearest cell considering the wanted direction
        for (int i = 0; i < cells.length; i++) {
            // skip the cell that we check for the nearest cell
            // and skip port cells
            if (cells[i] == selectedCell || cells[i] instanceof Port)
                continue;
            // the center of the next cell
            Point centerOfCell = getCenter(cells[i]);
            Point ref = new Point(centerOfCell.x, centerOfSelectedCell.y);
            double dist = centerOfSelectedCell.distance(centerOfCell);
            double absOfTan = centerOfCell.distance(ref) / ref.distance(centerOfSelectedCell);
            switch (direction) {
                case 1:
                    // DOWN
                    if (centerOfCell.y >= centerOfSelectedCell.y) {
                        if (dist < minDistance && absOfTan >= 1) {
                            minDistance = dist;
                            nearestCell = cells[i];
                        }
                    }
                    break;
                case 2:
                    // LEFT
                    if (centerOfCell.x <= centerOfSelectedCell.x) {
                        if (dist < minDistance && absOfTan <= 1) {
                            minDistance = dist;
                            nearestCell = cells[i];
                        }
                    }
                    break;
                case 3:
                    // RIGHT
                    if (centerOfCell.x >= centerOfSelectedCell.x) {
                        if (dist < minDistance && absOfTan <= 1) {
                            minDistance = dist;
                            nearestCell = cells[i];
                        }
                    }
                    break;
                default:
                    // UP
                    if (centerOfCell.y <= centerOfSelectedCell.y) {
                        if (dist < minDistance && absOfTan >= 1) {
                            minDistance = dist;
                            nearestCell = cells[i];
                        }
                    }
                    break;
            }
        }
        return nearestCell;
    }

    public Object findCell(Point p, Map propertyMap) {
        // retrieve all cells from model
        Set cellSet = BPDGraphModel.getAllCellsInModel(graphModel());
        // Set cellSet=PEGraphModel.getAllActivitiesInModel(graphModel());
        // if there is no any return null
        if (cellSet == null || cellSet.size() == 0)
            return null;
        Object[] cells = cellSet.toArray();
        // search for the nearest cell considering the wanted direction
        for (int i = 0; i < cells.length; i++) {
            // the center of the next cell
            Rectangle r = getBounds(cells[i], propertyMap).getBounds();// HM, JGraph3.4.1
            if (r.contains(p) && !(cells[i] instanceof Port)) {
                return cells[i];
            }
        }
        return null;
    }

    /**
     * Arranges processes after deletion of one of them, and sets the new insertion point for the processes that will be
     * inserted next.
     */
    private void arrangeProcesses(Map propertyMap, Rectangle deletedProcessBounds) {
        int nextHowMany = BPDGraphModel.getAllActivitiesInModel(graphModel()).size() - 1;
        // System.out.println("There is "+nextHowMany+" prcs");
        int deletedOrdNoX = (int) Math.round((deletedProcessBounds.x - BPDConstants.PROCESS_OFFSET) / (defProcessWidth * BPDConstants.PROCESS_DISTANCE_COEFF));
        int deletedOrdNoY = (int) Math.round((deletedProcessBounds.y - BPDConstants.PROCESS_OFFSET) / (defProcessHeight * BPDConstants.PROCESS_DISTANCE_COEFF));
        int deletedOrdNo = (deletedOrdNoX + 1) + deletedOrdNoY * BPDConstants.HOW_MANY_PROCESSES_IN_ONE_ROW;
        // System.out.println("x="+deletedOrdNoX+",y="+deletedOrdNoY+",
        // on="+deletedOrdNo);
        if (nextHowMany >= deletedOrdNo) {
            for (int i = deletedOrdNo + 1; i <= nextHowMany + 1; i++) {
                int xon = (i - 1) % BPDConstants.HOW_MANY_PROCESSES_IN_ONE_ROW;
                int yon = (i - 1) / BPDConstants.HOW_MANY_PROCESSES_IN_ONE_ROW;
                // System.out.println("xtp="+xon+", ytp="+yon);
                // find the cell for point corresponding to the current ordinal
                // number
                Point p = new Point((int) (BPDConstants.PROCESS_OFFSET + xon * BPDConstants.PROCESS_DISTANCE_COEFF * defProcessWidth + defProcessWidth / 2),
                        (int) (BPDConstants.PROCESS_OFFSET + yon * BPDConstants.PROCESS_DISTANCE_COEFF * defProcessHeight + defProcessHeight / 2));
                // System.out.println("FOC="+findCell(p,propertyMap).getClass().getName());
                com.ds.bpm.bpd.graph.Process pr = (com.ds.bpm.bpd.graph.Process) findCell(p, propertyMap);
                // System.out.println("Proc is "+pr);
                if (pr != null) {
                    // get old cell bounds
                    int xo = getBounds(pr, propertyMap).getBounds().x;// HM, JGraph3.4.1
                    int yo = getBounds(pr, propertyMap).getBounds().y;// HM, JGraph3.4.1
                    // calculate new cell bounds
                    int xn = (int) (BPDConstants.PROCESS_OFFSET + ((i - 2) % BPDConstants.HOW_MANY_PROCESSES_IN_ONE_ROW) * BPDConstants.PROCESS_DISTANCE_COEFF * defProcessWidth);
                    int yn = (int) (BPDConstants.PROCESS_OFFSET + ((i - 2) / BPDConstants.HOW_MANY_PROCESSES_IN_ONE_ROW) * BPDConstants.PROCESS_DISTANCE_COEFF * defProcessHeight);
                    // do translation of cell (virtually)
                    Map attrib = GraphConstants.cloneMap(pr.getAttributes());
                    Rectangle bounds = (Rectangle) GraphConstants.getBounds(attrib);// HM, JGraph3.4.1
                    bounds.translate(xn - xo, yn - yo);
                    propertyMap.put(pr, attrib);
                }
            }
        }
    }

    /**
     * 得到指定流程定义的权限组
     *
     * @param process 给定的流程
     * @return 指定流程的权限组
     */
    public static Collection getRightGroup(WorkflowProcess process) {
        Collection ret = new ArrayList();
        ExtendedAttributes extendedAttrs = (ExtendedAttributes) process.get("ExtendedAttributes");
        ExtendedAttribute extendedAttr = extendedAttrs.getExtendedAttribute("RightGroups");
        XMLCollection rightGroups = extendedAttr.getAnyXMLCollection();
        for (Iterator iter = rightGroups.toCollection().iterator(); iter.hasNext(); ) {
            RightGroup element = (RightGroup) iter.next();
            String code = element.get("Code").toValue().toString();
            if (!code.equals("PERFORMER") && !code.equals("READER")) {
                String[] aElem = new String[]{element.get("Name").toValue().toString(), element.get("Code").toValue().toString()};
                ret.add(aElem);
            }
        }
        return ret;
    }

    public static Collection getAllRightGroup(WorkflowProcess process) {
        Collection ret = new ArrayList();
        ExtendedAttributes extendedAttrs = (ExtendedAttributes) process.get("ExtendedAttributes");
        ExtendedAttribute extendedAttr = extendedAttrs.getExtendedAttribute("RightGroups");
        XMLCollection rightGroups = extendedAttr.getAnyXMLCollection();
        for (Iterator iter = rightGroups.toCollection().iterator(); iter.hasNext(); ) {
            RightGroup element = (RightGroup) iter.next();
            String[] aElem = new String[]{element.get("Name").toValue().toString(), element.get("Code").toValue().toString()};
            ret.add(aElem);
        }
        return ret;
    }

    public static String getRightGroupName(WorkflowProcess process, String code) {
        String rightGroupName = "";
        Collection rightGroups = getAllRightGroup(process);
        for (Iterator it = rightGroups.iterator(); it.hasNext(); ) {
            String[] rightGroup = (String[]) it.next();
            if (rightGroup[1].equals(code)) {
                rightGroupName = rightGroup[0];
                break;
            }
        }
        return rightGroupName;
    }

    /**
     * 得到指定package下的所有子流程
     *
     * @param pk 给定的package
     * @return
     */
    public static Collection getSubFlows(Package pk) {
        Collection ret = new ArrayList();
        WorkflowProcesses wps = (WorkflowProcesses) pk.get("WorkflowProcesses");
        Collection sfs = wps.toCollection();
        for (Iterator iter = sfs.iterator(); iter.hasNext(); ) {
            WorkflowProcess element = (WorkflowProcess) iter.next();
            if (element.isSubFlow()) {
                ret.add(element);
            }
        }
        return ret;
    }

    /**
     * /** 取得指定服务的所有流程集合
     *
     * @return
     * @throws AxisFault
     * @throws RemoteException
     */
    public static List<ProcessDef> getExtencwFlows() {

        // 获得流程定义信息集合
        List<ProcessDef> processInfoVector = BPD.getInstance().getRemoteProcessDefList(false);

        return processInfoVector;
    }

    // /**
    // * 取得指定服务的所有流程集合
    // *
    // * @return
    // * @throws AxisFault
    // * @throws RemoteException
    // */
    // public static Object[] getAllFlows() {
    //
    // Collection ret = new ArrayList();
    // try {
    // // 获得流程定义信息集合
    // Vector processInfoVector = BPD.getInstance().getRemoteProcessDefList(false);
    // if (processInfoVector != null) {
    // for (int i = 0; i < processInfoVector.size(); i++) {
    // Vector processInfo = (Vector) processInfoVector.get(i);
    // ProcessDef processDef = new ProcessDef(processInfo);
    // String processDefId = processDef.getProcessDefId();
    // String processDefName = processDef.getProcessDefName();
    // String[] temp = new String[] { processDefName, processDefId };
    // ret.add(temp);
    // }
    // }
    //
    // } catch (Exception e) {
    // return null;
    //
    // }
    // return ret.toArray();
    // }

    public static List<ProcessDef> getAllFlows() {

        // 获得流程定义信息集合
        List<ProcessDef> processInfoVector = BPD.getInstance().getRemoteProcessDefList(false);

        return processInfoVector;
    }

    public static Collection getOutFlows(Package pk) {
        Collection ofs = new ArrayList();
        return ofs;
    }

    // 更新流程中活动的位置信息
    public static void updateActivityPosition(WorkflowProcess wp) {
        Set startActs = getActivities(wp.getStartDescriptions());
        Set endActs = getActivities(wp.getEndDescriptions());
        Activities acts = (Activities) wp.get("Activities");
        for (Iterator it = acts.toCollection().iterator(); it.hasNext(); ) {
            com.ds.bpm.bpd.xml.activity.Activity act = (com.ds.bpm.bpd.xml.activity.Activity) it.next();
            String position = "NORMAL";
            if (startActs.contains(act.getID())) {
                position = "START";
            } else if (endActs.contains(act.getID())) {
                position = "END";
            }
            act.setPosition(position);
        }
    }

    // 获取与开始或者结束节点相连的活动
    private static Set getActivities(Set startOrEnd) {
        Set set = new HashSet();
        for (Iterator it = startOrEnd.iterator(); it.hasNext(); ) {
            String sd = (String) it.next();
            String[] startD = XMLUtil.tokenize(sd, ";");
            String[] ids = Utils.tokenize(startD[1], WorkflowProcess.ENDOFWORKFLOWACT_SEPARATE);
            if (ids != null && ids.length > 0) {
                for (int i = 0; i < ids.length; i++) {
                    set.add(ids[i]);
                }
            }
        }
        return new HashSet(set);
    }

    // 判断位置为xOff和yOff的结束节点是否存在，若存在则返回满足条件的节点，否则返回Null
    private End getEnd(int xOff, int yOff) {
        End end = null;
        Set ends = getEnds();
        for (Iterator it = ends.iterator(); it.hasNext(); ) {
            End tmp = (End) it.next();
            if (tmp.getXOff() == xOff && tmp.getYOff() == yOff) {
                end = tmp;
                break;
            }
        }
        return end;
    }

    public void updateSplit(Activity s) {
        // update split of source activity
        if (s != null && s.getUserObject() instanceof XMLElement) {
            com.ds.bpm.bpd.xml.activity.Activity a = (com.ds.bpm.bpd.xml.activity.Activity) s.getUserObject();
            int noft = s.getOutgoingTransitions().size();
            String type = a.getSplit().get("Type").toString().trim();
            if (type.length() == 0) {
                if (noft > 1) {
                    a.setSplitType(ResourceManager.getLanguageDependentString("XORKey"));
                }
            } else {
                if (noft <= 1) {
                    a.setSplitType("");
                }
            }
        }
    }

    public void updateJoin(Activity t) {
        // update join of target activity
        if (t != null && t.getUserObject() instanceof XMLElement) {
            com.ds.bpm.bpd.xml.activity.Activity a = (com.ds.bpm.bpd.xml.activity.Activity) t.getUserObject();
            int noft = t.getIncomingTransitions().size();
            String type = a.getJoin().get("Type").toString().trim();
            if (type.length() == 0) {
                if (noft > 1) {
                    a.setJoinType(ResourceManager.getLanguageDependentString("XORKey"));
                }
            } else {
                if (noft <= 1) {
                    a.setJoinType("");
                }
            }
        }
    }

}
/* End of WorkflowManager.java */
