/* ResourceManager.java
 *
 * Title : BPM工作流图形定义工具BPD
 * Class Desription：主界面UI的展现抽象类
 * Authors： wenzhang li
 * Company： 基督山BPM
 * CreatedTime：2005-12-6
 *
 */

package com.ds.bpm.bpd;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.net.URL;

import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.Box;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JToggleButton;
import javax.swing.JToolBar;
import javax.swing.KeyStroke;

import com.ds.bpm.bpd.actions.ChangeApplication;
import com.ds.bpm.bpd.actions.Close;
import com.ds.bpm.bpd.actions.CloseAll;
import com.ds.bpm.bpd.actions.Commission;
import com.ds.bpm.bpd.actions.ProcessProperties;

/**
 * 主界面菜单、工具栏、流程定义按钮以及右键菜单的构造类
 */
public class BarFactory {

	protected AbstractEditor editor;

	public BarFactory(AbstractEditor editor) {
		this.editor = editor;
	}

	// ********************** CREATE TOOLBARS AND THEIR COMPONENTS
	// *****************
	/**
	 * 创建图形界面工具栏组
	 * @see #createToolBars
	 */
	public Component createGraphToolBars() {
		return createToolBars(editor.graphbars());
	}
	/**
	 * 创建主界面的工具栏组 Creates application's toolbars. Reads resource to find out
	 * which toolbars to create. NOTE: this method is modified to accept a
	 * creation of a button after toolbars.
	 * 
	 * @see #createToolBars
	 */
	public Component createToolBars() {
		return createToolBars(editor.toolbarToLoad());
	}

	
	public Component createToolBars(String toolBarsName) {
		// 从资源文件获得工具栏组
		String[] toolBars = Utils.tokenize(ResourceManager
				.getResourceString(toolBarsName), " ");
		final JPanel panel = new JPanel();
		panel.setLayout(new BorderLayout());
		int i;
		Box box = Box.createHorizontalBox();
		for (i = 0; i < toolBars.length; i++) {
			// if I found "-" it means that button will be created
			if (!toolBars[i].equals("-")) {
				String label = ResourceManager
						.getLanguageDependentString(toolBars[i]
								+ BPDConstants.LABEL_SUFFIX);
				final Component c = createToolbar(toolBars[i], label);
				box.add(c);
				box.add(Box.createHorizontalStrut(8));
			} else {
				break;
			}
		}
		panel.add(BorderLayout.WEST, box);

		return panel;
	}
	
	/**
	 * 根据资源文件配置，创建一组工具栏 Create the toolbar. By default this reads the resource
	 * file for the definition of the toolbar.
	 * 
	 * @param key
	 *            The key in the resource file to serve as the basis
	 * @param label
	 *            The label of toolbar
	 * @see #createToolbar
	 */
	protected Component createToolbar(String key, String label) {
		JToolBar toolbar = new JToolBar(label);
		toolbar.putClientProperty("JToolBar.isRollover", Boolean.TRUE);
		toolbar.setFloatable(false);
		String[] toolKeys = Utils.tokenize(ResourceManager
				.getResourceString(key), " ");
		for (int i = 0; i < toolKeys.length; i++) {
			if (toolKeys[i].equals("-")) {
				toolbar.addSeparator();
			} else {
				toolbar.add(createTool(toolKeys[i]));
			}
		}
		toolbar.addSeparator();
		// 将此工具栏加入工具栏组的HashTable
		editor.putToolbarGroup(key, toolbar);
		return toolbar;
	}

	/**
	 * 根据资源文件配置，创建工具栏的按钮 Hook through which every toolbar item is created.
	 * 
	 * @see #createTool
	 */
	protected Component createTool(String key) {
		return createToolbarButton(key);
	}

	/**
	 * 根据资源文件配置，创建工具栏的按钮 Create a button to go inside of the toolbar. By default
	 * this will load an image resource. The image filename is relative to the
	 * classpath (including the '.' directory if its a part of the classpath),
	 * and may either be in a JAR file or a separate file.
	 * 
	 * @param key
	 *            The key in the resource file to serve as the basis of lookups.
	 */
	protected Component createToolbarButton(String key) {
		AbstractButton b = null;
		// 从资源文件获得按钮图标位置
		URL url = ResourceManager.getResource(key + BPDConstants.IMAGE_SUFFIX);
		if (url != null) {
			// 如果图标存在，创建带图标的按钮
			b = new JButton(new ImageIcon(url)) {
				public float getAlignmentY() {
					return 0.5f;
				}
			};
		} else {
			// 如果图标不存在，创建不带图标的按钮
			b = new JButton(ResourceManager.getLanguageDependentString(key
					+ BPDConstants.LABEL_SUFFIX)) {
				public float getAlignmentY() {
					return 0.5f;
				}
			};
		}
		// 设置按钮大小
		b.setMargin(new Insets(1, 1, 1, 1));
		b.setRequestFocusEnabled(false);
		// 设置按钮的行为侦听
		String astr = ResourceManager.getResourceString(key
				+ BPDConstants.ACTION_SUFFIX);
		if (astr == null) {
			astr = key;
		}
		Action a = editor.getAction(astr);
		if (a != null) {
			b.setActionCommand(astr);
			b.addActionListener(new EventRedirector(a));
			b.setEnabled(a.isEnabled());
			a
					.addPropertyChangeListener(new ActionChangedListener(
							b));
		} else {
			b.setEnabled(false);
		}
		// 设置按钮的鼠标经过提示
		String tip = ResourceManager.getLanguageDependentString(key
				+ BPDConstants.TIP_SUFFIX);
		if (tip != null) {
			b.setToolTipText(tip);
		}
		// 将此工具栏加入工具栏组的HashTable
		editor.putToolbarComponent(key, b);
		return b;
	}

	// ********************** CREATE ButtonGroup AND THEIR COMPONENTS
	// *****************
	/**
	 * 创建流程定义图形的按钮组 Creates WorkFlow's ButtonGroup. Reads resource to find out
	 * which ButtonGroup to create. NOTE: this method is modified to accept a
	 * creation of a button after ButtonGroup.
	 * 
	 * @see #createSpecialButtonGroup
	 */
	public Component createSpecialButtonGroup() {
		// 根据资源文件配置获得流程定义图形的按钮组
		String[] buttonGroup = Utils.tokenize(ResourceManager
				.getResourceString(editor.buttongroupToLoad()), " ");
		Box box = Box.createVerticalBox();
		JToolBar specialToolbar = new JToolBar();
		specialToolbar.putClientProperty("JToolBar.isRollover", Boolean.TRUE);
		specialToolbar.setFloatable(false);
		specialToolbar.setOrientation(JToolBar.VERTICAL);
		int i;
		final JPanel panel = new JPanel();
		panel.setLayout(new BorderLayout());
		for (i = 0; i < buttonGroup.length; i++) {
			// if I found "-" it means that button will be created
			if (!buttonGroup[i].equals("-")) {
				final Component button = createSpecialButtonGroupTool(buttonGroup[i]);
				specialToolbar.add(button);
			} else {
				break;
			}
		}
		box.add(specialToolbar);
		panel.add(BorderLayout.NORTH, box);
		return panel;
	}

	/**
	 * 根据资源文件配置，创建流程定义图形的按钮 Hook through which every buttongroup item is
	 * created.
	 * 
	 * @param key
	 *            The key in the resource file to serve as the basis of lookups.
	 * @see #createSpecialButtonGroupTool
	 */
	protected Component createSpecialButtonGroupTool(String key) {
		return createSpecialButtonGroupButton(key);
	}

	/**
	 * 根据资源文件配置，创建流程定义图形的按钮 Create a button to go inside of the buttongroup. By
	 * default this will load an image resource. The image filename is relative
	 * to the classpath (including the '.' directory if its a part of the
	 * classpath), and may either be in a JAR file or a separate file.
	 * 
	 * @param key
	 *            The key in the resource file to serve as the basis of lookups.
	 */
	protected Component createSpecialButtonGroupButton(String key) {
		// 从资源文件获得按钮图标的位置
		URL url = ResourceManager.getResource(key + BPDConstants.IMAGE_SUFFIX);
		// 创建按钮
		JToggleButton specialButton = new JToggleButton();
		// 设置图标
		specialButton.setIcon(new ImageIcon(url));
		
		(new ImageIcon(url)).getImage();
		// 设置大小
		specialButton.setMargin(new Insets(2, 2, 2, 2));
		// 设置鼠标提示
		String tip = ResourceManager.getLanguageDependentString(key
				+ BPDConstants.TIP_SUFFIX);
		if (tip != null) {
			specialButton.setToolTipText(tip);
		}
		// 初始化界面时，将此按钮组设成未激活
		if (BPD.getInstance().getActivedProcessEditor() == null) {
			specialButton.setEnabled(false);
		}
		// 将此按钮加入流程定义图形按钮组的HashTable
		editor.putSpecialButtonGroup(key, specialButton);
		return specialButton;
	}

	/**
	 * 将流程定义图形按钮组置为激活或未激活 Activate Or NoActivate SpecialButton Method
	 * 
	 * @param isActivate
	 *            是否激活流程定义图形按钮组
	 */
	public void activateSpecialButtonGroup(boolean isActivate) {
		String[] buttonGroup = Utils.tokenize(ResourceManager
				.getResourceString(editor.buttongroupToLoad()), " ");
		if (editor.specialButtonGroup.size() != 0) {
			for (int i = 0; i < buttonGroup.length; i++) {
				// if I found "-" it means that button will be created
				if (!buttonGroup[i].equals("-")) {
					JToggleButton specialButton = (JToggleButton) editor
							.getSpecialButtonGroup(buttonGroup[i]);
					specialButton.setEnabled(isActivate);
				} else {
					break;
				}
			}
		}
		// 点击选择按钮
		((JToggleButton) editor.getSpecialButtonGroup(BPDConstants.SELECT_TOOL))
				.doClick();
	}

	// ********************* CREATING MENUBAR AND IT'S COMPONENTS
	// ******************
	/**
	 * 创建主界面菜单 Create the menubar for the app. By default this pulls the
	 * definition of the menubar from the associated resource file.
	 */
	public JMenuBar createMenubar() {
		JMenuBar mb = new JMenuBar();
		// 从资源文件获得菜单组
		String[] menuKeys = Utils.tokenize(ResourceManager
				.getResourceString(editor.menubarToLoad()), " ");
		for (int i = 0; i < menuKeys.length; i++) {
			// 获得某组菜单的菜单项
			String[] itemKeys = Utils.tokenize(ResourceManager
					.getResourceString(menuKeys[i]), " ");
			JMenu m = createMenu(menuKeys[i], itemKeys);
			if (m != null) {
				mb.add(m);
			}
		}
		return mb;
	}
	
	
	/**
	 * 创建主界面菜单 Create the menubar for the app. By default this pulls the
	 * definition of the menubar from the associated resource file.
	 */
	public JMenuBar createFormMenubar(String key) {
		JMenuBar mb = new JMenuBar();
		// 从资源文件获得菜单组
		String[] menuKeys = Utils.tokenize(ResourceManager
				.getResourceString(key), " ");
		for (int i = 0; i < menuKeys.length; i++) {
			// 获得某组菜单的菜单项
			String[] itemKeys = Utils.tokenize(ResourceManager
					.getResourceString(menuKeys[i]), " ");
			JMenu m = createMenu(menuKeys[i], itemKeys);
			if (m != null) {
				mb.add(m);
			}
		}
		return mb;
	}


	/**
	 * 创建一组菜单 Create a menubar for the app. By default this pulls the definition of
	 * the menubar from the associated resource file.
	 */
	protected JMenu createMenu(String key, String[] itemKeys) {
		JMenu menu = new JMenu(ResourceManager.getLanguageDependentString(key
				+ BPDConstants.LABEL_SUFFIX));
		// 设置字体
		menu.setFont(BPDConfig.getInstance().getFont());
		// 创建菜单项
		for (int i = 0; i < itemKeys.length; i++) {
			if (itemKeys[i].equals("-")) {
				menu.addSeparator();
			} else {
				JMenuItem mi = createMenuItem(itemKeys[i], true);
				menu.add(mi);
			}
		}
		// 设置菜单图标
		URL url = ResourceManager.getResource(key + BPDConstants.IMAGE_SUFFIX);
		if (url != null) {
			menu.setHorizontalTextPosition(JButton.RIGHT);
			menu.setIcon(new ImageIcon(url));
		}
		// 设置组合键快捷方式
		setMnemonic(menu, ResourceManager.getLanguageDependentString(key
				+ BPDConstants.MNEMONIC_SUFFIX));

		menu.setActionCommand(key);

		return menu;
	}

	/**
	 * 创建菜单项 This is the hook through which all menubar items are created. It
	 * registers the result with the menuitem hashtable so that it can be
	 * fetched with getMenuItem().
	 */
	public JMenuItem createMenuItem(String cmd, boolean putIntoTable) {
		String subMenu = ResourceManager.getResourceString(cmd
				+ BPDConstants.MENU_SUFFIX);
		if (subMenu != null) {
			// 创建“最近打开文件”的子菜单
			String[] itemKeys = Utils.tokenize(subMenu, " ");
			JMenu mn = createMenu(cmd, itemKeys);
			if (putIntoTable) {
				editor.putMenuItem(cmd, mn);
			}
			return mn;
		} else {
			// 创建菜单项
			JMenuItem mi;
			Object specialItem = editor.getSpecialItem(cmd);
			if (specialItem instanceof JMenuItem) {
				mi = (JMenuItem) specialItem;
				mi.setText(ResourceManager.getLanguageDependentString(cmd
						+ BPDConstants.LABEL_SUFFIX));
				mi.setFont(BPDConfig.getInstance().getFont());
			} else {
				mi = new JMenuItem(ResourceManager
						.getLanguageDependentString(cmd
								+ BPDConstants.LABEL_SUFFIX));
				mi.setFont(BPDConfig.getInstance().getFont());
			}
			URL url = ResourceManager.getResource(cmd
					+ BPDConstants.IMAGE_SUFFIX);
			if (url != null) {
				mi.setHorizontalTextPosition(JButton.RIGHT);
				mi.setIcon(new ImageIcon(url));
			}
			// 设置组合键快捷方式
			setAccelerator(mi, ResourceManager.getLanguageDependentString(cmd
					+ BPDConstants.ACCEL_SUFFIX));
			setMnemonic(mi, ResourceManager.getLanguageDependentString(cmd
					+ BPDConstants.MNEMONIC_SUFFIX));
			// 设置菜单项的行为侦听
			if (specialItem == null) {
				String astr = ResourceManager.getResourceString(cmd
						+ BPDConstants.ACTION_SUFFIX);
				if (astr == null) {
					astr = cmd;
				}
				// if (astr.equals("OpenLocal")) astr = "OpenLocal";
				mi.setActionCommand(astr);
				Action a = editor.getAction(astr);
				if (a != null) {
					mi.addActionListener(new EventRedirector(a));
					a
							.addPropertyChangeListener(new ActionChangedListener(
									mi));
					mi.setEnabled(a.isEnabled());
				} else {
					mi.setEnabled(false);
				}
			}
			// 将菜单项加入菜单项HashTable
			if (putIntoTable) {
				editor.putMenuItem(cmd, mi);
			}
			return mi;
		}
	}

	/**
	 * 设置菜单项的组合键快捷方式
	 * 
	 * @param mi
	 *            菜单项
	 * @param mnemonic
	 *            快捷方式
	 */
	public static void setMnemonic(JMenuItem mi, String mnemonic) {
		if (mnemonic != null && mnemonic.length() > 0) {
			mi.setMnemonic(mnemonic.toCharArray()[0]);
		}
	}

	/**
	 * 设置菜单项的组合键快捷方式
	 * 
	 * @param mi
	 *            菜单项
	 * @param accel
	 *            快捷方式
	 */
	public static void setAccelerator(JMenuItem mi, String accel) {
		if (accel != null) {
			try {
				int mask = 0;
				if (accel.startsWith("CTRL")) {
					mask += ActionEvent.CTRL_MASK;
					accel = accel.substring(5);
				}
				if (accel.startsWith("SHIFT")) {
					mask += ActionEvent.SHIFT_MASK;
					accel = accel.substring(6);
				}
				if (accel.startsWith("ALT")) {
					mask += ActionEvent.ALT_MASK;
					accel = accel.substring(4);
				}
				int key = KeyEvent.class.getField("VK_" + accel).getInt(null);
				mi.setAccelerator(KeyStroke.getKeyStroke(key, mask));
			} catch (Exception e) {
				System.err
						.println("BarFactory Class--Error while assigning accelerator !!!");
			}
		}
	}
	 /**
	    * Create menubar item with appropriate icon.
	    */
	   public static JMenuItem createMenuItem( String cmd ) {
	      JMenuItem mi;
	      mi = new JMenuItem( ResourceManager.getLanguageDependentString( cmd +
	          BPDConstants.LABEL_SUFFIX ) );
	      URL url = ResourceManager.getResource( cmd + BPDConstants.IMAGE_SUFFIX );
	      if ( url != null ) {
	         mi.setHorizontalTextPosition( JButton.RIGHT );
	         mi.setIcon( new ImageIcon( url ) );
	      }
	      return mi;
	   }
	// ***************************** EVENTREDIRECTOR CLASS
	// ************************
	/**
	 * 改变图形的行为侦听源 This will change the source of the actionevent to graph.
	 */
	private class EventRedirector implements ActionListener {

		protected Action action;

		public EventRedirector(Action a) {
			this.action = a;
		}

		public void actionPerformed(ActionEvent e) {
			if (BPD.getInstance().getActivedProcessEditor() != null) {
				ProcessEditor activedProcessEditor = BPD.getInstance()
						.getActivedProcessEditor();
				e = new ActionEvent(activedProcessEditor.getGraph(), e.getID(),
						e.getActionCommand(), e.getModifiers());
			} else {
				e = new ActionEvent(new Object(), e.getID(), e
						.getActionCommand(), e.getModifiers());
			}
			action.actionPerformed(e);
		}

	}

	// ************************* END OF EVENTREDIRECTOR CLASS
	// *********************

	// ************************* ACTIONCHANGEDLISTENER CLASS
	// **********************
	/**
	 * 菜单项的行为侦听器 Class used to change enable state of buttons.
	 */
	private class ActionChangedListener implements PropertyChangeListener {
		AbstractButton button;

		ActionChangedListener(AbstractButton b) {
			super();
			button = b;
		}

		public void propertyChange(PropertyChangeEvent e) {
			String propertyName = e.getPropertyName();
			if (e.getPropertyName().equals(Action.NAME)
					&& button instanceof JMenuItem) {
				String text = (String) e.getNewValue();
				button.setText(text);
			} else {
				if (propertyName.equals("enabled")) {
					Boolean enabledState = (Boolean) e.getNewValue();
					button.setEnabled(enabledState.booleanValue());
				}
			}
		}
	}

	// ********************* END OF ACTIONCHANGEDLISTENER CLASS
	// ********************

	/**
	 * Creates popup menubar and adds a various actions (depending of where mouse
	 * was pressed - which process is selected). 工作流JTabbedPanel的右键菜单
	 */
	protected JPopupMenu createPopupMenuOfJTabbedPanel() {
		JPopupMenu menu = new JPopupMenu();
		menu.add(createMenuItem(Utils.getUnqualifiedClassName(Close.class),
				false));
		menu.add(createMenuItem(Utils.getUnqualifiedClassName(CloseAll.class),
				false));
		menu.addSeparator();
		menu.add(createMenuItem(Utils
				.getUnqualifiedClassName(ChangeApplication.class), false));
		menu.add(createMenuItem(
				Utils.getUnqualifiedClassName(Commission.class), false));
		menu.add(createMenuItem(Utils
				.getUnqualifiedClassName(ProcessProperties.class), false));
		return menu;
	}
}
