/**
 * $RCSfile: ProgressDialog.java,v $
 * $Revision: 1.1 $
 * $Date: 2016/09/30 14:56:42 $
 * <p>
 * Copyright (C) 2003 spk, Inc. All rights reserved.
 * <p>
 * This software is the proprietary information of spk, Inc.
 * Use is subject to license terms.
 */
package com.ds.common.swing;

import java.awt.Component;
import java.awt.Container;
import java.awt.Dialog;
import java.awt.Font;
import java.awt.Frame;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.*;
import javax.swing.border.Border;

import com.ds.common.swing.progress.Task;

/**
 * <p>Title: 常用代码打包</p>
 * <p>Description:
 * This class is a utility dialog. It can display the proceeding of a task's
 * performing thought a ProgressBar instance. It can automatically and timely
 * update the progress's current value and progress message according to the
 * task's current status.
 * <p>Copyright: Copyright (c) 2006</p>
 *
 * @author wenzhang li
 * @version 1.0
 */
public class ProgressDialog extends JDialog implements ActionListener {

    //----------------------------------------------------------------------------
    // Public methods
    //----------------------------------------------------------------------------

    public final static int UNDEFINED = -1;
    public final static int CANCEL = 1;
    public final static int ONE_SECOND = 1000;

    /**
     * @param owner                 the parent component for this dialog
     * @param title                 the progress dialog's title
     * @param note                  the note for progress
     * @param initialInfo           the initial message of the progress of the
     *                              task
     * @param task                  the task of progress
     * @param millisToDecideToPopup progress dialog decide to popup according to
     *                              how much millis passed
     * @param displayProgressBar    if display the progress bar
     */
    public ProgressDialog(Dialog owner, String title, String note, String initialInfo, Task task, int millisToDecideToPopup, boolean displayProgressBar) {
        super(owner, title != null ? title : "Progress Dialog", true);
        try {
            if (millisToDecideToPopup == 0) {
                millisToDecideToPopup = 500;
            }
            jbInit(owner, note, initialInfo, task, millisToDecideToPopup, displayProgressBar);
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }

    /**
     * @param owner                 the parent component for this dialog
     * @param title                 the progress dialog's title
     * @param note                  the note for progress
     * @param initialInfo           the initial message of the progress of the
     *                              task
     * @param task                  the task of progress
     * @param millisToDecideToPopup progress dialog decide to popup according to
     *                              how much millis passed
     * @param displayProgressBar    if display the progress bar
     */
    public ProgressDialog(Frame owner, String title, String note, String initialInfo, Task task, int millisToDecideToPopup, boolean displayProgressBar) {
        super(owner, title != null ? title : "Progress Dialog", true);

        try {
            if (millisToDecideToPopup == 0) {
                millisToDecideToPopup = 500;
            }
            jbInit(owner, note, initialInfo, task, millisToDecideToPopup, displayProgressBar);
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }

    /**
     * Gets the status attribute of the ProgressDialog object
     *
     * @return The status value
     */
    public int getStatus() {
        return status;
    }

    public void actionPerformed(ActionEvent e) {
        Object src = e.getSource();
        if (src == cancel) {
            status = CANCEL;
        }
        setVisible(false);
    }

    //----------------------------------------------------------------------------
    // Private methods
    //----------------------------------------------------------------------------

    private final static int WIDTH = 320;
    private final static int HEIGHT = 130;

    private GridBagLayout gridbag = new GridBagLayout();
    private JButton cancel = new JButton("取消");
    private JProgressBar progressBar;
    private JTextArea infoArea = new JTextArea(1, 35);
    private JLabel noteLabel = new JLabel();
    private JLabel imageLabel;
    private JPanel infoPanel = new JPanel();

    private int status = UNDEFINED;
    private Timer timer;

    private void jbInit(final Component owner, String note, String initialInfo, final Task task, int millisToDecideToPopup, final boolean displayProgressBar) {
        Container container = getContentPane();
        container.setLayout(gridbag);
        if (millisToDecideToPopup == 0) {
            millisToDecideToPopup = 500;
        }

        if (displayProgressBar) {
            progressBar = new JProgressBar(0, task.getLengthOfTask());
            progressBar.setValue(0);
            progressBar.setStringPainted(true);
        }

        noteLabel.setText(note);

        infoPanel.setLayout(gridbag);
        Border border = BorderFactory.createBevelBorder(3);
        infoArea.setBorder(border);
        infoArea.setText(initialInfo);
        infoArea.setEditable(false);
        infoArea.setLineWrap(false);
        infoArea.setFont(new Font("Dialog", 0, 14));
        infoArea.setBackground(getContentPane().getBackground());
        if (!displayProgressBar) {
            imageLabel = new JLabel(new ImageIcon(Images.getImage(ProgressDialog.class, "progress/connecting.gif")));
            infoPanel.add(imageLabel, new GridBagConstraints(0, 1, 1, 1, 1.0, 1.0, GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(0, 0, 5, 0), 0, 0));
            infoPanel.add(infoArea, new GridBagConstraints(1, 1, 1, 1, 1.0, 1.0, GridBagConstraints.WEST, GridBagConstraints.HORIZONTAL, new Insets(0, 0, 0, 0), 0, 0));
        } else {
            infoPanel.add(infoArea, new GridBagConstraints(0, 1, 1, 1, 1.0, 1.0, GridBagConstraints.WEST, GridBagConstraints.HORIZONTAL, new Insets(0, 0, 0, 0), 0, 0));
        }

        container.add(noteLabel, new GridBagConstraints(0, 1, 1, 1, 1.0, 1.0, GridBagConstraints.WEST, GridBagConstraints.HORIZONTAL, new Insets(0, 10, 0, 0), 0, 0));
        container.add(cancel, new GridBagConstraints(1, 1, 1, 1, 1.0, 1.0, GridBagConstraints.EAST, GridBagConstraints.NONE, new Insets(0, 0, 0, 10), 0, 0));
        container.add(infoPanel, new GridBagConstraints(0, 2, 1, 1, 1.0, 1.0, GridBagConstraints.WEST, GridBagConstraints.HORIZONTAL, new Insets(0, 15, 0, 0), 0, 0));

        if (displayProgressBar) {
            container.add(progressBar, new GridBagConstraints(0, 3, 1, 1, 1.0, 1.0, GridBagConstraints.WEST, GridBagConstraints.HORIZONTAL, new Insets(0, 0, 0, 0), 0, 0));
        }

        cancel.addActionListener(this);

        Point point;
        if (owner != null) {
            point = owner.getLocationOnScreen();
            int width = owner.getWidth();
            int x = (width - WIDTH) / 2;
            int height = owner.getHeight();
            int y = (height - HEIGHT) / 2;
            point.setLocation(point.x + x, point.y + y);
        } else {
            point = new Point(250, 250);
        }
        setLocation(point);
        setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);
        this.setSize(WIDTH, HEIGHT);

        //Create a timer.
        timer = new Timer(ONE_SECOND / 2, new ActionListener() {

            public void actionPerformed(ActionEvent evt) {
                if (status == CANCEL || task.getHasError() || task.done()) {
                    setVisible(false);
                    if (task.done()) {
                        task.complete();
                    } else {
                        task.cancel();
                    }
                    task.stop();
                    timer.stop();
                    if (task.getHasError()) {
                        JOptionPane.showMessageDialog(owner, task.getMessage(), "Error!", JOptionPane.ERROR_MESSAGE, null);
                    }
                } else {
                    infoArea.setText(task.getMessage());
                    if (displayProgressBar) {
                        progressBar.setValue(task.getCurrent());
                    }
                }
            }
        });
        timer.start();
        if (!task.done()) {
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    try {
                        ProgressDialog.this.setVisible(true);
                    } catch (Throwable te) {

                    }
                }
            });


        }

    }
}
