/* ActivitySet.java
 *
 * Authors:
 * Stefanovic Nenad  chupo@iis.ns.ac.yu
 * Bojanic Sasa      sasaboy@neobee.net
 * Puskas Vladimir   vpuskas@eunet.yu
 * Pilipovic Goran   zboniek@uns.ac.yu
 *
 */

package com.ds.bpm.bpd.xml.elements;

import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Node;

import com.ds.bpm.bpd.xml.XMLAttribute;
import com.ds.bpm.bpd.xml.XMLCollectionElement;
import com.ds.bpm.bpd.xml.XMLElement;
import com.ds.bpm.bpd.xml.activity.Activities;
import com.ds.bpm.bpd.xml.panels.XMLGroupPanel;
import com.ds.bpm.bpd.xml.panels.XMLPanel;

/**
 * Represents a WfMC DTD element that has the similar name.
 */
public class ActivitySet extends XMLCollectionElement {
	private Activities refActivities;

	private Transitions refTransitions;
	
	private XMLAttribute mainVersionId = new XMLAttribute("MainProcessVersionId"); // required	
	
	private XMLAttribute parentVersionId = new XMLAttribute("ParentVersionId"); // required		
	
	private XMLAttribute versionId = new XMLAttribute("VersionId"); // required
	
	/** Container for activity set. */
	private transient WorkflowProcess myWorkflow = null;

	/**
	 * Creates a new instance of the class.
	 */
	public ActivitySet(ActivitySets ass, WorkflowProcess myWorkflow) {
		super(ass);
		// the ID is controled other way
		ass.decrementID();
		this.myWorkflow = myWorkflow;
		//this.versionId.setValue(myWorkflow.getVersionID());
	
		this.parentVersionId.setValue(myWorkflow.getID());
		refActivities = new Activities(this);
		refTransitions = new Transitions(this);

		fillStructure();
	}
	
	public void setVersionId(String versionId){
	    this.versionId.setValue(versionId);
	}

	/**
	 * Defines the super-class method. Read the explanation for this method
	 * within XMLComplexElement class.
	 */
	protected void fillStructure() {
		super.fillStructure();
		attrId.setReadOnly(true);
		versionId.setReadOnly(true);
		complexStructure.add(versionId);
		complexStructure.add(mainVersionId);
		complexStructure.add(parentVersionId);
		complexStructure.add(refActivities);
		complexStructure.add(refTransitions);
		
	}

	/**
	 * Sets proper ID prefix for all Id generators (XMLCollection class
	 * instances).
	 */
	public void setIDPrefixForCollections() {
		String colPref = getID() + XMLCollectionElement.ID_DELIMITER;
		refActivities.setIDPrefix(colPref + "Act");
		refTransitions.setIDPrefix(colPref + "Tra");
	}

	/**
	 * Returns a workflow process that contains the set.
	 * 
	 * @return The workflow process that ownes the set.
	 */
	public WorkflowProcess getOwnerProcess() {
		return myWorkflow;
	}

	/**
	 * Called by Package object after importing of XML file to set the proper
	 * Participant objects for performer of activity, according on it's ID read
	 * from XML, to set proper reference to workflow process for activities of
	 * Subflow type, and to set proper user defined activity properties.
	 */
	protected void afterImporting(boolean importUserProperties) {
		refActivities.afterImporting(importUserProperties);
		refTransitions.afterImporting();
		refActivities.fillOutgoingTransitions();
		this.mainVersionId.setValue(myWorkflow.getMainProcess().getVersionID());
	}

	/**
	 * Prepares the tabbed panel to show editable fields of Activity. Panel
	 * consists of six tabs that logically comprises the Activity elements to be
	 * edited.
	 * 
	 * @return XMLPanel to be shown.
	 */
	public XMLPanel getPanel() {
		return new XMLGroupPanel(this, new XMLElement[] { attrId,
				refActivities, refTransitions }, toLabel());
	}

	/**
	 * Adds an user defined property to the activity. This is used when the new
	 * Activity is created by inserting it's visual object into the graph, to
	 * set all mandatory user properties defined, or when the new mandatory
	 * property is created.
	 * 
	 * @param up
	 *            The mandatory user property to add.
	 */
	public void addPropertyToAllActivities(UserProperty up) {
		refActivities.addPropertyToAllActivities(up);
	}

	/**
	 * Removes an user defined property upon it's removal from collection of
	 * properties.
	 * 
	 * @param up
	 *            The user property to remove.
	 */
	public void removePropertyFromAllActivities(UserProperty up) {
		refActivities.removePropertyFromAllActivities(up);
	}

	/**
	 * Adds an user defined property after the user changed it's type from
	 * <tt>OPTIONAL</tt> to <tt>MANDATORY</tt>.
	 * 
	 * @param up
	 *            The mandatory user property to add.
	 */
	public void refreshMandatories(UserProperty up) {
		refActivities.refreshMandatories(up);
	}

	/**
	 * Returns <tt>true</tt> if specified workflow process isn't referenced
	 * from any activity of subflow type.
	 */
	protected boolean canRemoveWorkflow(WorkflowProcess toRemove) {
		return refActivities.canRemoveWorkflow(toRemove);
	}

	/**
	 * Returns <tt>true</tt> if specified participant isn't referenced to be a
	 * performer of activity.
	 */
	protected boolean canRemoveParticipant(Participant toRemove) {
		return refActivities.canRemoveParticipant(toRemove);
	}

	/**
	 * Returns <tt>true</tt> if specified application isn't used anywhere.
	 */
	protected boolean canRemoveApplication(Application toRemove) {
		return refActivities.canRemoveApplication(toRemove);
	}

	/**
	 * Returns <tt>true</tt> if specified data field isn't used anywhere.
	 */
	protected boolean canRemoveDataFieldOrFormalParameter(
			XMLCollectionElement toRemove) {
		boolean cr = refActivities
				.canRemoveDataFieldOrFormalParameter(toRemove);
		if (cr == false) {
			return false;
		}
		return refTransitions.canRemoveDataFieldOrFormalParameter(toRemove);
	}

	/**
	 * Returns <tt>true</tt> if specified data field or formal parameter is
	 * used in transition condition expression of any transition within activity
	 * set's transitions.
	 */
	protected boolean isDataFieldOrFormalParameterUsedWithinTransitionConditions(
			XMLCollectionElement toRemove) {
		return !refTransitions.canRemoveDataFieldOrFormalParameter(toRemove);
	}

	public String toString() {
		return attrId.toString();
	}

	/**
	 * Overrides super-class method to realize this class specific writting to
	 * XML file.
	 */
	public void toXML(Node parent) throws DOMException {
		if (parent.getOwnerDocument() != null) {
			super.toXML(parent);
			// this happens when user wants to view XPDL presentation of the
			// graph
		} else {
			Node node = ((Document) parent).createElement(name);
			super.toXML(node);
			parent.appendChild(node);
		}
	}

	public void fromXML(Node node) {
		processAttributes(node);
		setIDPrefixForCollections();
		processElements(node);

		((ActivitySets) myCollection).updateID(getID());
	}

	// 实现clone方法
	public Object clone() {
		ActivitySet as = (ActivitySet) super.clone();
		as.refActivities = (Activities) this.refActivities.clone();
		as.refTransitions = (Transitions) this.refTransitions.clone();
		as.fillStructure();
		return as;
	}
}
