/* Activities.java
 *
 * Authors:
 * Stefanovic Nenad  chupo@iis.ns.ac.yu
 * Bojanic Sasa      sasaboy@neobee.net
 * Puskas Vladimir   vpuskas@eunet.yu
 * Pilipovic Goran   zboniek@uns.ac.yu
 *
 */

package com.ds.bpm.bpd.xml.activity;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import com.ds.bpm.bpd.enums.ActivityDefTypeEnums;
import com.ds.bpm.bpd.xml.XMLCollection;
import com.ds.bpm.bpd.xml.XMLCollectionElement;
import com.ds.bpm.bpd.xml.XMLComplexChoice;
import com.ds.bpm.bpd.xml.XMLComplexElement;
import com.ds.bpm.bpd.xml.XMLElement;
import com.ds.bpm.bpd.xml.elements.ActivitySet;
import com.ds.bpm.bpd.xml.elements.ActualParameter;
import com.ds.bpm.bpd.xml.elements.ActualParameters;
import com.ds.bpm.bpd.xml.elements.Application;
import com.ds.bpm.bpd.xml.elements.Participant;
import com.ds.bpm.bpd.xml.elements.UserProperty;
import com.ds.bpm.bpd.xml.elements.WorkflowProcess;
import com.ds.bpm.bpd.xml.panels.XMLGroupPanel;
import com.ds.bpm.bpd.xml.panels.XMLPanel;
import com.ds.bpm.bpd.xml.panels.XMLTableControlPanel;
import com.ds.bpm.bpd.xml.panels.XMLTablePanel;

/**
 * Represents a WfMC DTD element that has the similar name. This class is a collection of class {@link Activity}
 * instances.
 */
public class Activities extends XMLCollection {

    /**
     * Creates a new instance of the class.
     * 
     * @param wpOrAs
     *            WorkflowProcess or ActivitySet that holds Activities class.
     */
    public Activities(XMLComplexElement wpOrAs) {
	super(wpOrAs);
    }

    /**
     * Generates a new element of the class which instances are members of collection of this class.
     *
     * return The generated instance of class that makes collection.
     */
    public XMLElement generateNewElement() {
	WorkflowProcess wp;
	if (myOwner instanceof WorkflowProcess) {
	    wp = (WorkflowProcess) myOwner;
	} else {
	    wp = (WorkflowProcess) ((ActivitySet) myOwner).getCollection().getOwner();
	}
	Activity a = new Activity(this, wp, ActivityDefTypeEnums.Tool);
	a.setRequired(true);
	return a;
    }

    /**
     * Returns an XMLActivity specified by given ID.
     *
     * @param ID
     *            The ID attribute of wanted XMLActivity.
     * @return An instance of XMLActivity class that has specified ID. If collection of activities does not contain
     *         wanted Activity, the <tt>null</tt> is returned.
     */
    public Activity getActivity(String ID) {
	return (Activity) super.getCollectionElement(ID);
    }

    /**
     * Returns <tt>true</tt> if specified workflow process isn't referenced from any activity of subflow type.
     */
    public boolean canRemoveWorkflow(WorkflowProcess toRemove) {
	Activity a;
	Iterator it = refCollectionElements.iterator();
	while (it.hasNext()) {
	    a = (Activity) it.next();
	    if (a.getType().equals(ActivityDefTypeEnums.SubFlow)) {
		if (((XMLComplexChoice) a.getSubflow().get("WorkflowProcess")).getChoosen() == toRemove) {
		    return false;
		}
	    }
	}
	return true;
    }

    /**
     * Returns <tt>true</tt> if specified participant isn't referenced to be a performer of activity.
     */
    public boolean canRemoveParticipant(Participant toRemove) {
	Activity a;
	Iterator it = refCollectionElements.iterator();
	while (it.hasNext()) {
	    a = (Activity) it.next();
	    if (a.get("Performer").toValue() == toRemove) {
		return false;
	    }
	}
	return true;
    }

    /**
     * Returns <tt>true</tt> if specified application isn't used as a tool of activity which type is Tool.
     */
    public boolean canRemoveApplication(Application toRemove) {
	Activity a;
	Iterator it = refCollectionElements.iterator();
	while (it.hasNext()) {
	    a = (Activity) it.next();
	    if (a.getType().equals(ActivityDefTypeEnums.No)) {
		Tools ts = a.getTools();
		Iterator itTools = ts.toCollection().iterator();
		while (itTools.hasNext()) {
		    Tool t = (Tool) itTools.next();
		    if (((XMLComplexChoice) t.get("Application")).getChoosen() == toRemove) {
			return false;
		    }
		}
	    }
	}
	return true;
    }

    /**
     * Returns <tt>true</tt> if specified data field isn't used as actual parameter of activity which type is Tool or
     * Subflow.
     */
    public boolean canRemoveDataFieldOrFormalParameter(XMLCollectionElement toRemove) {
	Activity a;
	Iterator it = refCollectionElements.iterator();
	while (it.hasNext()) {
	    a = (Activity) it.next();
	    if (a.getType().equals(ActivityDefTypeEnums.No)) {
		Tools ts = a.getTools();
		Iterator itTools = ts.toCollection().iterator();
		while (itTools.hasNext()) {
		    Tool t = (Tool) itTools.next();
		    ActualParameters aps = (ActualParameters) t.get("ActualParameters");
		    Iterator itAPs = aps.toCollection().iterator();
		    while (itAPs.hasNext()) {
			ActualParameter ap = (ActualParameter) itAPs.next();
			if (ap.toValue() == toRemove) {
			    return false;
			}
		    }
		}
	    }
	    if (a.getType().equals(ActivityDefTypeEnums.SubFlow)) {
		SubFlow sfw = a.getSubflow();
		ActualParameters aps = (ActualParameters) sfw.get("ActualParameters");
		Iterator itAPs = aps.toCollection().iterator();
		while (itAPs.hasNext()) {
		    ActualParameter ap = (ActualParameter) itAPs.next();
		    if (ap.toValue() == toRemove) {
			return false;
		    }
		}
	    }
	}
	return true;
    }

    /**
     * Returns a set of Activity elements, which type is BlockActivity, contained within Activities object.
     *
     * @return A set of Activity object of BlockActivity type.
     */
    public Set getBlockActivities() {
	Set blockAs = new HashSet();
	Activity a;
	Iterator it = refCollectionElements.iterator();
	while (it.hasNext()) {
	    a = (Activity) it.next();
	    if (a.getType().equals(ActivityDefTypeEnums.Block)) {
		blockAs.add(a);
	    }
	}
	return blockAs;
    }

    /**
     * Called by owner object after importing of XML file to do neccessary things on activities after importing of XML.
     */
    public void afterImporting() {
	afterImporting(true);
    }

    /**
     * Called by owner object after importing of XML file to do neccessary things on activities after "afterImporting"
     * method is called both for activities and transitions
     */
    public void fillOutgoingTransitions() {
	Iterator it = refCollectionElements.iterator();
	while (it.hasNext()) {
	    Activity act = (Activity) it.next();
	    act.fillOutgoingTransitions();
	}
    }

    public int[] getInvisibleTableFieldOrdinals() {
	int[] itfo = new int[12];
	itfo[0] = 2;
	itfo[1] = 3;
	itfo[2] = 4;
	itfo[3] = 6;
	itfo[4] = 7;
	itfo[5] = 8;
	itfo[6] = 9;
	itfo[7] = 10;
	itfo[8] = 11;
	itfo[9] = 12;
	itfo[10] = 13;
	itfo[11] = 14;
	return itfo;
    }

    // First, the controlled panel must be created, and then the control panel
    public XMLPanel getPanel() {
	isReadOnly = true;
	controlledPanel = new XMLTablePanel(this, "", false, false);
	controlPanel = new XMLTableControlPanel(this, "", true, false);
	return new XMLGroupPanel(this, new XMLPanel[] { controlledPanel, controlPanel }, toLabel(), XMLPanel.BOX_LAYOUT, false, true);
    }

    public WorkflowProcess getWorkflowProcess() {
	Object own = getOwner();
	if (own instanceof ActivitySet) {
	    return ((ActivitySet) own).getOwnerProcess();
	} else {
	    return (WorkflowProcess) own;
	}
    }

    public Object toValue() {
	return String.valueOf(refCollectionElements.size());
    }

    public void addPropertyToAllActivities(UserProperty up) {
	Iterator it = refCollectionElements.iterator();
	while (it.hasNext()) {
	    ((Activity) it.next()).addUserProperty(up);
	}
    }

    public void removePropertyFromAllActivities(UserProperty up) {
	Iterator it = refCollectionElements.iterator();
	while (it.hasNext()) {
	    ((Activity) it.next()).removeUserProperty(up);
	}
    }

    public void refreshMandatories(UserProperty up) {
	Iterator it = refCollectionElements.iterator();
	while (it.hasNext()) {
	    ((Activity) it.next()).refreshMandatories(up);
	}
    }

    /**
     * Called by owner object after importing of XML file to do neccessary things on activities after importing of XML.
     */
    public void afterImporting(boolean importUserProperties) {
	Iterator it = refCollectionElements.iterator();
	while (it.hasNext()) {
	    Activity act = (Activity) it.next();
	    act.afterImporting(importUserProperties);
	}
    }

    /**
     * Overrides parent method to handle the case when the activities from WorkflowProcess collection are copied to
     * ActivitySet collection and vice-versa.
     */
    public void refreshCollection(Set elementsToAddOrRemove, boolean append) {
	if (append) {
	    for (Iterator it = elementsToAddOrRemove.iterator(); it.hasNext();) {
		Activity act = (Activity) it.next();
		// HM: generate new ID because of possible different id-prefixes
		if (act.getCollection() != this) {
		    decrementID();
		    act.get("Id").setValue(this.generateID());
		}
		act.setCollection(this);
		refCollectionElements.add(act);
	    }
	} else {
	    refCollectionElements.removeAll(elementsToAddOrRemove);
	}
    }

    /**
     * Overrides parent method to be accessible within WorkflowManager.
     */
    public void decrementID() {
	getWorkflowProcess().decrementActivityId();
    }

    public long getCurrentID() {
	return getWorkflowProcess().getCurrentActivityId();
    }

    public String generateID() {
	if (IDPrefix == null)
	    IDPrefix = "";
	String ID;
	do {
	    ID = IDPrefix + new Long(getWorkflowProcess().getNextActivityId()).toString();
	} while (getWorkflowProcess().getActivity(ID) != null);
	return ID;
    }

    protected void resetID() {
	getWorkflowProcess().resetActivityId();
    }

    protected void updateID(String someID) {
	// try to get ID as if it is ours
	try {
	    long val;
	    if (someID.startsWith(IDPrefix)) {
		String ID = someID.substring(IDPrefix.length(), someID.length());
		val = Long.parseLong(ID);
		if (val > getWorkflowProcess().getCurrentActivityId()) {
		    getWorkflowProcess().setCurrentActivityId(val);
		}
	    }
	    // then try to update someID as if it is only number
	} catch (Exception ex) {
	    return;
	}
    }

}
