package com.ds.bpm.bpd.misc;

import java.util.Enumeration;
import java.util.NoSuchElementException;
import java.util.Vector;

import javax.swing.tree.TreeNode;

import com.ds.common.database.metadata.ColInfo;
import com.ds.common.database.metadata.TableInfo;
import com.ds.esd.client.Project;
import com.ds.esd.tool.module.EUModule;
import com.ds.esd.tool.ui.component.Component;
import com.ds.iot.Area;
import com.ds.iot.Device;
import com.ds.iot.DeviceEndPoint;
import com.ds.iot.Place;
import com.ds.iot.ZNode;
import com.ds.org.Org;

public class OrgTreeNode implements TreeNode {

    private TreeNode parent;
    private Vector children;
    private boolean leaf;
    private Object userObject;

    public static final Enumeration EMPTY_ENUMERATION = new Enumeration() {
        public boolean hasMoreElements() {
            return false;
        }

        public Object nextElement() {
            throw new NoSuchElementException("No more elements");
        }
    };

    public OrgTreeNode(Object userObject) {
        super();
        this.parent = null;
        this.userObject = userObject;
    }

    public OrgTreeNode() {
        this(null);
    }

    /**
     * @return Returns the children.
     */
    public TreeNode[] getChildren() {
        if (children == null) {
            return new TreeNode[0];
        }
        return (TreeNode[]) children.toArray(new TreeNode[0]);
    }

    /**
     * @return Returns the parent.
     */
    public TreeNode getParent() {
        return parent;
    }

    /**
     * @param parent The parent to set.
     */
    public void setParent(TreeNode parent) {
        this.parent = parent;
    }

    /**
     * @return Returns the leaf.
     */
    public boolean isLeaf() {
        return leaf;
    }

    /**
     * @param leaf The leaf to set.
     */
    public void setLeaf(boolean leaf) {
        this.leaf = leaf;
    }

    public void insert(TreeNode newChild, int index) {
        if (!(newChild instanceof OrgTreeNode)) {
            throw new IllegalArgumentException("new child is not a OrgTreeNode");
        }
        if (newChild == null) {
            throw new IllegalArgumentException("new child is null");
        }
        OrgTreeNode oldParent = (OrgTreeNode) newChild.getParent();

        if (oldParent != null) {
            oldParent.remove(newChild);
        }
        ((OrgTreeNode) newChild).setParent(this);
        if (children == null) {
            children = new Vector();
        }
        children.add(index, newChild);
    }

    public void insert(TreeNode newChild) {
        OrgTreeNode oldParent = (OrgTreeNode) newChild.getParent();

        if (oldParent != null) {
            oldParent.remove(newChild);
        }
        ((OrgTreeNode) newChild).setParent(this);
        if (children == null) {
            children = new Vector();
        }
        children.add(newChild);
    }

    public void remove(TreeNode aChild) {
        if (aChild == null) {
            throw new IllegalArgumentException("argument is null");
        }

        if (!isNodeChild(aChild)) {
            throw new IllegalArgumentException("argument is not a child");
        }
        remove(getIndex(aChild)); // linear search
    }

    public void remove(int index) {
        OrgTreeNode child = (OrgTreeNode) getChildAt(index);
        children.remove(index);
        child.setParent(null);
    }

    public void removeAllChildren() {
        for (int i = getChildCount() - 1; i >= 0; i--) {
            remove(i);
        }
    }

    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }
        if (o instanceof OrgTreeNode) {
            OrgTreeNode oo = (OrgTreeNode) o;
            if (oo.userObject == null) {
                return false;
            }
            if (oo.userObject.equals(this.userObject)) {
                return true;
            }
        }
        return false;
    }

    /**
     * @return Returns the userObject.
     */
    public Object getUserObject() {
        return userObject;
    }

    /**
     * @param userObject The userObject to set.
     */
    public void setUserObject(Object userObject) {
        this.userObject = userObject;
    }

    public int getIndex(TreeNode aChild) {
        if (aChild == null) {
            throw new IllegalArgumentException("argument is null");
        }

        if (!isNodeChild(aChild)) {
            return -1;
        }
        return children.indexOf(aChild); // linear search
    }

    public TreeNode getChildAt(int index) {
        return (TreeNode) children.get(index);
    }

    public void removeFromParent() {
        OrgTreeNode parent = (OrgTreeNode) getParent();
        if (parent != null) {
            parent.remove(this);
        }
    }

    /**
     * Returns true if <code>aNode</code> is a child of this node. If <code>aNode</code> is null, this method returns
     * false.
     *
     * @return true if <code>aNode</code> is a child of this node; false if <code>aNode</code> is null
     */
    public boolean isNodeChild(TreeNode aNode) {
        boolean retval;

        if (aNode == null) {
            retval = false;
        } else {
            if (getChildCount() == 0) {
                retval = false;
            } else {
                retval = (aNode.getParent() == this);
            }
        }

        return retval;
    }

    /**
     * Returns the number of children of this node.
     *
     * @return an int giving the number of children of this node
     */
    public int getChildCount() {
        if (children == null) {
            return 0;
        } else {
            return children.size();
        }
    }

    public int getLevel() {
        TreeNode ancestor;
        int levels = 0;

        ancestor = this;
        while ((ancestor = ancestor.getParent()) != null) {
            levels++;
        }

        return levels;
    }

    public boolean isRoot() {
        return getParent() == null;
    }

    public Enumeration children() {
        if (children == null) {
            return EMPTY_ENUMERATION;
        } else {
            return children.elements();
        }
    }

    public boolean getAllowsChildren() {
        return true;
    }

    public String toString() {
        if (userObject == null) {
            return "root";
        }

        if (userObject instanceof Org) {
            return ((Org) userObject).getName();
        }


        if (userObject instanceof TableInfo) {
            return ((TableInfo) userObject).getName();
        }


        if (userObject instanceof Project) {
            return ((Project) userObject).getDesc() == null ? ((Project) userObject).getProjectName() : ((Project) userObject).getDesc();
        }
        if (userObject instanceof EUModule) {
            return ((EUModule) userObject).getDesc() == null ? ((EUModule) userObject).getName() : ((EUModule) userObject).getDesc();
        }
        if (userObject instanceof Component) {
            return ((Component) userObject).getProperties().getDesc() == null ? ((Component) userObject).getProperties().getName() : ((Component) userObject).getProperties().getDesc();
        }


        if (userObject instanceof ColInfo) {
            return ((ColInfo) userObject).getName();
        }

        if (userObject instanceof Place) {
            return ((Place) userObject).getName();
        }

        if (userObject instanceof Device) {
            return ((Device) userObject).getName();
        }

        if (userObject instanceof ZNode) {
            return ((ZNode) userObject).getName();
        }

        if (userObject instanceof DeviceEndPoint) {
            return ((DeviceEndPoint) userObject).getName();
        }

        if (userObject instanceof Area) {
            String name = ((Area) userObject).getName();
            if (name == null || name.equals("")) {
                name = ((Area) userObject).getName();
            }
            return name;
        }
        return "未知类型";
    }
}
