/* Participant.java
 *
 * Authors:
 * Stefanovic Nenad  chupo@iis.ns.ac.yu
 * Bojanic Sasa      sasaboy@neobee.net
 * Puskas Vladimir   vpuskas@eunet.yu
 * Pilipovic Goran   zboniek@uns.ac.yu
 *
 */

package com.ds.bpm.bpd.xml.elements;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import javax.swing.JTextField;

import com.ds.bpm.bpd.xml.XMLAttribute;
import com.ds.bpm.bpd.xml.XMLCollection;
import com.ds.bpm.bpd.xml.XMLCollectionElement;
import com.ds.bpm.bpd.xml.XMLComplexElement;
import com.ds.bpm.bpd.xml.XMLElement;
import com.ds.bpm.bpd.xml.XMLUtil;
import com.ds.bpm.bpd.xml.panels.XMLGroupPanel;
import com.ds.bpm.bpd.xml.panels.XMLPanel;
import com.ds.bpm.bpd.xml.panels.XMLTextPanel;

/**
 * Represents a WfMC DTD element that has the similar name. Participant is
 * entity involved in executing workflow process. Participant of workflow
 * process is a performer of workflow process activity. To extend its useability
 * Participant in BPD is visible too.
 * 
 * @see Participants
 */
public class Participant extends XMLCollectionElement {
	private ParticipantType refParticipantType = new ParticipantType();

	private ExternalReference refExternalReference = new ExternalReference();

	// min=0

	private XMLAttribute attrName = new XMLAttribute("Name");

	private int graphReferences = 0;

	/**
	 * Default participant to show if real participant is not selected as
	 * activity performer, or activity is of type that doesn't have performer.
	 */

	private static Participant freeTextExpressionParticipant = new Participant(
			null);
	static {
		freeTextExpressionParticipant
				.set("Id", "FreeTextExpressionParticipant");
		freeTextExpressionParticipant
				.set(
						"Name",
						XMLUtil
								.getLanguageDependentString("FreeTextExpressionParticipantKey"));
		freeTextExpressionParticipant.setReadOnly(true);
	}

	/**
	 * Returns default participant. This participant is never written to XML.
	 */
	public static Participant getFreeTextExpressionParticipant() {
		return freeTextExpressionParticipant;
	}

	/**
	 * Creates a new instance of the class.
	 * 
	 * @param p
	 *            The reference to collection of participants where this
	 *            instance will be put into.
	 */
	public Participant(Participants p) {
		super(p);

		fillStructure();
	}

	/**
	 * Defines the super-class method. Read the explanation for this method
	 * within XMLComplexElement class.
	 */
	protected void fillStructure() {
		super.fillStructure();
		// attrName.setRequired(true);
		complexStructure.add(attrName);
		attributes.add(attrName);
		refParticipantType.setRequired(true);
		complexStructure.add(refParticipantType);
	}

	/**
	 * Overrides the super class method to return the value of it's "Name"
	 * attribute value.
	 * 
	 * @return The value of this class "Name" attribute.
	 */
	public Object toValue() {
		return attrName.toValue();
	}

	/**
	 * Overrides super-class method to retreive the value of this class "Name"
	 * attribute plus three leters that represent the type of participant. This
	 * is used when displaying instance of this class within dialog or within
	 * the graph.
	 * 
	 * @return The "Name" attribute value of this class plus additional three
	 *         letters that represents the participant type.
	 */
	public String toString() {
		String partTypeString = refParticipantType.toValue().toString();
		if (partTypeString.equals("ROLE")) {
			partTypeString = " - rol";
		} else if (partTypeString.equals("ORGANIZATIONAL_UNIT")) {
			partTypeString = " - org";
		} else if (partTypeString.equals("RESOURCE_SET")) {
			partTypeString = " - set";
		} else if (partTypeString.equals("RESOURCE")) {
			partTypeString = " - res";
		} else if (partTypeString.equals("HUMAN")) {
			partTypeString = " - man";
		} else if (partTypeString.equals("SYSTEM")) {
			partTypeString = " - sys";
		}
		String disp = attrName.toString();
		if (disp.trim().length() == 0) {
			disp = attrId.toString().trim();
		}
		return disp + partTypeString;
	}

	/**
	 * Gets the tooltip for participant. The tooltip consists of property names
	 * and values.
	 * 
	 * @return The tooltip to be displayed when user holds the mouse above
	 *         participant's graph object.
	 */
	public String getTooltip() {
		return XMLUtil.makeTooltip(new XMLElement[] { attrId, attrName,
				refParticipantType });
	}

	/**
	 * Prepares panel to show editable fields of Participant. If participant
	 * does have children, panel will forbid user to change its type.
	 * 
	 * @return panel containing name, participant type and description fields
	 */
	public XMLPanel getPanel() {
		return new XMLGroupPanel(this, new XMLElement[] { attrId, attrName,
				refParticipantType, refExternalReference }, toLabel());
	}

	public Package getPackage() {
		if (myCollection != null) {
			return ((Participants) myCollection).getPackage();
		} else {
			return null;
		}
	}

	public void graphReferenceAdded() {
		graphReferences++;
	}

	public void graphReferenceRemoved() {
		graphReferences--;
	}

	public int getNoOfGraphReferences() {
		return graphReferences;
	}

	/**
	 * Returns <tt>true</tt> if specified Participant object has it's graph
	 * representation within any workflow process graph.
	 */
	public boolean isGraphContained() {
		return graphReferences > 0;
	}

	/**
	 * Checks if an ID entered by the user is unique.
	 */
	public boolean isIDUniqueAndValid(XMLPanel groupPanel) {
		if (this == freeTextExpressionParticipant)
			return true;
		XMLTextPanel tp = (XMLTextPanel) ((XMLGroupPanel) groupPanel)
				.getPanel(0);
		String IDToCheck = tp.getText();
		// if there is an element with given ID, return false
		XMLComplexElement p = getCollection().getCollectionElement(IDToCheck);
		boolean isOK = true;
		String message = null;
		String dialogTitle = null;
		if ((p != null && p != this)
				|| IDToCheck.equals(Participant
						.getFreeTextExpressionParticipant().getID())) {
			message = XMLUtil.getLanguageDependentString("ErrorIDMustBeUnique");
			dialogTitle = XMLUtil.getLanguageDependentString("DialogIDIsNotUnique");
			isOK = false;
		} else if (!XMLCollection.isIdValid(IDToCheck)) {
			message = XMLUtil.getLanguageDependentString("ErrorIDMustBeValid");
			dialogTitle = XMLUtil.getLanguageDependentString("DialogIDIsNotValid");
			isOK = false;
		} else {
			Participant pp = ((Participants) myCollection)
					.getParticipant(IDToCheck);
			if (pp != null && pp.isGraphContained() && pp != this) {
				message = XMLUtil
						.getLanguageDependentString("ErrorCannotRedefineParticipantThatIsShownInTheGraph");
				dialogTitle = XMLUtil.getLanguageDependentString("Title");
				isOK = false;
			}
		}
		if (!isOK) {
			XMLPanel.errorMessage(groupPanel.getDialog(), dialogTitle, "",
					message);
			((JTextField) tp.getComponent(2)).requestFocus();
		}
		return isOK;
	}

	/**
	 * This method is called only if user doesn't press Cancel button within the
	 * dialog for editing Participant properties, so the changes to the real
	 * collection of Participant are applied here.
	 * 
	 * @param groupPanel
	 *            The panel for editing parameters.
	 * @return always returns <tt>true</tt>.
	 */
	public boolean isValidEnter(XMLPanel groupPanel) {
		return true;
	}

	/**
	 * Overrides parent method to display participant type language specific.
	 */
	public Collection toComplexTypeValues() {
		java.util.List l = new ArrayList();
		Iterator it = complexStructure.iterator();
		while (it.hasNext()) {
			XMLElement el = (XMLElement) it.next();
			if (el instanceof XMLAttribute || el instanceof ParticipantType) {
				l.add(el.toString());
			} else {
				l.add(el.toValue());
			}
		}
		return l;
	}

	// 实现clone方法
	public Object clone() {
		Participant p = (Participant) super.clone();
		p.attrName = (XMLAttribute) this.attrName.clone();
		p.refParticipantType = (ParticipantType) this.refParticipantType
				.clone();
		p.fillStructure();
		return p;
	}
}
