/* Condition.java
*
* Authors:
* Stefanovic Nenad  chupo@iis.ns.ac.yu
* Bojanic Sasa      sasaboy@neobee.net
* Puskas Vladimir   vpuskas@eunet.yu
* Pilipovic Goran   zboniek@uns.ac.yu
*
*/

package com.ds.bpm.bpd.xml.elements;

import com.ds.bpm.enums.route.RouteEnums;
import org.w3c.dom.DOMException;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

import com.ds.bpm.bpd.xml.XML;
import com.ds.bpm.bpd.xml.XMLAttribute;
import com.ds.bpm.bpd.xml.XMLComplexElement;
import com.ds.enums.EnumsUtil;

/**
 * Represents a WfMC DTD element that has the similar name.
 *
 * @see XML
 */
public class Condition extends XMLComplexElement {

    // private Set refXpression=new HashSet(); // min=0, max=unbounded
    private Xpression refXpression = new Xpression(); // min=0, max=unbounded

    private XMLAttribute attrType = new XMLAttribute(EnumsUtil.getAttribute(RouteEnums.RouteCondition));

    /**
     * Creates a new instance of the class.
     */
    public Condition() {
	super();

	fillStructure();
    }

    /**
     * Defines the super-class method. Read the explanation for this method within XMLComplexElement class.
     */
    protected void fillStructure() {
	complexStructure.add(attrType);
	attributes.add(attrType);
	complexStructure.add(refXpression);
    }

    /**
     * Overrides super-class method to retrieve the defined condition.
     *
     * @return The condition of transition.
     */
    public String toString() {
	return refXpression.toString();
    }

    public Object toValue() {
	return refXpression.toValue();
    }

    /**
     * Overrides super-class method to indicate the emptiness of element. Here we use a trick: we temporarily remove the
     * attribute 'Type' from structure, so we can claim that element is empty although it's 'Type' attribute is not.
     *
     * @return The state of emptiness of element. The element is empty if all of it's members, except 'Type' attribute,
     *         are.
     */
    public boolean isEmpty() {
	complexStructure.remove(0);
	boolean isEmpty = super.isEmpty();
	complexStructure.add(0, attrType);
	if (isEmpty && !attrType.toValue().toString().equals("CONDITION")) {
	    isEmpty = false;
	}
	return isEmpty;
    }

    // NOTE: this must be corrected in a future, now, the Xpression
    // element is used only visualy to enter simple text, and when
    // writing to XML, this text is put into XML value and Xpression
    // tags aren't.
    // NOTE: maybe it should be defined the new basic class to suport
    // elements like Condition and Xpression
    /**
     * Overrides super-class method to realize this class specific writting to XML file.
     *
     * @return The string for a WfMC DTD Condition element tag.
     */
    public void toXML(Node parent) throws DOMException {
	if (isEmpty())
	    return;
	Node node = (parent.getOwnerDocument()).createElement(name);
	attrType.toXML(node);
	Node textNode = parent.getOwnerDocument().createTextNode(refXpression.toString());
	node.appendChild(textNode);
	parent.appendChild(node);
    }

    // this is just temporarily, when we implement expressions as
    // it should be, this must be changed
    /**
     * Overrides super-class method to realize this class specific reading from XML file.
     *
     * @param node
     *            root node
     */
    public void fromXML(Node node) {
	attrType.setValue("");

	// getting Type attrib.
	if (node.hasAttributes()) {
	    NamedNodeMap attribs = node.getAttributes();
	    Node type = attribs.getNamedItem("Type");
	    if (type != null) {
		attrType.fromXML(type);
	    }
	}
	// getting the node with text
	Object newVal;
	if (node.hasChildNodes()) {
	    newVal = node.getChildNodes().item(0).getNodeValue();
	} else {
	    newVal = node.getNodeValue();
	}
	if (newVal != null) {
	    value = newVal;
	}
	// passing the node content to Xpression
	refXpression.setValue(value);
    }

    /**
     * Used to create exact copy of instance of this class. The newly created instance will have all the properties same
     * as the copied one.
     *
     * @return The newly created instance of this class.
     */
    public Object clone() {
	Condition c = (Condition) super.clone();

	c.attrType = (XMLAttribute) this.attrType.clone();
	c.refXpression = (Xpression) this.refXpression.clone();
	c.fillStructure();

	return c;
    }

}
