/* XMLCollection.java
 *
 * Title : BPM工作流图形定义工具BPD
 * Class Desription：XML元素的XMLCollection类
 * Authors： wenzhang
 * Company： 基督山BPM
 * CreatedTime：2005-12-6
 *
 */

package com.ds.bpm.bpd.xml;

import com.ds.bpm.bpd.xml.activity.Activity;
import com.ds.bpm.bpd.xml.elements.WorkflowProcess;
import com.ds.bpm.bpd.xml.panels.*;
import org.apache.xml.utils.XMLChar;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import javax.swing.*;
import javax.swing.tree.DefaultMutableTreeNode;
import java.util.*;

/**
 * XMLCollection class instance represents program equivalence to the collection
 * of elements defined in some XML XML. It is used to enable user to visually
 * create new element and put it into collection, and also to edit it and
 * modify.
 * <p>
 * NOTE: Although this class is not declared abstract, it is uselles without
 * redefining it's methods (especially {@link #generateNewElement} method}.
 */
public class XMLCollection extends XMLElement {
	/** Used as a textual prefix for a generated numerical ID */
	protected transient String IDPrefix = "";

	/** Used to generate a unique ID for some element */
	private transient long nextID = 0;

	/** The collection of elements. */
	protected transient ArrayList refCollectionElements = new ArrayList();

	/**
	 * Visually presents the collection of elements. The default panel is
	 * instance of {@link XMLTablePanel} class.
	 */
	protected transient XMLPanel controlledPanel;

	/**
	 * The panel consisted of control buttons. It controls the process of
	 * adding, deleting and modifying the elements of collection. The default
	 * panel is instance of {@link XMLTableControlPanel} class.
	 */
	protected transient XMLControlPanel controlPanel;

	/** The owner class which element is this class instance. */
	protected transient XMLComplexElement myOwner;

	/**
	 * Create collection which is owned by specified owner.
	 * 
	 * @param myOwner
	 *            the program equivalence to an XML schema defined element that
	 *            holds the XML schema collection presented by this class
	 *            instance.
	 */
	public XMLCollection(XMLComplexElement myOwner) {
		super();
		this.myOwner = myOwner;
	}

	public XMLCollection(XMLComplexElement myOwner, String name) {
		super(name);
		this.myOwner = myOwner;
	}

	public XMLComplexElement getOwner() {
		return myOwner;
	}

	/** Adds new element to collection. */
	public void add(XMLElement el) {
		refCollectionElements.add(el);
	}

	/** Removes specified element from collection. */
	public void remove(Object el) {
		refCollectionElements.remove(el);
	}

	/** Gets the element that is placed at specified no. from collection. */
	public Object get(int no) {
		try {
			return refCollectionElements.get(no);
		} catch (Exception ex) {
			return null;
		}
	}

	/** Returns the number of elements within collection. */
	public int size() {
		return refCollectionElements.size();
	}

	/** Clears the collection. */
	public void clear() {
		refCollectionElements.clear();
	}

	/**
	 * Refreshes the collection.
	 * 
	 * @param elementsToAddOrRemove
	 *            Set of elements that has to be added to or removed from
	 *            collection.
	 * @param append
	 *            <tt>true</tt> if adding elements to collection,
	 *            <tt>false</tt> otherwise.
	 */
	public void refreshCollection(Set elementsToAddOrRemove, boolean append) {
		if (append) {
			refCollectionElements.addAll(elementsToAddOrRemove);
		} else {
			refCollectionElements.removeAll(elementsToAddOrRemove);
		}
	}

	/**
	 * Returnes the element specified by ID or Name attribute. If this is
	 * collection of XMLCollectionElement objects, element is searched by ID
	 * attribute, otherwise it is searched by Name attribute.
	 * 
	 * @param IDOrName
	 *            ID or Name attribute of wanted element.
	 * @return Wanted element if exist, null otherwise.
	 */
	public XMLComplexElement getCollectionElement(String IDOrName) {
		XMLComplexElement ce = null;
		String ceID = "";
		String ceName = "";
		Iterator it = refCollectionElements.iterator();
		while (it.hasNext()) {
			try {
				XMLComplexElement cetmp = (XMLComplexElement) it.next();
				if (cetmp instanceof XMLCollectionElement) {
					ceID = ((XMLCollectionElement) cetmp).getID();
				}
				if (cetmp.get("Name") != null) {
					ceName = cetmp.get("Name").toString();
				}
				if (ceName.equals(IDOrName) || ceID.equals(IDOrName)) {
					ce = cetmp;
					break;
				}
			} catch (ClassCastException cce) {
				cce.printStackTrace();
			} // just catch it
		}
		return ce;
	}

	/**
	 * Gets the structure of elements contained within collection, that is the
	 * all elements that element is made of.
	 */
	public Collection getElementStructure() {
		XMLElement el = generateNewElement();
		if (el instanceof XMLCollectionElement) {
			try {
				decrementID();
				((XMLCollectionElement) el).set("Id", "-1");
			} catch (Exception ex) {
			}
		}
		if (el instanceof XMLComplexElement) {
			return ((XMLComplexElement) el).toComplexType();
		} else {
			List l = new ArrayList();
			l.add(el);
			return l;
		}
	}

	/**
	 * Returns the ordinal numbers of elements not to be displayed within table
	 * panel. The ordinal numbers of elements corresponds to the inserting order
	 * within {@link XMLComplexElement#fillStructure} method.
	 */
	public int[] getInvisibleTableFieldOrdinals() {
		return null;
	}

	/**
	 * Sets the collection and all contained elements to be read only or not.
	 */
	public void setReadOnly(boolean ro) {
		isReadOnly = ro;
		Iterator it = refCollectionElements.iterator();
		while (it.hasNext()) {
			XMLElement el = (XMLElement) it.next();
			el.setReadOnly(ro);
		}
	}

	/** Returns the collection of all elements within collection. */
	public Collection toCollection() {
		return refCollectionElements;
	}

	/**
	 * Gets elements that can be choosed within table. Default implementation
	 * returns all elements within collection.
	 */
	public Collection getTableElements() {
		return refCollectionElements;
	}

	/**
	 * Gets elements that can be choosed within some combo box. Default
	 * implementation returns all elements within collection.
	 */
	public Collection getChoosable() {
		return refCollectionElements;
	}

	/**
	 * Generates the new element that made collection. Derived classes has to
	 * implement this method to create it's collection element.
	 */
	public XMLElement generateNewElement() {
		return new XMLElement();
	}

	/**
	 * Some specific things to be done after element is created. Default
	 * implementation is nothing to do, and derived classes should implement
	 * it's specific actions.
	 */
	public void onElementCreated(XMLElement el) {
		return;
	}

	/**
	 * Some specific things to be done after element is inserted. into
	 * collection. Default implementation is nothing to do, and derived classes
	 * should implement it's specific actions.
	 */
	public void onElementInserted(XMLElement el) {
		return;
	}

	/**
	 * Some specific things to be done after element from collection is
	 * modified. Default implementation is nothing to do, and derived classes
	 * should implement it's specific actions.
	 */
	public void onElementModified(XMLElement el) {
		return;
	}

	/**
	 * Some specific things to be done after element is deleted from collection.
	 * Default implementation is nothing to do, and derived classes should
	 * implement it's specific actions.
	 */
	public void onElementDeleted(XMLElement el) {
		return;
	}

	/**
	 * Some specific things to be done after element is removed from collection.
	 * Default implementation is nothing to do, and derived classes should
	 * implement it's specific actions.
	 */
	public void onElementRemoved(XMLElement el) {
		return;
	}

	/**
	 * Returns <tt>true</tt> if element can be inserted into collection.
	 * Default implementation returns <tt>true</tt>, and derived classes
	 * should implement it's specific check.
	 */
	public boolean canInsertElement(XMLElement el) {
		return true;
	}

	/**
	 * Returns <tt>true</tt> if element can be removed from collection.
	 * Default implementation returns <tt>true</tt>, and derived classes
	 * should implement it's specific check.
	 */
	public boolean canRemoveElement(XMLElement el) {
		return true;
	}

	public XMLPanel getControlledPanel() {
		return controlledPanel;
	}

	public XMLPanel getControlPanel() {
		return controlPanel;
	}

	/**
	 * Returns <tt>true</tt> if there is no elements within collection.
	 */
	public boolean isEmpty() {
		return size() == 0;
	}

	// First, the controlled panel must be created, and then the control panel
	public XMLPanel getPanel() {
		controlledPanel = new XMLTablePanel(this, "", false, false);
		controlPanel = new XMLTableControlPanel(this, "", true, false);
		return new XMLGroupPanel(this, new XMLPanel[] { controlledPanel,
				controlPanel }, toLabel(), XMLPanel.BOX_LAYOUT, false, true);
	}

	public void toXML(Node parent) throws DOMException {

		if (parent != null) {
			Node node=null;
			if (parent.getOwnerDocument()!=null){
				node = (parent.getOwnerDocument()).createElement(name);
			}else{
				node = ((Document) parent).createElement(name);
			}
			
			for (Iterator it = refCollectionElements.iterator(); it.hasNext();) {
				((XMLElement) it.next()).toXML(node);
			}
			parent.appendChild(node);
		}

	}

	public void fromXML(Node node) {
	
	 
		if (node != null) {
			String nameSpacePrefix = node.getPrefix();
			if (nameSpacePrefix != null) {
				nameSpacePrefix += ":";
			} else {
				nameSpacePrefix = "";
			}

			XMLElement newOne = generateNewElement();
			
			if (newOne instanceof XMLCollectionElement) {
				try {
					decrementID();
					((XMLCollectionElement) newOne).set("Id", "-1");
				} catch (Exception ex) {
				}
			}
			String elName = newOne.name;
			if (node.hasChildNodes()) {
				NodeList children = node.getChildNodes();
				int lng = children.getLength();
				for (int i = 0; i < lng; i++) {
					Node child = children.item(i);
					String xml=XMLUtil.getContent(child, false);
					if (child.getNodeName().equals(nameSpacePrefix + elName)) {
						newOne = generateNewElement();
						decrementID();
						newOne.fromXML(children.item(i));	
						
						refCollectionElements.add(newOne);
					}
				}
			}
		}
	}

	public String toString() {
		if (labelName != null) {
			return labelName;
		} else {
			return "";
		}
	}

	/**
	 * Returns the name of the message defined within property file that
	 * explains the reason why element marked as read only can't be deleted.
	 */
	public String getReadOnlyMessageName(XMLComplexElement el) {
		return "WarningCannotDeleteReadOnlyElement";
	}

	/**
	 * Returns the name of the message defined within property file that
	 * explains the reason why element that is in use can't be deleted.
	 */
	public String getInUseMessageName(XMLComplexElement el) {
		return "WarningCannotDeleteElementThatIsInUse";
	}

	// must make new collection that consists of the same elements as
	// previous one
	public Object clone() {
		XMLCollection d = (XMLCollection) super.clone();
		d.refCollectionElements = new ArrayList();
		Iterator it = this.refCollectionElements.iterator();
		while (it.hasNext()) {
			XMLElement el = (XMLElement) it.next();
			d.refCollectionElements.add(el.clone());
		}
		d.myOwner = this.myOwner;
		return d;
	}

	// INTERFACE FOR GENERATING IDS
	/** Sets the prefix for some ID generated by #generateID. */
	public void setIDPrefix(String idPref) {
		IDPrefix = idPref;
	}

	/** Returns the prefix of some ID generated by #generateID. */
	public String getIDPrefix() {
		return IDPrefix;
	}

	/** Sets the ID upon further IDs are generated. */
	protected void setCurrentID(long ID) {
		nextID = ID;
	}

	/** Gets the ID upon further IDs are generated. */
	public long getCurrentID() {
		return nextID;
	}

	/**
	 * Generates numerical ID for elements that need it, and converts that ID to
	 * string by adding it a prefix that is set using #setIDPrefix method. ID is
	 * generated by incrementing the previously generated numerical ID.
	 * 返回新生成的UUID
	 * 
	 * @return The string representation of ID.
	 */
	public String generateID() {
		/*
		 * if (IDPrefix==null) IDPrefix=""; String ID; do { ID=IDPrefix+new
		 * Long(++nextID).toString(); } while (getCollectionElement(ID)!=null);
		 * return ID;
		 */
		String uuid = UUID.randomUUID().toString();
		return uuid;
	}

	/**
	 * Resets an ID generator. Used when opening an existing XML file or when
	 * creating a new file.
	 */
	protected void resetID() {
		nextID = 0;
	}

	/**
	 * Decrements an ID. Used when the temporary element is created. This avoids
	 * unnecessary incrementation of IDs.
	 */
	protected void decrementID() {
		nextID--;
	}

	/**
	 * Updates an ID. Used when importing an existing XML file. The method
	 * updates ID and prepeares it for generation of new ID.
	 * 
	 * @param someID
	 *            the ID value upon the update is performed.
	 */
	protected void updateID(String someID) {
		// try to get ID as if it is ours
		try {
			long val;
			if (someID.startsWith(IDPrefix)) {
				String ID = someID
						.substring(IDPrefix.length(), someID.length());
				val = Long.parseLong(ID);
				if (val > nextID) {
					nextID = val;
				}
			}
			// then try to update someID as if it is only number
		} catch (Exception ex) {
			return;
		}
	}

	public void refreshLabelName() {
		super.refreshLabelName();
		Iterator it = refCollectionElements.iterator();
		while (it.hasNext()) {
			XMLElement el = (XMLElement) it.next();
			el.refreshLabelName();
		}
	}

	/**
	 * Checks if Id is valid NMTOKEN string.
	 */
	public static boolean isIdValid(String id) {
		return XMLChar.isValidNmtoken(id);
	}

	public int[] getEditableTableFieldOrdinals() {
		return null;
	}

	public DefaultCellEditor getEditableDefaultValue(int column) {
		return null;
	}

	public boolean isJCheckBox(int column) {
		return false;
	}

	// 得到所有子元素的Value值
	public Collection getValues() {
		List ret = new ArrayList();
		for (Iterator it = toCollection().iterator(); it.hasNext();) {
			XMLElement elem = (XMLElement) it.next();
			ret.add(elem.toValue());
		}
		return ret;
	}

	public JList getJList() {
		return ((XMLListPanel) controlledPanel).getList();
	}

	public void setOwner(XMLComplexElement myOwner) {
		this.myOwner = myOwner;
	}
	 public DefaultMutableTreeNode getNode() {
	      DefaultMutableTreeNode node = new ToNameMutableTreeNode(this);
	      for(int i = 0; i < this.refCollectionElements.size(); i++) {	     
			  node.add(					
	             ((XMLElement)this.refCollectionElements.get(i)).getNode()
	             );
	      }
	      return node;
	   }
	 public DefaultMutableTreeNode getXPDLNode() {	
	     DefaultMutableTreeNode node = new ToNameMutableTreeNode(this);	     
		  for(int i = 0; i < this.refCollectionElements.size(); i++) {	
			  XMLElement	xmel = (XMLElement)this.refCollectionElements.get(i);
			if  (xmel instanceof Activity ||
					xmel instanceof WorkflowProcess
			){
				node.add(xmel.getNode()
			             );
			}
			  
	      }
	      return node;
	   }
	
	
	 
}
