/* XMLComplexElement.java
 *
 * Authors:
 * Stefanovic Nenad  chupo@iis.ns.ac.yu
 * Bojanic Sasa      sasaboy@neobee.net
 * Puskas Vladimir   vpuskas@eunet.yu
 * Pilipovic Goran   zboniek@uns.ac.yu
 *
 */


package com.ds.bpm.bpd.xml;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import javax.swing.text.StyledDocument;
import javax.swing.tree.DefaultMutableTreeNode;

import com.ds.bpm.bpd.xml.panels.NewXMLGroupPanel;
import com.ds.bpm.bpd.xml.panels.NewXMLPanel;

import org.w3c.dom.DOMException;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

/**
 * Represents the complex XML element. For e.g., the following
 * definition within XML schema:
 * <pre>
 *  &lt;xsd:element name="DataField"&gt;
 *     &lt;xsd:complexType&gt;
 *        &lt;xsd:sequence&gt;
 *           &lt;xsd:element ref="DataType"/&gt;
 *           &lt;xsd:element ref="InitialValue" minOccurs="0"/&gt;
 *           &lt;xsd:element ref="Length" minOccurs="0"/&gt;
 *           &lt;xsd:element ref="Description" minOccurs="0"/&gt;
 *           &lt;xsd:element ref="ExtendedAttributes" minOccurs="0"/&gt;
 *        &lt;/xsd:sequence&gt;
 *        &lt;xsd:attribute name="Id" type="xsd:NMTOKEN" use="required"/&gt;
 *        &lt;xsd:attribute name="Name" type="xsd:string"/&gt;
 *        &lt;xsd:attribute name="IsArray" default="FALSE"&gt;
 *           &lt;xsd:simpleType&gt;
 *              &lt;xsd:restriction base="xsd:NMTOKEN"&gt;
 *                 &lt;xsd:enumeration value="TRUE"/&gt;
 *                 &lt;xsd:enumeration value="FALSE"/&gt;
 *              &lt;/xsd:restriction&gt;
 *           &lt;/xsd:simpleType&gt;
 *        &lt;/xsd:attribute&gt;
 *     &lt;/xsd:complexType&gt;
 *  &lt;/xsd:element&gt;
 * </pre>
 * will be presented as the object of the class derived from this class,
 * and the name of the class will be DataField. The DataField object will
 * have the other objects which are instances of XMLElement class that
 * will be placed within it's collection {@link #complexStructure}.
 * These elements, in this particular case, will be the instances of
 * {@link XMLAttribute} class (that will be also placed into another
 * collection - {@link #attributes}) which will represent the 'Id', 'Name'
 * and 'IsArray' XML attribute, the instances of classes <b>InitialValue</b>,
 * <b>Length</b>, <b>Description</b> which are all derived from {@link
 * XMLSimpleElement} class and represents corresponding XML simple elements,
 * and the instances of classes <b>DataType</b> and <b>ExtendedAttributes</b>,
 * which are also derived from <code>XMLComplexElement</code> class.
 * <p> The panels of elements that instance of <code>XMLComplexElement</code>
 * class contains in it's {@link #complexStructure} collection, will be placed
 * on it's panel, which allows it's editing.
 * <p> When writting or reading XML document, the {@link #complexStructure}
 * will be used to write/read all required elements.
 * <p> The methods for setting or getting the value of each element by it's name,
 * or by it's position within {@link #complexStructure} collectioin is provided.
 * <p>NOTE: Although this class is not declared abstract, it is
 *          uselles without redefining it's method {@link #fillStructure}.
 *          The classes derived from this class corresponds to the
 *          complex XML element.
 */
public class NewXMLComplexElement
    extends NewXMLElement {

   /**
    *  The list of all elements (attributes, and other elements) that
    *  complex element is consisted of.
    */
   protected List complexStructure = new ArrayList();
   /** The list of attributes that complex element contains. */
   protected List attributes = new ArrayList();

   public NewXMLComplexElement() {
      super();
   }

   /**
    * Returns the collection of elements this element is made of.
    */
   public Collection toComplexType() {
      return complexStructure;
   }

   /**
    * Returns the collection of strings that represents elements
    * that this element is made of.
    */
   public Collection toComplexTypeValues() {
      List l = new ArrayList();
      Iterator it = complexStructure.iterator();
      while ( it.hasNext() ) {
         XMLElement el = ( XMLElement ) it.next();
         if ( el instanceof XMLAttribute ) {
            l.add( el.toString() );
         }
         else {
            l.add( el.toValue() );
         }
      }
      return l;
   }

   /**
    * Sets the element and all elements it is made of to be
    * read only or not.
    */
   public void setReadOnly( boolean ro ) {
      super.setReadOnly( ro );
      Iterator it = complexStructure.iterator();
      while ( it.hasNext() ) {
         XMLElement el = ( XMLElement ) it.next();
         el.setReadOnly( ro );
      }
   }

   public boolean isEmpty() {
      boolean isEmpty = true;
      Iterator it = complexStructure.iterator();
      while ( it.hasNext() ) {
         XMLElement el = ( XMLElement ) it.next();
         isEmpty = isEmpty && el.isEmpty();
      }
      return isEmpty;
   }

   public boolean isValid() {
      boolean isValid = true;
      Iterator it = complexStructure.iterator();
      while ( it.hasNext() ) {
         XMLElement el = ( XMLElement ) it.next();
         isValid = isValid && el.isValid();
      }
      return isValid;
   }

   public void toXML( Node parent ) throws DOMException {
      if ( isEmpty() && !isRequired() )
         return;
      if ( parent != null ) {
         Node node = ( parent.getOwnerDocument() ).createElement( name );
         for ( Iterator it = complexStructure.iterator(); it.hasNext(); ) {
            XMLElement el = ( XMLElement ) it.next();
            el.toXML( node );
         }
         parent.appendChild( node );
      }
   }

   public void fromXML( Node node ) {
      processAttributes( node );
      processElements( node );
   }

   protected void processAttributes( Node node ) {
      if ( node != null ) {
         if ( node.hasAttributes() ) {
            NamedNodeMap attribs = node.getAttributes();
            for ( int i = 0; i < attribs.getLength(); ++i ) {
               Node attrib = ( Node ) attribs.item( i );
               try {
//System.out.println("Getting attrib "+attrib.getNodeName());
                  get( attrib.getNodeName() ).fromXML( attrib );
               }
               catch ( NullPointerException npe ) {
                  /*if (attribute.getNodeValue().trim().length() > 0 ) {
                     System.err.println("Processing attributes for "+ name
                                     +" element having problems with "
                                     + attribute.getNodeName()+" attribute\n"
                                     + attribute.getNodeValue().trim());
                                     }*/
               }
            }
         }
      }
   }

   protected void processElements( Node node ) {
      if ( node != null ) {
         if ( node.hasChildNodes() ) {
            XMLUtil.parseElements( node, complexStructure );
         }
      }
   }

   public NewXMLPanel getPanel() {
      NewXMLElement[] c = new NewXMLElement[complexStructure.size()];
      complexStructure.toArray( c );
      return new NewXMLGroupPanel( this, c, toLabel() );
   }

   /** Gets the element that is placed at specified no. within structure. */
   public XMLElement get( int no ) {
      try {
         return ( XMLElement ) complexStructure.get( no );
      }
      catch ( Exception ex ) {
         return null;
      }
   }

   /**
    * Sets the element that is placed at specified no. within structure
    * to the specified value.
    */
   public void set( int no, Object value ) {
      XMLElement el;
      try {
         el = get( no );
      }
      catch ( Exception ex ) {
         el = null;
      }
      if ( el != null ) {
         el.setValue( value );
      }
   }

   /** Gets the element with specified name from stucture. */
   public XMLElement get( String name ) {
      Iterator it = complexStructure.iterator();
      while ( it.hasNext() ) {
         XMLElement el = ( XMLElement ) it.next();
         if ( el.name.equals( name ) ) {
            return el;
         }
      }
      return null;
   }

   /**
    * Sets the element with specified name from stucture to
    * the specified value.
    */
   public void set( String name, Object value ) {
      XMLElement el = get( name );
      if ( el != null ) {
         el.setValue( value );
      }
   }

   public String toString() {
      if ( labelName != null ) {
         return labelName;
      }
      else {
         return "";
      }
   }

   /**
    * The classes that are derived from this class has to give its
    * definition for this method. It is used to insert all members
    * of those classes that are derived from XMLElement
    * class (or classes that are derived from that class) into
    * one list, and the members that are derived from
    * XMLAttribute class also to insert into other list.
    * Many methods of this class uses those lists:
    * <code>toXML</code> method uses them to properly write an element
    * tag to the XML file,
    * <code>fromXML</code> method uses them to properly read an element
    * tag from the XML file,
    * <code>getPanel</code> method uses them to properly display members
    * of this class,
    * <code>get</code> and <code>set</code> uses them to properly
    * retrieve/set the elements from list, ... and so on
    * <p>NOTE: The order of inserted elements is relevant for XML to be
    *          valid (members of classes derived from this class must be
    *          inserted into first mentioned list in the same order that
    *          they are within a corresponding tag for those classes
    *          within WfMC XML).
    */
   protected void fillStructure() {
      return;
   }

   // prepeares cloning for extended classes
   public Object clone() {
      NewXMLComplexElement d = ( NewXMLComplexElement )super.clone();
      d.complexStructure = new ArrayList();
      d.attributes = new ArrayList();

      return d;
   }

   public void refreshLabelName() {
      super.refreshLabelName();
      Iterator itCs = complexStructure.iterator();
      while ( itCs.hasNext() ) {
         XMLElement el = ( XMLElement ) itCs.next();
         el.refreshLabelName();
      }
   }

   /**
    * DefaultMutableTreeNode with all subnodes of this element.
    * @return return node with subelements.
    */
   public DefaultMutableTreeNode getNode() {
      DefaultMutableTreeNode node = new ToNameMutableTreeNode( this );
      for ( int i = 0; i < this.complexStructure.size(); i++ ) {
         if ( !this.attributes.contains( this.complexStructure.get( i ) ) ) {
            node.add(
                ( ( NewXMLElement )this.complexStructure.get( i ) ).getNode()
                );
         }
      }
      return node;
   }

   /**
    * Count child elements of this complex element.
    * @return count of subelements.
    */
   public int getElementsCount() {
      int elementsCount = 0;
      for ( int i = 0; i < this.complexStructure.size(); i++ ) {
         if ( !this.attributes.contains( this.complexStructure.get( i ) ) ) {
            elementsCount++;
         }
      }
      return elementsCount;
   }

   /**
    * Return element child nodes, without attributes.
    * @return list with subelements.
    */
   public List getChildElements() {
      List list = new ArrayList();
      for ( int i = 0; i < this.complexStructure.size(); i++ ) {
         if ( !this.attributes.contains( this.complexStructure.get( i ) ) ) {
            list.add( this.complexStructure.get( i ) );
         }
      }
      return list;
   }

   /**
    * Make print description of this element and all subelements and attributes.
    * @return print description of element
    */
   public String getPrintDescription( String indent, StyledDocument doc ) {
      String retVal = "";
      try {
         StringBuffer all = new StringBuffer();
         //attributes
         StringBuffer attributes = new StringBuffer();
         int size = this.attributes.size();
         for ( int i = 0; i < size; i++ ) {
            XMLAttribute attr = ( ( XMLAttribute )this.attributes.get( i ) );
            attributes.append( attr.toName() + " = " + attr.toString() );
            if ( i != size - 1 ) {
               attributes.append( ", " );
            }
         }

         all.append( toName() + " :   " + attributes.toString() );
         doc.insertString( doc.getLength(), toName() + " :   " + attributes.toString(), atts );

         //only extended nodes should be displayed
         if ( !this.isCollapsed() ) {
            //subelements
            String value = "";
            for ( int i = 0; i < this.complexStructure.size(); i++ ) {
               if ( !this.attributes.contains( this.complexStructure.get( i ) ) ) {
                  value = ( ( NewXMLElement )this.complexStructure.get( i ) ).getPrintDescription(
                      indent +
                      OFFSET, doc );
                  if ( value != null && !value.trim().equals( "" ) ) {
                     all.append( NEWLINE );
                     all.append( indent );
                     all.append( value );
                  }
               }
            }
         }
         retVal = all.toString();
      }
      catch ( Exception e ) {
         e.printStackTrace();
      }
      return retVal;
   }

}
