package com.ds.bpm.bpd.misc;

import com.ds.esd.plugins.api.node.APIPaths;
import com.ds.esd.tool.module.EUModule;
import com.ds.esd.tool.ui.component.Component;

import javax.swing.tree.TreeNode;
import java.util.Enumeration;
import java.util.NoSuchElementException;
import java.util.Vector;

public class APITreeNode<T> implements TreeNode {

    private TreeNode parent;
    private Vector<TreeNode> children;
    private boolean leaf;
    private T userObject;

    public static final Enumeration EMPTY_ENUMERATION = new Enumeration() {
        public boolean hasMoreElements() {
            return false;
        }

        public Object nextElement() {
            throw new NoSuchElementException("No more elements");
        }
    };

    public APITreeNode(T userObject) {
        super();
        this.parent = null;
        this.userObject = userObject;
    }

    public APITreeNode() {
        this(null);
    }

    /**
     * @return Returns the children.
     */
    public TreeNode[] getChildren() {
        if (children == null) {
            return new TreeNode[0];
        }
        return (TreeNode[]) children.toArray(new TreeNode[0]);
    }

    /**
     * @return Returns the parent.
     */
    public TreeNode getParent() {
        return parent;
    }

    /**
     * @param parent The parent to set.
     */
    public void setParent(TreeNode parent) {
        this.parent = parent;
    }

    /**
     * @return Returns the leaf.
     */
    public boolean isLeaf() {
        return leaf;
    }

    /**
     * @param leaf The leaf to set.
     */
    public void setLeaf(boolean leaf) {
        this.leaf = leaf;
    }

    public void insert(TreeNode newChild, int index) {
        if (!(newChild instanceof APITreeNode)) {
            throw new IllegalArgumentException("new child is not a OrgTreeNode");
        }
        if (newChild == null) {
            throw new IllegalArgumentException("new child is null");
        }
        APITreeNode oldParent = (APITreeNode) newChild.getParent();

        if (oldParent != null) {
            oldParent.remove(newChild);
        }
        ((APITreeNode) newChild).setParent(this);
        if (children == null) {
            children = new Vector();
        }
        children.add(index, newChild);
    }

    public void insert(TreeNode newChild) {
        APITreeNode oldParent = (APITreeNode) newChild.getParent();

        if (oldParent != null) {
            oldParent.remove(newChild);
        }
        ((APITreeNode) newChild).setParent(this);
        if (children == null) {
            children = new Vector();
        }
        children.add(newChild);
    }

    public void remove(TreeNode aChild) {
        if (aChild == null) {
            throw new IllegalArgumentException("argument is null");
        }

        if (!isNodeChild(aChild)) {
            throw new IllegalArgumentException("argument is not a child");
        }
        remove(getIndex(aChild)); // linear search
    }

    public void remove(int index) {
        APITreeNode child = (APITreeNode) getChildAt(index);
        children.remove(index);
        child.setParent(null);
    }

    public void removeAllChildren() {
        for (int i = getChildCount() - 1; i >= 0; i--) {
            remove(i);
        }
    }

    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }
        if (o instanceof APITreeNode) {
            APITreeNode oo = (APITreeNode) o;
            if (oo.userObject == null) {
                return false;
            }
            if (oo.userObject.equals(this.userObject)) {
                return true;
            }
        }
        return false;
    }

    /**
     * @return Returns the userObject.
     */
    public Object getUserObject() {
        return userObject;
    }

    /**
     * @param userObject The userObject to set.
     */
    public void setUserObject(T userObject) {
        this.userObject = userObject;
    }

    public int getIndex(TreeNode aChild) {
        if (aChild == null) {
            throw new IllegalArgumentException("argument is null");
        }

        if (!isNodeChild(aChild)) {
            return -1;
        }
        return children.indexOf(aChild); // linear search
    }

    public TreeNode getChildAt(int index) {
        return (TreeNode) children.get(index);
    }

    public void removeFromParent() {
        APITreeNode parent = (APITreeNode) getParent();
        if (parent != null) {
            parent.remove(this);
        }
    }

    /**
     * Returns true if <code>aNode</code> is a child of this node. If <code>aNode</code> is null, this method returns
     * false.
     *
     * @return true if <code>aNode</code> is a child of this node; false if <code>aNode</code> is null
     */
    public boolean isNodeChild(TreeNode aNode) {
        boolean retval;

        if (aNode == null) {
            retval = false;
        } else {
            if (getChildCount() == 0) {
                retval = false;
            } else {
                retval = (aNode.getParent() == this);
            }
        }

        return retval;
    }

    /**
     * Returns the number of children of this node.
     *
     * @return an int giving the number of children of this node
     */
    public int getChildCount() {
        if (children == null) {
            return 0;
        } else {
            return children.size();
        }
    }

    public int getLevel() {
        TreeNode ancestor;
        int levels = 0;

        ancestor = this;
        while ((ancestor = ancestor.getParent()) != null) {
            levels++;
        }

        return levels;
    }

    public boolean isRoot() {
        return getParent() == null;
    }

    public Enumeration children() {
        if (children == null) {
            return EMPTY_ENUMERATION;
        } else {
            return children.elements();
        }
    }

    public boolean getAllowsChildren() {
        return true;
    }

    public String toString() {
        if (userObject == null) {
            return "root";
        }

        if (userObject instanceof APIPaths) {
            APIPaths apiPaths = (APIPaths) userObject;
            return   apiPaths.getDesc() == null ?   apiPaths.getName(): apiPaths.getDesc();
        }

        if (userObject instanceof EUModule) {
            return ((Component) userObject).getAlias();
        }


        return "未知类型";
    }
}
