/* UserProperty.java
 *
 * Authors:
 * Stefanovic Nenad  chupo@iis.ns.ac.yu
 * Bojanic Sasa      sasaboy@neobee.net
 * Puskas Vladimir   vpuskas@eunet.yu
 * Pilipovic Goran   zboniek@uns.ac.yu
 *
 */

package com.ds.bpm.bpd.xml.elements;

import com.ds.bpm.bpd.xml.XML;
import com.ds.bpm.bpd.xml.XMLAttribute;
import com.ds.bpm.bpd.xml.XMLCollectionElement;
import com.ds.bpm.bpd.xml.XMLElement;
import com.ds.bpm.bpd.xml.panels.XMLGroupPanel;
import com.ds.bpm.bpd.xml.panels.XMLPanel;

/**
 * This is not a real WfMC DTD element, but it is introduced to help. It
 * represents a user defined activity property.
 * 
 * @see XML
 */
public class UserProperty extends XMLCollectionElement {
	private XMLAttribute vName = new XMLAttribute("Name");

	private XMLAttribute vDataType = new XMLAttribute("DataType", new String[] {
			"STRING", "INTEGER", "BOOLEAN", "DATETIME" }, 0);

	private XMLAttribute vRelevance = new XMLAttribute("Relevance",
			new String[] { "OPTIONAL", "MANDATORY", "MANDATORYFROMNOWON" }, 0);

	private XMLAttribute vDefaultValue = new XMLAttribute("DefaultValue");

	/**
	 * Creates a new instance of the class.
	 * 
	 * @param ups
	 *            The reference to collection of user properties where this
	 *            instance will be put into.
	 */
	public UserProperty(UserProperties ups) {
		super(ups);

		fillStructure();
	}

	/**
	 * Defines the super-class method. Read the explanation for this method
	 * within XMLComplexElement class.
	 */
	protected void fillStructure() {
		super.fillStructure();
		setRequired(true);
		attrId.setReadOnly(true);
		vName.setRequired(true);
		complexStructure.add(vName);
		vDataType.setRequired(true);
		complexStructure.add(vDataType);
		vRelevance.setRequired(true);
		complexStructure.add(vRelevance);
		complexStructure.add(vDefaultValue);
	}

	/**
	 * Indicates if user define activity property is mandatory for all
	 * activities within a package.
	 * 
	 * @return <tt>true</tt> if property is mandatory, <tt>false</tt>
	 *         otherwise.
	 */
	public boolean isMandatory() {
		return vRelevance.toValue().toString().equals("MANDATORY");
	}

	/**
	 * Indicates if user define activity property is mandatory for every
	 * activity within a package created after it's definition.
	 * 
	 * @return <tt>true</tt> if property is mandatory from it's creation,
	 *         <tt>false</tt> otherwise.
	 */
	public boolean isMandatoryFromNowOn() {
		return vRelevance.toValue().toString().equals("MANDATORYFROMNOWON");
	}

	/**
	 * Indicates if user define activity property is optional.
	 * 
	 * @return <tt>true</tt> if property is optional, <tt>false</tt>
	 *         otherwise.
	 */
	public boolean isOptional() {
		return vRelevance.toValue().toString().equals("OPTIONAL");
	}

	/**
	 * Overrides super-class method to retreive the value of this class "Name"
	 * attribute. This is used when displaying instance of this class within
	 * dialog.
	 * 
	 * @return The "Name" attribute value of this class.
	 */
	public String toString() {
		if (vName != null) {
			return vName.toString();
		} else {
			return "";
		}
	}

	/**
	 * Prepares the one of the group panels that contains some of this element
	 * editable fields.
	 * 
	 * @param no
	 *            The ordinal number of group panel to be shown.
	 * @return XMLPanel to be shown.
	 */
	public XMLPanel getPanel(int no) {
		XMLPanel p;
		switch (no) {
		case 0:
			p = attrId.getPanel();
			break;
		case 1:
			p = vName.getPanel();
			break;
		case 2:
			p = vDataType.getPanel();
			break;
		case 3:
			p = vRelevance.getPanel();
			break;
		case 4:
			p = vDefaultValue.getPanel();
			break;
		default:
			p = new XMLPanel();
		}
		return p;
	}

	/**
	 * Prepares the panel to show editable fields of UserProperty instance.
	 * 
	 * @return XMLPanel to be shown.
	 */
	public XMLPanel getPanel() {
		return new XMLGroupPanel(this, new XMLElement[] { attrId, vName,
				vDataType, vRelevance, vDefaultValue }, toLabel());
	}

	/**
	 * Used to create exact copy of instance of this class. The newly created
	 * instance will have all the properties same as the copied one.
	 * 
	 * @return The newly created instance of this class.
	 */
	public Object clone() {
		UserProperty up = (UserProperty) super.clone();
		up.vName = (XMLAttribute) this.vName.clone();
		up.vDataType = (XMLAttribute) this.vDataType.clone();
		up.vRelevance = (XMLAttribute) this.vRelevance.clone();

		up.fillStructure();

		return up;
	}

}
