/* DataFields.java
 *
 * Authors:
 * Stefanovic Nenad  chupo@iis.ns.ac.yu
 * Bojanic Sasa      sasaboy@neobee.net
 * Puskas Vladimir   vpuskas@eunet.yu
 * Pilipovic Goran   zboniek@uns.ac.yu
 *
 */

package com.ds.bpm.bpd.xml.elements;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import com.ds.bpm.bpd.xml.XML;
import com.ds.bpm.bpd.xml.XMLCollection;
import com.ds.bpm.bpd.xml.XMLCollectionElement;
import com.ds.bpm.bpd.xml.XMLComplexElement;
import com.ds.bpm.bpd.xml.XMLElement;
import com.ds.bpm.bpd.xml.activity.Activities;
import com.ds.bpm.bpd.xml.elements.formula.FormalParameter;
import com.ds.bpm.bpd.xml.elements.formula.FormalParameters;

/**
 * Represents a WfMC DTD element that has the similar name. This class is a
 * collection of class <b>DataField</b> instances.
 * 
 * @see XML
 */
public class DataFields extends XMLCollection {
	/**
	 * Creates a new instance of the class.
	 * 
	 * @param packageORWProcess
	 *            The class instance which member is this class instance.
	 */
	public DataFields(XMLComplexElement packageORWProcess) {
		super(packageORWProcess);
	}

	/**
	 * Generates a new element of the class which instances are members of
	 * collection of this class. return The generated instance of class that
	 * makes collection.
	 */
	public XMLElement generateNewElement() {
		Package p;
		if (myOwner instanceof Package) {
			p = (Package) myOwner;
		} else {
			p = ((WorkflowProcess) myOwner).getPackage();
		}
		DataField df = new DataField(this, p);
		df.setRequired(true);
		return df;
	}

	/**
	 * Returns all DataFields:
	 * <ul>
	 * <li> If DataFields instance is defined at the the process level, it gives
	 * all of it's data fields plus the ones defined at the package level.
	 * <li> If DataFields instance is defined at the the package level, it gives
	 * all of it's data fields.
	 * </ul>
	 */
	public Collection getTableElements() {
		ArrayList allDFs = new ArrayList();
		if (myOwner instanceof Package) {
			allDFs.addAll(refCollectionElements);
		} else {
			DataFields pdfs = (DataFields) (((WorkflowProcess) myOwner)
					.getPackage()).get("DataFields");
			allDFs.addAll(refCollectionElements);
			allDFs.addAll(pdfs.refCollectionElements);
		}
		if (!getPackage().isReadOnly()) {
			Iterator i = allDFs.iterator();
			while (i.hasNext()) {
				DataField df = (DataField) i.next();
				boolean isMine = isMine(df);
				df.setReadOnly(!isMine);
				// Checks for it's own data field if it is used inside some
				// transition condition expression, and if it isn't, set
				// it's ID attribute not to be read only.
				if (isMine && !isUsedWithinTransitionConditions(df)) {
					df.get("Id").setReadOnly(false);
				} else {
					df.get("Id").setReadOnly(true);
				}
			}
		}
		return allDFs;
	}

	/**
	 * <ul>
	 * <li> If DataFields instance is defined at the the process level, it gives
	 * all of it's data fields plus the ones defined at the package level
	 * (except the ones that have the ID identical to some DataField at the
	 * process level), plus all formal parameters of process .
	 * <li> If DataFields instance is defined at the the package level, it gives
	 * all of it's data fields.
	 * </ul>
	 */
	public Collection getChoosable() {
		ArrayList chos = new ArrayList();
		if (myOwner instanceof Package) {
			chos.addAll(refCollectionElements);
		} else {
			DataFields pdfs = (DataFields) (((WorkflowProcess) myOwner)
					.getPackage()).get("DataFields");
			chos.addAll(refCollectionElements);
			// adding the formal parameters of workflow process if this
			// is at workflow process level
			WorkflowProcess wp = (WorkflowProcess) myOwner;
			FormalParameters fps = (FormalParameters) wp
					.get("FormalParameters");
			Iterator i = fps.toCollection().iterator();
			while (i.hasNext()) {
				FormalParameter fp = (FormalParameter) i.next();
				if (getCollectionElement(fp.getID()) == null) {
					chos.add(fp);
				}
			}
			// adding data fields from package level
			i = pdfs.getChoosable().iterator();
			while (i.hasNext()) {
				DataField df = (DataField) i.next();
				if (getCollectionElement(df.getID()) == null) {
					chos.add(df);
				}
			}
		}

		if (!getPackage().isReadOnly()) {
			Iterator i = chos.iterator();
			while (i.hasNext()) {
				XMLCollectionElement dfOrFP = (XMLCollectionElement) i.next();
				if (dfOrFP instanceof DataField) {
					DataField df = (DataField) dfOrFP;
					boolean isMine = isMine(df);
					dfOrFP.setReadOnly(!isMine);
					// Checks for it's own data field if it is used inside some
					// transition condition expression, and if it isn't, set
					// it's ID attribute not to be read only.
					if (isMine && !isUsedWithinTransitionConditions(df)) {
						dfOrFP.get("Id").setReadOnly(false);
					} else {
						dfOrFP.get("Id").setReadOnly(true);
					}
				}
			}
		}
		return chos;
	}

	/**
	 * Returns a data field with specified ID.
	 * 
	 * @param ID
	 *            The 'Id' attribute of wanted instance of DataField element
	 *            defined within collection.
	 * @return The instance of DataField element that have the specified ID as
	 *         an 'Id' attribute if it exist among all defined data fields
	 *         within collection, if it doesn't exist, <tt>null</tt> is
	 *         returned.
	 */
	public DataField getDataField(String ID) {
		DataField toReturn = (DataField) super.getCollectionElement(ID);

		// if the data field haven't been found, and this is an instance of
		// data fields at workflow process level, search the package level
		if ((toReturn == null) && (myOwner instanceof WorkflowProcess)) {
			toReturn = ((DataFields) getPackage().get("DataFields"))
					.getDataField(ID);
		}

		return toReturn;
	}

	/**
	 * Checks if specified data field can be removed. DataField can be removed
	 * only if it is not used anywhere.
	 */
	public boolean canRemoveElement(XMLElement el) {
		boolean remove = true;
		DataField toRemove = (DataField) el;
		// if this is data field from workflow process, check if it is
		// used by any activity of subflow or generic type, or
		// if it is used in transition condition expressions
		// of any transition, but only within this workflow process
		if (myOwner instanceof WorkflowProcess) {
			Activities acts = (Activities) myOwner.get("Activities");
			remove = acts.canRemoveDataFieldOrFormalParameter(toRemove);
			// if it claims to be removed, check transitions
			if (remove == true) {
				Transitions ts = (Transitions) myOwner.get("Transitions");
				remove = ts.canRemoveDataFieldOrFormalParameter(toRemove);
			}

			// if it claims to be removed, check activity sets
			if (remove == true) {
				ActivitySets actsts = (ActivitySets) myOwner
						.get("ActivitySets");
				remove = actsts.canRemoveDataFieldOrFormalParameter(toRemove);
			}
			// else, check if it is used by any activity or transition within
			// any workflow process within package
		} else {
			WorkflowProcesses wps = (WorkflowProcesses) myOwner
					.get("WorkflowProcesses");
			Iterator it = wps.toCollection().iterator();
			while (it.hasNext()) {
				WorkflowProcess wp = (WorkflowProcess) it.next();
				Activities acts = (Activities) wp.get("Activities");
				remove = acts.canRemoveDataFieldOrFormalParameter(toRemove);
				if (!remove) {
					break;
				}
				Transitions ts = (Transitions) wp.get("Transitions");
				remove = ts.canRemoveDataFieldOrFormalParameter(toRemove);
				if (!remove) {
					break;
				}
				ActivitySets actsts = (ActivitySets) wp.get("ActivitySets");
				remove = actsts.canRemoveDataFieldOrFormalParameter(toRemove);
				if (!remove) {
					break;
				}
			}
		}
		return remove;
	}

	protected boolean isUsedWithinTransitionConditions(DataField df) {
		boolean inUse = false;
		// if this is data field from workflow process, check all transitions
		// within this workflow process
		if (myOwner instanceof WorkflowProcess) {
			Transitions ts = (Transitions) myOwner.get("Transitions");
			inUse = !ts.canRemoveDataFieldOrFormalParameter(df);
			// if it claims not to be used, check activity sets
			if (inUse == false) {
				ActivitySets actsts = (ActivitySets) myOwner
						.get("ActivitySets");
				inUse = actsts
						.isDataFieldOrFormalParameterUsedWithinTransitionConditions(df);
			}
			// else, check if it is used by any transition within
			// any workflow process within package
		} else {
			WorkflowProcesses wps = (WorkflowProcesses) myOwner
					.get("WorkflowProcesses");
			Iterator it = wps.toCollection().iterator();
			while (it.hasNext()) {
				WorkflowProcess wp = (WorkflowProcess) it.next();
				Transitions ts = (Transitions) wp.get("Transitions");
				inUse = !ts.canRemoveDataFieldOrFormalParameter(df);
				if (inUse) {
					break;
				}
				ActivitySets actsts = (ActivitySets) wp.get("ActivitySets");
				inUse = actsts
						.isDataFieldOrFormalParameterUsedWithinTransitionConditions(df);
				if (inUse) {
					break;
				}
			}
		}
		return inUse;
	}

	public boolean isMine(DataField df) {
		return refCollectionElements.contains(df);
	}

	public String getReadOnlyMessageName(XMLComplexElement el) {
		return "WarningWorkflowRelevantDataDefinedAtPackageLevelCannotBeDeletedFromProcessLevel";
	}

	public String getInUseMessageName(XMLComplexElement el) {
		return "WarningCannotDeleteWorkflowRelevantDataThatIsInUse";
	}

	/** Usefull to determine if some data field is from other package. */
	public Package getPackage() {
		Package p;
		if (myOwner instanceof Package) {
			p = (Package) myOwner;
		} else {
			p = ((WorkflowProcess) myOwner).getPackage();
		}
		return p;
	}

	public int[] getInvisibleTableFieldOrdinals() {
		int[] itfo = new int[5];
		itfo[0] = 2;
		itfo[1] = 4;
		itfo[2] = 5;
		itfo[3] = 6;
		itfo[4] = 7;
		return itfo;
	}

}
