/* Applications.java
 *
 * Authors:
 * Stefanovic Nenad  chupo@iis.ns.ac.yu
 * Bojanic Sasa      sasaboy@neobee.net
 * Puskas Vladimir   vpuskas@eunet.yu
 * Pilipovic Goran   zboniek@uns.ac.yu
 *
 */

package com.ds.bpm.bpd.xml.elements;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import com.ds.bpm.bpd.xml.XML;
import com.ds.bpm.bpd.xml.XMLCollection;
import com.ds.bpm.bpd.xml.XMLComplexElement;
import com.ds.bpm.bpd.xml.XMLElement;
import com.ds.bpm.bpd.xml.activity.Activities;

/**
 * Represents a WfMC DTD element that has the similar name. This class is a
 * collection of class <b>Application</b> instances.
 * 
 * @see XML
 */
public class Applications extends XMLCollection {
	private ArrayList externalApplications = new ArrayList();

	/**
	 * Creates a new instance of the class.
	 * 
	 * @param packageORWProcess
	 *            The class instance which member is this class instance.
	 */
	public Applications(XMLComplexElement packageORWProcess) {
		super(packageORWProcess);
	}

	/**
	 * Generates a new element of the class which instances are members of
	 * collection of this class. return The generated instance of class that
	 * makes collection.
	 */
	public XMLElement generateNewElement() {
		Package p = getPackage();
		Application ap = new Application(this, p);
		ap.setRequired(true);
		return ap;
	}

	/**
	 * It is overriden to give the caller all Applications:
	 * <ul>
	 * <li> If Applications instance is defined at the the workflow process
	 * level, it gives all of it's applications plus the ones defined at the
	 * package level (including all applications from externaly referenced
	 * packages).
	 * <li> If Applications instance is defined at the the package level, it
	 * gives all of it's applications plus the ones from externaly referenced
	 * packages).
	 * </ul>
	 */
	public Collection getTableElements() {
		ArrayList allApps = new ArrayList();
		if (myOwner instanceof Package) {
			allApps.addAll(refCollectionElements);
			allApps.addAll(externalApplications);
		} else {
			Applications apps = (Applications) (((WorkflowProcess) myOwner)
					.getPackage()).get("Applications");
			allApps.addAll(refCollectionElements);
			allApps.addAll(apps.refCollectionElements);
			allApps.addAll(apps.externalApplications);
		}

		// if this is collection within the package that is not external,
		// remove read-only attribute for all applications that belong
		// to the collection, and set to others
		if (!getPackage().isReadOnly()) {
			Iterator i = allApps.iterator();
			while (i.hasNext()) {
				Application app = (Application) i.next();
				app.setReadOnly(!isMine(app));
			}
		}
		return allApps;
	}

	/**
	 * It is overriden to give the caller all Applications except the ones with
	 * same Ids as applications on lower level:
	 * <ul>
	 * <li> If Applications instance is defined at the the workflow process
	 * level, it gives all of it's applications plus the ones defined at the
	 * package level (including all applications from externaly referenced
	 * packages).
	 * <li> If Applications instance is defined at the the package level, it
	 * gives all of it's applications plus the ones from externaly referenced
	 * packages).
	 * </ul>
	 */
	public Collection getChoosable() {
		ArrayList chos = new ArrayList();
		if (myOwner instanceof Package) {
			chos.addAll(refCollectionElements);
			Iterator i = externalApplications.iterator();
			while (i.hasNext()) {
				Application app = (Application) i.next();
				if (getCollectionElement(app.getID()) == null) {
					chos.add(app);
				}
			}
		} else {
			Applications apps = (Applications) (((WorkflowProcess) myOwner)
					.getPackage()).get("Applications");
			chos.addAll(refCollectionElements);
			Collection eaps = apps.getChoosable();
			Iterator i = eaps.iterator();
			while (i.hasNext()) {
				Application app = (Application) i.next();
				if (getCollectionElement(app.getID()) == null) {
					chos.add(app);
				}
			}
		}

		// if this is collection within the package that is not external,
		// remove read-only attribute for all applications that belong
		// to the collection, and set to others
		if (!getPackage().isReadOnly()) {
			Iterator i = chos.iterator();
			while (i.hasNext()) {
				Application app = (Application) i.next();
				app.setReadOnly(!isMine(app));
			}
		}
		return chos;
	}

	public Application getApplication(String ID) {
		Application toReturn = (Application) super.getCollectionElement(ID);

		// if the application haven't been found, and this is an instance of
		// applications at workflow process level, search the package level
		if ((toReturn == null) && (myOwner instanceof WorkflowProcess)) {
			toReturn = ((Applications) getPackage().get("Applications"))
					.getApplication(ID);
		}

		// if the application haven't been found, search external applications
		if (toReturn == null) {
			String extID;
			Iterator it = externalApplications.iterator();
			while (it.hasNext()) {
				Application atmp = (Application) it.next();
				extID = atmp.getID();
				if (extID.equals(ID)) {
					toReturn = atmp;
					break;
				}
			}
		}

		return toReturn;
	}

	/**
	 * Checks if specified application can be removed. Application can be
	 * removed only if it is not used anywhere.
	 */
	public boolean canRemoveElement(XMLElement el) {
		if (!isMine((Application) el))
			return false;
		Application toRemove = (Application) el;

		return canRemoveApplication(toRemove);
	}

	public boolean canRemoveApplication(Application toRemove) {
		boolean remove = true;
		// if this is application from workflow process, check if it is
		// used by any activity only within this workflow process
		if (myOwner instanceof WorkflowProcess) {
			Activities acts = (Activities) myOwner.get("Activities");
			remove = acts.canRemoveApplication(toRemove);
			// if it claims to be removed, check activity sets
			if (remove == true) {
				ActivitySets actsts = (ActivitySets) myOwner
						.get("ActivitySets");
				remove = actsts.canRemoveApplication(toRemove);
			}
			// else, check if it is used by any activity within
			// any workflow process within package
		} else {
			WorkflowProcesses wps = (WorkflowProcesses) myOwner
					.get("WorkflowProcesses");
			Iterator it = wps.toCollection().iterator();
			while (it.hasNext()) {
				WorkflowProcess wp = (WorkflowProcess) it.next();
				Activities acts = (Activities) wp.get("Activities");
				remove = acts.canRemoveApplication(toRemove);
				if (!remove) {
					break;
				}
				ActivitySets actsts = (ActivitySets) wp.get("ActivitySets");
				remove = actsts.canRemoveApplication(toRemove);
				if (!remove) {
					break;
				}
			}
		}
		return remove;
	}

	public boolean isMine(Application a) {
		return refCollectionElements.contains(a);
	}

	public String getReadOnlyMessageName(XMLComplexElement el) {
		if (!refCollectionElements.contains(el)) {
			if (myOwner instanceof Package) {
				return "WarningCannotDeleteExternalApplication";
			} else {
				return "WarningApplicationDefinedAtPackageLevelCannotBeDeletedFromProcessLevel";
			}
		}
		return "";
	}

	public String getInUseMessageName(XMLComplexElement el) {
		return "WarningCannotDeleteApplicationThatIsInUse";
	}

	/** Usefull to determine if some application is from other package. */
	public Package getPackage() {
		Package p;
		if (myOwner instanceof Package) {
			p = (Package) myOwner;
		} else {
			p = ((WorkflowProcess) myOwner).getPackage();
		}
		return p;
	}

	/**
	 *
	 */
	protected void insertFromExternal(Package ep) {
		Applications externalApps = (Applications) ep.get("Applications");
		// insert only the aps that owns package 'ep'
		// and are not referenced from some ext. pack. of 'ep'
		Iterator it = externalApps.refCollectionElements.iterator();
		while (it.hasNext()) {
			Application app = (Application) it.next();
			if (!externalApplications.contains(app)) {
				externalApplications.add(app);
			}
		}
	}

	/**
	 *
	 */
	protected void removeFromExternal(Package ep) {
		Applications externalApps = (Applications) ep.get("Applications");
		Iterator it = externalApps.toCollection().iterator();
		while (it.hasNext()) {
			Application app = (Application) it.next();
			// removes only applications that directly belong to specified
			// package
			externalApplications.remove(app);
		}
	}

	public int[] getInvisibleTableFieldOrdinals() {
		int[] itfo = new int[3];
		itfo[0] = 2;
		itfo[1] = 3;
		itfo[2] = 4;
		return itfo;
	}

	/**
	 * Overrides parent method to be accessible within application.
	 */
	protected void updateID(String someID) {
		super.updateID(someID);
	}

}
