package com.ds.common.util;

import com.alibaba.fastjson.util.TypeUtils;
import com.ds.common.logging.ChromeProxy;
import com.ds.common.util.java.DynamicClassLoader;
import com.ds.common.util.java.TmpJavaFileManager;
import com.ds.common.util.java.TmpJavaFileObject;
import com.ds.config.JDSConfig;
import com.ds.config.JDSUtil;

import javax.tools.*;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.*;

public class CompileJava {


    public static boolean compile(Set<File> list, Set<String> libPaths, Set<String> classPaths, ChromeProxy log) throws IOException {
        String libPath = JDSUtil.getJdsRealPath() + "lib";
        String dest = JDSUtil.getJdsRealPath() + "classes";
        if (libPaths == null) {
            libPaths = new HashSet<>();
        }
        libPaths.add(new File(libPath).getAbsolutePath());
        libPaths.add(new File(dest).getAbsolutePath());
        return compile(list, dest, libPaths, classPaths, log);
    }


    public static boolean compile(File file, Set<String> classPaths, ChromeProxy log) throws IOException {
        Set<File> list = new LinkedHashSet<>();
        list.add(file);
        return compile(list, null, classPaths, log);
    }


    public static boolean compile(File file, Set<String> libPaths, Set<String> classPaths, ChromeProxy log) throws IOException {
        Set<File> list = new LinkedHashSet<>();
        list.add(file);
        return compile(list, libPaths, classPaths, log);
    }


    private static boolean compile(Set<File> javaFiles, String des, Set<String> libPaths, Set<String> classSrcPaths, ChromeProxy log) throws IOException {
        String classpath = getLibPathFromDirs(libPaths);
        JavaCompiler javaCompiler = ToolProvider.getSystemJavaCompiler();
        if (javaCompiler == null) {
            log.printError("没有编译器,不能编译");
            throw new RuntimeException("没有编译器,不能编译");
        }
        File dir = new File(des);
        if (!dir.exists()) {
            dir.mkdirs();
        }


        List<String> optionList = new ArrayList();
        optionList.add("-d");
        optionList.add(dir.getCanonicalPath());
        optionList.add("-parameters");
        // optionList.add("-verbose");
        if (classpath != null) {
            optionList.add("-classpath");
            classpath = ".;" + JDSConfig.getAbsolutePath("") + ";" + classpath + dir.getCanonicalPath() + File.separator;
            for (String classSrcPath : classSrcPaths) {
                File file = new File(classSrcPath);
                if (file.exists()) {
                    classpath = classpath + ";" + file.getCanonicalPath() + File.separator;
                }
            }

            optionList.add(classpath);
        }
        List<TmpJavaFileObject> files = new ArrayList<>();
        for (File javaFile : javaFiles) {
            if (javaFile.exists()) {
                String fileName = javaFile.getName();
                fileName = fileName.substring(0, fileName.length() - ".java".length());
                FileInputStream inputStream = new FileInputStream(javaFile);
                String javaCode = IOUtility.toString(inputStream);
                IOUtility.shutdownStream(inputStream);
                TmpJavaFileObject fileObject = new TmpJavaFileObject(fileName, javaCode);
                files.add(fileObject);
            }

        }

        DiagnosticCollector<JavaFileObject> diagnostics = new DiagnosticCollector<JavaFileObject>();
        TmpJavaFileManager javaFileManager =
                new TmpJavaFileManager(javaCompiler.getStandardFileManager(diagnostics, null, null));


        if (!javaCompiler.getTask(null, javaFileManager, diagnostics, optionList, null, files).call()) {
            StringBuilder errorMsg = new StringBuilder();
            for (Diagnostic d : diagnostics.getDiagnostics()) {
                JavaFileObject fileObject = (JavaFileObject) d.getSource();
                String err = String.format("Compilation error: Line %d - %s%n", d.getLineNumber(),
                        d.getMessage(null));
                if (fileObject != null) {
                    String fileName = fileObject.getName();
                    String fileNameBuff = "编译 出错：" + fileName;
                    err = fileNameBuff + System.lineSeparator() + err;
                }
                errorMsg.append(err);
                System.err.print(err);
            }
            javaFileManager.close();
            javaFileManager.flush();
            throw new IOException(errorMsg.toString());
        }
        List<Class> classes = new ArrayList<>();
        Map<String, JavaFileObject> fileObjectMap = javaFileManager.getFileObjectMap();
        ClassUtility.getFileObjectMap().putAll(fileObjectMap);


        for (Map.Entry<String, JavaFileObject> entry : fileObjectMap.entrySet()) {
            String className = entry.getKey();
            TmpJavaFileObject bytesJavaFileObject = (TmpJavaFileObject) entry.getValue();
            DynamicClassLoader loader = ClassUtility.getDynamicClassLoader(className, bytesJavaFileObject, false);
            Class clazz = null;
            try {
                clazz = loader.findClass(entry.getKey());
            } catch (ClassNotFoundException e) {
                // e.printStackTrace();
                loader = ClassUtility.getDynamicClassLoader(className, bytesJavaFileObject, true);
                try {
                    clazz = loader.findClass(entry.getKey());
                } catch (ClassNotFoundException e1) {
                    e1.printStackTrace();
                }
            }
            if (clazz != null) {
                classes.add(clazz);
                ClassUtility.dynClassMap.put(className, clazz);
                TypeUtils.addMapping(className, clazz);
            }

        }


        javaFileManager.close();
        javaFileManager.flush();

        return true;
    }


    public static boolean compile(String src, Set<String> paths, ChromeProxy chrome) throws IOException {
        File srcFile = new File(src);
        Set<File> list = new LinkedHashSet<File>();
        getAllJavaFile(srcFile, list);
        return compile(list, null, paths, chrome);
    }


    public static Class dynCompile(String className, String classCode, String desc) throws IOException, ClassNotFoundException {

        JavaCompiler javaCompiler = ToolProvider.getSystemJavaCompiler();
        List<JavaFileObject> fileObjects = new ArrayList<JavaFileObject>();

        String simpleName = className;

        if (simpleName.indexOf(".") > -1) {
            simpleName = simpleName.substring(simpleName.lastIndexOf(".") + 1);
        }
        TmpJavaFileObject sourceJavaFileObject = new TmpJavaFileObject(simpleName, classCode);
        fileObjects.add(sourceJavaFileObject);

        String libPath = JDSUtil.getJdsRealPath() + "lib";
        String dest = JDSUtil.getJdsRealPath() + "classes";

        Set<String> paths = new HashSet<>();
        paths.add(libPath);
        paths.add(dest);

        String classpath = getLibPathFromDirs(paths);
        List<String> optionList = new ArrayList();
        File dir = new File(desc);
        if (!dir.exists()) {
            dir.mkdirs();
        }
        optionList.add("-d");
        optionList.add(dir.getCanonicalPath());
        optionList.add("-parameters");
        //  optionList.add("-verbose");
        optionList.add("-classpath");
        classpath = ".;" + JDSConfig.getAbsolutePath("") + ";" + classpath + dir.getCanonicalPath() + "/;" + desc;
        optionList.add(classpath);
        DiagnosticCollector<JavaFileObject> collector = new DiagnosticCollector<>();
        JavaFileManager javaFileManager =
                new TmpJavaFileManager(javaCompiler.getStandardFileManager(collector, null, null));

        if (!javaCompiler.getTask(null, javaFileManager, collector, optionList, null, fileObjects).call()) {
            StringBuilder errorMsg = new StringBuilder();
            for (Diagnostic d : collector.getDiagnostics()) {
                String err = String.format("Compilation error: Line %d - %s%n", d.getLineNumber(),
                        d.getMessage(null));
                errorMsg.append(err);
                System.err.print(err);
            }
            throw new IOException(errorMsg.toString());
        }

        TmpJavaFileObject bytesJavaFileObject = (TmpJavaFileObject) ((TmpJavaFileManager) javaFileManager).getFileObjectMap().get(className);
        if (bytesJavaFileObject != null) {
            sourceJavaFileObject = bytesJavaFileObject;
        }

        ClassUtility.getFileObjectMap().put(className, sourceJavaFileObject);
        DynamicClassLoader loader = new DynamicClassLoader(bytesJavaFileObject,className);//ClassUtility.getDynamicClassLoader(className, sourceJavaFileObject, false);
        Class clazz = loader.findClass(className);
        ClassUtility.getDynClassMap().put(className, clazz);
        TypeUtils.addMapping(className, clazz);
        return clazz;

    }


    public static boolean compile(String src, Set<String> paths, Set<String> classPaths, ChromeProxy chrome) throws IOException {
        File srcFile = new File(src);
        Set<File> list = new LinkedHashSet<>();
        getAllJavaFile(srcFile, list);
        return compile(list, paths, classPaths, chrome);
    }

    public static Set<File> getAllJavaFile(File srcFile) {
        return getAllJavaFile(srcFile, null);
    }

    private static Set<File> getAllJavaFile(File srcFile, Set<File> list) {
        if (list == null) {
            list = new LinkedHashSet<>();
        }
        if (srcFile.isDirectory()) {
            File[] fs = srcFile.listFiles();
            for (int i = 0; i < fs.length; i++) {
                getAllJavaFile(fs[i], list);
            }
        } else {
            if (srcFile.getName().endsWith(".java")) {
                list.add(srcFile);
            }

        }
        return list;
    }

    private static void getLibPathFromDir(File dir, StringBuilder sb) throws IOException {
        if (dir.isDirectory()) {
            File[] fs = dir.listFiles();
            for (int i = 0; i < fs.length; i++) {
                getLibPathFromDir(fs[i], sb);
            }
        } else {
            if (dir.getName().endsWith(".jar")) {
                sb.append(dir.getCanonicalPath());
                sb.append(";");
            }
        }
    }


    private static String getLibPathFromDirs(Set<String> dir) throws IOException {
        if (dir == null || dir.isEmpty()) {
            return null;
        }
        StringBuilder sb = new StringBuilder();
        for (String path : dir) {
            if (path != null) {
                File d = new File(path);
                if (d.isDirectory()) {
                    getLibPathFromDir(d, sb);
                }
            }
        }
        return sb.toString();
    }


    public static void main(String[] args) throws Exception {


    }

}
