/**
 * $RCSfile: ClassUtility.java,v $
 * $Revision: 1.1 $
 * $Date: 2014/07/08 00:25:55 $
 * <p>
 * Copyright (C) 2003 spk, Inc. All rights reserved.
 * <p>
 * This software is the proprietary information of spk, Inc.
 * Use is subject to license terms.
 */
package com.ds.common.util;

import com.ds.common.util.java.DynamicClassLoader;
import com.ds.common.util.java.TmpJavaFileObject;
import com.ds.esb.config.manager.EsbBeanFactory;

import javax.tools.JavaFileObject;
import java.io.InputStream;
import java.net.URL;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

/**
 * <p>Title: 常用代码打包</p>
 * <p>Description:
 * A utility class to assist with loading classes by name. Many application servers use
 * custom classloaders, which will break uses of:
 * <pre>
 *    Class.forName(className);
 * </pre>
 * <p>
 * This utility attempts to load the class using a number of different mechanisms to work
 * around this problem.
 * </p>
 * <p>Copyright: Copyright (c) 2006</p>
 * <p>Company: www.justdos.net</p>
 *
 * @author wenzhang li
 * @version 1.0
 */
public final class ClassUtility {

    private static ClassUtility instance = new ClassUtility();

    public static Map<String, Class<?>> dynClassMap = new ConcurrentHashMap<>();

    public static Map<String, Class<?>> fileClassMap = new ConcurrentHashMap<>();

    private static Map<String, JavaFileObject> fileObjectMap = new ConcurrentHashMap<>();

    public static Map<String, DynamicClassLoader> classLoaderMap = new ConcurrentHashMap<>();


    /**
     * Private constructor to prevent instantiation.
     */

    private ClassUtility() {
    }


    public synchronized static DynamicClassLoader getDynamicClassLoader(String className, TmpJavaFileObject fileObject, Boolean dyn) {
        DynamicClassLoader loader = classLoaderMap.get(className);
        if (loader == null || dyn) {
            loader = new DynamicClassLoader(fileObject);
            classLoaderMap.put(className, loader);
        } else {
            loader.setTmpJavaFileObject(fileObject);
        }
        return loader;
    }

    /**
     * Loads the class with the specified name.
     *
     * @param className the name of the class
     * @return the resulting <code>Class</code> object
     * @throws ClassNotFoundException if the class was not found
     */
    public synchronized static Class loadClass(String className) throws ClassNotFoundException {

        if (className.equals("int")) {
            return int.class;
        } else if (className.equals("long")) {
            return long.class;
        } else if (className.equals("char")) {
            return Character.class;
        } else if (className.equals("short")) {
            return short.class;
        } else if (className.equals("byte")) {
            return byte.class;
        } else if (className.equals("float")) {
            return float.class;
        } else if (className.equals("double")) {
            return double.class;
        } else if (className.equals("boolean")) {
            return boolean.class;
        }

        Class theClass = fileClassMap.get(className);

        if (theClass == null) {
            theClass = dynClassMap.get(className);
            TmpJavaFileObject fileObject = (TmpJavaFileObject) fileObjectMap.get(className);

            if (theClass == null && fileObject != null) {
                DynamicClassLoader loader = getDynamicClassLoader(className, fileObject, true);
                try {
                    theClass = loader.findClass(className);
                    dynClassMap.put(className, theClass);
                } catch (ClassNotFoundException e) {
                    e.printStackTrace();
                }
            }
        }

        if (theClass == null) {
            theClass = EsbBeanFactory.findClass(className);
        }


        if (theClass == null) {
            try {
                theClass = Class.forName(className);
            } catch (ClassNotFoundException e1) {
                try {
                    theClass = Thread.currentThread().getContextClassLoader().loadClass(className);
                } catch (ClassNotFoundException e2) {
                    theClass = instance.getClass().getClassLoader().loadClass(className);
                }
            }
        }

        return theClass;
    }


    public static Class loadClassByFile(String classPath, String className) throws ClassNotFoundException {
        FileClassLoader classLoader = new FileClassLoader(classPath);
        Class theClass = classLoader.findClass(className);
        fileClassMap.put(className, theClass);
        return theClass;
    }

    /**
     * Loads the resource with the specified name.
     *
     * @param name the name of the resource
     * @return the resulting <code>java.io.InputStream</code> object
     */
    public static InputStream loadResource(String name) {
        InputStream in = instance.getClass().getResourceAsStream(name);
        if (in == null) {
            in = Thread.currentThread().getContextClassLoader().getResourceAsStream(name);
            if (in == null) {
                in = instance.getClass().getClassLoader().getResourceAsStream(name);
            }
        }
        return in;
    }

    /**
     * Loads the resource with the specified name.
     *
     * @param name the name of the resource
     * @return the result <code>java.net.URL</code> object
     */
    public static URL loadResourceURL(String name) {
        URL url = instance.getClass().getResource(name);

        if (url == null) {
            url = Thread.currentThread().getContextClassLoader().getResource(name);
            if (url == null) {
                url = instance.getClass().getClassLoader().getResource(name);
            }
        }
        return url;
    }

    public static void clear(String className) {
        dynClassMap.remove(className);
        fileObjectMap.remove(className);
        classLoaderMap.remove(className);
    }

    public static Set<Package> getAllDynPacks() {
        Set<Package> packages = new HashSet<>();
        Set<String> keySet = dynClassMap.keySet();
        for (String key : keySet) {
            Class clazz = dynClassMap.get(key);
            packages.add(clazz.getPackage());
        }
        return packages;

    }


    public static Map<String, Class<?>> getDynClassMap() {
        return dynClassMap;
    }

    public static void setDynClassMap(Map<String, Class<?>> dynClassMap) {
        ClassUtility.dynClassMap = dynClassMap;
    }

    public static Map<String, JavaFileObject> getFileObjectMap() {
        return fileObjectMap;
    }

    public static void setFileObjectMap(Map<String, JavaFileObject> fileObjectMap) {
        ClassUtility.fileObjectMap = fileObjectMap;
    }
}
