package cn.ps1.aolai.utils;

import java.lang.reflect.Array;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Map;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 调整输出数据的格式
 * 
 * @author Aolai
 * @since 1.8 $Date: 2024.7.20
 *
 */
public class FmtUtil {

	private static Logger log = LoggerFactory.getLogger(FmtUtil.class);

	// 数值为空时
	private static final String NIL = "0.00";
	// 人民币显示格式
	private static final String RMB = "#,##0.00";
	// 模糊替换符号
//	private static final String[] STAR = { "", "*", "**", "***", "****" };
	// 匹配一个或多个由汉字、空格、数字或下划线组成的字符串
	private static final String WD = "[\\w+ \u4e00-\u9fa5]+";
	// 数字或小数点数字
	private static final String DD = "(-?\\d+)(\\.\\d+)?";

	/**
	 * 匹配一个或多个由汉字、空格、数字或下划线组成的字符串<br>
	 */
	public static boolean isValidWord(String val) {
		return Pattern.matches(WD, val);
	}

	/**
	 * 判断是否为浮点数：包含整数、小数、负数
	 */
	public static boolean isNumeric(Object str) {
		return isMatch(str, DD);
	}

	/**
	 * 字符串中是否有完全匹配：匹配字符或数字
	 */
	public static boolean isInteger(Object str) {
		return ifInt(str);
	}

	/**
	 * 把数串逐级累计（科目层级转换）<br>
	 * 如字符串(4332)分割为整数数组([4,3,3,2])，转换为[4,7,10,12]
	 * 
	 * @param numStr 数字的字符串
	 * @return Array 累计数组
	 */
	public static int[] num2Arr(String numStr) {
		int[] num = str2num(numStr);
		for (int i = 1; i < num.length; i++) {
			num[i] += num[i - 1];
		}
		return num;
	}

	/**
	 * 字符串(4332)分割为整数数组([4,3,3,2])
	 */
	public static int[] str2num(String numStr) {
		if (!isInteger(numStr))
			return new int[0];
		int[] num = new int[numStr.length()];
		for (int i = 0; i < numStr.length(); i++) {
			num[i] = Character.getNumericValue(numStr.charAt(i));
		}
		return num;
	}

	/**
	 * 判断字符串是否为“0”、“0.0+”
	 */
	public static boolean isZero(Object obj) {
		return Pattern.matches("(-?0)(\\.0+)?", String.valueOf(obj));
	}

	/**
	 * 对字符串进行模糊化处理：PN-电话号码，ID-身份证，EM-邮件，BA-银行账户<br>
	 * 默认对全局所有人都模糊化，仅放开对个别人不模糊化<br>
	 * 需要配置 META_STYLE = {"blue":"1"}
	 */
	public static String blur(String str, String diff) {
		if ("PN".equals(diff)) { // 电话号码phoneNumber
			str = blur(str, 4, 3);
		} else if ("ID".equals(diff)) {
			str = blur(str, 4, 4); // 身份证号
		} else if ("BA".equals(diff)) {
			str = blur(str, 4, 3); // 银行账号
		} else if ("EM".equals(diff)) {
			String[] arr = str.split("@");
			int len = arr[0].length();
			if (arr.length == 2 && len > 0) {
				arr[0] = len > 3 ? blur(arr[0], len - 4, 3) : blur(arr[0], 0, len - 1);
				str = String.join("@", arr);
			}
		}
		return str;
	}

	/**
	 * 字符串模糊化
	 * 
	 * @param str 字符串
	 * @param pos 居右的位置，补充*的位数
	 * @param num 补充*的位数
	 */
	public static String blur(String str, int pos, int num) {
		if ((pos = str.length() - pos - num) < 0)
			return str; // 长度不规范
		// { "", "*", "**", "***", "****" }
		String star = leftPad("", num, '*');
		return str.substring(0, pos) + star + str.substring(pos + num);
	}

	/**
	 * 时间（日期）格式转变
	 */
	public static String timeParse(String str, String pat0, String pat1) {
		try {
			Calendar cal = Calendar.getInstance();
			cal.setTime(toDate(str, pat0));
			return timeParse(cal, pat1);
		} catch (Exception e) {
			log.error("timeParse...{}", str);
			return str;
		}
	}

	/**
	 * 根据日期格式，获取当前日期时间：pattern格式
	 */
	public static String timeParse(String pattern) {
		return timeParse(Calendar.getInstance(), pattern);
	}

	/**
	 * 根据日期格式，获取当前日期时间：pattern格式
	 */
	public static String timeParse(Calendar cal, String pattern) {
		pattern = pattern == null ? Const.DTF : pattern;
		return new SimpleDateFormat(pattern).format(cal.getTime());
	}

	/**
	 * 缩进值为指定位数的左边距
	 */
	public static String indent(Object str, int num, char chr) {
		StringBuilder pad = new StringBuilder();
		while (num-- > 0)
			pad.append(chr);
		return pad.toString() + (str == null ? "" : str);
	}

	/**
	 * 缩进num个空格
	 */
	public static String indent(Object str, int num) {
		return indent(str, num, ' ');
	}

	/**
	 * 字符串左侧填充符号到指定长度
	 */
	public static String leftPad(Object str, int size, char chr) {
		int leftNum = str == null ? size : size - String.valueOf(str).length();
		return indent(str, leftNum, chr);
	}

	/**
	 * 字符串左侧填充符号到指定宽度
	 */
	public static String leftPad(Object str, int size) {
		return leftPad(str, size, '0');
	}

	/**
	 * 字符串左侧填充符号到指定宽度
	 */
	public static String toBase62(long num, int len) {
		return leftPad(Digest.enBase62(num), len);
	}

	/**
	 * 把整数的62进制的序列号自动加1
	 */
	public static String nextSeq(String seqNo) {
		return leftPad(Digest.nextSeq(seqNo), seqNo.length());
	}

	/**
	 * 数字以千分位展示，保留N位小数点
	 */
	public static String commaFmt(String str, int digit) {
		return commaFmt(toNumeric(str), digit);
	}

	/**
	 * 数字以千分位展示，保留N位小数点
	 */
	public static String commaFmt(String str, int digit, String def) {
		return commaFmt(toNumeric(str), digit, def);
	}

	/**
	 * 数字以千分位展示，保留N位小数点
	 * 
	 * @param num 精度数据
	 * @param digit 小数点精度的位数
	 */
	public static String commaFmt(double num, int digit) {
		return commaFmt(num, digit, "-");
	}

	/**
	 * 数字以千分位展示，保留N位小数点
	 * 
	 * @param num 精度数据
	 * @param digit 小数点精度的位数
	 * @param def 默认0的返回字符串
	 */
	public static String commaFmt(double num, int digit, String def) {
		String str = parse(num, "#,##0", digit);
		return isZero(str) ? def : str;
	}

	/**
	 * 把金额、数量、单价等数值转换成字指定格式的符串
	 */
	public static String parse(String str, int digit) {
		return parse(toNumeric(str), digit);
	}

	/**
	 * 把金额、数量、单价等数值转换成字指定格式的符串
	 */
	public static String parse(double num, int digit) {
		return parse(num, "#0", digit);
	}

	/**
	 * 把金额、数量、单价等数值转换成字指定格式的符串
	 * 
	 * @param num 数值
	 * @param pattern 格式
	 * @param digit 小数点精度的位数
	 */
	private static String parse(double num, String pattern, int digit) {
		pattern = digit > 0 ? pattern + Const.DOT : pattern;
		StringBuilder pad = new StringBuilder();
		while ((digit--) > 0)
			pad.append(Const.S_0);
		return parse(num, pattern + pad.toString());
	}

	/**
	 * 把金额、数量、单价等数值转换成字指定格式的符串
	 */
	private static String parse(double num, String pattern) {
		return new DecimalFormat(pattern).format(num);
	}

	/**
	 * 把符号“0”、“0.0+”转为空字符串
	 */
	public static String ifZero(String str, String def) {
		return (str == null || isZero(str)) ? def : str;
	}

	/**
	 * 字符串转换为Date类型
	 */
	public static Date toDate(String dateStr, String pat) {
		SimpleDateFormat sdf = new SimpleDateFormat(pat);
		try {
			sdf.setLenient(false); // 严格解析日期格式
			return sdf.parse(dateStr);
		} catch (Exception e) {
			return null;
		}
	}

	/**
	 * 字符串转换为Double类型
	 */
	public static double toNumeric(String str) {
		try {
			return isNumeric(str) ? Double.parseDouble(str) : 0.0;
		} catch (Exception e) {
			log.error("toNumeric...{}", str);
			return 0.0;
		}
	}

	/**
	 * 分变成带千分位格式的金额（元），四舍五入
	 */
	public static String toYuan(String str, String def) {
		str = parse(toNumeric(str) / 100, RMB);
		return def != null && NIL.equals(str) ? def : str;
	}

	/**
	 * 人民币汉字大写，传入数值可大于1万亿元
	 */
	public static String toRmb(String money) {
		return isInteger(money) ? toRmb(Long.parseLong(money)) : "";
	}

	/**
	 * 人民币汉字大写，传入数值可大于1万亿元
	 */
	public static String toRmb(long num) {
		String[] digit = { "零", "壹", "贰", "叁", "肆", "伍", "陆", "柒", "捌", "玖" };
		String[][] units = { { "分", "角", "元", "万", "亿", "万" }, { "", "拾", "佰", "仟" } };
		String head = num < 0 ? "负" : "";
		// String s = "";
		num = Math.abs(num);
		StringBuilder rmb = new StringBuilder();
		for (int i = 0; i < 2 && num > 0; i++) {
			// s = digit[(int) (num % 10)] + units[0][i] + s;
			rmb.insert(0, digit[(int) (num % 10)] + units[0][i]);
			num = num / 10;
		}
		for (int i = 2; i < units[0].length && num > 0; i++) {
			// String p = "";
			StringBuilder sb = new StringBuilder();
			for (int j = 0; j < units[1].length && num > 0; j++) {
				// p = digit[(int) (num % 10)] + units[1][j] + p;
				sb.insert(0, digit[(int) (num % 10)] + units[1][j]);
				num = num / 10;
			}
			String p = sb.toString();
			p = p.replaceAll("(零.)*零$", "").replaceAll("^$", "零");
			// s = p + units[0][i] + s;
			rmb.insert(0, p + units[0][i]);
		}
		String s = rmb.toString();
		s = s.replaceAll("(零.)*零元", "元").replaceAll("(零.)+", "零");
		return head + s.replaceAll("零$", "整");
	}

	/**
	 * 字符对称引用号“str”
	 */
	public static String quote(Object str, String ch) {
		return ch + str + ch;
	}

	/**
	 * 占位符处理：placeHolder
	 * 
	 * @param opr 运算符号
	 */
	public static String pHolder(Object key, String opr) {
		return "{" + key + "}" + quote(opr, " ");
	}

	/**
	 * 字符串中是否有完全匹配：匹配字符或数字
	 * 
	 * @return boolean 成功与否
	 */
	public static boolean isMatch(Object str, String regex) {
		return ifStr(str) && Pattern.matches(regex, (String) str);
	}

	/**
	 * 字符串中是否包含指定的字符串，如："\\d+"
	 */
	public static boolean findIn(String str, String regex) {
		if (str == null)
			throw new FailedException();
		return Pattern.compile(regex).matcher(str).find();
	}

	/**
	 * 获取配置参数的值
	 */
	static int getInt(String val, int def) {
		return ifInt(val) ? Integer.parseInt(val) : def;
	}

	/**
	 * 字符串中是否有完全匹配：匹配字符或数字："^-?\\d+$"
	 */
	private static boolean ifInt(Object val) {
		return ifStr(val) && Pattern.matches("-?\\d+", (String) val);
	}

	/**
	 * 对象是否为字符串
	 */
	private static boolean ifStr(Object obj) {
		return obj instanceof CharSequence;
	}

	/**
	 * 判断字符串是否为空
	 */
	public static boolean isEmpty(Object obj) {
		if (obj == null)
			return true;
		if (obj instanceof CharSequence) // 字符串
			return ((CharSequence) obj).length() == 0;
		if (obj instanceof Map)
			return ((Map<?, ?>) obj).isEmpty();
		if (obj instanceof Collection) // List列表、Set等
			return ((Collection<?>) obj).isEmpty();
		if (obj.getClass().isArray()) // 数组
			return Array.getLength(obj) == 0;
//		if (obj instanceof Optional) // java8引入为了解决null值判断问题
//			return !((Optional<?>) obj).isPresent();
		return false;
	}

	private FmtUtil() {
		throw new IllegalStateException("Utility class");
	}

}
