package cn.ps1.aolai.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringJoiner;

import cn.ps1.aolai.entity.User;
import cn.ps1.aolai.utils.ConfUtil;
import cn.ps1.aolai.utils.Const;
import cn.ps1.aolai.utils.Digest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import redis.clients.jedis.Jedis;
import redis.clients.jedis.ShardedJedis;
import redis.clients.jedis.ShardedJedisPool;

/**
 * 针对Redis的增删改查及Token相关基础业务操作
 *
 * @author Aolai
 * @version 1.0
 * @since 1.7 $Date: 2017.6.17
 */

@Service
public class RedisService {
	
	private static Logger log = LoggerFactory.getLogger(RedisService.class);

	@Autowired
	private ShardedJedisPool jedisPool;

	/** 控制同一账户多人同时登录，未配置默认为仅限单用户登录使用 */
	private boolean singleUser() {
		return !ConfUtil.isMultLogin();
	}

	/**
	 * 新建用户Token，再写入Cookies并返回前端
	 */
	public Map<String, String> newToken(String userId, String spec) {
		Map<String, String> token = new HashMap<>();
		token.put(ConfUtil.CERTID, spec); // 32位证书
		token.put(ConfUtil.TOKEN, Digest.uuid8()); // 8位时间戳字符串
		token.put(User.ID, userId); // 未编码的userId
		// 缓存用户Token，这里的userId为加密前的数据
		log.debug("newToken...");
		setToken(token);
		// 编码处理
		// 写入Cookies为编码处理后的userId信息
		token.put(User.ID, Digest.sm4Encrypt(userId, spec));
		return token;
	}

	/**
	 * 缓存用户Token，控制同一账户多次登录，默认不配置时仅限单一用户登录使用。
	 */
	public void setToken(Map<String, String> token) {
		/**
		 * 多次登录仅通过uuid获取token即可，单一登录须校验userId
		 * 数据例如：uuid=eA8bTjyw、t.eA8bTjyw={token}
		 */
		String uuid = token.get(ConfUtil.TOKEN);
		hmset(Const.RDS_TOKEN + uuid, token, ConfUtil.cacheTime());
		if (singleUser()) {
			// 例如：userId=18666677888
			String userId = token.get(User.ID);
			// 仅单用户时配置了：k.18666677888=eA8bTjyw
			set(Const.RDS_CERT + userId, uuid, ConfUtil.cacheTime());
		}
	}

    /**
     * 验证token是否有效：单用户无法对应多个token，多个token可缓存同一个userid
     * <p>
     * {userId=BMr2LhF2LhF2LhF=,token=QKDMZJXWSFV, ...}
     */
	public Map<String, String> verifyToken(String uuid) {
		/**
		 * 多次登录仅通过uuid获取token即可，单一登录须校验userId
		 */
		String tid = Const.RDS_TOKEN + uuid;
		Map<String, String> token = hmget(tid);
		// 正常时可以通过uuid获取到{token}信息
//		if (token != null && expire(tid)) {
		if (!token.isEmpty() && expire(tid)) {
			if (singleUser()) {
				// 仅单用户时配置了：k.18666677888=eA8bTjyw
				String uid = Const.RDS_CERT + token.get(User.ID);
				if (uuid.equals(get(uid)) && expire(uid))
					return token;
			} else {
				// 多用户跳过校验：k.18666677888
				return token;
			}
			log.debug("verifyToken...^rds_uuid={}", tid);
		}
		return null;
	}

    /**
     * 验证token是否有效：单用户无法对应多个token，多个token可缓存同一个userid
     * <p>
     * {userId=BMr2LhF2LhF2LhF=,token=QKDMZJXWSFV, ...}
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
     */
	@Deprecated
	public Map<String, String> verifyToken(Map<String, String> cookies) {
		return verifyToken(cookies.get(ConfUtil.TOKEN));
	}

	/**
	 * 验证手机动态验证码是否有效
	 * 
	 * @param mobile 手机号码
	 * @param vCode 动态验证码
	 */
	public boolean checkVCode(Object mobile, Object vCode) {
		return exists(Const.RDS_CODE + mobile + vCode);
	}

	/**
	 * 缓存手机动态验证码（默认缓存十分钟）
	 * [时效从配置中取，如果没有配置，则取默认值]
	 * 
	 * @param mobile 手机号码
	 * @param vCode 动态验证码
	 */
	public void setVCode(Object mobile, Object vCode) {
		String rdsVcode = Const.RDS_CODE + mobile + vCode;
		// 不推荐在这里处理动态码的有效时间
		set(rdsVcode, Const.S_1, ConfUtil.vcodeDue());
	}

	/**
     * 退出登录、清除token信息：{userId,token,certId}
     */
	public void clearToken(Map<String, String> token) {
		// 前面已经验证了token有效性
		log.debug("clearToken...{}", token);
		// 只需删除这个基于uuid的token信息即可
		// 基于userId的信息再登录后自动覆盖
		del(Const.RDS_TOKEN + token.get(ConfUtil.TOKEN));
	}

	/**
	 * 缓存当前的路由页面的用户访问权限，主键为：应用、公司、岗位
	 * 
	 * @param key
	 * @param uriMap 这里的值都是'1'
	 */
	void setActionRole(String key, Map<String, String> uriMap) {
		hmset(Const.RDS_ROLE + key, uriMap, ConfUtil.cacheTime());
	}

	/**
	 * 获取当前路由页面的用户访问权限，主键为：应用、公司、岗位
	 *
	 * @param key 主键
	 * @param actUri 页面路由
	 * @return String 当前路由的访问权限，有访问权限返回'1'
	 */
	String getActionRole(String key, String actUri) {
		String rk = Const.RDS_ROLE + key;
		// 注意这里获取actUri值的可能状态：null\值‘0’\值‘1’
		return expire(rk) ? hget(rk, actUri) : null;
	}

    /**
     * 缓存当前用户信息，以备在访问接口时鉴权时使用
     *
     * @param user 用户信息
     */
    public void setUserInfo(Map<String, String> user) {
		String uid = Const.RDS_USER + user.get(User.ID);
		hmset(uid, user, ConfUtil.cacheTime());
    }

    /**
     * 获取当前登录用户信息，在访问接口时鉴权时使用
     *
     * @param userId 用户编号
     * @return Map 当前用户信息
     */
	public Map<String, String> getUserInfo(Object userId) {
		String rdsKey = Const.RDS_USER + userId;
		return expire(rdsKey) ? hmget(rdsKey) : new HashMap<>();
	}

	/**
	 * 使用分布式锁（简易锁），写入成功是“OK”，写入失败返回空
	 */
	public boolean setWithLock(String key, String uuid, int sec) {
		try (ShardedJedis jedis = jedisPool.getResource()) {
			// nx: not exists，只有key不存在时把值set到redis
			// xx: is exists，只有key存在时
			// ex: seconds，px: milliseconds
			return "OK".equals(jedis.set(key, uuid, "NX", "EX", sec));
		} catch (Exception e) {
			log.error(e.getMessage());
			return false;
		}
	}

    /**
     * 设置单个键值，永久有效（设置过期时间为0秒）
     *
     * @param key 主键
     * @param val 值
     */
    public boolean set(String key, String val) {
    	// 成功则返回“OK”字符串
        return set(key, val, 0);
    }

    /**
     * 设置单个键值，并设置有效时间
     *
     * @param key 主键
     * @param val 值
     * @param sec 有效期（秒）
     */
	public boolean set(String key, String val, int sec) {
		log.debug("set...{}={}", key, val);
		try (ShardedJedis jedis = jedisPool.getResource()) {
			// 设置过期时间sec，这里返回值：“OK”=成功
			jedis.setex(key, sec, val);
			return true;
		} catch (Exception e) {
			log.error(e.getMessage());
			return false;
		}
	}

	/**
     * 设置map对象，永久有效
     *
     * @param key 主键
     * @param map 对象
     */
    public boolean hmset(String key, Map<String, String> map) {
        return hmset(key, map, 0);
    }

    /**
     * 设置map对象
     *
     * @param key 主键
     * @param map 对象
     * @param sec 有效期（秒）
     * @return String
     */
    public boolean hmset(String key, Map<String, String> map, int sec) {
		log.debug("hmset...{}={}", key, map);
		try (ShardedJedis jedis = jedisPool.getResource()) {
			// 这里返回值=“OK”，成功
			jedis.hmset(key, map);
			if (sec > 0)
				expire(key, sec);
			return true;
		} catch (Exception e) {
			log.error(e.getMessage());
			return false;
		}
    }

    /**
     * 获取map对象
     */
	public Map<String, String> hmget(String key) {
		try (ShardedJedis jedis = jedisPool.getResource()) {
			return jedis.hgetAll(key);
		} catch (Exception e) {
			log.error(e.getMessage());
		}
		log.debug("hmget...{} is Empty.", key);
		return new HashMap<>();
	}

    /**
     * 获取map对象单个域的键值
     */
    public String hget(String key, String field) {
		log.debug("hset...{}.{}", key, field);
        try (ShardedJedis jedis = jedisPool.getResource()) {
            return jedis.hget(key, field);
        } catch (Exception e) {
			log.error(e.getMessage());
            return null;
        }
    }

    /**
     * 设置map对象单个域的键值，永久有效
     */
    public boolean hset(String key, String field, String val) {
        return hset(key, field, val, 0);
    }

    /**
     * 设置map对象单个域的键值
     */
    public boolean hset(String key, String field, String val, int sec) {
		log.debug("hset...{}.{}", key, field);
		try (ShardedJedis jedis = jedisPool.getResource()) {
			// 这里新建时返回1，覆盖返回0
			jedis.hset(key, field, val);
			if (sec > 0)
				expire(key, sec);
			return true;
		} catch (Exception e) {
			log.error(e.getMessage());
			return false;
		}
    }

    /**
     * 获取单个键值
     */
    public String get(String key) {
		try (ShardedJedis jedis = jedisPool.getResource()) {
			return jedis.get(key);
		} catch (Exception e) {
			log.error("get...{} is null.", key);
			return null;
		}
    }

	/**
	 * 根据关键字模糊匹配的结果
	 */
	public List<String> getKeys(String pattern) {
		List<String> keyList = new ArrayList<>();
		try (ShardedJedis jedis = jedisPool.getResource()) {
			Collection<Jedis> allShards = jedis.getAllShards();
			for (Jedis j : allShards) {
				// 模糊匹配如：keys + "*"
				keyList.addAll(j.keys(pattern));
			}
		} catch (Exception e) {
            log.error("getKeys...{}", pattern);
		}
		return keyList;
	}

	/**
     * 根据key删除键值对数据
     */
	public boolean del(String key) {
		try (ShardedJedis jedis = jedisPool.getResource()) {
			return jedis.del(key) > 0;
		} catch (Exception e) {
            log.error("del...{}", key);
			return false;
		}
	}

	/**
	 * 批量删除按关键字模糊匹配的结果
	 */
	public boolean delKeys(String pattern) {
		Long count = 0L;
		try (ShardedJedis jedis = jedisPool.getResource()) {
			Collection<Jedis> allShards = jedis.getAllShards();
			for (Jedis jds : allShards) {
				Set<String> set = jds.keys(pattern);
				for (String key : set)
					count += jedis.del(key);
			}
		} catch (Exception e) {
            log.error("delKeys...{}", pattern);
		}
		return count > 0;
	}

	/**
	 * 批量删除按关键字模糊匹配的结果
	 */
	public boolean delKeys(List<String> keyList) {
		Long count = 0L;
		try (ShardedJedis jedis = jedisPool.getResource()) {
			for (String key : keyList) {
				count += jedis.del(key);
			}
		} catch (Exception e) {
            log.error("delKeys...{}", e.getMessage());
		}
		return count > 0;
	}

	/**
     * 根据key删除键值对数据
     */
    public boolean hdel(String key, String[] fields) {
        try (ShardedJedis jedis = jedisPool.getResource()) {
        	return jedis.hdel(key, fields) > 0;
        } catch (Exception e) {
            log.error("hdel...{}", e.getMessage());
            return false;
        }
    }

    /**
     * 判断key是否存在
     */
    public boolean exists(String key) {
        try (ShardedJedis jedis = jedisPool.getResource()) {
            return jedis.exists(key);
        } catch (Exception e) {
            log.error("exists...{}", key);
            return false;
        }
    }

    /**
     * 设置key的默认N小时有效期，或延续有效期时间
     */
    public boolean expire(String key) {
        return expire(key, ConfUtil.cacheTime());
    }

	/**
	 * 设置key的默认有效期，或延续有效期时间
	 */
	public boolean expire(String key, int sec) {
		try (ShardedJedis jedis = jedisPool.getResource()) {
			// 返回1-成功，0-失败
			return jedis.expire(key, sec) > 0;
		} catch (Exception e) {
			log.error("expire...{}", key);
			return false;
		}
	}

	/**
     * 阻止指定ip访问，指定时间之内（1分钟、30分钟）限制IP访问次数（10次、50次）
     *
     * @param ip 客户端IP地址
     */
	public boolean isBlocked(String ip) {
		int[] times = { Const.ONE_MM, 3 * Const.TEN_MM, 10, 50 };
		return isBlocked(ip, times);
	}

    /**
     * 对短时间内尝试登陆的ip次数进行限制
     *
     * @param ip  客户端IP地址
     * @param times 一定时间内及次数限制
     * @return boolean
     */
    public boolean isBlocked(String ip, int[] times) {
        String rk = Const.RDS_DENY + ip;
        Map<String, String> addr = hmget(rk);
        // 记录当前访问时间戳，如：“1693814052623”
		StringJoiner s = new StringJoiner(",");
		long nowTime = new Date().getTime();
		int loop = times.length / 2; // 对标两次
		if (addr.isEmpty()) {
			// 初次访问，未锁
			addr = new HashMap<>();
			addr.put(Const.LOCK, "0");
		} else if ("1".equals(addr.get(Const.LOCK))) { // 已锁定
			return true;
		} else {
        	// 缓存了一组时间戳，时间戳的数量，即：访问次数
            String[] arr = addr.get(Const.INFO).split(","); // 时间戳计数
			int i = 0;
			while (i < arr.length) {
				// 时间差 = 当前时间 - 遍历时间戳：1693814052623,1693814052789,...
				long diff = nowTime - Long.parseLong(arr[i]);
				// 对标两次参数loop=2，如：1分钟10次，30分钟50次
                for (int n = 0; n < loop; n++) {
                	// 第一次对标60秒，即：指定时间1分钟内小于10次
                	// 时间小于60秒内，数量大于限定的10次（加当前1次）
					if (diff < times[n] && arr.length + 1 - i > times[n + loop]) {
						// 第二次对标180秒，即：指定时间30分钟内小于50次
						// 时间小于180秒内，数量大于限定的50次
						addr.put(Const.LOCK, "1"); // 锁定30分钟
						hmset(rk, addr, times[loop - 1]);
						return true;
					}
                }
				// 对标两次无超出30分钟的时间戳，则跳出循环
				if (diff < times[1]) {
					// 保留有效的时间戳
					while (i < arr.length) {
						s.add(arr[i++]);// 1693814052623,1693814052789
					}
					break;
				}
				i++;
            }
        }
		// 初次访问缓存时间戳，如：“1693814052623”
        // 第二次：“1693814052623,1693814052789”
		s.add(String.valueOf(nowTime));
		addr.put(Const.INFO, s.toString());
		hmset(rk, addr, times[loop - 1]); // 循环次数
        return false;
    }

	/**
	 * 生成一个66位国密SM2公钥，并缓存私钥 sec秒后失效
	 * <p>
	 * 仅登录初始化用在了getTicket()中，缓存2分钟
	 */
	public String getSm2PubKey(Object ticket, int sec) {
		Map<String, String> keyPair = Digest.genSm2Pair();
		String pubk = keyPair.get(Digest.PUB_KEY); // 66位公钥
		String prik = keyPair.get(Digest.PRI_KEY); // 64位私钥
		/**
		 * 用了两种方式来缓存私钥（sec秒后失效）<br>
		 * 1.以ticket为key暂存私钥<br>
		 * 2.以临时公钥为key暂存私钥
		 */
		set(Const.RDS_APPID + ticket, prik, sec);
		set(Const.RDS_CERT + pubk, prik, sec);
		/** 返回前端公钥 */
		return pubk;
	}

	/**
	 * 读取临时缓存的64位国密SM2私钥
	 */
	String getCertKey(Object key) {
		/**
		 * 支持两种方式获取私钥<br>
		 * 1.以ticket为key获取私钥<br>
		 * 2.以临时公钥为key获取私钥（目前使用第2种方式）
		 */
		String prik = get(Const.RDS_APPID + key);
		return prik == null ? get(Const.RDS_CERT + key) : prik;
	}

	/**
	 * 清除单个表的元数据对象dto
	 */
	void clearDto(Object table, Object rdsDbid) {
		del(Const.RDS_META + table + rdsDbid);
	}

}
