package cn.ps1.aolai.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.ps1.aolai.entity.Field;
import cn.ps1.aolai.entity.Layout;
import cn.ps1.aolai.entity.User;
import cn.ps1.aolai.utils.ConfUtil;
import cn.ps1.aolai.utils.Const;
import cn.ps1.aolai.utils.FailedException;
import cn.ps1.aolai.utils.FmtUtil;

/**
 * 针对TARIN表、FEILD表的扩展业务操作
 *
 * @author Taishan
 * @since 2024年10月02日
 */

@Service
public class LayoutService {

	private static Logger log = LoggerFactory.getLogger(LayoutService.class);

	/** layoutSet */
	private static Set<String> layoutSet = new HashSet<>();
	static {
		layoutSet.add(Layout.HEAD);
		layoutSet.add(Layout.OPTS);
		layoutSet.add(Layout.JOINT);
	}

	@Autowired
	private AolaiService aolai;
	@Autowired
	private RedisService redis;
	@Autowired
	private UtilsService utils;
	@Autowired
	HttpServletRequest req; // 单实例可以注入

	/**
	 * 表头数据的LAYOUT展示控制项，默认GCONF配置项：meta.custom=‘1’
	 */
/*	Map<String, String> getAliasArray(Object uriKey, String key) {
		// key = Const.BLUR
		return getAliasArray(getLayoutBy(uriKey), key);
	}*/

	/**
	 * 这里的key=permit、blur<br>
	 * 例如场景数据：layoutOpts = {"permit":"1","blur":"*","freeze":"1" ...}<br>
	 * 返回有权限的：{...xxxId:null, xxxMc:null, xxxMobi:‘*’, ...}<br>
	 */
	Map<String, String> getAliasArray(Map<String, String> item, String key) {
		List<String> head = utils.json2List(item.get(Layout.HEAD));
		List<Map<String, String>> opts = utils.json2List(item.get(Layout.OPTS));
		// 遍历配置规则，从 layoutOpts 找出
		Map<String, String> map = new LinkedHashMap<>();
		for (int i = 0; i < head.size() && i < opts.size(); i++) {
			Map<String, String> opt = opts.get(i);
			// 仅返回有权限（permit=‘1’）的数据
			if (Const.S_1.equals(opt.get(Const.PERMIT))) {
				// 返回值格式：{userId:'1',userName:null,userOpUid='0',...}
				map.put(head.get(i), opt.get(key)); // key=Const.BLUR
			}
		}
		log.debug("getAliasArray...{}", map);
		if (map.isEmpty())
			log.debug("--->>> not data found in LAYOUT");
		return map;
	}

	/**
	 * 选项配置： {"hide":"0", "width":"180", "freeze":"1", "permit":"1",
	 * "align":"left", "title":"0", "filter":""}
	 */
	private Map<String, Map<String, String>> getAliasOpts(Map<String, String> item) {
		List<String> head = utils.json2List(item.get(Layout.HEAD));
		List<Map<String, String>> opts = utils.json2List(item.get(Layout.OPTS));
		// 遍历配置规则
		Map<String, Map<String, String>> map = new LinkedHashMap<>();
		for (int i = 0; i < head.size() && i < opts.size(); i++) {
			Map<String, String> opt = opts.get(i);
			if (Const.S_1.equals(opt.get(Const.PERMIT))) {
				// 选项配置值，layout表每个字段对应的opts：
				map.put(head.get(i), opt);
			}
		}
		return map;
	}

	/**
	 * 按用户查询页面表头数据的LAYOUT展示控制项
	 */
	Map<String, String> getLayoutBy(Object uriKey) {
		return getLayoutBy(uriKey, req.getAttribute(User.ID));
	}

	/**
	 * 表头数据的LAYOUT展示控制项，默认GCONF配置项：meta.custom=‘1’
	 */
	private Map<String, String> getLayoutBy(Object uriKey, Object userId) {
		// 执行到此处，已经是默认条件：ConfUtil.isCustomMeta()
		String rk = Const.RDS_PAGE + uriKey + aolai.rdsDbid() + userId;
		Map<String, String> layout = redis.hmget(rk);

		// 从数据库获取信息
		// 此处如果redis存储的是对象，redis自动回转换成map。
		if (layout.isEmpty()) {
			String[] layoutArr = layoutSet.toArray(new String[layoutSet.size()]);
			Map<String, String> item;
			// userId 为空时，选择一条默认数据，不为空时选当前用户的设置
			if ("".equals(userId)) {
				item = getLayoutOne(uriKey);
			} else {
				// 查询当前操作用户或默认用户“”的布局
				item = getLayoutOne(uriKey, userId);
			}

			// 获取LAYOUT共通数据，并缓存数据：layoutHead\layoutOpts\layoutJoint
			// { layoutHead:[...],layoutOpts:[...],layoutJoint:{} }
			layout = utils.newMap(item, layoutArr);
			// 缓存数据请求对象（缓存2小时改为1小时）
			redis.hmset(rk, layout, Const.ONE_HH);
		}
		return layout;
	}

	/**
	 * 获取LAYOUT公共的通用控制项：查询条件操作用户为空<br>
	 * 这个方法比较特殊，获取公共的通用配置，与用户配置无关<br>
	 */
	private Map<String, String> getLayoutOne(Object uriKey) {
		Map<String, Object> cond = new HashMap<>();
		cond.put(Layout.URI, uriKey);
		cond.put(Layout.OPER, "");
		return aolai.findOne(Layout.TABLE, layoutSet, cond);
	}

	/**
	 * 获取LAYOUT定制化的私有控制项：查询条件为当前操作用户，或默认用户“”<br>
	 * 注意：此处必须 args[2] = "", 否则可能循环调用getAliasArray()
	 */
	private Map<String, String> getLayoutOne(Object uriKey, Object userId) {
		// 按用户降序，用户自定义数据优先
		Map<String, String> sort = new HashMap<>();
		sort.put(Layout.OPER, "DESC");
		String[] args = { null, Layout.TABLE, "", "1" };
		// 当前操作用户或默认用户“”的布局
		Map<String, Object> cond = layoutCond(uriKey, userId);
		List<Map<String, String>> list = aolai.findAll(cond, sort, args);
		log.debug("getLayoutOne...{}", list);

		if (list.isEmpty())
			throw new FailedException(ConfUtil.NOT_FOUND);
		return list.get(0);
	}

	/**
	 * 根据用户查询当前用户或系统默认的（userId或“”）布局条件
	 */
	private Map<String, Object> layoutCond(Object uriKey, Object userId) {
		Map<String, Object> cond = new HashMap<>();
		cond.put(Layout.URI, uriKey);
		String[] userArr = { (String) userId, "" };
		cond.put(utils.pHolder(Layout.OPER, Const.IN), userArr);
		return cond;
	}

	// 需要解决跨账套问题
	void clearLayout(Object uriKey) {
		// 缓存刷新支持通配符“*”
//		Object userId = req.getAttribute(User.ID);
		redis.delKeys(Const.RDS_PAGE + uriKey + aolai.rdsDbid() + "*");
	}

	/**
	 * 增加一条FIELD元数据
	 */
	public Map<String, String> addFieldItem() {
		String[] keys = { Field.URI, Field.ORIGIN, Field.ALIAS };
		Map<String, Object> data = utils.jsonParams(keys);

		// 先查询已有全部数据
		Object uriKey = data.get(Field.URI);
		List<Map<String, String>> list = getFields(uriKey);
		// 递增序号、默认为定制化数据“1”
		data.put(Field.SORT, list.size());
		data.put(Field.CUSTOM, "1");

		// 设置数据权限：默认零权限、先隐藏显示
		Map<String, String> opts = utils.obj2Map(data.get(Field.OPTS));
		if (!"1".equals(opts.get(Const.PERMIT)))
			opts.put(Const.PERMIT, "0");

		// 增加一条FIELD数据
		if (utils.isFailed(aolai.addRecord(Field.TABLE, data, true)))
			throw new FailedException();

		// 已有数据列表中，补一条新数据
		Map<String, String> item = new HashMap<>();
		item.put(Field.ALIAS, String.valueOf(data.get(Field.ALIAS)));
		item.put(Field.OPTS, utils.obj2Str(opts));
		list.add(item);

		// 更新布局LAYOUT表的数据
		return setLayoutItem(uriKey, list);
	}

	/**
	 * 删除一条FIELD元数据
	 */
	public Map<String, String> delFieldItem() {
		String[] keys = { Field.URI, Field.ALIAS };
		Map<String, Object> where = utils.jsonParams(keys);

		// 先查询全部已有数据
		Object uriKey = where.get(Field.URI);
		List<Map<String, String>> list = getFields(uriKey);
		int idx = 0;
		for (idx = 0; idx < list.size(); idx++) {
			// 遍历全部已有FIELD数据，字段已存在，则不能删除
			if (list.get(idx).get(Field.ALIAS).equals(where.get(Field.ALIAS))) {
				// 另外、非自定义的数据也不能删除
				if (!"1".equals(list.get(idx).get(Field.CUSTOM)))
					throw new FailedException(ConfUtil.CANT_REMOVE);
				list.remove(idx);
				break;
			}
		}
		// 先删除当前数据
		if (utils.isFailed(aolai.delete(Field.TABLE, where)))
			throw new FailedException();

		/** 递减已有其他数据的序号 */
		Map<String, Object> cond = new HashMap<>();
		cond.put(Field.URI, uriKey);
		cond.put(utils.pHolder(Field.SORT, ">"), idx);
		aolai.decrSort(Field.TABLE, Field.SORT, cond);

		// 更新布局LAYOUT表数据
		return setLayoutItem(uriKey, list);
	}

	/**
	 * 更新布局LAYOUT表数据，增、删、改皆需更新
	 */
	private <T> Map<String, String> setLayoutItem(Object uriKey, List<Map<String, T>> list) {
		// 更新据项
		Map<String, Object> params = new HashMap<>();
		params.put(Layout.URI, uriKey);
		params.put(Layout.HEAD, utils.obj2Str(utils.getArr(list, Field.ALIAS)));
		params.put(Layout.OPTS, utils.obj2Str(utils.getArr(list, Field.OPTS)));
		// 默认授权全部操作员
		params.put(Layout.OPER, "");
		params.put(Layout.STATE, "1");
		return setLayoutItem(params);
	}

	/**
	 * 新增并更新一条LAYOUT布局数据
	 */
	private Map<String, String> setLayoutItem(Map<String, Object> params) {
		Map<String, String> result = aolai.addRecord(Layout.TABLE, params, true);
		if (utils.isSuccess(result))
			clearLayout(params.get(Layout.URI));
		return result;
	}

	/**
	 * 根据用户个人喜好自定义显示字段列表
	 */
	public Map<String, String> setLayoutItem() {
		// 当前用户设置的布局数据
		Object userId = req.getAttribute(User.ID);
		String[] keys = { Layout.URI, Layout.HEAD, Layout.OPTS };
		Map<String, Object> params = utils.jsonParams(keys);
		// 对照原有数据进行处理，查询当前操作用户或默认用户“”的布局
		Map<String, String> layout = getLayoutOne(params.get(Layout.URI), userId);
		// 新配置
		List<String> head = utils.obj2List(params.get(Layout.HEAD));
		List<Map<String, String>> opts = utils.obj2List(params.get(Layout.OPTS));
		// 原配置有数据的读取权限需要保留，注意不能修改数据权限
		// 获取有权限的数据：
		Map<String, String> permit = getAliasArray(layout, Const.PERMIT);
		// 批处理opts
		for (int i = 0; i < head.size() && i < opts.size(); i++) {
			Map<String, String> opt = opts.get(i);
			opt.put(Const.PERMIT, permit.get(head.get(i)) == null ? "0" : "1");
		}

		// 批量修改配置属性
		params.put(Layout.JOINT, layout.get(Layout.JOINT));
		params.put(Layout.OPTS, utils.obj2Str(opts));
		params.put(Layout.HEAD, utils.obj2Str(head));
		params.put(Layout.OPER, userId);
		return setLayoutItem(params);
	}

	/**
	 * 仅设置自定义LAYOUT的展示属性<br>
	 * 这里的opts为单一head及单一Field的单一属性
	 */
	public Map<String, String> setLayoutOpts() {
		// 当前用户设置的布局数据
		Object userId = req.getAttribute(User.ID);
		String[] keys = { Layout.URI, Layout.HEAD, Layout.OPTS };
		Map<String, Object> params = utils.jsonParams(keys);
		// 对照原有数据进行处理，查询当前操作用户或默认用户“”的布局
		Map<String, String> layout = getLayoutOne(params.get(Layout.URI), userId);

		// 需保留原配置的权限
		List<String> head = utils.json2List(layout.get(Layout.HEAD));
		List<Map<String, String>> opts = utils.json2List(layout.get(Layout.OPTS));
		// 新配置的保留权限
		int idx = head.indexOf(params.get(Layout.HEAD));
		if (idx == -1)
			throw new FailedException();
		Map<String, String> opt = opts.get(idx);
		// 请求参数
		String permit = opt.get(Const.PERMIT);
		Map<String, String> tmp = utils.obj2Map(params.get(Layout.OPTS));
		for (Map.Entry<String, String> e : tmp.entrySet()) {
			opt.put(e.getKey(), e.getValue());
		}
		// 保留原数据的权限
		opt.put(Const.PERMIT, permit);
		log.debug("setLayoutOpts..." + opts);

		params.put(Layout.JOINT, layout.get(Layout.JOINT));
		params.put(Layout.OPTS, utils.obj2Str(opts));
		params.put(Layout.HEAD, utils.obj2Str(head));
		params.put(Layout.OPER, userId);
		return setLayoutItem(params);
	}

	/**
	 * 设置一条FIELD元数据，即：自定义字段信息表的字段属性信息
	 */
	public Map<String, String> setFieldItem() {
		String[] keys = { Field.URI, Field.ALIAS, Field.OPTS };
		Map<String, Object> params = utils.jsonParams(keys);

		// 设置数据权限：默认零权限、先隐藏显示
		Map<String, String> opts = utils.obj2Map(params.get(Field.OPTS));
		if (!"1".equals(opts.get(Const.PERMIT)))
			opts.put(Const.PERMIT, "0");

		// 修改配置参数
		Map<String, Object> item = new HashMap<>();
		item.put(Field.OPTS, utils.obj2Str(opts));

		// 更新FIELD数据的条件：Field.URI, Field.ALIAS
		Map<String, Object> cond = utils.newMap(params, new String[] { Field.URI, Field.ALIAS });
		Map<String, String> result = aolai.update(Field.TABLE, item, cond);

		// 更新LAYOUT布局数据
		if (utils.isSuccess(result)) {
			Object uriKey = params.get(Field.URI);
			return setLayoutItem(uriKey, getFields(uriKey));
		}
		return result;
	}

	/**
	 * 维护全部FIELD元数据扩展选项属性，包含排序、各种开关项等
	 * <p>
	 * 每个账套中一个FIELD表
	 */
	public Map<String, String> setFieldItems() {
		String[] keys = { Field.URI, Const.ITEMS };
		Map<String, Object> params = utils.jsonParams(keys);

		Object uriKey = params.get(Field.URI);
		// 先查询全部已有数据
		List<Map<String, String>> list = getFields(uriKey);
		// 更新数据
		List<Map<String, Object>> items = utils.obj2List(params.get(Const.ITEMS));
		if (list.isEmpty() || list.size() != items.size())
			return utils.invalidParams();

		// 获取已有数据原来的表名
		String table = list.get(0).get(Field.ORIGIN);
		Set<String> set = utils.toSet(list, Field.ALIAS);

		// 更新数据
		for (int i = 0; i < items.size(); i++) {
			Map<String, Object> item = items.get(i);
			if (!set.contains(item.get(Field.ALIAS)))
				utils.invalidParams();
			// 更新的数据配置
			Object opts = item.get(Field.OPTS);
			if (opts instanceof Map)
				item.put(Field.OPTS, utils.obj2Str(opts));
			item.put(Field.ORIGIN, table); // 原来的表名不能变
			item.put(Field.URI, uriKey); // 补充uri字段
			item.put(Field.SORT, i); // 重新排序
		}
		// 批量更新数据
		Map<String, String> result = aolai.batchAdd(Field.TABLE, items, true);
		if (utils.isSuccess(result))
			return setLayoutItem(uriKey, items);
		return result;
	}

	/**
	 * 根据uriKey查询FIELD全部信息一览
	 */
	private List<Map<String, String>> getFields(Object uriKey) {
		Map<String, Object> cond = new HashMap<>();
		cond.put(Field.URI, uriKey);
		Map<String, String> sort = new HashMap<>();
		sort.put(Field.SORT, "");
		return aolai.findAll(Field.TABLE, "", cond, sort);
	}

	/**
	 * 根据fieldUri查询，获取FIELD元数据字段信息，返给前端布局
	 * <p>
	 * 因为每个账套中一个FIELD表，所以需要携带base参数
	 */
	public List<Map<String, String>> getFieldsBy() {
		String[] keys = { Field.URI };
		Map<String, Object> params = utils.jsonParams(keys);

		// 查询全部 FIELD 元数据字段，如："getFieldList"
		Object uriKey = params.get(Field.URI);
		List<Map<String, String>> fieldList = getFieldsBy(uriKey, Const.LIST);
		log.debug("getFieldsBy...{}", fieldList);

		// 统一列设置页面，返回所有字段，不用返回每个用户的数据
		if (Const.S_1.equals(params.get(Const.ALL))) {
			return fieldList;
		}
		return layout(fieldList, uriKey, params);
	}

	/**
	 * 在当前账套内获取FIELD表中的元数据字段信息<br>
	 * 根据uriKey查询fieldAlias所有列名<br>
	 * <p>
	 * 这里查询 FIELD 关联 GMETA 表，META_STYLE 支持枚举配置<br>
	 *  如： META_STYLE={"enumType": "dwlx", "filterType": "enum"}。<br>
	 * 过滤类型配置：{"filterType": "string"}
	 */
	public List<Map<String, String>> getFieldsBy(Object uriKey, String key) {
		Map<String, Map<String, String>> tables = utils.joinCond(Field.TABLE);
		Map<String, String> cond1 = new HashMap<>();
		cond1.put(Field.ORIGIN, Const.TABLE);
		cond1.put(Field.ALIAS, Const.ALIAS);
		// 关联GMETA元数据表
		tables.put(Const.GMETA, cond1);

		Map<String, Object> cond = new HashMap<>();
		cond.put(Field.URI, uriKey);
		if (!utils.isEmpty(key))
			cond.put(utils.jsonExt(Field.OPTS, key), Const.S_1);

		// 按序号排列
		Map<String, String> sort = new HashMap<>();
		sort.put(Field.SORT, Const.SORTBY[0]);

		// 查询全部元数据：具体展示格式放在前端处理
		return aolai.findAll(tables, "getFieldList", cond, sort);
	}

	/**
	 * 处理布局数据，返给前端布局
	 */
	private List<Map<String, String>> layout(List<Map<String, String>> list, Object uriKey,
			Map<String, Object> params) {
		//boolean isShow = Const.S_1.equals(params.get(Const.SHOW)); // 展示全部
		// 选项配置，忽略隐藏{"hide":"1"}的值
		// { layoutHead:[...],layoutOpts:[...],layoutJoint:{} }
		// 根据 layoutHead 设置每个字段的配置：{ alias: {layoutOpts} }
		Map<String, Map<String, String>> opts = getAliasOpts(getLayoutBy(uriKey));
		Map<String, Map<String, String>> fieldItems = utils.list2Map(list, Field.ALIAS);
		log.debug("layout...{}", opts);
		// 按alias重新排序
		List<Map<String, String>> fields = new ArrayList<>();
		for (Map.Entry<String, Map<String, String>> e : opts.entrySet()) {
			Map<String, String> field = fieldItems.get(e.getKey());
			// item为空，说明layout配置的参数数量多了
			if (field == null) {
				log.error("> No data in FIELD or too many field in LAYOUT.");
				throw new FailedException("systemError");
			}

			// 忽略隐藏{"hide":"1"}的值
			// 设置页面需要展示hide的字段，此处添加前端传入参数 {show:1}则返回隐藏字段；否则不返回隐藏字段
//			Map<String, String> opt = e.getValue();
//			if (isShow || !Const.S_1.equals(opt.get(Const.HIDE))) {
				// 选项配置
				field.put(Layout.OPTS, utils.obj2Str(e.getValue()));
				fields.add(field);
//			}
		}
		return fields;
	}

	/**
	 * 根据前台参数，获取布局参数，并拼接生成多表联合查询参数<br>
	 * joint:{userName:{key:“A”,val:“阿三”}, fzflMc:{key:“B”,val:"绿色"}...}
	 */
	public Map<String, Map<String, String>> jointTables(Map<String, Object> params) {
		/**
		 * 注意区分 params.get(Const.JOINT) 与  layoutOne.get(Layout.JOINT)<br>
		 * 这里params.get(Const.JOINT)是前台传递的页面“过滤条件”参数<br>
		 */
		Map<String, Map<String, Object>> joint = utils.obj2Map(params.get(Const.JOINT));
		// 根据参数fieldUri去查询FIELD表中相关字段数据
//		Map<String, Object> where = utils.newMap(params, Field.URI);
		Map<String, Object> where = new HashMap<>();
		Object uri = params.get(Const.URI);
		where.put(Field.URI, uri == null ? params.get(Field.URI) : uri);
		// 此处加入判空处理
		if (!utils.isEmpty(joint.keySet())) {
			where.put(utils.pHolder(Field.ALIAS, Const.IN), joint.keySet());
		}
		// 获取相关字段数据一览
		List<Map<String, String>> fieldItems = aolai.findAll(Field.TABLE, where);

		// 根据uriKey获取通用（userId=“”）的布局参数
		Map<String, String> layoutOne = getLayoutBy(where.get(Field.URI), "");

		/**
		 * 注意区分 params.get(Const.JOINT) 与  layoutOne.get(Layout.JOINT)<br>
		 * 前面params.get(Const.JOINT)是前台传递的页面“过滤条件”参数<br>
		 * 这里获取的 layoutJoint 为多表关联条件，tables 很可能 NULL 或 '{}' 为错误
		 */
		Map<String, Map<String, String>> tables = utils.json2Map(layoutOne.get(Layout.JOINT));

		// 获取FIELD表中的字段别名fieldAlias及关联表名fieldOrigin
		for (Map<String, String> field : fieldItems) {
			// 获取字段别名如：fzflMC
			String alias = field.get(Field.ALIAS);
			// 获取转换参数，格式如：{key:“A”,val:“阿三”}
			Map<String, Object> map = joint.get(alias);
			if (!utils.isEmpty(map)) {
				// 获取关联的表名，并根据表名获取关联条件，如：ZDFZFL、USER
				Map<String, String> cond = tables.get(field.get(Field.ORIGIN));

				// 获取运算符主键：A、B、C、..
				Object key = map.get(Const.KEY);
				// 把map转变成SQL数值："%{}"
				Object val = ConfUtil.opExpr(key, map);

				// 字段的格式转换为：“{fzflMC} like”
				String fieldKey = FmtUtil.pHolder(alias, ConfUtil.getOpr(key));
				// 增加检索过滤条件
				if (cond != null && val instanceof CharSequence) {
					cond.put(fieldKey, (String) val);
					// 这里增加每个表的动态连结方式：“left”\“inner”\“right”
					// 在joinTables()方法中会用到此参数
					cond.put(null, "inner");
				} else {
					// 如果是对应主表的字段，参数值放到params条件中
					// 拓展in和between场景，参数也放到params条件中
					params.put(fieldKey, val);
				}
			}
		}
		return tables;
	}

	/**
	 * 为关联表的拼接精准查询条件
	 */
	public Map<String, Map<String, Object>> jointCond(String alias, Object opKey, Object opVal) {
		Map<String, Map<String, Object>> joint = new HashMap<>();
		Map<String, Object> map0 = new HashMap<>();
		map0.put(Const.KEY, opKey);
		map0.put(Const.VAL, opVal);
		joint.put(alias, map0);
		return joint;
	}

	/**
	 * 为关联表的拼接精准查询条件
	 */
	public Map<String, Map<String, Object>> jointCond(Map<String, Map<String, Object>> joint,
			String alias, Object opKey, Object opVal) {
		Map<String, Object> map1 = new HashMap<>();
		map1.put(Const.KEY, opKey);
		map1.put(Const.VAL, opVal);
		joint.put(alias, map1);
		return joint;
	}

	/**
	 * 检索获取分页数据一览，内置实现分页逻辑的处理逻辑<br>
	 */
	public Map<String, Object> getPagedData(Map<String, Object> params) {
		// 注意：这里必须先处理参数params，因为方法内部含有处理params的逻辑
		Map<String, Map<String, String>> tables = jointTables(params);
		// 再传入参数params，不能同时传入getPagedData()
		return getPagedData(tables, params, null);
	}

	/**
	 * 检索获取分页数据一览，内置实现分页逻辑的处理逻辑<br>
	 * 这是用于先预处理好tables的情况，用于分库和主库的表混合使用的场景
	 */
	public Map<String, Object> getPagedData(Map<String, Map<String, String>> tables,
			Map<String, Object> params) {
		// 再传入参数params
		return getPagedData(tables, params, null);
	}

	/**
	 * 检索获取分页数据一览，内置实现分页逻辑的处理逻辑<br>
	 */
	public Map<String, Object> getPagedData(Map<String, Map<String, String>> tables,
			Map<String, Object> params, Map<String, Object> cond) {
		// 默认使用参数中的uri，如果没有此参数则使用http的请求参数
		Object uri = params.get(Const.URI);
		String uriKey = uri == null ? utils.getRequestURI() : String.valueOf(uri);
		// 兼容了分页和不分页的两种情况，不分页时，系统限制默认最大输出行数
		// 这里是从前台传递来的排序sort参数
		Map<String, String> sort = utils.getSort(params);
		cond = cond == null ? params : cond;
		return aolai.queryList(tables, uriKey, cond, sort);
	}

	/**
	 * 补充显示数据一览中用于前台展示的名称
	 */
	public Map<String, Object> setEchoName(Map<String, Object> data,
			Map<String, Map<String, String>> echo) {
		return setEchoName(data, echo, "_name"); // 默认拼一个“_name”
	}

	/**
	 * 补充显示数据一览中用于前台展示的名称
	 */
	public Map<String, Object> setEchoName(Map<String, Object> data,
			Map<String, Map<String, String>> echo, String tag) {
		setEchoName(getListData(data), echo, tag);
		return data;
	}

	/**
	 * 补充显示数据一览中用于前台展示的名称
	 */
	public List<Map<String, String>> setEchoName(List<Map<String, String>> items,
			Map<String, Map<String, String>> echo, String tag) {
		if (items.isEmpty() || echo.isEmpty())
			return items;
		// 补充显示数据一览中用于前台展示的名称
		for (Map<String, String> row : items) {
			// {wyfwKfdwid=>wyfwKfdwid_name, wyfwWydwid=>wyfwWydwid_name}
			for (String key : echo.keySet()) {
				// 获取的数值，根据值再获取名称
				if (row.containsKey(key)) {
					String val = echo.get(key).get(row.get(key));
					row.put(key + tag, val == null ? "" : val);
				}
			}
		}
		return items;
	}

	/**
	 * 读取列表中的数据一览
	 */
	public List<Map<String, String>> getListData(Object obj) {
		if (obj instanceof Map)
			return getListData(utils.obj2Map(obj).get(Const.ITEMS));
		if (obj instanceof Collection)
			return utils.obj2List(obj);
		return new ArrayList<>();
	}

	/**
	 * 列表中的数据分页展示
	 */
	public List<Map<String, String>> splitPage(List<Map<String, String>> data,
			Map<String, Object> cond) {
		int pSize = utils.getInt(cond, Const.PAGE_SIZE);
		if (pSize > 0) {
			// 分页
			List<Map<String, String>> pageList = new ArrayList<>();
			// 页面号码、及每页展示的行数
			int pNo = utils.getInt(cond, Const.PAGE_NO) - 1;
			int floor = pNo > 0 ? pNo * pSize : 0;
			int total = data.size();
			int end = total > floor + pSize ? floor + pSize : total;
			for (int i = floor; i < end; i++) {
				pageList.add(data.get(i));
			}
			return pageList;
		}
		return data;
	}

}
