package cn.ps1.aolai.utils;

import java.util.HashMap;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;

import cn.ps1.aolai.entity.User;
import cn.ps1.aolai.service.AolaiService;
import cn.ps1.aolai.service.RedisService;
import cn.ps1.aolai.service.ThirdService;
import cn.ps1.aolai.service.UtilsService;

/**
 * 通用controller接口拦截器
 * 
 * @author Aolai
 * @version 1.0
 *
 */
@Component
public class Interceptor implements HandlerInterceptor {

	private static Logger LOG = LoggerFactory.getLogger(Interceptor.class);

	private static final String REQ_TIME = "REQ_TIME";

	@Autowired
	private RedisService redis;
	@Autowired
	private ThirdService third;
	@Autowired
	private UtilsService utils;

	@Autowired
	private AolaiService aolai;

	/**
	 * 调用 Controller方法之前经过拦截器统一处理所有web请求
	 */
	@Override
	public boolean preHandle(HttpServletRequest req, HttpServletResponse rsp,
			Object handler) throws Exception {
		// 预定义配置参数
		defineConfig(req);

		/**
		 * 首先进行verifyToken验证，携带的cookies信息如下：<br>
		 * {token=dzFSVJhg,userId=f3d785bdb1e54ee9f042f03115c827fd, ...}<br>
		 * 与redis中的userId,token,certId等信息校验
		 */
		Map<String, String> token = verifyToken(req);
		if (token != null) {
			/** token有效，继续对userInfo进行鉴权 */
			return third.authAccess(req, rsp, token);
		}

		/**
		 * checkToken无效后，再验证携带的 k参数（相当于appId）
		 * 需特殊处理的三个方法：getTicket、checkToken、signIn
		 * <p>
		 * getTicket > {ticket: 'BNQTJMFP', k: 'www.ps1.cn', i18n=ZH} <br>
		 * checkToken > {appCode: 'XX'} <br>
		 * params = {user: culai[1], pass: this.hexMd5(culai[0]), ticket} <br>
		 * signIn > {jsonstr: this.encoding(params, chrArr), k: certKey}
		 */
		// 验证 k参数，验证无效后返回失败！
		return third.verifySecret(req) ? true : third.invlidToken(rsp);

		// 第三方单点登录认证（待扩展）
//		if (third.thirdLogin(req, rsp, cookies))
//			return true;
	}

	/**
	 * 定义配置参数
	 */
	private void defineConfig(HttpServletRequest req) {
		// 对应配置文件中设置一个值‘true’或‘1’即可
		if (ConfUtil.IS_TRACKING) {
			// 正常情况下，tid基本不可能“”
			String tid = req.getHeader(ConfUtil.TRACEID);
			MDC.put(ConfUtil.TRACEID, tid == null ? Digest.uuid() : tid);
		}
		// 记录请求时间
		req.setAttribute(REQ_TIME, System.currentTimeMillis());
	}

	/**
	 * 根据请求携带参数获取Token信息
	 */
	private Map<String, String> verifyToken(HttpServletRequest req) {
		String uuid = utils.getCookie(req, ConfUtil.TOKEN);
		if (uuid == null) {
			uuid = req.getHeader(ConfUtil.TOKEN);
			if (uuid == null)
				uuid = req.getParameter(ConfUtil.TOKEN);
		}
		if (uuid == null) {
			LOG.debug("verifyToken...NULL");
			return null;
		}
		return redis.verifyToken(uuid);
	}

	/**
	 * 调用 Controller方法执行完成之后处理
	 */
	@Override
	public void postHandle(HttpServletRequest req, HttpServletResponse rsp,
			Object handler, ModelAndView modelAndView) throws Exception {

		// 请求响应耗时（毫秒）
		long msec = System.currentTimeMillis()
				- (long) req.getAttribute(REQ_TIME);

		// 获取当前用户：req.getAttribute(Const.USERID))
		Map<String, String> user = utils.userSelf(req);

		Map<String, Object> logInfo = new HashMap<>();
		logInfo.put("logUid", user.get(User.ID));
		logInfo.put("logComp", user.get(User.COMP));
		logInfo.put("logUri", utils.getRequestURI(req));

		// 用户网络的IP地址，与 user.get(Const.IPADDR)一致
		logInfo.put("logIp", utils.getClientIp(req));
		logInfo.put("logMsec", msec);

		// 请求响应的数据结果，响应AdviceResponse中的处理
		Object body = req.getAttribute("PLAIN_BODY");
		if (body instanceof Map) {
			Map<String, ?> map = utils.obj2Map(body);
			Object status = map.get(Const.STS);
			// 只记录错误信息
			if (!Const.STR_1.equals(status))
				logInfo.put("logInfo", map.get(Const.INFO));
			logInfo.put("logStatus", status);
		}

		// 请求时携带的json参数，如：{"appCode":"","i18n":"ZH"}
		Map<?, ?> json = utils.obj2Map(req.getAttribute(Const.JSON));
		logInfo.put(Const.I18N, json.get(Const.I18N)); // 请求参数

		// 日志要不要保存“请求参数”的最大长度
		if (ConfUtil.IS_LOG_ARGS > 0) {
			String args = utils.obj2Str(json);
			// 增删时请求参数较大，超过最大字节暂不记录请求参数
			// 最大支持字节：65,535
			if (args != null && args.length() < ConfUtil.IS_LOG_ARGS)
				logInfo.put("logArgs", args);
		}

		LOG.debug("logInfo:{}", logInfo);

		// 要不要保存“响应日志”记录
		if (!ConfUtil.IS_LOG_RESP)
			return;

		try {
			aolai.addRecord(json.get(Const.BASE), "OPERLOG", logInfo);
		} catch (Exception e) {
			LOG.warn("OPERLOG...{}", e.getMessage());
		}
	}

	/**
	 * DispatcherServlet进行视图的渲染之后处理
	 */
	@Override
	public void afterCompletion(HttpServletRequest req,
			HttpServletResponse rsp, Object handler, Exception ex)
			throws Exception {
		// 请求完毕，需要销毁，第二次请求过来重新生成。
        MDC.remove(ConfUtil.TRACEID);
	}

}
