package cn.ps1.aolai.utils;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;
import java.util.ResourceBundle;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 全局通用的配置参数类（config.properties）
 * 
 * @author Aolai
 * @since 1.7 $Date: 2017.6.17
 * @version 1.0
 */

public class ConfUtil {

	private static Logger LOG = LoggerFactory.getLogger(ConfUtil.class);

	/** 以下为系统约定的全局变量定义及默认配置 */

	/** 应用的配置文件：config.properties */
	private static ResourceBundle CONF = ResourceBundle.getBundle("config");
	/** 前台传递的参数：config.properties */
	private static ResourceBundle PARAM = ResourceBundle.getBundle("params");
	/** 需要校验的参数：config.properties */
	private static ResourceBundle VALID = ResourceBundle.getBundle("valid");

	public static Map<String, String> GCONG = new HashMap<>();

	/** 用于保存密钥的键值，公钥主键：“pubk” */
	public static final String PUB_KEY = "pubk"; // 密钥对：公钥
	/** 私钥主键：“prik” */
	public static final String PRI_KEY = "prik"; // 密钥对：私钥

	/** 约定为 “jsonstr”字符串 */
	public static final String JSONSTR = "jsonstr";
	/** 主键：“appCode” */
	public static final String APPCODE = "appCode";
	/** 保存在token中的客户端的IP地址的主键“ipaddr” */
	public static final String IPADDR = "ipaddr";
	/** “ticket” */
	public static final String TICKET = "ticket";
	/** “token” */
	public static final String TOKEN = "token";

	/** 前端传递的k参数，注意与certKey区别，与RDS_CERT无关 */
	public static final String CERT_K = "k";
	public static final String COMMA = ",|;";
	// 跟踪编号
	public static final String TRACEID = "traceId";
	// 证书编号
	public static final String CERTID = "certId";
	// 公司编号
	public static final String COMPID = "compId";
	// 用户编号
	public static final String USERID = "userId";

	/** "duplicateName" */
	public static final String DUPL_NAME = "duplicateName";
	/** "cantRemove" */
	public static final String CANT_REMOVE = "cantRemove";	
	/** "cantUpdate" */
	public static final String CANT_UPDATE = "cantUpdate";	
	/** "overCounts" */
	public static final String OVER_COUNTS = "overCounts";	

	/**
	 * 以下约定的参数需要与config.properties中的配置相匹配
	 */

	// 默认未启用GCONF
	private static boolean usingGconf = false;

	/** 应用接口互访KEY */
	public static String APP_KEY = "";
	/** 应用接口互访KEY */
	public static String ENC_KEY = "";
	/** 应用缩写的编码，如：“FF” */
	public static String APP_CODE = "";
	/** 主应用的请求地址，如：http://host.ps1.cn */
	public static String APP_CORE = "";
	/** 主应用，如："/doyea" */
	public static String APP_MAIN = "";
	/** 访问主应用的证书 */
	public static String[] APP_SPEC = {};	

	/** 多租户分库时的账套编号，如："dbid" 或 "baseId"，未配置默认"baseId" */
	public static String BASE_DBID = "";
	/** 多租户的分割符号"." 或 "_"，未配置默认"." */
	public static String BASE_DOT = "";
	/** 多租户的账套名称的前缀，如："culai"、"base" */
	public static String BASE_NAME = "";
	/** 用于区分Mysql、PG数据库，如：mysql\public， */
	public static String BASE_MAIN = "";
	/** 支持多语言参数，如：ZH,EN */
	public static String[] I18N_LOCALES = {};

	/** 单点SSO请求单点登录的远程服务地址，如：http://sso.ps1.cn/aolai/s */
	public static String SSO_URL = "";
	/** 各个接口通用校验项目，可不用配置 */
	public static String VALID_MUST = "";
	/** 查询数据行数的限制，未配置默认limit.rows=''不限制数据 */
	public static String LIMIT_ROWS = "";
	/** Http请求超时时间，默认2分钟超时，120秒 */
	public static int HTTP_TIMEOUT = 120000;
	// Redis默认的用户缓存时间 2小时
	public static int CACHE_TIME = Const.TWO_HH;

	/** API是否需要对应三方应用授权才能访问 */
	public static boolean API_PERMIT = false;
	/** 应用请求路径跟踪标识，可不用配置 */
	public static boolean IS_TRACKING = false;
	/** 开放API接口标识，可不用配置，若需要配置为：true */
	public static boolean IS_API_OPEN = false;
	/** 响应内容加密，可不用配置 */
	public static boolean IS_ENC_RESP = false;
	/** 响应日志记录，可不用配置 */
	public static boolean IS_LOG_RESP = false;
	/** 日志要不要保存“请求参数”的最大长度 */
	public static int IS_LOG_ARGS = 0;

	/** 请求内容是否忽略加密，默认加密（不忽略），可不用配置 */
	public static boolean IS_ENC_OMIT = false;

	/** 控制同一账户多人同时登录，未配置默认为仅限单用户登录使用 */
	public static boolean IS_MULTI_LOGIN = false;
	/** 是否支持自定义设置标签的隐藏和显示的数量 */
	public static boolean IS_CUSTOM_COLS = false;

	/** 正则映射 */
	public static String REGEXP = " ";
	/** rlike 映射 */
	public static String RLIKE = " ";

	/** 有效的 SQL表达式，暂时禁用: "!","&&","||","XOR" */
	public static String[] SQL_EXPR = { "=", ">", "<", ">=", "<=", "<>", "!=",
			"regexp", "rlike", "is", "not", "null", "and", "or", "like", "in",
			"(", ")", "between", "length(", "json_extract(" };

	static {
		// 初始化系统配置参数
		initAppConf();

		APP_CODE = getConf("app.code");
		APP_KEY = getConf("app.key", "www.ps1.cn");
		ENC_KEY = Digest.md5(APP_KEY);
		APP_CORE = getConf("app.core");
		APP_MAIN = getConf("app.main");
		// 格式为：ticket:apec
		// VJEFKISG:12044b9cbfb8eeaeba3e21d1d7a26611
		APP_SPEC = getConf("app.spec").split(Const.COLON);

		BASE_DBID = getConf("base.dbid", "baseId");
		BASE_DOT = getConf("base.dot", Const.DOT);
		BASE_NAME = getConf("base.name");
		BASE_MAIN = getConf("base.main", "public"); // mysql时可配置为空
		I18N_LOCALES = getConf("i18n.locales", "ZH,EN").split(COMMA);

		SSO_URL = getConf("sso.url");
		VALID_MUST = getConf("valid.must");
		LIMIT_ROWS = getConf("limit.rows");
		HTTP_TIMEOUT = Integer.parseInt(getConf("http.timeout", "90000"));
		String conf = getConf("cache.time");
		if (conf.length() > 0)
			CACHE_TIME = Integer.parseInt(conf);

		API_PERMIT = getConf("api.permit").length() > 0;
		IS_TRACKING = getConf("app.trace").length() > 0;
		IS_API_OPEN = getConf("api.open").length() > 0; // 开放API模式:"true"
		IS_ENC_RESP = getConf("enc.resp").length() > 0;
		IS_LOG_RESP = getConf("log.resp").length() > 0;
		// 日志要不要保存“请求参数”的最大长度
		IS_LOG_ARGS = Integer.parseInt(getConf("log.args", "0"));
		IS_ENC_OMIT = getConf("enc.omit").length() > 0;

		IS_MULTI_LOGIN = getConf("multi.login").length() > 0;
		IS_CUSTOM_COLS = getConf("cols.custom").length() > 0;

		REGEXP += getConf("sql.regexp", "regexp");
		RLIKE += getConf("sql.rlike", "rlike");

		// 可以自定义SQL注入的关键字
		conf = getConf("sql.expr");
		if (conf.length() > 0)
			SQL_EXPR = conf.split(COMMA);
	}

	@SuppressWarnings("unchecked")
	private static void initAppConf() {
		/**
		 * 下面除了APP_GCONF之外的配置参数全部从数据库中获取
		 * 
		 * 如果从数据库获取数据，正常配置为：app.gconf=,GCONF,,0
		 * 
		 * 如果各应用使用相同的数据库，不同应用可配置不同GCONF名
		 */
		String args = getConf("app.gconf");

		// 根据配置判断是否启用GCONF
		usingGconf = args.length() > 0;
		if (usingGconf) {
			Object res = invoke(null, "appGconf", args);
			if (res == null)
				usingGconf = false;
			else
				GCONG = (Map<String, String>) res;
		}
	}

	/**
	 * 获取数据库中的系统配置参数值
	 */
	private static String appConf(String key) {
		return appConf(key, "");
	}

	/**
	 * 获取数据库中的系统配置参数值
	 */
	private static String appConf(String key, String def) {
		String val = GCONG.get(key);
		return val == null ? def : val;
	}

	/**
	 * 调用：根据Bean名称，发起第三方服务接口调用<br>
	 */
	public static Object invoke(Object bean, String method, Object args) {
		try {
			// 获取bean
			if (bean == null)
				bean = SpringContext.getBean("gconfService");
			// 获取公开方法
			Method m = bean.getClass().getMethod(method, Object.class);
			// 调用方法，根据参数进行处理
			return m.invoke(bean, args);
		} catch (Exception e) {
			LOG.error("invoke....{}({})", method, args);
			return null;
		}
	}

	/**
	 * 获取配置参数
	 */
	public static String getConf(String key, String def) {
		return usingGconf ? appConf(key, def) : getProp(CONF, key, def);
	}

	/**
	 * 获取配置参数
	 */
	public static String getConf(String key) {
		return usingGconf ? appConf(key) : getProp(CONF, key, "");
	}

	public static String getParam(String key) {
		return getProp(PARAM, key, "");
	}

	public static String getValid(String key) {
		return getProp(VALID, key, "");
	}

	private static String getProp(ResourceBundle rb, String key, String def) {
		try {
			return rb.getString(key);
		} catch (Exception e) {
			LOG.debug("getProp...{}", e.getMessage());
			return def;
		}
	}

}
