package cn.ps1.aolai.service;

import java.io.Closeable;
import java.lang.reflect.Array;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.DecimalFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;

import cn.ps1.aolai.entity.User;
import cn.ps1.aolai.utils.ConfUtil;
import cn.ps1.aolai.utils.Const;
import cn.ps1.aolai.utils.Digest;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * 全局通用的公共方法Utils类
 * 
 * @author Fulin
 * @since  1.7 $Date: 2017.6.17
 * @version 1.0
 */

@Service
public class UtilsService {

	private static Logger LOG = LoggerFactory.getLogger(UtilsService.class);
	private static ObjectMapper jsonMapper = new ObjectMapper();

	private class typeRef<T> extends TypeReference<T> {
	}

	@Autowired
	private ApplicationContext appCtx;

	/**
	 * 构造函数
	 */
	public UtilsService() {
	}

	/**
	 * 当前数据源是否为PostgreSQL
	 */
	boolean isPostgres() {
		// 注意区分数据库驱动，这里与AolaiDao.xml中的databaseId命名保持一致
		// 在加载配置文件，注册SqlSessionFactoryBean时获取配置属性
		return appCtx.containsBean("dsPostgres");
	}

	/**
	 * 获取HTTP请求从前端传递来的参数
	 */
	public Map<String, String> getParams(HttpServletRequest req) {
		return obj2Map(jsonParams(req));
	}

	/**
	 * 获取从前端HTTP请求传递来的参数：梳理功能已移到ThirdService.setReqAttr()
	 */
	public Map<String, Object> jsonParams(HttpServletRequest req) {
		// 从前端传递来的请求参数
		return obj2Map(req.getAttribute(Const.JSON));
	}

	/**
	 * 请求参数为无效参数
	 */
	public Map<String, String> invalidParams() {
		return result(Const.STR_2);
	}

	/**
	 * 当前操作用户自己，相当于getSession()
	 */
	public Map<String, String> userSelf(HttpServletRequest req) {
		return obj2Map(req.getAttribute(User.KEY));
	}

	/**
	 * 设置当前操作用户的公司ID后返回公司ID
	 */
	public String setUserComp(HttpServletRequest req,
			Map<String, Object> params, String key) {
		String userComp = userSelf(req).get(User.COMP);
		if (userComp != null)
			params.put(key, userComp);
		return userComp;
	}

	/**
	 * 用于重新设置查询条件的数据状态：一般默认设置为“0”“1”的情况
	 */
	public void resetState(Map<String, Object> cond, String key) {
		cond.put(key, isNil(cond.get(key)) ? Const.STR_0 : Const.STR_1);
	}

	/**
	 * 根据父节点新建一个新节点，如ID：0102 (bitW=2)
	 * 
	 * @param pId 父级编码
	 * @param id 子级编号
	 * @param bitW 编号层级的宽度，如：bitW=3
	 * @return String
	 */
	public String idCode(Object pId, int id, int bitW) {
		// 第一个子节点必为“1”，拼接父节点的ID后，变为0x0x01
		// 第二个子节点之后则是“x0x02”，只需前面补充一个“0”即可
		String pid = isNil(pId) ? "" : String.valueOf(pId);
		if (id > 1) {
			bitW += pid.length();// + bitW;
			pid = "";
		}
		return pid + String.format("%0" + bitW + "d", id);
	}

	/**
	 * 拼装检查数据重复的条件，默认“第一个”值是主键，首先要排除主键
	 */
	public <T> Map<String, Object> sameIf(Map<String, T> cond, String[] keys) {
		// 排除自己
		Map<String, Object> where = sameIf(cond, keys, 1);
		if (cond.containsKey(keys[0]))
			where.put(pHolder(keys[0], Const.NEQ), cond.get(keys[0]));
		return where;
	}

	private <T> Map<String, Object> sameIf(Map<String, T> cond, String[] keys,
			int offset) {
		// 不能有重名的数据
		Map<String, Object> where = new HashMap<>();
		for (int i = offset; i < keys.length; i++)
			where.put(keys[i], cond.get(keys[i]));
		where.put(Const.I18N, cond.get(Const.I18N));
		return where;
	}

	/**
	 * 检索条件转换
	 */
	public <T> Map<String, Object> sameId(Map<String, T> cond, String key) {
		return sameIf(cond, new String[] { setComp(key), key + "Id" }, 0);
	}

	/**
	 * 检索条件转换
	 */
	public <T> Map<String, Object> sameNo(Map<String, T> cond, String key) {
		return sameIf(cond, new String[] { setComp(key), key + "No" }, 0);
	}

	/** 公司主键 */
	private String setComp(String key) {
		return key + "Comp";
	}

	/**
	 * 校验request请求参数是否存在
	 */
	public boolean availParams(Map<?, ?> map, String key) {
		if (key == null)
			return availParams(map, new String[] { key });
		return availParams(map, key.split(ConfUtil.COMMA));
	}

	/**
	 * 检查请求（输入）参数无效
	 * 
	 * @param params
	 * @param keys
	 * @return boolean 成功与否
	 */
	public boolean availParams(Map<?, ?> params, String[] keys) {
		if (params == null)
			return false;
		for (int i = 0; i < keys.length; i++) {
			if (!params.containsKey(keys[i])) {
				LOG.debug("invalidParams={}", keys[i]);
				return false;
			}
		}
		return true;
	}

	/**
	 * 把数串逐级累计（科目层级转换）
	 * 
	 * @param numStr 数字的字符串
	 * @return Array 累计数组
	 */
	public int[] num2Arr(String numStr) {
		int[] num = str2num(numStr);
		for (int i = 1; i < num.length; i++) {
			num[i] += num[i - 1];
		}
		return num;
	}

	/**
	 * 字符串分割为整数数组
	 */
	public int[] str2num(String numStr) {
		if (!isInteger(numStr))
			return new int[0];
		String[] arr = numStr.split("");
		int[] num = new int[arr.length];
		for (int i = 0; i < arr.length; i++) {
			num[i] = Integer.parseInt(arr[i]);
		}
		return num;
	}

	/**
	 * 数组合并拼接为字符串
	 */
	public String arr2Str(Object[] strArr) {
		return join(strArr, null);
	}

	/**
	 * 数组合并拼接为字符串
	 */
	public String join(Object[] strArr, String joint) {
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < strArr.length; i++) {
			if (joint != null && i > 0)
				sb.append(joint);
			sb.append(strArr[i]);
		}
		return sb.toString();
	}

	/**
	 * 从对象中获取整数值
	 */
	public <T> int getInt(Map<String, T> map, String key) {
		if (map != null) {
			String val = String.valueOf(map.get(key));
			if (isInteger(val))
				return Integer.valueOf(val);
		}
		return 0;
	}

	/**
	 * 将字符串的首字母转大写
	 * <p>
	 * 用字母的ascii编码前移，效率要高于截取字符串进行转换的操作
	 */
	public String initCap(Object str) {
		char[] cs = String.valueOf(str).toLowerCase().toCharArray();
		cs[0] -= 32;
		return String.valueOf(cs);
	}

	/**
	 * Map对象数据互转
	 */
	public Map<String, Object> map2Obj(Map<String, String> map) {
		Map<String, Object> obj = new HashMap<>();
		if (map != null)
			for (Map.Entry<String, String> e : map.entrySet()) {
				obj.put(e.getKey(), e.getValue());
			}
		return obj;
	}

	/**
	 * 通过参数映射解析器，进行参数主键转换
	 */
	public <T> List<Map<String, T>> mapParser(Map<String, String> parser,
			List<Map<String, T>> list) {
		// 处理结果数据
		if (!isEmpty(parser))
			for (int i = 0; i < list.size(); i++) {
				if (list.get(i).isEmpty())
					break;
				list.set(i, mapParser(parser, list.get(i)));
			}
		return list;
	}

	/**
	 * 通过参数映射解析器，进行参数主键转换<br>
	 * parser={empUid=userId, empComp=userComp, empName=userName}
	 */
	public <T> Map<String, T> mapParser(Map<String, String> parser,
			Map<String, T> data) {
		if (isEmpty(parser))
			return data;
		Map<String, T> item = new HashMap<>();
		for (Map.Entry<String, String> e : parser.entrySet()) {
			if (data.containsKey(e.getValue()))
				item.put(e.getKey(), data.get(e.getValue()));
		}
		return item;
	}

	/**
	 * 对象转换为Map对象
	 */
	@SuppressWarnings("unchecked")
	public <T> Map<String, T> obj2Map(Object data) {
		if (data == null) return new HashMap<>();
		return data instanceof Map ? (Map<String, T>) data : new HashMap<>();
	}

	/**
	 * 对象转换为List对象
	 */
	@SuppressWarnings("unchecked")
	public <T> List<T> obj2List(Object data) {
		if (data == null) return new ArrayList<>();
		return data instanceof Collection ? (List<T>) data : new ArrayList<>();
	}

	/**
	 * Map对象数据互转
	 */
	public Map<String, String> obj2Map(Map<String, Object> obj) {
		Map<String, String> map = new HashMap<>();
		if (obj != null)
			for (Map.Entry<String, Object> e : obj.entrySet()) {
				Object val = e.getValue();
				map.put(e.getKey(), val == null ? null : String.valueOf(val));
			}
		return map;
	}

	/**
	 * 把对象（Map、List）转为jsonStr字符串
	 * 
	 * @param obj
	 * @return jsonStr
	 */
	public String obj2Str(Object obj) {
		try {
			return jsonMapper.writeValueAsString(obj);
		} catch (Exception e) {
			// 这里基本不可能出错
			LOG.error("obj2Str...{}", e.getMessage());
			return null;
		}
	}

	/**
	 * 从Json字符串中找出Map对象节点
	 */
	public Map<String, Object> node2Obj(String jsonStr, String path) {
		try {
			if (!isEmpty(jsonStr)) {
				JsonNode node = jsonMapper.readTree(jsonStr);
				return node2Obj(path == null ? node : node.path(path));
			}
		} catch (Exception e) {
			error("node2Obj", jsonStr);
		}
		return new HashMap<>();
	}

	/**
	 * Json对象字符串转换为Map对象
	 * 
	 * @param jsonStr
	 * @return Map
	 */
	public Map<String, Object> node2Obj(String jsonStr) {
		return node2Obj(jsonStr, null);
	}

	/**
	 * Json对象转换为Map对象
	 * 
	 * @param jsonNode
	 * @return Map
	 */
	public Map<String, Object> node2Obj(JsonNode jsonNode) {
		Map<String, Object> map = new HashMap<>();
		if (jsonNode.isValueNode()) {
			map.put("", jsonNode.asText());
		} else if (jsonNode.isArray()) {
			List<Object> list = new ArrayList<>();
			Iterator<JsonNode> it = jsonNode.iterator();
			while (it.hasNext()) {
				Map<String, Object> child = node2Obj(it.next());
				if (child.keySet().size() == 1 && child.keySet().contains("")) {
					list.add(child.get(""));
				} else {
					list.add(child);
				}
			}
			map.put("", list);
		} else {
			Iterator<Map.Entry<String, JsonNode>> it = jsonNode.fields();
			while (it.hasNext()) {
				Map.Entry<String, JsonNode> entity = it.next();
				Map<String, Object> child = node2Obj(entity.getValue());
				if (child.keySet().size() == 1 && child.keySet().contains("")) {
					map.put(entity.getKey(), child.get(""));
				} else {
					map.put(entity.getKey(), child);
				}
			}
		}
		return map;
	}

	/** 记录错误 */
	void error(String method, String str) {
		LOG.error("{}...'{}'", method, omitStr(str));
	}

	/** 记录日志 */
	void debug(String method, String str) {
		LOG.debug("{}...'{}'", method, omitStr(str));
	}

	/**
	 * 省略字符串
	 */
	private String omitStr(String str) {
		return str.length() < 200 ? str : str.substring(0, 200) + "...";
	}

	/**
	 * 拼接SQL字符串的值
	 */
	public String sqlVal(Object val) {
		return "'" + val + "'";
	}

	/**
	 * 拼接json格式的SQL语句
	 */
	public String jsonExt(String field, Object key) {
		return jsonExt(field, key, Const.EQU);
	}

	/**
	 * 拼接json格式的SQL语句
	 */
	public String jsonExt(String field, Object key, String opr) {
		String[] keys = { " json_extract({", field, "},'$." };
		return arr2Str(keys) + key + "') " + opr;
	}

	/**
	 * 把jsonStr转换成为Map对象
	 * 
	 * @param str
	 * @return Map
	 */
	public <T> Map<String, T> json2Map(String str) {
		try {
			if (str != null)
				return jsonMapper.readValue(str, new typeRef<Map<String, T>>());
		} catch (Exception e) {
			// 例如：HTTP/1.1 500 Internal Server Error
			error("json2Map", str);
		}
		return new HashMap<>();
	}

	/**
	 * 把json字符串转换成为list对象
	 */
	public <T> List<T> json2List(Object str) {
		try {
			if (str != null)
				return jsonMapper.readValue((String) str,
						new typeRef<List<T>>());
		} catch (Exception e) {
			LOG.error("json2List...{}" + e.getMessage());
		}
		return new ArrayList<>();
	}

	/**
	 * list转换为键值对格式的map对象
	 * 
	 * @param <T>
	 */
	public <T> Map<String, Map<String, T>> list2Map(List<Map<String, T>> list,
			String key) {
		Map<String, Map<String, T>> map = new HashMap<>();
		if (list != null && key != null) {
			for (int i = 0; i < list.size(); i++) {
				Map<String, T> item = list.get(i);
				map.put((String) item.get(key), item);
			}
		}
		return map;
	}

	/**
	 * list转换为键值对格式的map对象
	 */
	public Map<String, String> list2Map(List<Map<String, String>> list,
			String key, String val) {
		Map<String, String> map = new HashMap<>();
		if (list != null && key != null && val != null) {
			for (int i = 0; i < list.size(); i++) {
				Map<String, String> item = list.get(i);
				map.put(item.get(key), item.get(val));
			}
		}
		return map;
	}

	/**
	 * 返回状态结果
	 * 
	 * @param status "1".成功，"0".失败、"2".参数错误、"3".Token失效
	 * @return Map
	 */
	public Map<String, String> result(String status) {
		Map<String, String> result = new HashMap<>();
		result.put(Const.STS, status);
		return result;
	}

	/**
	 * 返回状态结果：false-失败、true-成功
	 */
	public Map<String, String> result(boolean status) {
		return result(status ? Const.STR_1 : Const.STR_0);
	}

	/**
	 * 返回状态结果
	 * 
	 * @param status "1".成功，"0".失败、"2".参数错误、"3".Token失效
	 * @param info 返回信息或数据
	 */
	public Map<String, String> result(String status, String info) {
		Map<String, String> result = result(status);
		if (info != null)
			result.put(Const.INFO, info);
		return result;
	}

	public <T> Object result(Map<String, T> map) {
		return map.containsKey(Const.STS) ? map : success(map);
	}

	/**
	 * 返回失败消息
	 */
	public Map<String, String> failed(String info) {
		return result(Const.STR_0, info);
	}

	public <T> boolean isFailed(Map<String, T> result) {
		return isNil(result.get(Const.STS));
	}

	public <T> boolean isSuccess(Map<String, T> result) {
		return Const.STR_1.equals(result.get(Const.STS));
	}

	/**
	 * 返回成功
	 */
	public Map<String, String> success(String info) {
		return result(Const.STR_1, info);
	}

	public Map<String, String> success() {
		return result(Const.STR_1);
	}

	/**
	 * 返回数据对象
	 */
	public Map<String, Object> success(Object obj) {
		Map<String, Object> result = new HashMap<>();
		result.put(Const.STS, Const.STR_1);
		result.put(Const.INFO, obj);
		return result;
	}

	/**
	 * 返回前台加密后的信息（注意：应用场景比较少，可能在个别地方单独使用）
	 */
	public Map<String, String> success(Object obj, HttpServletRequest req) {
		Object cert = req.getAttribute(ConfUtil.CERTID);
		return success(Digest.decrypt(obj2Str(obj), String.valueOf(cert)));
	}

	/**
	 * 找出List列表中匹配的指定字符串
	 */
	public Map<String, String> findIn(List<Map<String, String>> list,
			String key, String val) {
		if (list != null && val != null)
			for (int i = 0; i < list.size(); i++) {
				if (val.equals(list.get(i).get(key)))
					return list.get(i);
			}
		return null;
	}

	/**
	 * 找出List列表中是否存在指定字符串
	 */
	public boolean findIn(List<String> list, Object str) {
		return list != null && list.contains(str);
	}

	/**
	 * 字符串数组中是否存在指定的字符串
	 */
	public boolean findIn(String[] arr, Object str) {
		if (str != null && arr != null)
			for (int i = 0; i < arr.length; i++) {
				if (str.equals(arr[i]))
					return true;
			}
		return false;
	}

	/**
	 * 字符串中是否包含指定的字符串，如："\\d+"
	 */
	public boolean findIn(String str, String regex) {
		if (str == null || regex == null)
			return false;
		return Pattern.compile(regex).matcher(str).find();
	}

	/**
	 * 判断字符串是否为空
	 */
	public boolean isNil(Object obj) {
		return Const.STR_0.equals(obj);
	}

	/**
	 * 判断字符串是否为空
	 */
	public boolean isBlank(Object obj) {
		return "".equals(obj);
	}

	/**
	 * 判断字符串是否为空
	 */
	public boolean isEmpty(Object obj) {
		if (obj == null)
			return true;
		if (obj instanceof CharSequence) // 字符串
			return ((CharSequence) obj).length() == 0;
		if (obj instanceof Map)
			return ((Map<?, ?>) obj).isEmpty();
		if (obj instanceof Collection) // List列表、Set等
			return ((Collection<?>) obj).isEmpty();
		if (obj.getClass().isArray()) // 数组
			return Array.getLength(obj) == 0;
		return false;
	}

	/**
	 * 判断对象是否为空
	 */
	public <T> boolean isEmpty(Map<String, T> map) {
		return (map == null || map.isEmpty());
	}

	/**
	 * 字符串中是否有完全匹配：匹配字符或数字
	 * 
	 * @return boolean 成功与否
	 */
	public boolean isMatch(String input, String regex) {
		if (input == null || regex == null)
			return false;
		return Pattern.matches(regex, input);
	}

	/**
	 * 判断是否为整数、负数
	 */
	public boolean isInteger(String str) {
		// ^[-\\+]?[\\d]+$匹配前面的子表达式一次或多次
		return isMatch(str, "-?\\d+"); // "^-?\\d+$"
	}

	/**
	 * 判断是否为浮点数：包含整数、小数、负数
	 */
	public boolean isNumeric(String str) {
		// "^(-?\\d+)(\\.\\d+)?$"
		return isMatch(str, "(-?\\d+)(\\.\\d+)?");
	}

	/**
	 * 判断是否为身份证号码："^\\d{17}[\\dXx]$"
	 */
	public boolean isIdcard(String str) {
		return isMatch(str, "\\d{17}[\\dXx]");
	}

	/**
	 * 判断是否为手机号码："^1[3-9]\\d{9}"
	 */
	public boolean isMobile(String str) {
		return isMatch(str, "1[3-9]\\d{9}");
	}

	/**
	 * 判断是否为时间格式
	 */
	public boolean isDatetime(String dateStr, String pat) {
		try {
			if (dateStr != null) {
				new SimpleDateFormat(pat).parse(dateStr);
				return true;
			}
		} catch (ParseException e) {
			error("isDatetime", dateStr);
		}
		return false;
	}

	/**
	 * 时间（日期）格式转变
	 */
	public String timeFormat(String timeStr, String pat0, String pat1) {
		try {
			Calendar cal = Calendar.getInstance();
			cal.setTime(new SimpleDateFormat(pat0).parse(timeStr));
			return timeFmt(pat1, cal);
		} catch (ParseException e) {
			error("timeFormat", timeStr);
			return timeStr;
		}
	}

	/**
	 * 计算两个日期之间的差数多少（毫秒数）：begin开始时间，end结束时间
	 */
	public long timeDiff(Date begin, Date end) {
		Calendar cal = Calendar.getInstance();
		cal.setTime(end);
		long msec = cal.getTimeInMillis();
		cal.setTime(begin);
		// 返回毫秒数
		return msec - cal.getTimeInMillis();
	}

	/**
	 * 计算两个日期之间的差数多少（毫秒数）：begin开始时间，end结束时间
	 */
	public long timeDiff(String begin, String end) {
		// yyyy-MM-dd HH:mm:ss
		return diffMsec(begin, end, Const.DTF);
	}

	/** 两个时间之间的毫秒间隔 */
	public long diffMsec(String begin, String end) {
		// yyyy-MM-dd HH:mm:ss.SSS
		return diffMsec(begin, end, Const.MSEC);
	}

	/** 两个时间之间的毫秒间隔 */
	private long diffMsec(String begin, String end, String pat) {
		try {
			if (!isEmpty(begin)) {
				SimpleDateFormat sdf = new SimpleDateFormat(pat);
				Date endDate = end == null ? new Date() : sdf.parse(end);
				return timeDiff(sdf.parse(begin), endDate);
			}
		} catch (ParseException e) {
			LOG.error("obj2Str...{}", e.getMessage());
		}
		return 0;
	}

	/**
	 * 根据日期格式，获取当前日期时间：pattern格式
	 */
	public String today(String pattern) {
		return timeFmt(pattern, Calendar.getInstance());
	}

	/**
	 * 根据日期格式，获取当前日期时间：pattern格式
	 */
	private String timeFmt(String pattern, Calendar cal) {
		pattern = pattern == null ? Const.DTF : pattern;
		return new SimpleDateFormat(pattern).format(cal.getTime());
	}

	/**
	 * 当前时间节点向后（或向前）推迟n年
	 */
	public String nextYear(String pattern, int n) {
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.YEAR, n);
		return timeFmt(pattern, cal);
	}

	/**
	 * 获得下个月期间
	 */
	public String nextMonth(String month, int m) {
		SimpleDateFormat sdf = new SimpleDateFormat(Const.MON);
		Calendar cal = Calendar.getInstance();
		try {
			month = isEmpty(month) ? today(Const.MON) : month;
			cal.setTime(sdf.parse(month)); // 设置当前月份
		} catch (ParseException e) {
			error("nextMonth", month);
		}
		cal.add(Calendar.MONTH, m);
		return sdf.format(cal.getTime());
	}

	/**
	 * 获得下个月期间
	 */
	public String nextMonth(String month) {
		return nextMonth(isEmpty(month) ? today(Const.MON) : month, 1);
	}

	/**
	 * 获得上个月期间，可以用替代方法：nextMonth(month, -m)
	 */
	public String prevMonth(String month, int m) {
		return nextMonth(month, -m);
	}

	/**
	 * 获得上个月期间
	 */
	public String prevMonth(String month) {
		return nextMonth(month, -1);
	}

	/**
	 * 获取host、Port
	 * 
	 * @param uri
	 * @return String[]
	 */
	String[] getHost(String uri) {
		int idx = uri.indexOf("://");
		uri = idx > 0 ? uri.substring(idx + 3) : uri;
		idx = uri.indexOf('/');
		uri = idx > 0 ? uri.substring(0, idx) : uri;
		return uri.split(Const.COLON);
	}

	/**
	 * 获取请求URI参数，返回值带了斜线，如：uri="/wsSaveTest1"
	 */
	public String getRequestURI(HttpServletRequest req) {
		// 获取数据的示例：“/ws/wsSaveTest1”
		String uri = req.getServletPath();
		// 返回数据的示例：“/wsSaveTest1”
		return uri.substring(uri.lastIndexOf('/'));
	}

	/**
	 * 获取域名http://xxx.ps1.cn:8080/xxx，放置cookie使用
	 */
	private String getDomain(HttpServletRequest req) {
		// 泉州同时包含域名、IP登录，需要动态获取。
		String ref = req.getHeader("Referer");
		if (isEmpty(ref))
			ref = req.getRequestURL().toString();
		LOG.debug("getDomain...{}", ref);
		// 获取的结果如：{"xxx.ps1.cn","8080"}
		String host = getHost(ref)[0];
		String[] arr = host.split("\\.");
		// 返回结果如：ps1.cn
		return arr.length == 3 ? arr[1] + Const.DOT + arr[2] : host;
	}

	/**
	 * 设置cookies
	 */
	public void setCookies(HttpServletRequest req, HttpServletResponse rsp,
			Map<String, String> map) {
		if (isEmpty(map))
			return;
		String domain = getDomain(req);
		for (Map.Entry<String, String> ent : map.entrySet()) {
			String k = ent.getKey();
			// 不缓存证书信息
			if (k.equals(ConfUtil.CERTID))
				continue;
			String v = ent.getValue();
			Cookie c = new Cookie(k, Digest.urlEncode(v));
			setCookiePath(c, domain);
			// 若需要安全保护：c.setHttpOnly(true);
			rsp.addCookie(c);
		}
	}

	/**
	 * 获取所有Cookie键值对
	 */
	public Map<String, String> getCookies(HttpServletRequest req) {
		Map<String, String> map = new HashMap<>();
		Cookie[] cookies = req.getCookies();
		if (cookies != null)
			for (Cookie c : cookies) {
				map.put(c.getName(), Digest.urlDecode(c.getValue()));
			}
		return map;
	}

	/**
	 * 获取指定的Cookie
	 */
	public String getCookie(HttpServletRequest req, String key) {
		return getCookies(req).get(key);
	}

	/**
	 * 刪除指定的Cookie
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	public void delCookie(HttpServletRequest req, HttpServletResponse rsp,
			String key) {
		Cookie[] cookies = req.getCookies();
		if (cookies != null)
			for (Cookie c : cookies) {
				if (c.getName().equals(key)) {
					setCookiePath(c, getDomain(req));
					c.setMaxAge(0); // 立即销毁cookie
					rsp.addCookie(c);
					break;
				}
			}
	}

	/**
	 * 清除全部Cookie的键值
	 */
	public void clearCookies(HttpServletRequest req, HttpServletResponse rsp) {
		Cookie[] cookies = req.getCookies();
		if (cookies != null) {
			String domain = getDomain(req);
			for (Cookie c : cookies) {
				setCookiePath(c, domain);
				c.setMaxAge(0); // 立即销毁cookie
				rsp.addCookie(c);
			}
		}
	}

	/**
	 * 设置CookiePath
	 */
	private void setCookiePath(Cookie c, String domain) {
		c.setDomain(domain);
		c.setPath("/");
	}

	/**
	 * 获取服务端本地的IP地址
	 */
	public String getLocalIp() {
		try {
			return InetAddress.getLocalHost().getHostAddress();
		} catch (UnknownHostException e) {
			LOG.error("getLocalIp...{}", e.getMessage());
			return "127.0.0.1";
		}
	}

	/**
	 * 获取用户请求者的IP地址
	 */
	public String getClientIp(HttpServletRequest req) {
		String ip = req.getHeader("X-Forwarded-For");
		LOG.debug("X-Forwarded-For...{}", ip);
		if (isEmpty(ip) || "unknown".equalsIgnoreCase(ip)) {
			ip = req.getRemoteAddr();
			if (isEmpty(ip))
				return "";
		}
		// 多个代理多个IP以逗号分割，第一个IP为客户端真实IP
		int p = ip.indexOf(',');
		return p > 0 ? ip.substring(0, p) : ip;
	}

	/**
	 * 浏览器请求的默认语言（国家CN）
	 */
	public String getLocale(HttpServletRequest req) {
		// 标准输出格式（这里不添加下划线符号“_”，在i18n方法中再单独处理）
		return req.getLocale().getLanguage();
	}

	/**
	 * 构建URI参数
	 */
	public String buildUri(Map<String, String> params) {
		StringBuilder sb = new StringBuilder();
		if (params != null) {
			for (Map.Entry<String, String> e : params.entrySet()) {
				if (isEmpty(e.getValue()) || isEmpty(e.getKey()))
					continue;
				if (sb.length() > 0)
					sb.append("&");
				String val = Digest.urlEncode(e.getValue());
				sb.append(e.getKey()).append(Const.EQU).append(val);
			}
		}
		return sb.toString();
	}

	/**
	 * 判断字符数组，不为空
	 *
	 * @param values 字符数组
	 * @return true or false
	 */
	public boolean notEmpty(String[] values) {
		if (values == null || values.length == 0)
			return false;
		for (int i = 0; i < values.length; i++) {
			if (isEmpty(values[i]))
				return false;
		}
		return true;
	}

	/**
	 * 数字步进，按step累加
	 */
	public int toStep(int min, int max, int step) {
		int num = min + step;
		return num > max ? max : num;
	}

	/**
	 * 关闭对象
	 */
	public void close(Closeable[] closeables) {
		if (closeables != null) {
			for (Closeable closeable : closeables) {
				close(closeable);
			}
		}
	}

	/**
	 * 关闭对象
	 */
	public void close(Closeable closeable1, Closeable closeable2) {
		close(new Closeable[] { closeable1, closeable2 });
	}

	/**
	 * 关闭对象
	 */
	public void close(Closeable closeable) {
		try {
			if (closeable != null)
				closeable.close();
		} catch (Exception e) {
			closeable = null;
			LOG.error("close...{}", e.getMessage());
		}
	}

	/**
	 * 分变成带千分位格式的金额（元），四舍五入
	 */
	public String toYuan(String money) {
		return toYuan(money, null);
	}

	/**
	 * 分变成带千分位格式的金额（元），四舍五入
	 */
	public String toYuan(String money, String blank) {
		try {
			long l = isInteger(money) ? Long.parseLong(money) : 0;
			if (blank != null && l == 0)
				return blank;
			return new DecimalFormat("#,##0.00").format(l / 100.0);
		} catch (Exception e) {
			error("toYuan", money);
		}
		return blank == null ? "0.00" : blank; // 理论上也不会再执行到这里
	}

	/**
	 * 带千分位格式的金额（元）变成分，四舍五入
	 */
	public String toCent(String money) {
		return toCent(money, null);
	}

	/**
	 * 带千分位格式的金额（元）变成分，四舍五入
	 */
	public String toCent(String money, String blank) {
		String val = Const.STR_0;
		try {
			double d = 0.0;
			if (!isEmpty(money)) {
				money = money.replace(Const.COMMA, "");
				if (isNumeric(money))
					d = Double.valueOf(money) * 100;
			}				
//			double d = isNumeric(money) ? Double.valueOf(money) * 100 : 0.0;
			val = new DecimalFormat("#0").format(d);
		} catch (Exception e) {
			error("toCent...{}", money);
		}
		return blank != null && isNil(val) ? blank : val;
	}

	/**
	 * 人民币汉字大写，传入数值要小于1万亿元
	 */
	public String toRmb(String money) {
		return isInteger(money) ? toRmb(Long.parseLong(money)) : "";
	}

	/**
	 * 人民币汉字大写，传入数值可大于1万亿元
	 */
	public String toRmb(long n) {
		String[] digit = { "零", "壹", "贰", "叁", "肆", "伍", "陆", "柒", "捌", "玖" };
		String[][] units = { { "分", "角", "元", "万", "亿", "万" },
				{ "", "拾", "佰", "仟" } };
		String s = "", head = n < 0 ? "负" : "";
		n = Math.abs(n);
		for (int i = 0; i < 2 && n > 0; i++) {
			s = digit[(int) (n % 10)] + units[0][i] + s;
			n = n / 10;
		}
		for (int i = 2; i < units[0].length && n > 0; i++) {
			String p = "";
			for (int j = 0; j < units[1].length && n > 0; j++) {
				p = digit[(int) (n % 10)] + units[1][j] + p;
				n = n / 10;
			}
			p = p.replaceAll("(零.)*零$", "").replaceAll("^$", "零");
			s = p + units[0][i] + s;
		}
		s = s.replaceAll("(零.)*零元", "元").replaceAll("(零.)+", "零");
		return head + s.replaceAll("零$", "整");
	}

    /**
     * 生成树形结构的叶子项目编号
     *
     * @param list 已有的叶子项目列表
     * @param key 项目主键
     * @param pid 父级项目编号
     * @param bitW 单层项目编号的位数
     */
	public String newLeafId(Object pid, List<Map<String, String>> list,
			String key, int bitW) {
		if (isEmpty(list))
			return idCode(pid, 1, bitW);
		// 获取最大编号
		int len = list.size();
		String maxId = list.get(len - 1).get(key);
		// 获取当前剔除父级编号的最大编号
		maxId = maxId.substring(maxId.length() - bitW);
		// 没有断号，直接返回
		if (Integer.parseInt(maxId) == len)
			return idCode(pid, len + 1, bitW);
		// 遍历找出断号
		for (int i = 0; i < len; i++) {
			String itemId = list.get(i).get(key);
			itemId = itemId.substring(itemId.length() - bitW);
			if (Integer.parseInt(itemId) - i > 1)
				return idCode(pid, i + 1, bitW);
		}
		// 正常情况，理论上不可能执行到此处
		return idCode(pid, len + 1, bitW);
	}

	/**
	 * 占位符处理：placeHolder
	 * @param opr 云算符号
	 */
	public String pHolder(Object key, String opr) {
		return "{" + key + "} " + opr;
	}

}
