package cn.ps1.aolai.service;

import java.io.PrintWriter;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import cn.ps1.aolai.entity.Action;
import cn.ps1.aolai.entity.Duty;
import cn.ps1.aolai.entity.User;
import cn.ps1.aolai.utils.ConfUtil;
import cn.ps1.aolai.utils.Const;
import cn.ps1.aolai.utils.Digest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * 提供第三方对接（微信相关的登录认证）安全服务认证
 * 
 * @author Fulin
 * @since  1.7 $Date: 2020.6.17
 * @version 1.0
 * 
 */

@Service
public class ThirdService {

	private static Logger LOG = LoggerFactory.getLogger(ThirdService.class);

	@Autowired
	private AolaiService aolai;
	@Autowired
	private RedisService redis;
	@Autowired
	private UtilsService utils;

	/**
	 * 支持第三方（如微信）快速授权登录
	 * 
	 * @return boolean 成功与否
	 */
	public boolean thirdLogin(HttpServletRequest req, HttpServletResponse rsp,
			Map<String, String> cookies) {
		// 微信自动登录
		if (cookies.containsKey("bindId") && cookies.containsKey(User.ID)) {
			/*
			Map<String, String> map = wxLogin(req, rsp, cookies);
			if (map != null) {
				// 生成Token，返回前端并写到Cookie中
				map = utils.newToken(req, rsp, map.get("bindUid"));
				redis.setToken(map);
				return true;
			}
			*/
		}
		return false;
	}

	/**
	 * 校验本应用接口证书
	 */
	private boolean verifyAppKey(String K) {
		return ConfUtil.APP_KEY.equals(K) || ConfUtil.ENC_KEY.equals(K);
	}

	/**
	 * 校验11位（或10位、8位）通行证 k码（相当于appId）
	 * <p>
	 * 接口：getTicket，参数：{ticket="LCEvNBmyjO", k="www.ps1.cn"} <br>
	 * signIn加密参数：{jsonstr:{user,pass,spec=密钥,ticket=通行证}, k="公钥"}
	 */
	public boolean verifySecret(HttpServletRequest req) {
		// 来自前台请求 signIn的json对象或加密字符串
		String jsonStr = req.getParameter(ConfUtil.JSONSTR);
		/**
		 * 如getTicket的参数：{ticket=BNQTJMFP, k=www.ps1.cn}<br>
		 * 这里是checkToken失败后，未携带公钥“k”返回失败
		 */
		String K = req.getParameter(ConfUtil.CERT_K);

		if (verifyAppKey(K)) {
			/**
			 * 这里处理未加密参数<br>
			 * 处理getTicket： k="www.ps1.cn" 或 MD5("www.ps1.cn")<br>
			 * Aolai本平台的接口调用时默认携带参数：k="www.ps1.cn"
			 */
			// 可直接跳过处理非 jsonStr格式参数
			if (jsonStr == null) return true;
			// 鉴权成功：设置未加密的 json对象
			return setReqAttr(req, utils.json2Map(jsonStr));
		}
		/** 验证ticket */
		String ticket = req.getParameter(ConfUtil.TICKET);
		if (ticket != null) {
			/**
			 * 这里处理三方接口调用SM4对称加密的参数
			 */
			return verifyApiKey(req, jsonStr, ticket);
		} else if (K == null) {
			return false;
		}

		/**
		 * 这里处理signIn携带的加密参数<b>
		 * {jsonStr:{user,pass,spec,ticket="LCEvNBmyjO"},k=66位}
		 * <p>
		 * 第三方应用跨平台的接口调用：<br>
		 * 根据携带的动态公钥（certKey） k=66位字符串（certKey）<br>
		 * 获取缓存的动态私钥 priKey=64位字符串
		 */
		// 用携带的66位公钥获取缓存的64位动态私钥
		String priKey = redis.getCertKey(K);

		// 用私钥（priKey）解码signIn携带的参数
		// 这里必须携带请求参数jsonStr，否则无效
		if (priKey != null && jsonStr != null) {
			jsonStr = Digest.sm2Decrypt(jsonStr, priKey);
			if (jsonStr != null)
				return setReqAttr(req, utils.json2Map(jsonStr));
		}

		LOG.warn("Auth failed > {} : {}", K, ConfUtil.APP_KEY);
		// 权限验证失败
		return false;
	}

	/**
	 * 第三方接口校验：获取32位对称证书（spec）对参数解码处理
	 */
	private boolean verifyApiKey(HttpServletRequest req, String jsonStr,
			String ticket) {
		// 必须开放三方调用接口时才能用
		if (!ConfUtil.IS_API_OPEN || jsonStr == null)
			return false;
		String spec = redis.get(Const.RDS_APPID + ticket);
		if (spec == null) {
			/** 根据唯一的 ticket获取各自的 certKey通行证 */
			Object res = ConfUtil.invoke(null, "appCert", ticket);
			Map<String, String> cert = utils.obj2Map(res);
			spec = cert.get("certKey");
			if (spec == null)
				return false;
			int sec = getSecond(req.getParameter("sec"));
			redis.set(Const.RDS_APPID + ticket, spec, sec); // 缓存
		}
		/**
		 * 第三方接口调用必须携带请求参数jsonStr，否则范围太大<br>
		 * 用传递的spec参数进行解密校验，如果解密失败，说明spec参数无效
		 */
		jsonStr = Digest.sm4Decrypt(jsonStr, spec);
		Map<String, Object> params = utils.json2Map(jsonStr);
		if (params.isEmpty())// || !spec.equals(params.get("spec")))
			return false;

		// 缓存请求参数
		return setReqAttr(req, params);
	}

	/**
	 * 公共方法： 梳理请求参数
	 */
	private boolean setReqAttr(HttpServletRequest req,
			Map<String, Object> params) {
		if (params.isEmpty()) {
			Enumeration<String> map = req.getParameterNames();
			while (map.hasMoreElements()) {
				String key = map.nextElement();
				params.put(key, req.getParameter(key));
			}
		} else if (params.containsKey(Const.BASE)) {
			// 禁止从前端页面携带base参数，有则删除
			// 否则可能导致越权
			params.remove(Const.BASE);
		}

		// 拼接数据库或表名，dbid必须为数字或字母或下划线
		Object dbid = params.get(ConfUtil.BASE_DBID);
		if (dbid != null && Pattern.matches("\\w+", String.valueOf(dbid))) {
			String base = ConfUtil.BASE_NAME + dbid;
			params.put(Const.BASE, base + ConfUtil.BASE_DOT);
		}

		// 先以传递的参数为主，再获取当前用户（user）设定的默认语言
		// 如果从前端传入i18n，则必须是I18N_LOCALES中的值
		if (!utils.findIn(ConfUtil.I18N_LOCALES, params.get(Const.I18N)))
			params.put(Const.I18N, getUserLang(req));

		// 整理完毕后缓存到Attribute中，在拦截器记录日志时使用
		req.setAttribute(Const.JSON, params);

		LOG.debug("setReqAttr...{}", params);
		return true;
	}

	/**
	 * 操作用户的默认语言（国家CN）
	 */
	private String getUserLang(HttpServletRequest req) {
		// 当前用户（user）设置的默认语言，理论上每个用户都设了默认语言
		String lang = utils.userSelf(req).get(User.LANG);
		// 一般浏览器请求语言为小写: zh
		if (utils.isEmpty(lang)) {
			// signIn用户还未登录成功时，用req的语言值
			lang = utils.getLocale(req).toUpperCase();
		}
		return lang;
	}

	/**
	 * 对访问用户进行鉴权，如果是非法登录，拒绝访问
	 * <p>
	 * {userId=BMr2LhF2LhF2LhF=,token=QKDMZJXWSFV, ...} <br>
	 * token信息：{userId,token,certId="29809e2c85e839a63b5928d56dbc5f8e"}
	 */
	public boolean authAccess(HttpServletRequest req, HttpServletResponse rsp,
			Map<String, String> token) throws Exception {
		// 通过token获取当前会话中的登录用户信息
		String userId = token.get(User.ID);
		Map<String, String> user = redis.getUserInfo(userId);
		// 用户不存在（或token失效）则鉴权失败，返回false

		if (user == null) {
			redis.clearToken(token);
			return invlidToken(rsp);
		}
		/**
		 * 获取前台的请求参数（同时进行解码处理）<br>
		 * 解码报错时返回null，证书可能过期或失效，需重新登录认证<br>
		 * 获取用户后，用32位对称证书（spec）解码处理
		 */
		String spec = token.get(ConfUtil.CERTID);
		Map<String, Object> params = decryptParams(req, user, spec);
		if (params == null)
			return invlidToken(rsp);

		// 其他应用访问公共接口（如culai、xcell）时，
		// 需要携带appCode，否则不能通过鉴权
		Object appCode = params.get(ConfUtil.APPCODE); // 前台携带appCode
		String svrApp = ConfUtil.APP_CODE; // 后台服务appCode
		// 当前访问接口（路由）
		String uri = utils.getRequestURI(req); // 例如：'/generalLedger'

		// 例如：访问接口checkToken或getMenuRole时，svrApp="M", appCode="ABC"
		// 前台携带参数为空时，使用后台的参数
		if (utils.isEmpty(appCode)) {
			// 鉴权：根据用户岗位（dutyId）鉴权，验证接口（uri）的访问权限
			// 要求为每个用户都要设置一个culai、xcell等公共接口的访问权限
			if (svrApp.length() > 0) {
				params.put(ConfUtil.APPCODE, svrApp);
				String role = trustedRole(params, user, uri);
				LOG.debug("trustedRole...{}", role);
				if (utils.isEmpty(role)) {
					// 返回false，无权限：denyAccess
					return denyAccess(rsp);
				}
			}
		} else if (!havePermit(svrApp, String.valueOf(appCode), uri)) {
			/**
			 * 开放接口：根据配置好的公共服务接口进行鉴权处理<br>
			 * svrApp 存在时，验证是否有访问权限，否则跳过此校验规则
			 */
			return denyAccess(rsp); // 返回false，无权限：denyAccess
		}
		LOG.debug("request from {}.{}", uri, params);

		// 鉴权：并缓存用户信息、传递参数
		if (!checkParams(params, uri.substring(1))) // 先过滤掉斜线
			return invlidParams(rsp); // 返回false，无效参数

		/** * 以下将对应用（app.code）、账套，进行鉴权处理 * */

		// 如果携带dbid参数时，需要判断该用户是否有账套授权
		String dbKey = ConfUtil.BASE_DBID;
		if (svrApp.length() > 0 && params.containsKey(dbKey)) {
			// 用户账套权限如： { App1:"101",App2:"101",App3: "64" }
			Map<String, String> userBase = utils.json2Map(user.get(User.BASE));

			String dbid = userBase.get(svrApp);
			// 如果获取userBase中dbid为“”或“0”，则参数有效
			// 访问主数据库时可忽略校验，当dbid="0"时也忽略
			if (!utils.isEmpty(dbid) && !Const.STR_0.equals(dbid)
					&& !dbid.equals(params.get(dbKey)))
				return invlidParams(rsp); // 返回false，鉴权失败
		}

		// 鉴权成功：设置解码后的属性参数“json”
		return setReqAttr(req, params);
	}

	/**
	 * 获取HTTP请求从前端传递来的参数并解密
	 * 
	 * @param req 请求参数
	 * @param user 当前登录用户的userInfo
	 * @param cert 默认为11位或10位通行证certId
	 */
	private Map<String, Object> decryptParams(HttpServletRequest req,
			Map<String, String> user, String spec) {
		// 缓存用户基本信息
		req.setAttribute(User.KEY, user);
		req.setAttribute(User.ID, user.get(User.ID));
		req.setAttribute(ConfUtil.CERTID, spec);

		// 解码从前端传递来的jsonStr参数
		return decryptParams(req.getParameter(ConfUtil.JSONSTR), spec);
	}

	/**
	 * 解码从前端传递来的jsonStr参数
	 */
	private Map<String, Object> decryptParams(String jsonStr, String spec) {
		// 默认请求参数都加密（不忽略），需要在此解码
		if (!ConfUtil.IS_ENC_OMIT && !utils.isEmpty(jsonStr)) {
			LOG.debug("decryptParams...{}", jsonStr);
			jsonStr = Digest.sm4Decrypt(jsonStr, spec);
			// 解码报错时返回null，证书可能过期或失效，需重新登录认证
			if (jsonStr == null)
				return null;
		}
		// 转为对象并返回，这里返回非空（!=null）对象
		return utils.json2Map(jsonStr);
	}

	/**
	 * 此处统一处理参数校验，根据URI校验参数是否有效
	 * <p>
	 * 例如默认: valid.must=dbid; 不带斜线的路由如：uri="wsSaveTest1"
	 * 
	 * @param params
	 * @param uri 不带斜线的路由
	 */
	private boolean checkParams(Map<?, ?> params, String uri) {
		String keys = ConfUtil.VALID_MUST + ConfUtil.getValid(uri);
		String[] keyArr = keys.split(ConfUtil.COMMA);
		return keys.length() == 0 || utils.availParams(params, keyArr);
	}

	/**
	 * 开放接口：根据配置好的公共服务接口进行鉴权处理
	 * <p>
	 * 系统预设的开放接口，无需鉴权（默认三方应用都可访问） <br>
	 * 另外：多个应用之间接口鉴权须通过isSecretKey实现鉴权 
	 */
	private boolean havePermit(String srvApp, String appCode, String uri) {

		// 正常应用都应鉴权：如果后台应用未配置app.code，则当前应用无需账套鉴权
		if (srvApp.length() == 0) return true;

		// 获取通行证权限，如：a.M.FF
		// 注意：新增的开放接口要先清除缓存（appCode=M）才能用
		String[] rks = {Const.RDS_APPID, srvApp, Const.DOT, appCode};
		String rk = utils.arr2Str(rks);
		Map<String, String> permit = redis.hmget(rk);
		if (permit == null) {
			// 访问culai公共服务平台时，需要校验是否具有访问权限
			permit = getActionPermit(srvApp, appCode);
			LOG.debug("permit...{}", permit);

			// 无可用的开放接口
			if (permit.isEmpty()) return false;
			redis.hmset(rk, permit, Const.TWO_HH);
		}
		LOG.debug("havePermit={}", permit);

		return permit.containsKey(uri); // 判断是是否有效
	}

	/**
	 * 这里访问公共服务平台，获取通行证权限一览
	 */
	private Map<String, String> getActionPermit(String srvApp, String appCode) {
		Map<String, Object> where = new HashMap<>();
		where.put(Action.APP, srvApp);
		// 这里访问公共服务平台：默认都是访问标记为‘0’的公共接口
		where.put(Action.MARK, Const.STR_0); // 分类标记： 1.私有资源 0.公共资源
		where.put(Action.STATE, Const.STR_1); // 有效状态： 1.有效 0.禁用

		// 增加api的对于三方应用的权限控制
		if (ConfUtil.API_PERMIT) {
			// 这里必须是数组，数量与{V}一致
			String[] V = { Const.VAL_1, Const.VAL_1 };
//			String[] arr = { "({", Action.PERMIT, "} is NULL OR {",
//					Action.PERMIT, "}->>'$.", appCode, "' ={} OR {",
//					Action.PERMIT, "}->>'$.all' ={})" };
			String[] arr = { "({", Action.PERMIT, "} is NULL OR ",
					utils.jsonExt(Action.PERMIT, appCode), "{} OR ",
					utils.jsonExt(Action.PERMIT, "all"), "{})" };
			where.put(utils.arr2Str(arr), V);
		}
		LOG.debug("getActionList...{}", where);

		// 获取通行证
		List<Map<String, String>> list = aolai.findList(Action.TABLE, where);
		return utils.list2Map(list, Action.URI, Action.STATE);
	}

	/**
	 * 根据appCode、岗位（dutyId）鉴权，验证接口（uri）的访问权限
	 * @return String 有权限‘1’，无效token‘’，无权限null
	 */
	private String trustedRole(Map<String, Object> params,
			Map<String, String> user, String uri) {
		// 这里params={appCode:M}。依赖参数： appCode，userComp、userDuty
		// 返回：compId、dutyId
		Map<String, Object> duty = aolai.getRoleParams(params, user);

		// 这里的appCode是应用获取后台的配置参数
		// 用户无权限duty，则 duty.get("appCode")为空，这里用了params的参数
		Object[] keys = { params.get(ConfUtil.APPCODE),
				duty.get(ConfUtil.COMPID), duty.get(Duty.ID) };
		// 正常返回：有权限‘1’
		return redis.getActionRole(utils.join(keys, Const.DOT), uri);
	}

	/**
	 * 验证无效的参数
	 */
	private boolean invlidParams(HttpServletResponse rsp) throws Exception {
		return invalidResult(rsp, Const.STR_2);
	}

	/**
	 * 验证无效的token
	 */
	public boolean invlidToken(HttpServletResponse rsp) throws Exception {
		return invalidResult(rsp, "3");
	}

	/**
	 * 越权访问或无权限
	 */
	private boolean denyAccess(HttpServletResponse rsp) throws Exception {
		return invalidResult(rsp, "5");
	}

	/**
	 * 验证无效的token
	 */
	private boolean invalidResult(HttpServletResponse rsp, String status)
			throws Exception {
		rsp.setContentType("application/json;charset=UTF-8");
		rsp.setCharacterEncoding(Const.UTF8);
		PrintWriter out = rsp.getWriter();
		// token失效，状态status="3"
		out.write(utils.obj2Str(utils.result(status)));
		out.flush();
		out.close();
		return false;
	}

	/**
	 * 从前端传递的一个缓存时效，默认缓存2小时，测试可改sec值
	 */
	private int getSecond(String sec) {
		// 前端传递的一个缓存时效，默认缓存10小时
		return utils.isInteger(sec) ? Integer.parseInt(sec) : ConfUtil.CACHE_TIME;
	}

}
