package cn.ps1.aolai.service;

import java.io.*;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpMessage;
import org.apache.http.NameValuePair;
import org.apache.http.StatusLine;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.ps1.aolai.utils.ConfUtil;
import cn.ps1.aolai.utils.Const;
import cn.ps1.aolai.utils.Digest;
import cn.ps1.aolai.utils.FailedException;

/**
 * 通过 http服务远程调用的业务操作
 *
 * @author Fulin
 * @version 1.0
 * @since 1.7 $Date: 2019.12.21
 */

@Service
public class HttpsService {

	private static Logger LOG = LoggerFactory.getLogger(HttpsService.class);

	private Map<String, String> headers = new HashMap<>();
	{
		headers.put("Accept", "*/*");
		headers.put("Cache-Control", "no-cache");
		headers.put("Connection", "keep-alive");
//		headers.put("Accept-Encoding", "gzip, deflate, br");
//		headers.put("Accept-Language", "zh-CN, zh, en-US");
//		headers.put("User-Agent", "Chrome/97.0.4692.71");
	}
	// 禁止了自动重定向
	private static RequestConfig DEF_CONF = null;
	// 主应用服务的请求地址
	private static String MAIN_URL = null;

	// 默认配置
	private RequestConfig requestConf() {
		if (DEF_CONF == null)
			DEF_CONF = RequestConfig.custom().setRedirectsEnabled(false)
					.setConnectTimeout(ConfUtil.HTTP_TIMEOUT)
					.setConnectionRequestTimeout(ConfUtil.HTTP_TIMEOUT)
					.setSocketTimeout(ConfUtil.HTTP_TIMEOUT).build();
		return DEF_CONF;
	}
	/** 缓存响应头信息 */
	public Header[] getHeaders = null;

	@Autowired
	UtilsService utils;

	/**
	 * 返回当前响应的Cookies
	 */
	public Set<String> getCookies() {
		// 浏览器向服务器提交携带“Cookie”，服务器向浏览器设置“Set-Cookie”
		return getHeader("Set-Cookie");
	}

	/**
	 * 返回当前响应的header信息
	 */
	public Set<String> getHeader(String name) {
		Set<String> values = new HashSet<>();
		for (Header header : getHeaders) {
			if (header.getName().equals(name)) {
				values.add(header.getValue());
			}
		}
		return values;
	}

	/**
	 * Http 处理Get请求的响应结果
	 * 
	 * @param url
	 * @param params
	 * @return String
	 */
	public String httpGet(String url, Map<String, String> params) {
		return httpGet(url, null, params);
	}

	/**
	 * Http 处理Get请求的响应结果
	 * 
	 * @param url
	 * @param cookies
	 * @return String
	 */
	public String httpGet(String url, Set<String> cookies) {
		return httpGet(url, cookies, new HashMap<>());
	}

	/**
	 * Http 处理Get请求的响应结果
	 * 
	 * @param url
	 * @param cookies
	 * @param params
	 * @return String
	 */
	public String httpGet(String url, Set<String> cookies,
			Map<String, String> params) {
		CloseableHttpClient client = HttpClientBuilder.create().build();
		CloseableHttpResponse resp = null;
		try {
			// 拼接Get请求参数
			HttpGet httpGet = new HttpGet(buildUrl(url, params));
			// 配置信息头
			httpGet.setConfig(requestConf());
			setHeaders(httpGet, url, cookies, null);
			// 发送请求
			resp = client.execute(httpGet);
			return getResult(doResponse(resp));
		} catch (Exception e) {
			LOG.error("httpGet...{}", url);
			throw new FailedException("httpFailed");
		} finally {
			utils.close(resp, client);
		}
	}

	/**
	 * Http 处理Post请求
	 */
	public String httpPost(String url, String jsonStr, String key) {
		return httpPost(url, withKey(jsonStr, key));
	}

	/**
	 * Http 处理Post请求
	 */
	public String httpPost(String url) {
		return httpPost(url, null, new HashMap<>(), Const.UTF8);
	}

	/**
	 * Http 处理Post请求
	 * 
	 * @param url
	 * @param cookies
	 * @return String
	 */
	public String httpPost(String url, Set<String> cookies) {
		return httpPost(url, cookies, new HashMap<>(), Const.UTF8);
	}

	/**
	 * Http 处理Post请求的响应结果
	 * 
	 * @param url 服务地址
	 * @param params 请求参数
	 * @return String 返回数据
	 */
	public String httpPost(String url, Map<String, String> params) {
		return httpPost(url, null, params, Const.UTF8);
	}

	/**
	 * Http 处理Post请求
	 * 
	 * @param url 服务地址
	 * @param cookies 携带Cookies信息
	 * @param params 请求参数
	 * @return String 返回数据
	 */
	public String httpPost(String url, Set<String> cookies,
			Map<String, String> params) {
		return httpPost(url, cookies, params, Const.UTF8);
	}

	/**
	 * Http 处理Post请求
	 * 
	 * @param url 服务地址
	 * @param cookies 携带Cookies信息
	 * @param params 请求参数
	 * @param encoding 编码
	 * @return String 返回数据
	 */
	public String httpPost(String url, Set<String> cookies,
			Map<String, String> params, String encoding) {
		return httpPost(url, cookies, params, encoding, null);
	}

	/**
	 * Http 处理Post请求，扩展携带了自定义Headers
	 * @param url 服务地址
	 * @param cookies 携带Cookies信息
	 * @param params 请求参数
	 * @param encoding 编码
	 * @param exHeadres 扩展头信息
	 * @return String 返回数据
	 */
	public String httpPost(String url, Set<String> cookies,
			Map<String, String> params, String encoding,
			Map<String, String> exHeadres) {
		try {
			// 设置携带的请求参数
			List<NameValuePair> nvpList = setParams(params);
			HttpEntity entity = new UrlEncodedFormEntity(nvpList, encoding);
			return httpPost(url, cookies, entity, exHeadres);
		} catch (Exception e) {
			throw new FailedException("httpFailed");
		}
	}

	/**
	 * Http 处理Post请求
	 */
	public String httpPost(String url, Set<String> cookies, HttpEntity entity,
			Map<String, String> exHeaders) {
		CloseableHttpClient client = HttpClientBuilder.create().build();
		CloseableHttpResponse resp = null;
		try {
			LOG.info("> httpPost...{}", url);
			// 配置Post请求参数
			HttpPost httpPost = new HttpPost(url);
			// 配置信息头
			httpPost.setConfig(requestConf());
			setHeaders(httpPost, url, cookies, exHeaders);
			// 设置携带的请求参数
			httpPost.setEntity(entity);
			// 发送请求
			resp = client.execute(httpPost);
			return getResult(doResponse(resp));
		} catch (Exception e) {
			LOG.error("httpPost...{}", url);
			throw new FailedException("httpFailed");
		} finally {
			utils.close(resp, client);
		}
	}

	/**
	 * Http 处理Post请求
	 */
	public String httpPost(String url, Set<String> cookies, HttpEntity entity) {
		return httpPost(url, cookies, entity, null);
	}

	/**
	 * JsonPost, 纯粹以json格式发送数据
	 */
	public String postData(String url, String json) {
		// 设置请求参数
		return httpPost(url, null, stringEntity(json));
	}

	/**
	 * 设置请求参数
	 */
	private StringEntity stringEntity(String json) {
		try {
			// 设置请求参数
			StringEntity entity = new StringEntity(json, StandardCharsets.UTF_8);
			entity.setContentType("application/json");
			// entity.setContentEncoding(Const.UTF8);
			return entity;
		} catch (Exception e) {
			LOG.error("stringEntity...{}", e.getMessage());
			throw new FailedException();
		}
	}

	/**
	 * HTTP响应
	 * 
	 * @param resp
	 * @return HttpEntity
	 */
	private HttpEntity doResponse(CloseableHttpResponse resp) {
		StatusLine sLine = resp.getStatusLine();
		// 200=正常返回，302=网址跳转重定向
		if (sLine.getStatusCode() == 200 || sLine.getStatusCode() == 302) {
			getHeaders = resp.getAllHeaders();
			return resp.getEntity();
		} else {
			LOG.error("> doResponse...{}", sLine);
			throw new FailedException("httpFailed");
		}
	}

	/**
	 * 对HTTP响应内容的处理
	 * 
	 * @param entity
	 * @return String
	 */
	private String getResult(HttpEntity entity) throws Exception {
		String result = null;
		if (entity != null) {
			result = EntityUtils.toString(entity, StandardCharsets.UTF_8);
			if (result.length() > 200)
				LOG.debug("{}{}", result.substring(0, 200), "...");
			else
				LOG.debug(result);
		}
		return result;
	}

	/**
	 * 拼接请求参数
	 * 
	 * @param url
	 * @param params
	 * @return String
	 */
	private String buildUrl(String url, Map<String, String> params) {
		if (url != null) {
			String str = utils.buildUri(params);
			// 检查最后一个字符
			boolean islast = url.lastIndexOf('?') == url.length() - 1;
			url += islast ? str : "&" + str;
		}
		return url;
	}

	/**
	 * 设置协议消息标头
	 */
	private void setHeaders(HttpMessage msg, String url, Set<String> cookies,
			Map<String, String> exHeaders) {
		for (Map.Entry<String, String> e : headers.entrySet()) {
			msg.addHeader(e.getKey(), e.getValue());
		}
		if (cookies != null) {
			for (String cookie : cookies)
				msg.addHeader("Cookie", cookie);
			LOG.info("setCookies...{}", cookies);
		}
		// 如果有自定义请求头，则采用自定义请求头
		if (exHeaders != null) {
			for (Map.Entry<String, String> e : exHeaders.entrySet()) {
				msg.addHeader(e.getKey(), e.getValue());
			}
		}
		// 设置域名及端口，或IP地址及端口号
//		msg.addHeader("Host", utils.getHost(url)[0]);
		msg.addHeader("Referer", url); // referer);
		
		// 放入请求唯一链路 traceId
		String tid = MDC.get(ConfUtil.TRACEID);
		// 从上个tid跟踪而来，基本不可能“”
		if (tid != null) {
			msg.addHeader(ConfUtil.TRACEID, tid);
		}
	}

	/**
	 * 创建参数队列
	 */
	private List<NameValuePair> setParams(Map<String, String> params)
			throws UnsupportedEncodingException {
		List<NameValuePair> nvpList = new ArrayList<NameValuePair>();
		if (params != null) {
			for (Map.Entry<String, String> e : params.entrySet()) {
				String key = e.getKey();
				String val = e.getValue();
				if (val != null)
					nvpList.add(new BasicNameValuePair(key, val));
			}
		}
		return nvpList;
	}

	/**
	 * 基于摘要认证的HTTP请求数据下载
	 * 
	 * @param userName
	 * @param password
	 * @param url
	 * @param filePath
	 * @return boolean
	 */
	public boolean digestDownload(String userName, String password, String url,
			String filePath) {
		CloseableHttpClient client = digestHttpClient(userName, password);
		CloseableHttpResponse resp = null;
		try {
			// 请求、响应
			HttpGet request = new HttpGet(url);
			resp = client.execute(request);
			HttpEntity entity = resp.getEntity();
			InputStream ins = entity.getContent();
			FileOutputStream fos = new FileOutputStream(filePath, false);
			int data = ins.read();
			while (data != -1) {
				fos.write(data);
				data = ins.read();
			}
			fos.close();
			EntityUtils.consume(entity);
			return true;
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			utils.close(resp, client);
		}
		return false;
	}

	/**
	 * 基于摘要认证的HTTP请求
	 * 
	 * @param userName
	 * @param password
	 * @param url
	 * @param json
	 * @param method
	 * @return String
	 */
	public String digestJsonRequest(String userName, String password,
			String url, String json, String method) {
		RequestBuilder builder = setBuilder(stringEntity(json), method);
		HttpUriRequest request = builder.setUri(url).build();
		json = digestRequest(userName, password, request);
		LOG.info("> digestJsonRequest...{}", json);
		return json;
	}

	/**
	 * 基于摘要认证的HTTP请求
	 *
	 * @param userName
	 * @param password
	 */
	public String digestRequest(String userName, String password,
			HttpUriRequest uriRequest) {
		CloseableHttpClient client = digestHttpClient(userName, password);
		CloseableHttpResponse resp = null;
		String result = null;
		try {
			// 请求、响应
			resp = client.execute(uriRequest);
			result = getResult(doResponse(resp));
		} catch (Exception e) {
			throw new FailedException("httpFailed");
		} finally {
			utils.close(client, resp);
		}
		return result;
	}

	/**
	 * 设置请求参数
	 */
	private RequestBuilder setBuilder(HttpEntity entity, String method) {
		RequestBuilder request = null;
		if ("POST".equals(method)) {
			request = RequestBuilder.post();
		} else if ("GET".equals(method)) {
			request = RequestBuilder.get();
		} else if ("PUT".equals(method)) {
			request = RequestBuilder.put();
		} else if ("DELETE".equals(method)) {
			request = RequestBuilder.delete();
		}
		// 添加参数
		return request.setEntity(entity);
	}

	/**
	 * 实现HTTP摘要认证
	 */
	private CloseableHttpClient digestHttpClient(String user, String pass) {
		// 摘要认证参数
		UsernamePasswordCredentials creds = new UsernamePasswordCredentials(
				user, pass);
		CredentialsProvider credsProvider = new BasicCredentialsProvider();

		credsProvider.setCredentials(AuthScope.ANY, creds);
		HttpClientBuilder http = HttpClientBuilder.create();
		http.setDefaultCredentialsProvider(credsProvider);

		return http.build();
	}

	/**
	 * 向指定 URL发送GET请求
	 */
	public String urlConnect(String url, Map<String, String> params) {
		return urlConnect(buildUrl(url, params), new byte[0], "GET");
	}

	/**
	 * 向指定 URL发送POST请求
	 */
	public String urlConnect(String url, String str, String method) {
		if (str == null) return null;
		return urlConnect(url, str.getBytes(StandardCharsets.UTF_8), method);
	}

	/**
	 * 通过HTTP方法请求数据
	 */
	public String urlConnect(String url, byte[] bytes, String method) {
		HttpURLConnection conn = null;
		try {
			LOG.debug("urlConnect...{}", url);
			URL httpUrl = new URL(url);
			conn = (HttpURLConnection) httpUrl.openConnection();
			setProperty(conn).setRequestMethod(method);

			conn.getOutputStream().write(bytes); // 输出流

			// PrintWriter out = new PrintWriter(conn.getOutputStream());
			// out.print(s); // 发送请求参数
			// out.flush(); // 输出流的缓冲

			// 发送请求
			if (conn.getResponseCode() == 200 || conn.getResponseCode() == 204) {
				return respAsString(conn.getInputStream());
			} else {
				return respAsString(conn.getErrorStream());
			}
		} catch (Exception e) {
			LOG.error("> urlConnect...{}", url);
			return null;
		}
	}

	/** 以下通过远程调用业务逻辑，获取信息 */

	private HttpURLConnection setProperty(HttpURLConnection conn) {
		requestConf();
		// 设置相应请求时间：120秒
		conn.setConnectTimeout(ConfUtil.HTTP_TIMEOUT);
		// 设置读取超时时间：120秒
		conn.setReadTimeout(ConfUtil.HTTP_TIMEOUT);
		conn.setUseCaches(false); // 默认是true
		conn.setDoOutput(true); // 默认是false
		conn.setDoInput(true); // 默认是true
		for (Map.Entry<String, String> e : headers.entrySet()) {
			conn.setRequestProperty(e.getKey(), e.getValue());
		}
		return conn;
	}

	/**
	 * 读取输入流，返回字符串
	 *
	 * @param inp
	 * @return String
	 */
	private String respAsString(InputStream inp) {
		StringBuilder strBuf = new StringBuilder();
		InputStreamReader isr = null;
		BufferedReader br = null;
		try {
			isr = new InputStreamReader(inp, StandardCharsets.UTF_8);
			br = new BufferedReader(isr);
			String line;
			while ((line = br.readLine()) != null) {
				strBuf.append(line);
			}
		} catch (IOException e) {
			LOG.error("> respAsString...{}", e.getMessage());
		} finally {
			Closeable[] keys = {inp, isr, br};
			utils.close(keys);
		}
		return strBuf.toString();
	}

	/**
	 * 远程post调用，传递params参数的请求
	 */
	public <T> Map<String, T> doPost(String url, Map<?, ?> params, String key) {
		return doPost(url, utils.obj2Str(params), key);
	}

	/**
	 * 传递params参数的远程post请求，返回json字符串
	 * <p>
	 * 参数以jsonStr格式传递，非jsonStr格式时可直接调用httpPost
	 */
	public <T> Map<String, T> doPost(String url, Map<?, ?> params) {
		return doPost(url, utils.obj2Str(params), null);
	}

	/**
	 * 传递params参数的远程post请求，返回json字符串
	 * <p>
	 * 参数以jsonStr格式传递，非jsonStr格式时可直接调用httpPost
	 */
	public <T> Map<String, T> doPost(String url, String jsonStr) {
		return doPost(url, jsonStr, null);
	}

	/**
	 * 传递params参数的远程post请求，返回json字符串，并增加了k值校验
	 * <p>
	 * 参数以jsonStr格式传递，非jsonStr格式时可直接调用httpPost
	 */
	public <T> Map<String, T> doPost(String url, String jsonStr, String key) {
		Map<String, String> params = withKey(jsonStr, key);
		return utils.json2Map(httpPost(url, null, params, Const.UTF8));
	}

	/**
	 * 重新拼接params参数，携带k参数
	 * 
	 * @param jsonStr 请求参数
	 * @param key 携带k参数
	 * @return Map 返回结果
	 */
	private Map<String, String> withKey(String jsonStr, String key) {
		Map<String, String> params = new HashMap<>();
		if (jsonStr != null)
			params.put(ConfUtil.JSONSTR, jsonStr);
		// 这里的k=www.ps1.cn\ZxCrOplK
		params.put(ConfUtil.CERT_K, key == null ? ConfUtil.APP_KEY : key);
		return params;
	}

	/**
	 * 传递params参数的远程post请求，返回byte[]格式数据
	 */
	public byte[] sendPost(String url, String jsonStr) {
		return sendPost(url, withKey(jsonStr, null));
	}

	/**
	 * 传递params参数的远程post请求，返回byte[]格式数据
	 */
	public byte[] sendPost(String url, String jsonStr, String key) {
		return sendPost(url, withKey(jsonStr, key));
	}

	/**
	 * 传递params参数的远程post请求，返回byte[]格式数据
	 */
	public byte[] sendPost(String url, Map<String, String> params) {
		return sendPost(url, null, params);
	}

	/**
	 * 传递params参数的远程post请求，返回byte[]格式数据
	 */
	public byte[] sendPost(String url, Set<String> cookies,
			Map<String, String> params) {
		return sendPost(url, cookies, params, Const.UTF8);
	}

	/**
	 * 传递params参数的远程post请求，返回byte[]格式数据
	 */
	public byte[] sendPost(String url, Set<String> cookies,
			Map<String, String> params, String encoding) {
		CloseableHttpClient client = HttpClientBuilder.create().build();
		CloseableHttpResponse resp = null;
		try {
			// 配置Post请求参数
			HttpPost httpPost = new HttpPost(url);
			// 配置信息头
			httpPost.setConfig(requestConf());
			setHeaders(httpPost, url, cookies, null);

			// 设置携带的请求参数
			List<NameValuePair> nvpList = setParams(params);
			HttpEntity entity = new UrlEncodedFormEntity(nvpList, encoding);
			httpPost.setEntity(entity);

			// 发送请求
			resp = client.execute(httpPost);
			return getStream(doResponse(resp));
		} catch (Exception e) {
			LOG.error("sendPost...{}", url);
			throw new FailedException("httpFailed");
		} finally {
			// resp 在响应方法内关闭
			utils.close(resp, client);
		}
	}

	/**
	 * HTTP响应
	 */
	private byte[] getStream(HttpEntity entity) throws Exception {
		return entity == null ? null : EntityUtils.toByteArray(entity);
	}

	/** 以下通过远程调用业务逻辑，获取信息 */

	/**
	 * 构造函数（为了测试用）
	 */
	public HttpsService() {
		utils = new UtilsService();
	}

	/**
	 * 单点SSO请求单点登录的远程服务地址：http://sso.ps1.cn/culai/s
	 * 
	 * @param ws
	 * @param params
	 * @return Map
	 */
	public <T> Map<String, T> ssoPost(String ws, Map<?, ?> params) {
		return doPost(ssoUrl(ws), utils.obj2Str(params), null);
	}

	/**
	 * 单点SSO请求单点登录的远程服务地址，如：http://sso.ps1.cn/aolai/s
	 * 
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 * @param ws 请求接口服务
	 * @param jsonStr 请求参数
	 * @return String 返回数据
	 */
	public <T> Map<String, T> ssoPost(String ws, String jsonStr) {
		return doPost(ssoUrl(ws), jsonStr, null);
	}

	/**
	 * 转发对SSO服务的请求，如：远程调用用户验证
	 */
	public <T> Map<String, T> ssoPost(HttpServletRequest req) {
		// 带斜线如："/wsSaveTest1"
		return ssoPost(utils.getRequestURI(req), utils.jsonParams(req));
	}

	/**
	 * 转发对SSO服务的请求，如：远程调用用户验证
	 */
	public <T> Map<String, T> forward(HttpServletRequest req) {
		// 带斜线如："/wsSaveTest1"
//		return ssoPost(utils.getRequestURI(req), utils.jsonParams(req));
		String url = mainUrl(utils.getRequestURI(req));
		return utils.obj2Map(forward(req, url));
	}

	/**
	 * 单纯的转发处理：携带原来的 Cookies和参数转发原请求，重新查询获取数据
	 */
	public String forward(HttpServletRequest req, String url) {
		String jsonStr = req.getParameter(ConfUtil.JSONSTR);

//		Map<String, String> params = withKey(jsonStr, null);
		// 原参数不变转发
		Map<String, String> params = new HashMap<>();
		params.put(ConfUtil.JSONSTR, jsonStr);		

		// 浏览器向服务器提交携带“Cookie”，服务器向浏览器设置“Set-Cookie”
		Set<String> cookies = new HashSet<>();
		Enumeration<String> enu = req.getHeaders("Cookie");
		while (enu.hasMoreElements())
			cookies.add(enu.nextElement());
		// 携带“Cookie”转发
		LOG.info("forward with cookies={}", cookies);
		return httpPost(url, cookies, params, Const.UTF8);
	}

	/**
	 * 单纯的转发处理：携带原来的 Cookies和参数转发原请求，重新查询获取数据
	 */
	public <T> Map<String, T> forward(String url, HttpServletRequest req) {
		return utils.json2Map(forward(req, url));
	}

	/**
	 * 获取主应用的服务地址
	 */
	public String ssoUrl(String ws) {
		LOG.debug("ssoUrl...{}{}", ConfUtil.SSO_URL, ws);
		return ConfUtil.SSO_URL + ws;
	}


	/**
	 * 获取主应用的服务地址
	 */
	public String mainUrl(String ws) {
		if (MAIN_URL == null)
			MAIN_URL = ConfUtil.APP_CORE + ConfUtil.APP_MAIN;
		return MAIN_URL + ws;
	}

	/**
	 * 以下调用当前应用管理服务，获取信息
	 */
	public <T> Map<String, T> corePost(String ws, Map<?, ?> params) {
		if (ConfUtil.APP_SPEC.length > 1) {
			String res = httpPost(ConfUtil.APP_SPEC, mainUrl(ws), params);
			return utils.obj2Map(res);
		}
		return doPost(mainUrl(ws), params);
	}

	/**
	 * 调用基于Aolai平台的远程服务，返回结果<br>
	 */
	public String httpPost(Map<String, String> host, Map<?, ?> cond) {
		// 请求参数
		String[] keys = { ConfUtil.TICKET, "spec", "url" };
		for (int i = 0; i < keys.length; i++)
			keys[i] = host.get(keys[i]);
		if (utils.notEmpty(keys)) {
			String[] spec = { keys[0], keys[1] };
			return httpPost(spec, keys[2], cond);
		}
		return null;
	}

	/**
	 * 调用基于Aolai平台的远程服务，返回结果<br>
	 * spec={"VJEFKIGS","12044b9cbfb8eeaeba3e21d1d7a26611"}
	 */
	private String httpPost(String[] spec, String url, Map<?, ?> cond) {
		/** 调用远程服务，返回结果 */
		return httpPost(url, withSpec(utils.obj2Str(cond), spec));
	}

	/**
	 * 重新拼接params参数，携带ticket参数
	 * 
	 * @param jsonStr 请求参数
	 */
	private Map<String, String> withSpec(String jsonStr, String[] spec) {
		Map<String, String> params = new HashMap<>();
		if (jsonStr != null) {// && spec.length > 1) {
			params.put(ConfUtil.TICKET, spec[0]);
			params.put(ConfUtil.JSONSTR, Digest.sm4Encrypt(jsonStr, spec[1]));
		}
		return params;
	}

}
