package cn.ps1.aolai.service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.ps1.aolai.dao.AolaiDao;
import cn.ps1.aolai.entity.Action;
import cn.ps1.aolai.entity.Duty;
import cn.ps1.aolai.entity.Field;
import cn.ps1.aolai.entity.Page;
import cn.ps1.aolai.entity.User;
import cn.ps1.aolai.utils.ConfUtil;
import cn.ps1.aolai.utils.Const;
import cn.ps1.aolai.utils.FailedException;

/**
 * Aolai基础框架相关服务层，基于Mybatis整合的元数据的基础操作类(CRUD)
 * 
 * @author Fulin
 * @version 1.0 $Date: 2019.7.6
 * @since 1.7
 */

@Service
public abstract class GmetaService {

	private static Logger LOG = LoggerFactory.getLogger(GmetaService.class);
	// 元数据表
	static final String META = "GMETA";
	// 排序字段
	static final String META_SORT = "META_SORT";

	// 匹配普通的 {fieldKey} 关键字字段
	static final Pattern P_QUOTE = Pattern.compile("\\{ *\\w+ *\\}"); // 引用
	static final Pattern P_VALUE = Pattern.compile("\\{ *V* *\\}"); // 固定值
	// 正则匹配JSON查询条件，只支持2层："{userDuty}->>'$.appCode.way' ="
//	static final Pattern P_JSON = Pattern.compile("^->>'\\$(\\.\\w+)+'$");
	static final Pattern P_JSON = Pattern.compile("^,'\\$(\\.\\w+)+'\\)$");

	// 这里base为映射中的主键名
	static final String TYPE = "type";
	static final String PKEY = "pkey";
	static final String DUPL = "dupl";

	static final String M_PKEY = "!p"; // mark_Pkey
	static final String M_I18N = "!i"; // mark_I18n
	static final String M_TYPE = "!t"; // mark_Type
	static final String NULL = "null";
	static final String LEFT = "left";
	static final String DEFT = "default";

//	static final String SPACE = " ";
//	static final String BLANK = "";
//	static final String NOT = "!";
	static final String WHERE = "where";
//	static final String JOIN = " join ";
//	static final String ONSTR = " on ";
	static final String ORDER = "order";
	static final String APPCODE = "appCode";

	static final String _S = "_";
	static final String CAST = "CAST(";
	static final String R_PAREN = "')";

	@Autowired
	AolaiDao dao;
	@Autowired
	RedisService redis;
	@Autowired
	UtilsService utils;

	/** 元数据的元数据 */
	static Map<String, String> gmeta = new HashMap<>();
	static {
		gmeta.put(Const.FIELD, "META_FIELD");
		gmeta.put(Const.ALIAS, "META_ALIAS");
		gmeta.put("name", "META_NAME");
		gmeta.put(TYPE, "META_TYPE");
		gmeta.put(NULL, "META_NULL");
		gmeta.put(Const.I18N, "META_I18N");
		gmeta.put(PKEY, "META_PKEY");
		gmeta.put("width", "META_WIDTH");
		gmeta.put(DEFT, "META_DEFAULT");
		gmeta.put("sort", META_SORT);
	}

	/*********************************************************
	 * 往“table”表中增加一条记录
	 */

	/**
	 * 往业务表中插入一条数据，并判断重复是否更新！
	 * 
	 * @param table 表名
	 * @param dto 字段映射
	 * @param data 数据
	 * @param dupl 更新标记
	 * @return result 返回状态
	 */
	Map<String, String> addRecord(Object base, String table,
			Map<String, String> dto, Map<String, Object> data, boolean dupl) {
		// 根据table表名获取“别名”“字段”的映射关系
		data = setFieldData(dto, data, true); // 键值对：{USER_NAME_ZH : "张三"}

		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(base, table));
		// 为了兼容PostgreSQL，识别出table的主键
		if (dupl)
			map.put(DUPL, utils.isPostgres() ? getMetaPks(dto) : "");

		// 返回执行结果
		LOG.debug("addRecord...{}", data);
		return utils.result(dao.addOne(map, data) > 0);
	}

	/**
	 * 识别出单表table的主键pkey
	 */
	private String getMetaPks(Map<String, String> dto) {
		String pks = "", key;
		for (Map.Entry<String, String> e : dto.entrySet()) {
			key = e.getKey();
			// 获取元数据中table的约定的主键名（pkey）
			if (key.contains(M_PKEY) && Const.STR_1.equals(e.getValue())) {
				if (pks.length() > 0)
					pks += Const.COMMA;
				pks += dto.get(key.split("!")[0]);
			}
		}
		return pks;
	}

	/**
	 * 往业务表中插入一条数据，并判断重复是否更新！
	 */
	public Map<String, String> addRecord(Object base, String table,
			Map<String, Object> data, boolean dupl) {
		return addRecord(base, table, getDto(table), data, dupl);
	}

	/**
	 * 往业务表中插入一条数据，并判断重复是否更新！
	 */
	public Map<String, String> addRecord(String table,
			Map<String, Object> data, boolean dupl) {
		return addRecord(null, table, data, dupl);
	}

	/**
	 * 往基础表中插入一条数据，并判断重复是否更新！
	 */
	public Map<String, String> addRecord(String table, Map<String, Object> data) {
		return addRecord(null, table, data, false);
	}

	/**
	 * 往业务表中插入一条数据，并判断重复是否更新！
	 */
	public Map<String, String> addRecord(Object base, String table,
			Map<String, Object> data) {
		return addRecord(base, table, data, false);
	}

	/*********************************************************
	 * 批量往业务表中插入多条数据
	 */
	private Map<String, String> batchAdd(Object base, String table,
			Map<String, String> dto, List<Map<String, Object>> items,
			Object lang, boolean dupl) {
		if (items.size() == 0)
			throw new FailedException();

		// 取第一行数据作为参数
		Map<String, Object> data = items.get(0);
		data.put(Const.I18N, lang);
		// 别名映射：{USER_NAME_ZH : "userName"}
		data = setFieldData(dto, data, false);

		// 数据库执行参数
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(base, table));
		// 为了兼容PostgreSQL，识别出table的主键
		if (dupl)
			map.put(DUPL, utils.isPostgres() ? getMetaPks(dto) : "");

		// 返回执行结果
		LOG.debug("batchAdd...{}", map);
		return utils.result(dao.batchAdd(map, data, items) > 0);
	}

	/**
	 * 批量往业务表中插入多条数据
	 */
	public Map<String, String> batchAdd(Object base, String table,
			List<Map<String, Object>> items, Object lang, boolean dupl) {
		return batchAdd(base, table, getDto(table), items, lang, dupl);
	}

	/**
	 * 批量往业务表中插入多条数据
	 */
	public Map<String, String> batchAdd(String table,
			List<Map<String, Object>> items, Map<String, Object> map,
			boolean dupl) {
		return batchAdd(getBase(map), table, items, i18n(map), dupl);
	}

	/**
	 * 批量往业务表中插入多条数据
	 */
	public Map<String, String> batchAdd(String table,
			List<Map<String, Object>> items, Map<String, Object> map) {
		return batchAdd(table, items, map, false);
	}

	/**
	 * 批量往业务表中插入多条数据
	 */
	public Map<String, String> batchAdd(String table,
			List<Map<String, Object>> items, Object lang, boolean dupl) {
		return batchAdd(null, table, items, lang, dupl);
	}

	/**
	 * 批量往业务表中插入多条数据
	 */
	public Map<String, String> batchAdd(String table,
			List<Map<String, Object>> items, Object lang) {
		return batchAdd(null, table, items, lang, false);
	}

	/**
	 * 批量往业务表中插入多条数据
	 */
	public Map<String, String> batchAdd(String table,
			List<Map<String, Object>> items) {
		return batchAdd(table, items, new HashMap<>());
	}

	/**
	 * 批量往业务表中插入多条数据：注意参数items、dupl（可选）
	 */
	public Map<String, String> batchAdd(Map<String, String> map) {
		// 整理“数据转换对象DTO”，并检查参数：items、dupl
		List<Map<String, Object>> list = utils.json2List(map.get("items"));
		boolean dupl = map.containsKey(DUPL); // 重复则更新
		return batchAdd(getBase(map), getTable(map), list, i18n(map), dupl);
	}

	/*********************************************************
	 * 删除表中的数据
	 */
	Map<String, String> delete(Object base, String table,
			Map<String, String> dto, Map<String, Object> where, Object joint) {
		// 梳理删除条件
		Map<String, Object> cond = setCondition(dto, where, joint);
		// 删除条件不能为空（全部删除）
		if (cond.isEmpty())
			throw new FailedException();

		// 数据库执行参数
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(base, table));
		return utils.result(dao.delete(map, cond) > 0);
	}

	/**
	 * 删除表中的数据
	 */
	public Map<String, String> delete(Object base, String table,
			Map<String, Object> where, Object joint) {
		return delete(base, table, getDto(table), where, joint);
	}

	/**
	 * 删除表中的数据
	 */
	public Map<String, String> delete(String table, Map<String, Object> where,
			Object joint) {
		return delete(null, table, where, joint);
	}

	/**
	 * 删除表中的数据
	 */
	public Map<String, String> delete(String table, Map<String, Object> where) {
		return delete(null, table, where, null);
	}

	/**
	 * 删除表中的数据
	 */
	public Map<String, String> delete(Object base, String table,
			Map<String, Object> where) {
		return delete(base, table, where, null);
	}

	/**
	 * 删除表中的数据
	 * 
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	public Map<String, String> delete(Map<String, String> params) {
		// 整理“数据转换对象DTO”，并检查参数
		Map<String, Object> where = utils.json2Map(params.get(WHERE));
		return delete(getBase(params), getTable(params), where);
	}

	/*********************************************************
	 * 批量删除数据
	 */

	/**
	 * 批量删除数据，其中list中的数据为逐条删除数据的条件
	 */
	Map<String, String> batchDelete(Object base, String table,
			Map<String, String> dto, List<Map<String, Object>> list,
			String joint) {
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(base, table));
		// 遍历数据并逐条删除
		int count = 0;
		for (int i = 0; i < list.size(); i++) {
			// list中的数据为逐条删除数据的条件
			Map<String, Object> cond = setCondition(dto, list.get(i), joint);
			if (!cond.isEmpty())
				count += dao.delete(map, cond); // 计数
		}
		// if (count == list.size()) // 数据一致性
		return utils.result(count > 0);
	}

	/**
	 * 批量删除表中的数据，其中list中的数据为逐条删除数据的条件
	 */
	public Map<String, String> batchDelete(Object base, String table,
			List<Map<String, Object>> list) {
		return batchDelete(base, table, getDto(table), list, null);
	}

	/**
	 * 批量删除表中的数据，其中where中的数据为删除数据的条件
	 * 
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	public Map<String, String> batchDelete(Object base, String table,
			Object where) {
		String whr = String.valueOf(where);
		return batchDelete(base, table, utils.json2List(whr));
	}

	/**
	 * 批量删除表中的数据，其中list中的数据为逐条删除数据的条件
	 */
	public Map<String, String> batchDelete(String table,
			List<Map<String, Object>> list) {
		return batchDelete(null, table, list);
	}

	/**
	 * 批量删除表中的数据，其中list中的数据为逐条删除数据的条件
	 */
	public Map<String, String> batchDelete(Object base, String table,
			List<Map<String, Object>> list, String joint) {
		return batchDelete(base, table, getDto(table), list, joint);
	}

	/**
	 * 批量删除表中的数据，其中where中的数据为删除数据的条件
	 * 
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	public Map<String, String> batchDelete(String table, Object where) {
		return batchDelete(null, table, where);
	}

	/**
	 * 批量删除表中的数据
	 * 
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	public Map<String, String> batchDelete(Map<String, Object> params) {
		if (params == null)
			return utils.result(false);
		String table = String.valueOf(params.get(Const.TABLE));
		return batchDelete(params.get(Const.BASE), table, params.get(WHERE));
	}

	/*********************************************************
	 * 更新表中的数据
	 */
	private Map<String, String> update(Object base, String table,
			Map<String, String> dto, Map<String, Object> fields,
			Map<String, Object> where, Object joint) {
		Map<String, Object> cond = setCondition(dto, where, joint);
		Map<String, Object> values = setCondition(dto, fields, null);
		if (cond.isEmpty() || values.isEmpty())
			throw new FailedException();

		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(base, table));
		// 返回执行结果
		return utils.result(dao.update(map, values, cond) > 0);
	}

	/**
	 * 更新数据
	 */
	public Map<String, String> update(Object base, String table,
			Map<String, Object> fields, Map<String, Object> where, Object joint) {
		// 根据table表名，获取表的列名与map的映射关系
		// DTO = { uid:'USER_ID', uid!t:'INTEGER', uid!i:'1', uid!p:'1' }
		return update(base, table, getDto(table), fields, where, joint);
	}

	/**
	 * 更新数据
	 */
	public Map<String, String> update(Object base, String table,
			Map<String, Object> fields, Map<String, Object> where) {
		return update(base, table, fields, where, null);
	}

	/**
	 * 更新数据
	 */
	public Map<String, String> update(String table, Map<String, Object> fields,
			Map<String, Object> where, Object joint) {
		return update(null, table, fields, where, joint);
	}

	/**
	 * 更新数据
	 */
	public Map<String, String> update(String table, Map<String, Object> fields,
			Map<String, Object> where) {
		return update(null, table, fields, where, null);
	}

	/**
	 * 更新数据
	 * 
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	public Map<String, String> update(Map<String, String> map, Object lang) {
		// 整理“数据转换对象DTO”，并检查参数
		Map<String, Object> fields = utils.json2Map(map.get("fields"));
		Map<String, Object> where = utils.json2Map(map.get(WHERE));
		fields.put(Const.I18N, lang);
		where.put(Const.I18N, lang);
		return update(getBase(map), getTable(map), fields, where);
	}

	/*********************************************************
	 * 支持多张表 join关联查询：left="left"为左连接
	 */
	public List<Map<String, String>> findList(Object base,
			Map<String, Map<String, String>> tables, String uriKey,
			Map<String, Object> where, Map<String, String> order, String limit) {
		return findList(base, tables, uriKey, where, order, LEFT, limit);
	}

	/**
	 * 支持多张表 join关联查询：left=""为内连接
	 * 
	 * @param tables 两表名为key，两表的关联主键的映射，表1设为null
	 * @param uriKey 前台上下文或接口名，仅缓存数据用
	 * @param where 综合多表的查询条件
	 * @param limit 限制每页输出行数，为“0”时不限行数
	 */
	public List<Map<String, String>> findList(Object base,
			Map<String, Map<String, String>> tables, String uriKey,
			Map<String, Object> where, Map<String, String> order, String left,
			String limit) {
		// 这里的dto为查询多个表相关联的字段
		Map<String, String> dto = getMetaDto(base, uriKey);
		String table = joinTable(base, dto, tables, left);
		if (utils.isEmpty(table))
			return new ArrayList<>();

		// 数据库查询参数：table、alias、order、joint、limit、floor
		Map<String, String> args = new HashMap<>();
		args.put(Const.TABLE, table);
		args.put(Const.ALIAS, uriKey);
		args.put(Const.LIMIT, limit);
		// 处理args.get("alias")和查询条件
		Map<String, Object> cond = setArgsAndCond(args, dto, where, order);
		// 返回执行结果
		return dao.findList(args, cond);
	}

	/**
	 * 联合多表查询，这里的dto为查询多个表相关联的字段
	 */
	String joinTable(Object base, Map<String, String> dto,
			Map<String, Map<String, String>> tables, String left) {
		if (tables == null)
			return null;
		String T0 = "", jT = "";
		// 以表名为主键，其中第一个表为空，第二个表与第一个表关联，第三个表与第二个表关联
		// { T01:<null>, T02<T02.colA=T01.colB> }
		for (String table : tables.keySet()) {
			// 根据表名table获取左连接的字段映射join关系
			Map<String, String> cond = tables.get(table);
			// 表的全名
			table = getTable(base, table);
			if (utils.isEmpty(cond)) {
				// 一般第一个表的关联必为空，后面一般不能为空
				T0 = table; // 主表
			} else {
				// 左连接关联表
				if (!utils.isEmpty(left))
					jT += " " + left;
				jT += " join " + table + " on ";

				// 两个表关联的关系条件
				int n = 0;
				for (Map.Entry<String, String> e : cond.entrySet()) {
					String val = e.getValue();
					if (val == null)
						continue;
					if (n++ > 0)
						jT += " and ";
					// 两个表关联的对应主键，如果右侧的元数据主键不存在，直接使用右侧的值
					// 多个表关联特殊情况时，可以返回一个纯粹带引号的“字符串”参数
					// 使用单引号以支持PGSQL，其中保证key不能有SQL注入非法字符
					val = dto.containsKey(val) ? i18n(dto, val, null) : (utils
							.isMatch(val, "\\w+") ? "'" + val + "'" : "''");
					jT += i18n(dto, e.getKey(), null) + Const.EQU + val;
				}
			}
		}
		return T0.length() == 0 ? null : T0 + jT;
	}

	/**
	 * 限定了最大返回记录数（limit.rows）
	 */
	private void setLimitRows(Map<String, String> args,
			Map<String, Object> where) {
		if (where != null) {
			// 页面号码、及每页展示的行数
			String pageNo = String.valueOf(where.get("pageNo"));
			String pageSize = String.valueOf(where.get("pageSize"));
			if (utils.isInteger(pageNo) && utils.isInteger(pageSize)) {
				long pNo = Long.parseLong(pageNo);
				long pSize = Long.parseLong(pageSize);
				args.put("floor", String.valueOf((pNo - 1) * pSize));
				args.put(Const.LIMIT, String.valueOf(pSize));
				return;
			}
		}
		String limit = args.get(Const.LIMIT);
		// 当limit==null或''，不限制数据行数
		if (!utils.isInteger(limit)) {
			// 如果limit.rows=''，默认不限制数据，有可能仅limit.rows=''
			args.put(Const.LIMIT, ConfUtil.LIMIT_ROWS);
		} else if ("0".equals(limit)) {
			// 如果limit==0，删除参数，设置为空不限行数，其他时保留
			args.remove(Const.LIMIT);
		}
	}

	/**
	 * 为了适配处理返回分页信息，把查询条件集中处理：args, dto, where, order
	 */
	Map<String, Object> setArgsAndCond(Map<String, String> args,
			Map<String, String> dto, Map<String, Object> where,
			Map<String, String> order) {

		// 数据库查询参数：table、alias、order、joint、limit、floor
		Object lang = i18n(where);

		// 传回前端的数据映射的主键参数数组
		args.put(Const.ALIAS, setAlias(dto, args.get(Const.ALIAS), lang)); // 无SQL注入风险
		// 根据“数据转换对象DTO”，整理排序字段
		args.put(ORDER, getOrder(dto, order, lang));
		// 限定了最大返回记录数，设置参数：limit、floor
		setLimitRows(args, where);

		// 拼接查询条件
		return setCondition(dto, where, args.get("joint"));
	}

	/*********************************************************
	 * 查询数据一览，限定了最大返回记录数（limit.rows）
	 * <p>
	 * 数据库查询参数：table、alias、order、joint、limit、floor
	 */
	public List<Map<String, String>> findList(Map<String, String> args,
			Map<String, Object> where, Map<String, String> order) {
		String table = getTable(args);
		Map<String, String> dto = getDto(table);
		// 替换表名
		args.put(Const.TABLE, getTable(getBase(args), table));
		// 整合处理args.get("alias")和查询条件
		Map<String, Object> cond = setArgsAndCond(args, dto, where, order);
		// 返回查询结果
		return dao.findList(args, cond);
	}

	/**
	 * 查询数据一览，限定了最大返回记录数（limit.rows）
	 */
	public List<Map<String, String>> findList(Map<String, String> args,
			Map<String, Object> where) {
		return findList(args, where, null);
	}

	/**
	 * 查询数据一览
	 */
	public List<Map<String, String>> findList(Object base, String table,
			String uriKey, Map<String, Object> where, Map<String, String> order) {
		// uriKey为传回前端的参数数组
		String db = base == null ? null : String.valueOf(base);
		String[] args = { db, table, uriKey };
		return findAll(where, order, args);
	}

	/**
	 * 查询数据一览
	 */
	public List<Map<String, String>> findList(Object base, String table,
			String uriKey, Map<String, Object> where) {
		// uriKey为传回前端的参数数组
		return findList(base, table, uriKey, where, null);
	}

	/**
	 * 查询数据一览
	 */
	public List<Map<String, String>> findList(String table, String uriKey,
			Map<String, Object> where) {
		// uriKey为传回前端的参数数组
		return findList(null, table, uriKey, where, null);
	}

	/**
	 * 查询数据一览
	 */
	public List<Map<String, String>> findList(String table,
			Map<String, Object> where) {
		return findList(table, null, where);
	}

	/**
	 * 转换参数
	 */
	Map<String, String> putArgs(String[] params) {
		// 无效参数
		if (params == null || params.length < 2)
			return null;
		// 有效参数传递
		String[] keys = { Const.BASE, Const.TABLE, Const.ALIAS, Const.LIMIT };
		Map<String, String> args = new HashMap<>();
		for (int i = 0; i < params.length && i < keys.length; i++) {
			args.put(keys[i], params[i]);
		}
		return args;
	}

	/**
	 * 基于查询条件args：base,table,alias,limit的数据集，用以支持limit="0"的情况
	 */
	public List<Map<String, String>> findAll(Map<String, Object> where,
			Map<String, String> order, String[] params) {
		Map<String, String> args = putArgs(params);
		return args == null ? new ArrayList<>() : findList(args, where, order);
	}

	/*********************************************************
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	private Map<String, String> findOne(Object base, String table,
			Map<String, String> dto, Map<String, Object> where, String uriKey) {
		// 从数据库查询所需的参数
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(base, table));
		// 这里的dto是基于一张表的元数据映射对象
		map.put(Const.ALIAS, setAlias(dto, uriKey, i18n(where))); // 无SQL注入风险

		// 返回执行结果，限定了只返回一条记录
		map = dao.findOne(map, setCondition(dto, where, null));
		return utils.isEmpty(map) ? utils.result(false) : map;
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	public Map<String, String> findOne(Object base, String table,
			Map<String, Object> where, String uriKey) {
		return findOne(base, table, getDto(table), where, uriKey);
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	public Map<String, String> findOne(Object base, String table,
			Map<String, Object> where) {
		return findOne(base, table, where, null);
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 * 
	 * @param table 表名
	 * @param where 查询条件
	 * @param uriKey 返回值的别名
	 */
	public Map<String, String> findOne(String table, Map<String, Object> where,
			String uriKey) {
		return findOne(null, table, where, uriKey);
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	public Map<String, String> findOne(String table, Map<String, Object> where) {
		return findOne(null, table, where, null);
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 * 
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	public Map<String, String> findOne(Map<String, String> map, String uriKey) {
		Map<String, Object> where = utils.json2Map(map.get(WHERE));
		return findOne(getBase(map), getTable(map), where, uriKey);
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	public Map<String, String> findOne(Map<String, String> map) {
		return findOne(map, null);
	}

	/*********************************************************
	 * 判断表中是否有数据
	 */
	private boolean exists(Object base, String table, Map<String, String> dto,
			Map<String, Object> where, Object joint) {
		// 数据库执行参数
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(base, table));
		// 返回执行结果
		return dao.countRows(map, setCondition(dto, where, joint)) > 0;
	}

	/**
	 * 判断表中是否有数据
	 */
	public boolean exists(Object base, String table, Map<String, Object> where,
			Object joint) {
		// 根据table表名，获取表的列名与map的映射关系
		// DTO = { uid:'USER_ID', uid!t:'INTEGER', uid!i:'1', uid!p:'1' }
		return exists(base, table, getDto(table), where, joint);
	}

	/**
	 * 判断表中是否有数据
	 */
	public boolean exists(Object base, String table, Map<String, Object> where) {
		return exists(base, table, where, null);
	}

	/**
	 * 判断表中是否有数据
	 */
	public boolean exists(String table, Map<String, Object> where, Object joint) {
		return exists(null, table, where, joint);
	}

	/**
	 * 判断表中是否有数据
	 */
	public boolean exists(String table, Map<String, Object> where) {
		return exists(null, table, where, null);
	}

	/**
	 * 判断表中是否有数据
	 * 
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	public boolean exists(Map<String, String> map) {
		// 整理“数据转换对象DTO”，并检查参数
		Map<String, Object> where = utils.json2Map(map.get(WHERE));
		return exists(getBase(map), getTable(map), where);
	}

	/**
	 * 根据条件获取记录的行数
	 */
	public int count(String table, Map<String, Object> where) {
		return count(null, table, where, null);
	}

	/**
	 * 根据条件获取记录的行数
	 */
	public int count(Object base, String table, Map<String, Object> where) {
		return count(base, table, where, null);
	}

	/**
	 * 根据条件获取记录的行数
	 */
	public int count(Object base, String table, Map<String, Object> where,
			String joint) {
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(base, table));
		// 注意这里的条件转换
		where = setCondition(getDto(table), where, joint);
		return count(map, where);
	}

	int count(Map<String, String> map, Map<String, Object> where) {
		return dao.countRows(map, where);
	}

	/** * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/**
	 * 根据用户的应用、公司、岗位（dutyId）获取页面上的功能事件控制权限
	 */
	private Map<String, String> cacheActionRole(Map<String, Object> map) {

		// 获取各关联表的相关的字段
		map.put(Const.ALIAS, setAlias(Action.TABLE, null)); // 无SQL注入风险
		// 查询条件：应用、公司、岗位、与页面路由相关
		List<Map<String, String>> list = dao.getActionRole(map);

		// 只要actUri有一次权限，其他页面actRoute都有权访问
		// 或则actUri有一无许可，则其他页面都无许可
		// 这里 actState='1'
		Map<String, String> actMap = utils.list2Map(list, Action.URI,
				Action.STATE);
		// 必须包含的数据信息
		if (!actMap.isEmpty()) {
			// 根据第一行判断参数是否有效
			// 注意这里key需要与third.trustedRole()中一致
			Object[] keys = { map.get(APPCODE), map.get(ConfUtil.COMPID),
					map.get(Duty.ID) };
			String key = utils.join(keys, Const.DOT);
			// 缓存信息
			redis.setActionRole(key, actMap);
			LOG.debug("cacheActionRole...{}", actMap);
		}
		return actMap;
	}

	/**
	 * 根据应用（appCode）公司（compId）岗位（dutyId）获取页面菜单的控制权限（ROLE_PAGE、ROLE_ACTION）
	 * 
	 * @param params 参数中包含了appCode、compId、dutyId、i18n信息
	 */
	public List<Map<String, String>> getMenuRole(Map<String, Object> params) {
		// 关联应用(appCode)公司（compId）岗位(dutyId)信息
		if (utils.isEmpty(params))
			return new ArrayList<>();

		// 缓存鉴权信息：此处缓存所有页面上的功能事件控制权限
		cacheActionRole(params);

		// 获取各关联表的相关的字段
		params.put(Const.ALIAS, getAlias("getMenuRole", i18n(params)));

		// 查询条件：应用、公司、岗位
		// pageId;pageName;pageRoute;pageCmpt;pageType;pageGroup;pageAction;pageIcon
		// ;roleId;(rolePage;)roleAction
		List<Map<String, String>> list = dao.getMenuRole(params);

		// 根据第一行判断参数是否有效，是否包含必须的数据信息
		mergeMenuRole(list);

		return list;
	}

	/**
	 * 梳理用户岗位：依赖 params中的 appCode，userComp、userDuty
	 */
	public Map<String, Object> getRoleParams(Map<String, Object> params,
			Map<String, String> user) {
		Map<String, Object> map = new HashMap<>();
		Object appCode = params.get(APPCODE);
		if (appCode != null) {
			// 从当前登录用户获取信息: "compId"、"dutyId"
			map.put(ConfUtil.COMPID, user.get(User.COMP));
			map.put(Const.I18N, i18n(params));

			// 在登录的时候，已经处理过userDuty，并把具体某个账套中的岗位信息写入userDuty中。
			// 所以，在此用户岗位信息必须从userDuty中获取，如果无岗位信息则框架鉴权就过不来
			Map<String, String> uDuty = utils.json2Map(user.get(User.DUTY));
			// 正常应该有权限
			if (uDuty.containsKey(appCode)) {
				map.put(Duty.ID, uDuty.get(appCode));
				map.put(APPCODE, appCode);
			}
		}
		return map;
	}

	/**
	 * 合并多角色的菜单权限，应用、公司、岗位
	 */
	private Map<String, String> mergeMenuRole(List<Map<String, String>> list) {
		String[] keys = new String[] { Page.ID, Page.ROUTE, Page.ACTION };
		if (list.size() == 1)
			return utils.list2Map(list, keys[1], keys[2]);
		// 遍历数据，合并多角色role的多个action控制权限
		Map<String, String> menu = new HashMap<>();
		for (int i = list.size() - 1; i > 0; i--) {
			// 当前行，如：“P01”、当前行的前一行，如：“P00”
			Map<String, String> tmp0 = list.get(i);
			Map<String, String> tmp1 = list.get(i - 1);
			// 当前行，如：“P01”、当前行的前一行，如：“P00”
			String page0 = tmp0.get(keys[0]); // "pageId"
			String page1 = tmp1.get(keys[0]); // "pageId"

			// 取得当前行roleAction的值
			int ra = Integer.valueOf(tmp0.get(keys[2])); // "roleAction"
			if (page1.equals(page0)) {
				ra = Integer.valueOf(tmp1.get(keys[2])) | ra; // "roleAction"
				// 合并权限后，给前一行（重复行）赋新值
				tmp1.put(keys[2], String.valueOf(ra)); // "roleAction"
				// 删除当前（重复的）行
				list.remove(i);
			} else {
				// 相邻的两行pageId不同时，处理当前有效行（仅仅当前行的权限或是合并后的权限）
				// menu.put(page0, String.valueOf(ra));
				menu.put(tmp0.get(keys[1]), tmp0.get(keys[2])); // "roleAction"
			}
			// 单独处理最后一行（有效行）
			if (i == 1) {
				// 这里使用了"pageRoute"，如总账页面：'/generalLedger'
				menu.put(tmp1.get(keys[1]), tmp1.get(keys[2])); // "roleAction"
			}
		}
		return menu;
	}

	/**
	 * 根据约定的规则获取系统配置参数
	 * 
	 * @param base 帐套名
	 * @param paramKey 参数主键
	 */
	public String getParam(Object base, String paramKey) {
		Map<String, Object> where = new HashMap<>();
		Map<String, String> result = new HashMap<>();
		// 获取系统配置参数
		where.put("paramKey", paramKey);
		result = findOne(base, "PARAM", where, "getParamVal");
		return result.get("paramVal");
	}

	/**
	 * 根据约定的规则获取系统配置参数
	 */
	public String getParam(Map<String, Object> params, String paramKey) {
		return getParam(getBase(params), paramKey);
	}

	/** * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/**
	 * 增删改查，字段自动支持国际化
	 */
	private String i18n(Map<String, String> dto, String key, Object lang) {
		String field = dto.get(key);
		// 根据约定（i18n）判断是否支持国际化
		if (dto.containsKey(key + M_I18N) && !utils.isEmpty(lang))
			field += _S + lang; // 获取clientLang，添加“_”符号
		return field;
	}

	/**
	 * 当前查询语言
	 */
	private <T> Object i18n(Map<String, T> params) {
		return params == null ? null : params.get(Const.I18N);
	}

	/**
	 * 查询排序条件：根据“数据转换对象DTO”，拼接排序条件
	 */
	private String getOrder(Map<String, String> dto, Map<String, String> order,
			Object lang) {

		// 如果排序条件为空，默认按主键pkey（!p）排序
		if (utils.isEmpty(order))
			return getMetaPks(dto);

		// 按传递的排序参数（order）排序
		String str = "", key;
		for (Map.Entry<String, String> e : order.entrySet()) {
			key = e.getKey();
			if (!dto.containsKey(key))
				continue;
			if (str.length() > 0)
				str += Const.COMMA;
			// 支持国际化，如：USER_NAME_ZH ASC
			str += i18n(dto, key, lang) + " " + e.getValue();
		}
		return str;
	}

	/**
	 * 查询条件：字段名对应的数值
	 */
	<T> Map<String, T> setCondition(Map<String, String> dto,
			Map<String, T> where, Object joint) {
		Map<String, T> cond = new HashMap<>();
		if (where == null)
			return cond;
		Object lang = i18n(where);
		for (Map.Entry<String, T> ent : where.entrySet()) {
			T val = ent.getValue();
			if (val == null)
				continue;
			String key = ent.getKey();
			if (dto.containsKey(key)) {
				// 这里的key只能是单纯的“fieldKey”
				key = i18n(dto, key, lang);
				key += joint == null ? Const.EQU : " " + joint;
				cond.put(key, val);
			} else if (P_QUOTE.matcher(key).find()) {
				/**
				 * 这里匹配参数如：“{fieldKey} <>” 之类参数<br>
				 * 转换后的格式如：{“USER_NAME_ZH <>”, “张三”}<br>
				 * 这里需要注意防SQL注入，如：{userId}='1';select * from t1;<br>
				 * 这里val必须是数组参数，支持同时传递多个 “数值”={"aaa","bbb"}
				 */
				// 携带变量参数{V}时的特殊处理
				boolean noneVar = P_VALUE.matcher(key).find();
				// 带 “{fieldKey}”的一般也携带joint=“”参数
				// 用“数值”={"aaa","bbb"}置换key里的多个{V}
				key = getFieldKey(dto, key, val, lang);
				if (key != null)
					cond.put(key, noneVar ? null : val);
			}
		}
		LOG.debug("setCondition...{}", utils.obj2Str(cond));
		return cond;
	}

	/**
	 * 拼接SQL语句的查询条件
	 * 
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	public String getWhere(Map<String, String> dto, String expr, Object lang) {
		return getFieldKey(dto, expr, new String[] {}, lang);
	}

	/**
	 * 把字符串中以“{}”标识出来的关键词替换成“表字段”名和数值{V}
	 * <p>
	 * 替换条件“{key}>={V}”中的key和V（可能有多个），校验表达式避免SQL注入风险
	 * <p>
	 * 注意预防SQL注入，如：{userId}='1';select * from t1;
	 * 
	 * @param <T>
	 */
	private <T> String getFieldKey(Map<String, String> dto, String expr,
			T valArr, Object lang) {
		// 例如：expr=“( {key} IS NULL OR {key} in ( {V} )”
		// 如：{ userLock }='0' and { userId } in ( '111','222' ) and ...
		String[] arr = expr.split("\\{ *"); // 分割“{key} in (”
		int v = 0;
		boolean isArr = valArr.getClass().isArray();
		String[] vArr = isArr ? (String[]) valArr : new String[] {};
		// arr.length>1时，一般arr[0]=""为有效表达式，如：arr[]=["","userId} in ("]
		// 一般arr[0]=""，应从 arr[1]开始处理
		if (arr.length > 1 && isValidExpr(arr[0])) {
			// 从 arr[1]开始处理，如：arr[1] = "userId} in ("
			for (int i = 1; i < arr.length; i++) {
				// 从arr[1]再分析，如：key[] = ["userId"," in ("]
				String[] key = arr[i].split(" *\\}");
				// 字段名{key[0]}必须存在、赋值无SQL注入风险，则继续处理。如：“ in (”为有效
				if (key.length == 0) {
					if (vArr.length > v)
						arr[i] = vArr[v++];

				} else if (dto.containsKey(key[0])) {
					// 替换：userId =>>USER_ID
					arr[i] = i18n(dto, key[0], lang);
					if (key.length > 1 && isValidExpr(key[1]))
						arr[i] += key[1];

				} else if (vArr.length > v) {
					// 约定{V}用""或大写字母V标识
					if ("".equals(key[0]) || "V".equals(key[0]))
						arr[i] = vArr[v++] + (key.length > 1 ? key[1] : "");
				}
			}
			return utils.arr2Str(arr);
		}
		return null;
	}

	/**
	 * 检查SQL注入，不在规定范围内的表达式无效
	 * 
	 * @param expr 已剔除空格的表达式，关键字请用小写字母
	 */
	private boolean isValidExpr(String expr) {
		if (expr.length() > 0) {
			// 从表达式拆分识别出关键字，如：“ in (”
			String[] arr = expr.split(" +"); // 以空格“”分割
			for (String str : arr) {
				// 必须能匹配规定范围内的表达式，或满足JSON类型操作的表达式格式："->>'$.inspur.eyun'"
				if (str.length() == 0
						|| utils.findIn(ConfUtil.SQL_EXPR, str.toLowerCase())
						|| P_JSON.matcher(str).find())
					continue;
				// 若有未匹配的表达式，则返回无效
				LOG.error("invalid expr...{}", str);
				throw new FailedException();
			}
		}
		return true;
	}

	/**
	 * 添加数据时：字段名对应数值的键值对，根据isKV判断是键值对、还是别名映射
	 */
	private Map<String, Object> setFieldData(Map<String, String> dto,
			Map<String, Object> params, boolean isKV) {
		// 添加数据时：数据不能为空
		if (params == null)
			throw new FailedException();

		Map<String, Object> data = new HashMap<>();
		Object lang = i18n(params);
		for (Map.Entry<String, Object> e : params.entrySet()) {
			String key = e.getKey();
			if (!dto.containsKey(key))
				continue;
			key = i18n(dto, key, lang);
			data.put(key, isKV ? e.getValue() : e.getKey());
		}
		// 添加数据时：数据不能为空
		if (data.isEmpty())
			throw new FailedException();
		return data;
	}

	/**
	 * 根据table表名获取元数据字段的别名
	 */
	public String setAlias(String table, Object lang) {
		return setAlias(table, null, lang);
	}

	/**
	 * 根据table表名获取元数据字段的别名
	 */
	public String setAlias(String table, String uriKey, Object lang) {
		return setAlias(getDto(table), uriKey, lang);
	}

	/**
	 * 根据转换成数据库“字段 as 别名”的映射字符串（无SQL注入风险）
	 */
	private String setAlias(Map<String, String> dto, String uriKey, Object lang) {
		// 增加空置判断，为空则返回全部dto字段（2022.5.2）
		// 如果支持自定义设置标签的隐藏和显示的数量，则返回全部dto字段（2023.8.27）
		// 自定义设置标签IS_CUSTOM_COLS：其中fieldShow="1"的数据，在返给前台后再处理
		if (ConfUtil.IS_CUSTOM_COLS || utils.isEmpty(uriKey))
			return fieldAsAlias(dto, lang);

		// 首先从配置文件读取参数，未定义时默认返回空（""）
		String val = ConfUtil.getParam(uriKey);
		// 如果返回“”，则返回全部表字段的别名
		if (val.length() == 0)
			return fieldAsAlias(dto, lang);
		// 返回字段与别名的映射
		return fieldAsAlias(dto, val.split(ConfUtil.COMMA), lang);
	}

	/**
	 * 根据转换成数据库“字段 as 别名”的映射字符串（无SQL注入风险）
	 */
	private String fieldAsAlias(Map<String, String> dto, String[] arr,
			Object lang) {
		String alias = "";
		for (int i = 0; i < arr.length; i++) {
			if (i > 0)
				alias += Const.COMMA;
			alias += fieldAsAlias(dto, arr[i], lang);
		}
		// 返回字段与别名的映射
		return alias;
	}

	/**
	 * 转换成数据库“字段 as 别名”的映射字符串（无SQL注入风险）
	 */
	private String fieldAsAlias(Map<String, String> dto, Object lang) {
		String alias = "";
		// 为约定区分type\i18n\pkey，标注为!t,!i,!p
		for (String key : dto.keySet()) {
			// 忽略标注了“!”的主键
			if (key.contains("!"))
				continue; // 忽略
			if (alias.length() > 0)
				alias += Const.COMMA;
			alias += fieldAsAlias(dto, key, lang);
		}
		return alias;
	}

	/**
	 * 拼接别名（无SQL注入风险），注意支持的数据类型，注意日期格式勿用TIME
	 */
	private String fieldAsAlias(Map<String, String> dto, String key, Object lang) {
		String val = dto.get(key);
		// 约定为区分数据类型type、国际化语言i18n，分别标注为!t,!i
		String type = dto.get(key + M_TYPE);
		String keyStr = "\"" + key + "\"";
		if (type != null) {
			// 根据 META_FIELD的不同数据类型进行处理
			// 增加了支持JSON格式
			if (type.matches("[IDBSJ].+") || type.matches("TIN.+")) {
				// INTEGER\DOUBLE\DECIMAL\DATE+\BIGINT\SMALLINT\TINYINT\JSON
				// 补充了适配 PG(PostgreSQL)
				// mysql中的ifnull()函数对应postgresql的COALESCE
				if (utils.isPostgres()) {
					val = CAST + val + " as TEXT)"; // 暂不处理NULL
				} else {
					val = CAST + val + " as CHAR)"; // 暂不处理NULL
				}
			} else if (type.matches("T.+")) { // TIMESTAMP(3)\勿用TIME
				// 追加PG(PostgreSQL) 适配
				if (utils.isPostgres()) {
					val = "to_char(" + val + ",'yyyy-MM-dd hh24:mi:ss"
							+ (type.length() > 9 ? ".ms" : "") + R_PAREN;
				} else {
					val = "DATE_FORMAT(" + val + ",'%Y-%m-%d %T"
							+ (type.length() > 9 ? ".%f" : "") + R_PAREN;
				}
			} else if (dto.containsKey(key + M_I18N) && !utils.isEmpty(lang)) {
				// 只有字符串（VARCHAR\CHAR）格式，才有可能支持国际化
				val += _S + lang;
			}
		} else if (val == null) {
			// 当携带的fieldKey不存在时，直接使用参数本身key
			val = keyStr;
		}
		// 这里为了兼容PgSQL使用双引号
		return val + " as " + keyStr;
	}

	/**
	 * 数据库与前台交互传递的参数，根据请求key获取字段及别名映射串
	 */
	public String getAlias(String uriKey, Object lang) {
		return getAlias(null, uriKey, lang);
	}

	/**
	 * 数据库与前台交互传递的参数，根据请求key获取字段及别名映射串
	 */
	private String getAlias(Object base, String uriKey, Object lang) {
		// 这里把元数据转换为字段的映射
		// 如：“USER_ID as "userId", USER_LANG as "userLang"”
		return fieldAsAlias(getMetaDto(base, uriKey), lang); // 无SQL注入风险
	}

	/**
	 * 数据库与前台交互传递的参数，与getAlias的区别是参数为table
	 */
	public String getField(String table, Object lang) {
		return fieldAsAlias(getDto(table), lang);
	}

	/**
	 * 从数据库中查询并返回字段与别名的映射
	 */
	private Map<String, String> getFields(String table, String[] arr,
			Object lang) {
		Map<String, String> map = new HashMap<>();
		if (arr != null) {
			Map<String, String> dto = getDto(table);
			// 例如key=userName，转为为{ userName: USER_NAME }
			for (String key : arr) {
				if (dto.containsKey(key))
					map.put(key, i18n(dto, key, lang));
			}
		}
		return map;
	}

	/**
	 * 数据库与前台交互传递的参数
	 */
//	public String getFieldAlias(List<Map<String, String>> metaList, Object lang) {
//		return fieldAsAlias(getDto(metaList), lang);
//	}

	/**
	 * 获取元数据对象：“别名”与“字段”的映射关系、类型、国际化
	 * <p>
	 * 如：{ uid:'USER_ID', uid!p:'1', uid!i:'1', uid!t:'VARCHAR(18)' }
	 */
	Map<String, String> getDto(String table) {
		String rk = Const.RDS_META + table;
		Map<String, String> dto = redis.hmget(rk);
		if (dto == null) {
			dto = getDto(getTableMeta(table));
			// 缓存数据请求对象（缓存10小时）
			redis.hmset(rk, dto, Const.TEN_HH);
		}
		return dto;
	}

	/**
	 * 获取元数据对象：“别名”与“字段”的映射关系、类型、国际化
	 * <p>
	 * 注意：getDto按表名查询，getMetaDto按主键查询，且有可能跨表查询
	 */
	Map<String, String> getMetaDto(Object base, String uriKey) {
		boolean isCustom = ConfUtil.IS_CUSTOM_COLS;
		// 按每个base和uriKey缓存数据
		String rk = Const.RDS_META + (isCustom ? base : "") + uriKey;
		Map<String, String> dto = redis.hmget(rk);
		if (dto == null) {
			// 如果支持自定义设置标签的隐藏和显示的数量
			if (isCustom) {
				dto = getDto(getMeta(getFieldList(base, uriKey)));
			} else {
				dto = getDto(getMeta(uriKey));
			}
			// 缓存数据请求对象（缓存10小时）
			redis.hmset(rk, dto, Const.TEN_HH);
		}
		return dto;
	}

	/**
	 * 获取元数据字段信息，根据fieldUri查询fieldAlias所有列名
	 */
	public List<Map<String, String>> getFieldList(Object base, String uriKey) {
		Map<String, Object> where = new HashMap<>();
		where.put(Field.URI, uriKey);
		// 按序号排列
		Map<String, String> order = new HashMap<>();
		order.put(Field.SORT, "ASC");
		// 查询全部元数据的字段，具体展示格式放在前端处理
		return findList(base, Field.TABLE, null, where, order);
	}

	/**
	 * 元数据对象：约定数据类型type(!t)\主键pkey(!p)\国际化i18n(!i)
	 */
	private Map<String, String> getDto(List<Map<String, String>> metaList) {
		Map<String, String> dto = new HashMap<>();
		for (int i = 0; i < metaList.size(); i++) {
			Map<String, String> meta = metaList.get(i);
			String alias = meta.get(Const.ALIAS);
			dto.put(alias, meta.get(Const.FIELD));
			dto.put(alias + M_TYPE, meta.get(TYPE));
			dto.put(alias + M_PKEY, meta.get(PKEY));
			if (Const.STR_1.equals(meta.get(Const.I18N)))
				dto.put(alias + M_I18N, meta.get(Const.I18N));
		}
		if (dto.isEmpty())
			throw new FailedException();
		return dto;
	}

	/**
	 * 根据 META_ALIAS从GMETA表中获取Meta元数据一览
	 */
	public List<Map<String, String>> getMeta(String uriKey) {
		String val = ConfUtil.getParam(uriKey);
		// 查询条件为多个字段的别名：[ 'userLang', 'userMail', 'userName', ...]
		String[] keys = val.length() == 0 ? null : val.split(ConfUtil.COMMA);
		return getMeta(keys);
	}

	/**
	 * 获取元数据字段信息，根据uriKey查询FIELD、GMETA关联数据
	 */
	private List<Map<String, String>> getMeta(
			List<Map<String, String>> fieldList) {
		String[] keys = new String[fieldList.size()];
		// 转换格式：[ userId, userLang, userName, userMail, ...]
		for (int i = 0; i < fieldList.size(); i++)
			keys[i] = fieldList.get(i).get(Field.ALIAS);
		return getMeta(keys);
	}

	private List<Map<String, String>> getMeta(String[] keys) {
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(null, META));
		map.put(Const.ALIAS, fieldAsAlias(gmeta, null)); // 无SQL注入风险
		// 查询结果：[{field:UDER_ID,pkey:'1',i18n:'1',alias:'userId',...}, ...]
		return dao.getMeta(map, keys);
	}

	/**
	 * 为兼容旧版单独加的处理，后期会弃用（用getTableMeta替代）
	 * 
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	public List<Map<String, String>> getMetaList(String table) {
		return getTableMeta(table);
	}

	/**
	 * 根据表名table从GMETA表中获取Meta元数据全部信息一览
	 * <p>
	 * 这里表名称table无SQL注入风险
	 */
	public List<Map<String, String>> getTableMeta(Object table) {
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(null, META));
		map.put(Const.ALIAS, fieldAsAlias(gmeta, null)); // 无SQL注入风险
		map.put(ORDER, META_SORT);
		Map<String, Object> cond = new HashMap<>();
		cond.put("META_TABLE=", table);
		return dao.findList(map, cond);
	}

	/**
	 * 账套（base）及表名组合拼接符号移到UtilsService的jsonParams处理
	 */
	public String getTable(Object base, String table) {
		// 如果base为空，补充PG数据库时的信息
		if (utils.isPostgres()) {
			if (base != null) {
				table = base + table; // base123_
				String db = String.valueOf(base); // base123.
				if (db.charAt(db.length() - 1) == '.')
					return table;
			}
			String[] keys = { ConfUtil.BASE_MAIN, table };
			return utils.join(keys, Const.DOT);
		} else {
			// 这里 base携带了拼接符号，符号拼接移到了UtilsService的jsonParams处理
			return base == null ? table : base + table;
		}
	}

	private String getTable(Map<String, String> params) {
		return params == null ? null : params.get(Const.TABLE);
	}

	private <T> Object getBase(Map<String, T> params) {
		return params == null ? null : params.get(Const.BASE);
	}

	/**
	 * 从表中获取最大值
	 */
	public int getMaxCode(Object base, String table, String field) {
		return getMaxCode(base, table, field, null);
	}

	/**
	 * 从表中获取最大值
	 */
	public int getMaxCode(Object base, String table, String field,
			Map<String, Object> where) {
		Map<String, String> dto = getDto(table);
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(base, table));
		map.put(Const.FIELD, dto.get(field));
		return dao.getMaxCode(map, setCondition(dto, where, null));
	}

	/**
	 * 归集数据行数，用于统计和分组数据
	 */
	public List<Map<String, String>> groupCount(Object base, String table,
			String[] keys, Map<String, Object> where) {
		Map<String, String> dto = getDto(table);
		Map<String, String> map = getFields(table, keys, null);
		table = getTable(base, table);
		return dao.groupCount(table, map, setCondition(dto, where, null));
	}

}
