package cn.ps1.aolai.service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import cn.ps1.aolai.entity.Field;
import cn.ps1.aolai.utils.ConfUtil;
import cn.ps1.aolai.utils.Const;
import cn.ps1.aolai.utils.FailedException;

/**
 * Aolai基础框架相关服务层，基于Mybatis整合的元数据的基础操作类(CRUD)
 * 
 * @author Fulin
 * @version 1.0 $Date: 2023.8.20
 * @since 1.7
 */

@Service
public class AolaiService extends GmetaService {

	private static Logger LOG = LoggerFactory.getLogger(AolaiService.class);

	/**
	 * 从GMETA元数据表中获取元数据字段属性信息一览
	 */
	public Map<String, Object> getMetaList(HttpServletRequest req) {
		Map<String, Object> params = utils.jsonParams(req);
		Object table = params.get(Const.TABLE);
		if (table == null)
			throw new FailedException();

		// 根据表名获取获取元数据字段
		return utils.success(getTableMeta(table));
	}

	/**
	 * 增加元数据项，或修改GMETA元数据表的信息
	 */
	public Map<String, String> setMetaItem(HttpServletRequest req) {
		Map<String, Object> params = utils.jsonParams(req);
		// 获取GMETA元数据的主键，并判断传递参数是否存在元数据主键
		// 主键一个也不能少
		String[] metaKeys = { Const.FIELD, Const.ALIAS, "name", TYPE, NULL, Const.I18N,
				PKEY, "width", DEFT, "sort", Const.TABLE };
		if (!utils.availParams(params, metaKeys))
			throw new FailedException();

		// 仅往GMETA表中插入一条数据，并判断重复是否更新
		Map<String, String> dto = getMetaDto();
		Map<String, String> result = addRecord(null, META, dto, params, true);
		// 清除与meta信息相关的dto缓存
		return eraseMetaDto(result);
	}

	/**
	 * 删除GMETA元数据表中的一条数据记录
	 */
	public Map<String, String> delMetaItem(HttpServletRequest req) {
		Map<String, Object> params = utils.jsonParams(req);
		// 获取删除数据的条件
		String whr = String.valueOf(params.get(WHERE));
		Map<String, Object> where = utils.json2Map(whr);
		if (where.isEmpty())
			throw new FailedException();

		// 删除一条数据
		Map<String, String> dto = getMetaDto();
		Map<String, String> result = delete(null, META, dto, where, null);
		// 清除与meta信息相关的dto缓存
		return eraseMetaDto(result);
	}

	/**
	 * 删除GMETA元数据表中的多条数据记录
	 */
	public Map<String, String> delMetaItems(HttpServletRequest req) {
		Map<String, Object> params = utils.jsonParams(req);
		// 获取删除数据的条件
		String whr = String.valueOf(params.get(WHERE));
		List<Map<String, Object>> list = utils.json2List(whr);
		if (list.size() == 0)
			throw new FailedException();

		// list中的数据为逐条删除数据的条件
		Map<String, String> dto = getMetaDto();
		Map<String, String> result = batchDelete(null, META, dto, list, null);
		// 清除与meta信息相关的dto缓存
		return eraseMetaDto(result);
	}

	/**
	 * 获取T_FIELD元数据字段信息，根据fieldUri查询fieldAlias所有列名
	 * <p>
	 * 因为T_FIELD表在每个base中，所以需要携带base参数
	 */
	public List<Map<String, String>> getFieldList(HttpServletRequest req) {
		Map<String, Object> params = utils.jsonParams(req);
		// 查询全部元数据的字段
		return getFieldList(params.get(Const.BASE), utils.getRequestURI(req));
	}

	/**
	 * 设置T_FIELD元数据自定义字段信息表的字段属性信息
	 */
	public Map<String, String> setFieldItem(HttpServletRequest req) {
		Map<String, Object> params = utils.jsonParams(req);
		// 获取T_FIELD元数据的主键
		// 主键一个也不能少
		// [fieldUri, fieldSort, fieldAlias, fieldLabel, fieldAlign, fieldShow,
		// fieldWidth, fieldFreeze, fieldCustom, fieldFormat]
		if (!utils.availParams(params, getMetaKeys(Field.TABLE, NULL)))
			throw new FailedException();

		// 仅往T_FIELD表中插入一条数据，并判断重复是否更新
		Object base = params.get(Const.BASE);
		Map<String, String> result = addRecord(base, Field.TABLE, params, true);
		// 清除所有base中与meta信息相关的dto缓存
		return eraseMetaDto(result, base);
	}

	/**
	 * 设置T_FIELD元数据自定义字段信息表的字段属性信息
	 * <p>
	 * 每个base分库中都有一个T_FIELD表，缓存了“m.base*"信息
	 */
	public Map<String, String> setFieldItems(HttpServletRequest req) {
		Map<String, Object> params = utils.jsonParams(req);
		if (!params.containsKey(Const.TABLE))
			throw new FailedException();
		// Object数据对象转换
		Map<String, String> items = utils.obj2Map(params);
		items.put(Const.TABLE, Field.TABLE);
		items.put(DUPL, "true");
		// 批量增加items，重复并覆盖旧数据
		Map<String, String> result = batchAdd(items);
		// 清除所有base中与meta信息相关的dto缓存
		return eraseMetaDto(result, params.get(Const.BASE));
	}

	/**
	 * 删除T_FIELD元数据字段表中的数据记录
	 */
	public Map<String, String> delFieldInfo(HttpServletRequest req) {
		Map<String, Object> params = utils.jsonParams(req);
		// 获取删除数据字段的条件
		Object obj = params.containsKey(WHERE);
		if (utils.isEmpty(obj))
			throw new FailedException();

		String whr = String.valueOf(obj);
		List<Map<String, Object>> list = utils.json2List(whr);
		// list中的数据为逐条删除数据的条件
		Object base = params.get(Const.BASE);
		Map<String, String> result = batchDelete(base, Field.TABLE, list);
		return eraseMetaDto(result);
	}

	/**
	 * 清除所有base中与GMETA信息相关的dto缓存
	 */
	private <T> Map<String, T> eraseMetaDto(Map<String, T> result) {
		return eraseMetaDto(result, "");
	}

	/**
	 * 清除所有base中与T_FIELD信息相关的dto缓存
	 */
	private <T> Map<String, T> eraseMetaDto(Map<String, T> result, Object base) {
		if (Const.STR_1.equals(result.get("status")))
			eraseMetaDto(ConfUtil.IS_CUSTOM_COLS ? base : "");
		return result;
	}

	/**
	 * 清除所有base中与GMETA元数据相关的缓存
	 */
	private void eraseMetaDto(Object rk) {
		redis.delKeys(Const.RDS_META + rk + "*");
	}

	/*********************************************************
	 * 分页查询：支持多张表 join关联查询：left="left"为左连接
	 */
	public Map<String, Object> queryList(Object base,
			Map<String, Map<String, String>> tables, String uriKey,
			Map<String, Object> where, Map<String, String> order, String limit) {
		return queryList(base, tables, uriKey, where, order, LEFT, limit);
	}

	/**
	 * 支持多张表 join关联查询：left=""为内连接，查询条件需放到别名列表中
	 * @param tables 两表名为key，两表的关联主键的映射，表1设为null
	 * @param uriKey 前台上下文或接口名，仅缓存数据用
	 * @param where 综合多表的查询条件，分页时需要pageNo、pageSize
	 * @param limit 限制每页输出行数，为“0”时不限行数
	 */
	public Map<String, Object> queryList(Object base,
			Map<String, Map<String, String>> tables, String uriKey,
			Map<String, Object> where, Map<String, String> order, String left,
			String limit) {
		// 这里的dto为查询多个表相关联的字段
		Map<String, String> dto = getMetaDto(base, uriKey);
		String table = joinTable(base, dto, tables, left);
		if (utils.isEmpty(table))
			return splitPage(0, new ArrayList<>());

		// 数据库查询参数：table、alias、order、joint、limit、floor
		Map<String, String> args = new HashMap<>();
		args.put(Const.TABLE, table);
		args.put(Const.ALIAS, uriKey);
		args.put(Const.LIMIT, limit);
		// 处理args.get("alias")和查询条件
		Map<String, Object> cond = setArgsAndCond(args, dto, where, order);

		// 返回数据总行数：T1 left join T2 on T2.A=T1.B
//		int total = count(null, table, cond);
		int total = count(args, cond);
		LOG.debug("> total...{}", total);

		// 返回查询结果
		List<Map<String, String>> items = dao.findList(args, cond);
		return splitPage(total, items);
	}

	/**
	 * 数据分页显示
	 */
	private Map<String, Object> splitPage(int total,
			List<Map<String, String>> items) {
		Map<String, Object> result = new HashMap<>();
		result.put("total", total);
		result.put("items", items);
		return result;
	}

	/*********************************************************
	 * 查询数据一览，限定了最大返回记录数（limit.rows）
	 */
	public Map<String, Object> queryList(Map<String, String> args,
			Map<String, Object> where, Map<String, String> order) {
		String table = args.get(Const.TABLE);
		Map<String, String> dto = getDto(table);
		table = getTable(args.get(Const.BASE), table);
		
		// 处理args.get("alias")和查询条件
		Map<String, Object> cond = setArgsAndCond(args, dto, where, order);
		args.put(Const.TABLE, table);
		// 返回数据总行数
		int total = count(args.get(Const.BASE), table, where);

		// 返回查询结果
		List<Map<String, String>> items = dao.findList(args, cond);
		return splitPage(total, items);
	}

	/**
	 * 查询数据一览，限定了最大返回记录数（limit.rows）
	 */
	public Map<String, Object> queryList(Map<String, String> args,
			Map<String, Object> where) {
		return queryList(args, where, null);
	}

	/**
	 * 查询数据一览
	 */
	public Map<String, Object> queryList(Object base, String table,
			String uriKey, Map<String, Object> where, Map<String, String> order) {
		// uriKey为传回前端的参数数组
		String db = base == null ? null : String.valueOf(base);
		String[] keys = { db, table, uriKey };
		return queryAll(where, order, keys);
	}

	/**
	 * 查询数据一览
	 */
	public Map<String, Object> queryList(Object base, String table,
			String uriKey, Map<String, Object> where) {
		// uriKey为传回前端的参数数组
		return queryList(base, table, uriKey, where, null);
	}

	/**
	 * 查询数据一览
	 */
	public Map<String, Object> queryList(String table, String uriKey,
			Map<String, Object> where) {
		// uriKey为传回前端的参数数组
		return queryList(null, table, uriKey, where, null);
	}

	/**
	 * 查询数据一览
	 */
	public Map<String, Object> queryList(String table, Map<String, Object> where) {
		return queryList(table, null, where);
	}

	/**
	 * 基于查询条件args：base,table,alias,limit的数据集，用以支持limit="0"的情况
	 */
	public Map<String, Object> queryAll(Map<String, Object> where,
			Map<String, String> order, String[] params) {
		Map<String, String> args = putArgs(params);
		if (args == null)
			return splitPage(0, new ArrayList<>());
		else
			return queryList(args, where, order);
	}

	/**
	 * 默认关联多表查询时，检索出全部数据
	 */
	public List<Map<String, String>> findAll(Object base,
			Map<String, Map<String, String>> tables, String uriKey,
			Map<String, Object> where, Map<String, String> order) {
		return findAll(base, tables, uriKey, where, order, LEFT);
	}

	/**
	 * 默认关联多表查询时，检索出全部数据
	 */
	public List<Map<String, String>> findAll(Object base,
			Map<String, Map<String, String>> tables, String uriKey,
			Map<String, Object> where, Map<String, String> order, String left) {
		return findList(base, tables, uriKey, where, order, left, "0");
	}

	/** * * * * * * * 以下对GMETA的基本操作 * * * * * * * **/

	/**
	 * 仅往GMETA表中插入一条数据，并判断重复是否更新
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	public Map<String, String> addMeta(Map<String, Object> data, boolean dupl) {
		return addRecord(null, META, getMetaDto(), data, dupl);
	}

	/**
	 * 仅从GMETA表中批量删除数据，其中list中的数据为逐条删除数据的条件
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	public Map<String, String> delMeta(List<Map<String, Object>> list) {
		return batchDelete(null, META, getMetaDto(), list, null);
	}

	/**
	 * 仅操作GMETA表的元数据“键值对”
	 */
	private Map<String, String> getMetaDto() {
		Map<String, String> map = new HashMap<>();
		// 赋值元数据的元值对
		map.putAll(gmeta);
		// 这里的table代表字段GMETA表中的META_TABLE字段
		map.put(Const.TABLE, "META_TABLE");
		return map;
	}

	/**
	 * 仅为基础维护校验参数时用
	 * <p>
	 * 先根据tableName获取“元数据对象DTO”，再获取元数据的属性alias数组
	 */
	private String[] getMetaKeys(String table, String key) {
		// 获取某表所有元数据，如：
		// [{field:USER_ID, alias:userId, pkey:'1', i18n:'1', null:'1',
		// name:'用户编号'},
		// {field:USER_NAME, alias:userName, pkey:'1', i18n:'1', null:'1',
		// name:'用户名称'}, ...]
		List<Map<String, String>> list = getTableMeta(table);
		Set<String> keySet = new HashSet<>();
		for (Map<String, String> item : list) {
			if (Const.STR_1.equals(item.get(key)))
				keySet.add(item.get(Const.ALIAS));
		}
		// 转换为：[ userId, userName, userLang ]
		String[] keys = new String[list.size()];
		return keySet.toArray(keys);
	}

	/**
	 * 判断某表名是否存在（暂未使用：暂无SQL注入风险）
	 */
	public boolean tableExists(Object base, String table) {
		List<String> list = dao.showTables(getTable(base, table));
		return list.size() > 0;
	}

	/**
	 * 从GMETA元数据表中获取元数据的表名一览（暂未使用）
	 */
	public List<String> getMetaTable() {
		return dao.getMetaTable(getTable(null, META));
	}

	/**
	 * 判断某表的字段列名是否存在（暂未使用：暂无SQL注入风险）
	 */
	public List<Map<String, String>> getColumns(Object base, String table,
			String field) {
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(base, table));
		map.put(Const.FIELD, field);
		return dao.showColumns(map);
	}

	/*********************************************************
	 * 删除表（不推荐），如果table参数从前台传入，请检查SQL注入风险
	 * 
	 * @param table
	 */
	public Map<String, String> dropTable(Object base, String table) {
		if (!utils.isMatch(table, "\\w+")) // "[_a-zA-Z0-9]+$"
			throw new FailedException();

		// 动态删除表
		dao.drop(getTable(base, table));

		// 清除所有base中与GMETA元数据相关的缓存
		eraseMetaDto("");

		return utils.success();
	}

	/*********************************************************
	 * 创建表（不推荐），如果table参数从前台传入，请检查SQL注入风险
	 */
	public Map<String, String> createTable(Object base, String table) {
		// 表名不符合规则
		if (!utils.isMatch(table, "\\w+")) // "[_a-zA-Z0-9]+$"
			throw new FailedException();

		// 先从META表中获取元数据
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(base, table));
		map.put("fields", getMetaSql(table));
		// 动态建表
		dao.create(map);
		LOG.debug("> createTable...{}", map);
		// 返回成功
		return utils.success();
	}

	/**
	 * 根据Meta元数据整理建表Sql
	 */
	private String getMetaSql(String table) {
		List<Map<String, String>> meta = getTableMeta(table);
		String sql = "", field, pkey = "";
		String[] lang = ConfUtil.I18N_LOCALES; // 国际化支持
		String notNULL = " NOT NULL", deFault = " DEFAULT ";
		for (int i = 0; i < meta.size(); i++) {
			if (i > 0)
				sql += Const.COMMA;
			Map<String, String> map = meta.get(i);
			// 支持国际化的字段
			if (Const.STR_1.equals(map.get(Const.I18N))) {
				for (int j = 0; j < lang.length; j++) {
					if (j > 0)
						sql += Const.COMMA;
					// 拼接国际化语言标识
					field = map.get(Const.FIELD) + _S + lang[j];
					sql += field + " " + map.get(TYPE);
					if (Const.STR_1.equals(map.get(NULL)))
						sql += notNULL;
					if (Const.STR_1.equals(map.get(PKEY))) {
						if (pkey.length() > 0)
							pkey += Const.COMMA;
						pkey += field;
					}
					if (map.get(DEFT).length() > 0)
						sql += deFault + map.get(DEFT);
				}
			} else {
				field = map.get(Const.FIELD);
				sql += field + " " + map.get(TYPE);
				if (Const.STR_1.equals(map.get(NULL)))
					sql += notNULL;
				if (Const.STR_1.equals(map.get(PKEY))) {
					if (pkey.length() > 0)
						pkey += Const.COMMA;
					pkey += field;
				}
				if (map.get(DEFT).length() > 0)
					sql += deFault + map.get(DEFT);
			}

		}
		if (pkey.length() > 0)
			sql += ",PRIMARY KEY (" + pkey + ")";
		return sql;
	}

	/**
	 * 这里增加树状结构的一个节点，在父级节点下增加一个子节点
	 */
	public Map<String, String> addTreeNode(Map<String, Object> item,
			String[] keys, int tierW) {
		/**
		 * 表名：0-TABLE, 公司编号：1-COMP, 节点编号：2-ID, 父级节点：3-PID, 叶子节点：4-LEAF,
		 * 层级：5-LEVEL, 当前操作用户：6-OPUID
		 */
		Map<String, Object> where = new HashMap<>();
		where.put(keys[1], item.get(keys[1]));
		where.put(keys[3], item.get(keys[3])); // 同一个PID下的节点
		where.put(Const.I18N, item.get(Const.I18N));
		// 按节点编号升序排序
		Map<String, String> order = new HashMap<>();
		order.put(keys[2], "ASC");

		// 查询数据一览
		String base = keys.length > 7 ? keys[7] : null;
		String[] args = { base, keys[0], null, Const.STR_0 };
		List<Map<String, String>> list = findAll(where, order, args);

		// 生成部门编号，默认部门编号：DEFAULT_DEPT=001
		// 每层的部门编号位数 001001 类似
		// 部门数量超限
		int w = tierW, limit = 1;
		while (w-- > 0)
			limit = limit * 10;
		if (list.size() == limit - 1)
			throw new FailedException(ConfUtil.OVER_COUNTS);

		// 根据父级编号，生成新节点编号
		Object pid = item.get(keys[3]); // PID
		String itemId = utils.newLeafId(pid, list, keys[2], tierW);
		// 赋值节点
		item.put(keys[2], itemId); // ID
		item.put(keys[4], Const.STR_1); // LEAF
		item.put(keys[5], itemId.length() / tierW); // LEVEL
		// 增加一条新记录
		try {
			Map<String, String> result = addRecord(base, keys[0], item, true);
			if (!utils.isSuccess(result))
				throw new RuntimeException();
		} catch (Exception e) {
			LOG.error("addTreeNode...{}", item);
			throw new RuntimeException();
		}
		// 如果不是根节点，则需要更新父节点为“非叶子”状态
		if (!Const.STR_0.equals(pid)) {
			Map<String, Object> fields = new HashMap<>();
			fields.put(keys[4], Const.STR_0); // LEAF
			fields.put(keys[6], item.get(keys[6])); // OPUID
			// 把“叶子”状态，更新为“非叶子”状态
			Map<String, Object> cond1 = new HashMap<>();
			cond1.put(keys[1], item.get(keys[1])); // COMP
			cond1.put(keys[2], pid); // item.get(keys[3])); // ID\PID
			cond1.put(keys[4], Const.STR_1); // LEAF
			// 更新父节点
			update(base, keys[0], fields, cond1);
		}
		return utils.success();
	}

}
