package cn.ps1.aolai.utils;

import java.awt.AlphaComposite;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.image.BufferedImage;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Iterator;
import java.util.List;

import javax.imageio.ImageIO;
import javax.imageio.ImageReadParam;
import javax.imageio.ImageReader;
import javax.imageio.stream.ImageInputStream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 登录页面处理图形验证码相关业务处理
 * 
 * @author Aolai
 * @since 1.8 $Date: 2023.5.1
 * @version 1.0
 */
public class Captcha {

	private static Logger log = LoggerFactory.getLogger(Captcha.class);

	// 默认图形模板文件类型
	private static final String PNG = "png";
	private static final String EXT = ".png";

	private static int width;
	private static int height;

	/** 任意随机数 */
	static SecureRandom rand = new SecureRandom();

	/**
	 * 从资源图上根据模板尺寸切图
	 */
	public static List<String> imageCutout(File templateFile, File originFile, File blockFile) {
		List<String> imgList = new ArrayList<>();
		try {
			// 标准模板图尺寸如：{562, 342}
			BufferedImage templateImage = ImageIO.read(templateFile);
			width = templateImage.getWidth();
			height = templateImage.getHeight();
			// 资源大图，在ImageIO.read里校验文件为空
			BufferedImage bufImage = ImageIO.read(originFile);
			int oriW = bufImage.getWidth();
			int oriH = bufImage.getHeight();
			// 随机抠图的位置坐标，注意保持X轴距离右端、Y轴距离底部的偏移
			Point point = randomPoint(oriW - width, oriH - height);
			// 获取抠图的尺寸
			Rectangle rec = new Rectangle(point.x, point.y, width, height);

			// 从资源图上根据模板尺寸切图
			// 随机点开始切取的目标区域: Point{x,y} >> imgSize={width,height}
			BufferedImage target = cutTargetArea(originFile, rec);
			// 抠图用的随机滑块（如：五边形、心形、星形等）
			BufferedImage block = ImageIO.read(blockFile);

			// 随机切点
			Point pt = getCutPoint(block, 3, 5);
			// 首先，参考block的尺寸生成遮罩图案
			BufferedImage maskImg = getMaskImage(target, block, pt);
			// 目标区域上抠图后的新图像
			if (maskImg != null) {
				BufferedImage cutout = cutoutByBlock(maskImg, block, pt);

				// 仅测试用：saveToFile(target, "ori"); saveToFile(newImage, "cut");

				imgList.add(imgEncode(toByteArray(cutout)));
				imgList.add(imgEncode(toByteArray(maskImg)));
				imgList.add(String.valueOf(pt.y));
			}
		} catch(Exception e) {
			throw new FailedException(ConfUtil.RUN_FAILED);
		}
		return imgList;
	}

	/**
	 * 从资源图上根据模板尺寸切图，获取目标区域
	 */
	private static BufferedImage cutTargetArea(File originFile, Rectangle rec) {
		try (InputStream fis = new FileInputStream(originFile);
				ImageInputStream iis = ImageIO.createImageInputStream(fis)) {
			// 资源图的数据流
//			fis = new FileInputStream(originFile);
//			iis = ImageIO.createImageInputStream(fis);
			Iterator<ImageReader> imgReaderList = ImageIO.getImageReadersByFormatName(PNG);
			ImageReader imageReader = imgReaderList.next();
			// 输入源中的图像将只按顺序读取
			imageReader.setInput(iis, true);
			ImageReadParam param = imageReader.getDefaultReadParam();
			// 设定目标区域大小
			param.setSourceRegion(rec);
			// 返回数据
			return imageReader.read(0, param);
		} catch (Exception e) {
			// 抛出异常
			throw new FailedException(ConfUtil.RUN_FAILED);
		}
	}

	/**
	 * 随机生成一个切点
	 */
	private static Point getCutPoint(BufferedImage blockImage, int top, int left) {
		// 随机抠图的坐标点，注意保持X轴距离右端、Y轴距离底部的偏移
		int x = width - blockImage.getWidth() - width / left;
		int y = height - blockImage.getHeight() - top; // 顶部预留3px距离
		// 随机坐标点
		Point point = randomPoint(x, y);
		x = point.x + width / left; // 位置略偏右
		y = point.y + top; // 顶部预留3px距离
		return new Point(x, y);
	}

	/**
	 * 随机抠图处理：在背景图上随机位置处，把原图透明化处理
	 */
	private static BufferedImage cutoutByBlock(BufferedImage origin, BufferedImage block, Point p) {
		// 图像处理
		Graphics2D g2D = origin.createGraphics();
		// 图像叠加合成：0.7f为透明度，值从0-1.0，依次变得不透明
		g2D.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_ATOP, 0.7f));
		// 在背景图上随机位置处，叠加block透明水印
		g2D.drawImage(block, p.x, p.y, block.getWidth(), block.getHeight(), null);
		// 在背景图上随机位置处，把原图透明化处理
		// 透明化处理方法：setWaterMark(origin, block, p.x, p.y, 100);

		g2D.dispose(); // 处置
		return origin;
	}

	/**
	 * 生成遮罩图案
	 */
	public static BufferedImage getMaskImage(BufferedImage origin, BufferedImage block, Point p) {
		BufferedImage newImage = null;
		if (origin != null && block != null) {
			// 生成最终遮罩图案
			int w = block.getWidth();
			int h = block.getHeight(); // 图片高度

			// 这里高度用了origin.getHeight()
			newImage = new BufferedImage(w, origin.getHeight(), origin.getType());
			// 重构水印的遮罩图案
			for (int i = 0; i < w; i++) { // 从左到右，每一列
				for (int j = 0; j < h; j++) { // 从上到下，每一行
					// 如果图像当前像素点有图案色(!=0)，则复制源文件信息到目标图片中
					// 黑色像素对应值"-16777216"，蓝色像素"-16755216"
					if (block.getRGB(i, j) < 0) {
						newImage.setRGB(i, p.y + j, origin.getRGB(p.x + i, p.y + j));
					}
				}
			}
		}
		return newImage;
	}

	/**
	 * 增加透明水印（未使用）
	 */
	public static void setWaterMark(BufferedImage origin, BufferedImage block,
			int x, int y, int alpha) {
		// 添加水印图像
		for (int i = 0; i < block.getWidth(); i++) {
			// 图片高度
			for (int j = 0; j < block.getHeight(); j++) {
				// 如果图像当前像素点有图案色(!=0)，则叠加到源文件信息到目标图片中
				// 黑色像素对应值"-16777216"，蓝色像素"-16755216"
				if (block.getRGB(i, j) < 0) {
					Color color = new Color(origin.getRGB(x + i, y + j)); // 源像素
					Color newC = new Color(color.getRed(), color.getGreen(),
							color.getBlue(), alpha);
					origin.setRGB(x + i, y + j, newC.getRGB());
				}
			}
		}
	}

	/**
	 * 图片数据格式转换
	 */
	public static String imgEncode(byte[] bytes) {
		// 获取编码器
		Base64.Encoder encoder = Base64.getEncoder();
		String base64 = encoder.encodeToString(bytes);
		return "data:image/" + PNG + ";base64," + base64;
	}

	/**
	 * 图片数据格式转换：解码为字节数组
	 */
	public static byte[] imgDecode(String base64Str) {
		Base64.Decoder decoder = Base64.getDecoder();
		String[] arr = base64Str.split(",");
		base64Str = arr.length == 1 ? arr[0] : arr[1];
		return decoder.decode(base64Str); // 解码为字节数组
	}

	/**
	 * 图片数据格式转换
	 */
	public static BufferedImage toImage(String base64) {
		Base64.Decoder decoder = Base64.getMimeDecoder();
		try {
			String[] arr = base64.split(",");
			byte[] bytes = decoder.decode(arr[1]);
			InputStream bais = new ByteArrayInputStream(bytes);
			return ImageIO.read(bais);
		} catch (Exception e) {
			log.error("toImage...{}", e.getMessage());
			return null;
		}
	}

	/**
	 * 随机生成位置坐标点
	 */
	public static Point randomPoint(int widthDiff, int heightDiff) {
		int x = widthDiff <= 0 ? 0 : rand.nextInt(widthDiff);
		int y = heightDiff <= 0 ? 0 : rand.nextInt(heightDiff);
		return new Point(x, y);
	}

	/**
	 * 把图片转换为二进制数据
	 */
	private static byte[] toByteArray(BufferedImage bufImg) {
		// 新建流
		try (ByteArrayOutputStream os = new ByteArrayOutputStream()) {
			// 利用ImageIO类提供的write方法，以png图片的数据模式写入流
			ImageIO.write(bufImg, PNG, os);
			// 从流中获取数据数组
			return os.toByteArray();
			// 写随机文件：String f = Digest.uuid8() + EXT;
			// 写入流：os.writeTo(new FileOutputStream(f));
		} catch (Exception e) {
			// 抛出异常
			throw new FailedException(ConfUtil.RUN_FAILED);
		}
	}

	/**
	 * 轮廓边线比对，根据透明度计算像素位移
	 */
	public static int getBlockShift(String shadeStr, String blockStr, int y) {
		return getBlockShift(toImage(shadeStr), toImage(blockStr), y);
	}

	/**
	 * 轮廓边线比对，根据透明度计算像素位移
	 */
	public static int getBlockShift(BufferedImage shade, BufferedImage block, int y) {
		List<Point> points = getBlockSide(block);
		int w = shade.getWidth() - block.getWidth();
		int left = 0;
		int offset = 0;
		for (int x = 1; x < w; x++) {
			int i = 0;
			for (Point p : points) {
				if (alphaDiff(shade, x + p.x, y + p.y)) {
					i++; // 透明度不同
				}
			}
			if (left < i) {
				left = i;
				offset = x;
			}
		}
		return offset;
	}

	/**
	 * 轮廓边线比对，根据颜色值计算像素位移
	 */
	public static int getBlockShift(String shadeStr, String blockStr, int y, int diff) {
		return getBlockShift(toImage(shadeStr), toImage(blockStr), y, diff);
	}

	/**
	 * 轮廓边线比对，根据颜色值计算像素位移
	 */
	public static int getBlockShift(BufferedImage shade, BufferedImage block, int y, int diff) {
		List<Point> points = getBlockSide(block);
		int pixel = shade.getWidth() - block.getWidth();
		int left = 0;
		int offset = 0;
		for (int x = 1; x < pixel; x++) {
			int i = 0;
			for (Point p : points) {
				if (colorDiff(shade, x + p.x, y + p.y, diff)) {
					i++; // 找到相似度最低的边界
				}
			}
			if (left < i) {
				left = i;
				offset = x;
			}
		}
		return offset;
	}

	/**
	 * 获取block边缘的像素点
	 */
	private static List<Point> getBlockSide(BufferedImage block) {
		List<Point> list = new ArrayList<>();
		for (int y = 0; y < block.getHeight(); y++) {
			for (int x = 0; x < block.getWidth(); x++) {
				if (block.getRGB(x, y) < 0) {
					list.add(new Point(x, y));
					break;
				}
			}
		}
		return list;
	}

	/**
	 * 获颜色的差值
	 */
	private static boolean colorDiff(BufferedImage shade, int x, int y, int d) {
		try {
			Color c0 = new Color(shade.getRGB(x, y));
			Color c1 = new Color(shade.getRGB(x - 1, y - 1));
			Color c2 = new Color(shade.getRGB(x + 1, y + 1));
			if (Math.abs(c1.getRed() - c0.getRed()) > d
					|| Math.abs(c1.getGreen() - c0.getGreen()) > d
					|| Math.abs(c1.getBlue() - c0.getBlue()) > d
					|| Math.abs(c2.getRed() - c0.getRed()) > d
					|| Math.abs(c2.getGreen() - c0.getGreen()) > d
					|| Math.abs(c2.getBlue() - c0.getBlue()) > d)
				return true;
		} catch (Exception e) {
			log.error("colorDiff...{}", e.getMessage());
		}
		return false;
	}

	/**
	 * 获取透明度的差值
	 */
	private static boolean alphaDiff(BufferedImage shade, int x, int y) {
		int diff = 0;
		try {
			Color c0 = new Color(shade.getRGB(x, y), true);
			Color c1 = new Color(shade.getRGB(x - 1, y - 1), true);
			Color c2 = new Color(shade.getRGB(x + 1, y + 1), true);
			diff = Math.abs(c1.getAlpha() - c0.getAlpha());
			diff += Math.abs(c2.getAlpha() - c0.getAlpha());
		} catch (Exception e) {
			log.error("alphaDiff...{}", e.getMessage());
		}
		return diff > 0;
	}

	/** * * 以下为测试内容 * * */

	public static void main(String[] args) {
		try {
			/**
			 * 可以用以下测试文件：bag.png，origin.png，block.png<br>
			 * 模板：File tempFile = new File("D:\\javaApp\\doyea\\bag.png");<br>
			 * 滑块：File blockFile = new File("D:\\javaApp\\doyea\\block.png");<br>
			 * 源：File originFile = new File("D:\\javaApp\\doyea\\origin.png");<br>
			 * 分割：imageCutout(templateFile, originFile, blockFile);
			 */

//			File f0 = new File("D:\\javaApp\\aolai\\block1.png");
//			File f1 = new File("D:\\javaApp\\aolai\\shade2.png");
//			BufferedImage block = ImageIO.read(f0);
//			BufferedImage shade = ImageIO.read(f1);
//			/**
//			 * 挖孔：cutoutByBlock(shade, block, new Point(80, 47));<br>
//			 * 点：Point pt = new Point(80, 47);<br>
//			 * 新图：BufferedImage newImg = getMaskImage(shade, block, pt);<br>
//			 * 保存：saveToFile(newImg, "newImg");
//			 */
//			getBlockShift(shade, block, 47);

			log.debug("sucess!");
		} catch (Exception e) {
			log.error("test...{}", e.getMessage());
		}
	}

	/**
	 * 仅测试用
	 */
	public static void saveToFile(BufferedImage bufImg, String name) {
		// 新建流
		try (ByteArrayOutputStream os = new ByteArrayOutputStream()){
			// 利用ImageIO类提供的write方法，以png图片的数据模式写入流
			ImageIO.write(bufImg, PNG, os);
			os.writeTo(new FileOutputStream(randFileName(name)));
		} catch (Exception e) {
			log.error("saveToFile...{}", e.getMessage());
		}
	}

	/**
	 * 仅测试用
	 */
	public static void saveToFile(byte[] bytes, String name) {
		File file = new File(randFileName(name));
		try (FileOutputStream fos = new FileOutputStream(file);
				BufferedOutputStream bos = new BufferedOutputStream(fos)) {
			bos.write(bytes);
		} catch (Exception e) {
			// 抛出异常
			throw new FailedException(ConfUtil.RUN_FAILED);
		}
	}

	/**
	 * 生成一个随机文件名
	 */
	private static String randFileName(String name) {
		return name + System.currentTimeMillis() + EXT;
	}

}
