package cn.ps1.aolai.service;

import java.io.Closeable;
import java.lang.reflect.Array;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.DecimalFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringJoiner;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.ps1.aolai.entity.User;
import cn.ps1.aolai.utils.ConfUtil;
import cn.ps1.aolai.utils.Const;
import cn.ps1.aolai.utils.Digest;
import cn.ps1.aolai.utils.FailedException;
import cn.ps1.aolai.utils.FmtUtil;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * 全局通用的公共方法Utils类
 * 
 * @author Aolai
 * @since  1.7 $Date: 2017.6.17
 * @version 1.0
 */

@Service
public class UtilsService {

	private static Logger log = LoggerFactory.getLogger(UtilsService.class);
	private static ObjectMapper objMapper = new ObjectMapper();

	private class TypeRef<T> extends TypeReference<T> {
	}

	@Autowired
	HttpServletRequest req; // 单实例可以注入

	/**
	 * 构造函数
	 */
	public UtilsService() {
		// 这里仅为了测试用
	}

	public void doFailed() {
		throw new FailedException(ConfUtil.RUN_FAILED);
	}

	/**
	 * 获取从前端HTTP请求传递来的请求参数
	 */
	public Map<String, Object> jsonParams() {
		return jsonParams(req);
	}

	/**
	 * 获取从前端HTTP请求传递来的请求参数：梳理功能已移到ThirdService.setJsonAttr()
	 */
	public Map<String, Object> jsonParams(HttpServletRequest req) {
		try {
			return obj2Map(req.getAttribute(Const.JSON));
		} catch (Exception e) {
			// 尚未完成初始化（未登录）之前，系统参数并未设置，需要增加抛出异常处理
			return new HashMap<>();
		}
	}

	/**
	 * 获取从前端HTTP请求传递来的请求参数：梳理功能已移到ThirdService.setJsonAttr()
	 */
	public Map<String, Object> jsonParams(String[] keys) {
		Map<String, Object> params = jsonParams();
		for (String key : keys) {
			if (!params.containsKey(key)) // 无效参数
				// String.join("_", ConfUtil.INVD_PARAMS, key)
				throw new FailedException(null, key);
		}
		return params;
	}

	/**
	 * 检查必须携带的参数
	 */
/*	boolean checkParams(Map<?, ?> map, String[] keys) {
		if (map == null) // 无效参数
			throw new FailedException();
		for (String key : keys) {
			if (!map.containsKey(key)) // 无效参数
				throw new FailedException(String.join("_", ConfUtil.INVD_PARAMS, key));
		}
		return true;
	}*/

	/**
	 * 有效检查：请求（输入）参数是否有效存在
	 */
	public boolean availParams(Map<?, ?> map, String[] keys) {
		if (map == null) // 无效参数
			return false;
		for (String key : keys) {
			if (!map.containsKey(key)) // 无效参数
				return false;
		}
		return true;
	}

	/**
	 * 请求参数为无效参数
	 */
	public Map<String, String> invalidParams() {
		return result(Const.S_2);
	}

	/**
	 * 当前操作用户自己，相当于getSession()
	 */
	public Map<String, String> userSelf() {
		return obj2Map(req.getAttribute(User.KEY));
	}

	/**
	 * 当前操作用户自己，相当于getSession()
	 */
	public Map<String, String> userSelf(HttpServletRequest req) {
		return obj2Map(req.getAttribute(User.KEY));
	}

	/**
	 * 设置当前操作用户的公司ID后返回公司ID
	 */
	public String setUserComp(Map<String, Object> params, String key) {
		return setUserComp(req, params, key);
	}

	/**
	 * 设置当前操作用户的公司ID后返回公司ID
	 */
	public String setUserComp(HttpServletRequest req, Map<String, Object> params, String key) {
		String userComp = userSelf(req).get(User.COMP);
		params.put(key, userComp);
		return userComp;
	}

	/**
	 * 清除无效的状态数据
	 */
	public void cleanState(Map<String, Object> cond, String key) {
		Object state = cond.get(key);
		if (!Const.S_0.equals(state) && !Const.S_1.equals(state))
			cond.remove(key);
	}

	/**
	 * 默认查询条件为状态为“1”<br>
	 * 用于重新设置查询条件的数据状态：一般默认设置为“0”“1”的情况
	 */
	public void resetState(Map<String, Object> cond, String key) {
		Object state = cond.get(key);
		if (Const.ALL.equals(state)) {
			cond.put(FmtUtil.pHolder(key, Const.NEQ), Const.S_9);
			cond.remove(key);
		} else {
			cond.put(key, state == null ? Const.S_1 : Const.S_0);
		}
	}

	/**
	 * 根据父节点新建一个新节点，如ID：0102 (bitW=2)
	 * 
	 * @param pId 父级编码
	 * @param id 子级编号
	 * @param bitW 编号层级的宽度，如：bitW=3
	 * @return String
	 */
	public String idCode(Object pId, int id, int bitW) {
		// 第一个子节点必为“1”，拼接父节点的ID后，变为0x0x01
		// 第二个子节点之后则是“x0x02”，只需前面补充一个“0”即可
		String pid = isNil(pId) ? "" : String.valueOf(pId);
		// 此处修复从数据库查询出最大的编码格式 与直接截取编码中最大的编码
		// 分两种情况 如果从数据库中查询的最大编码 例如 0114 ，层级宽度为 2 ，则id会变为 114，此时截取子集编码为14 
		// 第二种情况，外部处理好编码，id原始编码为 001014，层级宽度为三级， 此时直接传入id为 14, 此时不做处理即可
		String maxId = String.valueOf(id);
		if (maxId.length() > bitW) {
			// 如果bitW=2位数，maxId=100则超数量限制
			return pid + maxId.substring(maxId.length() - bitW);
		}
		String fmt = "%0" + bitW + "d";
		return pid + String.format(fmt, id);
	}

	/**
	 * 拼装检查数据重复的条件，默认“第一个”值是主键，首先要排除主键
	 */
	public Map<String, Object> sameIf(Map<String, ?> cond, String[] keys) {
		// 排除自己
		Map<String, Object> where = sameIf(cond, 1, keys);
		if (cond.containsKey(keys[0]))
			where.put(pHolder(keys[0], Const.NEQ), cond.get(keys[0]));
		return where;
	}

	/**
	 * 检索条件转换
	 */
	public Map<String, Object> sameIf(Map<String, ?> cond, int offset, String[] keys) {
		// 不能有重名的数据
		Map<String, Object> where = newCond(cond);
		for (int i = offset; i < keys.length; i++)
			where.put(keys[i], cond.get(keys[i]));
		return where;
	}

	/**
	 * 转换新检索条件
	 */
	public <T> Map<String, T> newMap(Map<String, T> params, String key) {
		return newMap(params, new String[] { key });
	}
	
	/**
	 * 转换新检索条件
	 */
	public <T> Map<String, T> newMap(Map<String, T> params, String[] keys) {
//		Map<String, Object> cond = newCond(params);
		Map<String, T> cond = new HashMap<>();
		for (String key : keys) {
			if (params.containsKey(key))
				cond.put(key, params.get(key));
		}
		return cond;
	}

	/**
	 * 转换新检索条件
	 */
	public Map<String, Object> newCond(Map<String, ?> params) {
		Map<String, Object> cond = new HashMap<>();
		cond.put(Const.I18N, params.get(Const.I18N));
		if (ConfUtil.dbid().length() > 0)
			cond.put(ConfUtil.dbid(), params.get(ConfUtil.dbid()));
//		if (params.containsKey(Const.BASE))
			cond.put(Const.BASE, params.get(Const.BASE));
		return cond;
	}

	/**
	 * 分页查询条件
	 */
	public Map<String, Object> pageCond(Map<String, Object> params) {
		String[] keys = { Const.PAGE_NO, Const.PAGE_SIZE };
		return newMap(params, keys);
	}

	/**
	 * 联合查询条件
	 */
	public Map<String, Map<String, String>> joinCond(String table) {
		Map<String, Map<String, String>> tables = new LinkedHashMap<>();
		tables.put(table, null);
		return tables;
	}

	/**
	 * 提取排序参数
	 */
	public Map<String, String> getSort(Map<String, Object> where) {
		// 排序参数
		Object sort = where.get(Const.SORT);
		return sort == null ? null : obj2Map(sort);
	}

	/**
	 * 检索条件转换，仅适用如randId格式的数据
	 */
	public Map<String, Object> sameId(Map<String, ?> cond, String key) {
		return sameIf(cond, 0, new String[] { setComp(key), key + "Id" });
	}

	/**
	 * 检索条件转换，仅适用如nodeNo格式的数据
	 */
	public Map<String, Object> sameNo(Map<String, ?> cond, String key) {
		return sameIf(cond, 0, new String[] { setComp(key), key + "No" });
	}

	/** 公司主键 */
	private String setComp(String key) {
		return key + "Comp";
	}

	/**
	 * 把数串逐级累计（科目层级转换）
	 * 
	 * @param numStr 数字的字符串
	 * @return Array 累计数组
	 */
	public int[] num2Arr(String numStr) {
		int[] num = str2num(numStr);
		for (int i = 1; i < num.length; i++) {
			num[i] += num[i - 1];
		}
		return num;
	}

	/**
	 * 字符串分割为整数数组
	 */
	public int[] str2num(String numStr) {
		if (!isInteger(numStr))
			return new int[0];
		int[] num = new int[numStr.length()];
		for (int i = 0; i < numStr.length(); i++) {
			num[i] = Integer.parseInt(numStr.substring(i, i + 1));
		}
		return num;
	}

	/**
	 * 数组合并拼接为字符串
	 * @deprecated 这个方法已被弃用，并且在未来版本用join()替换。
	 */
	@Deprecated
	public <T> String arr2Str(T[] strArr) {
		return join(strArr, "");
	}

	/**
	 * 数组合并拼接为字符串
	 */
	public <T> String join(T[] strArr) {
		return join(strArr, "");
	}

	/**
	 * 数组合并拼接为字符串
	 */
	public <T> String join(T[] strArr, String joint) {
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < strArr.length; i++) {
			sb.append(i > 0 ? joint + strArr[i] : strArr[i]);
		}
		return sb.toString();
	}

	/**
	 * 从对象中获取整数值
	 */
	public int getInt(Map<String, ?> map, String key) {
		if (map != null) {
			String val = String.valueOf(map.get(key));
			if (isInteger(val))
				return Integer.parseInt(val);
		}
		return 0;
	}

	/**
	 * 从对象中获取整数值
	 */
	public long getLong(Map<String, ?> map, String key) {
		if (map != null) {
			String val = String.valueOf(map.get(key));
			if (isInteger(val))
				return Long.parseLong(val);
		}
		return 0;
	}

	/**
	 * 将字符串的首字母转大写
	 * <p>
	 * 用字母的ascii编码前移，效率要高于截取字符串进行转换的操作
	 */
	public String initCap(Object str) {
		char[] cs = String.valueOf(str).toLowerCase().toCharArray();
		cs[0] -= 32;
		return String.valueOf(cs);
	}

	/**
	 * Map对象数据互转
	 */
	public Map<String, Object> map2Obj(Map<String, ?> map) {
		Map<String, Object> obj = new HashMap<>();
		if (map != null)
			for (Map.Entry<String, ?> e : map.entrySet()) {
				obj.put(e.getKey(), e.getValue());
			}
		return obj;
	}

	/**
	 * 通过参数映射解析器，进行参数主键转换<br>
	 * 转换：newMap.put(parser.getKey(), oldMap.get(parser.getValue()));<br>
	 * 映射示例：parser={empUid=userId, empComp=userComp, empName=userName}<br>
	 */
	public <T> List<Map<String, T>> parse(Map<String, ?> parser, List<Map<String, T>> list) {
		// 处理结果数据
		if (!isEmpty(parser))
			for (int i = 0; i < list.size(); i++) {
				if (list.get(i).isEmpty())
					break;
				list.set(i, parse(parser, list.get(i)));
			}
		return list;
	}

	/**
	 * 通过参数映射解析器，进行参数主键转换<br>
	 * 转换：newMap.put(parser.getKey(), oldMap.get(parser.getValue()));<br>
	 * 映射示例：parser={empUid=userId, empComp=userComp, empName=userName}<br>
	 */
	public <T> Map<String, T> parse(Map<String, ?> parser, Map<String, T> map) {
		if (isEmpty(parser))
			return map;
		Map<String, T> item = new HashMap<>();
		for (Map.Entry<String, ?> e : parser.entrySet()) {
			if (map.containsKey(e.getValue()))
				item.put(e.getKey(), map.get(e.getValue()));
		}
		return item;
	}

	/**
	 * 通过参数映射解析器，进行参数主键转换<br>
	 * @deprecated 这个方法已被弃用，并且在未来版本用parse()替换。
	 */
	@Deprecated
	public <T> Map<String, T> mapParser(Map<String, ?> parser, Map<String, T> map) {
		return parse(parser, map);
	}

	/**
	 * 通过参数映射解析器，进行参数主键转换<br>
	 * @deprecated 这个方法已被弃用，并且在未来版本用parse()替换。
	 */
	@Deprecated
	public <T> List<Map<String, T>> mapParser(Map<String, ?> parser, List<Map<String, T>> list) {
		return parse(parser, list);
	}

	/**
	 * 对象转换为Map对象
	 */
	@SuppressWarnings("unchecked")
	public <T> Map<String, T> obj2Map(Object obj) {
		return obj instanceof Map ? (Map<String, T>) obj : new HashMap<>();
	}

	/**
	 * 对象转换为List对象
	 */
	@SuppressWarnings("unchecked")
	public <T> List<T> obj2List(Object obj) {
		return obj instanceof Collection ? (List<T>) obj : new ArrayList<>();
	}

	/**
	 * Map对象数据互转化，含对象拷贝
	 */
	public Map<String, String> obj2Map(Map<String, ?> obj) {
		Map<String, String> map = new HashMap<>();
		if (obj != null)
			for (Map.Entry<String, ?> e : obj.entrySet()) {
				Object val = e.getValue();
				map.put(e.getKey(), val == null ? null : String.valueOf(val));
			}
		return map;
	}

	/**
	 * 把对象（Map、List）转为jsonStr字符串
	 * 
	 * @param obj
	 * @return jsonStr
	 */
	public String obj2Str(Object obj) {
		try {
			return objMapper.writeValueAsString(obj);
		} catch (Exception e) {
			// 这里基本不可能出错
			log.error("obj2Str...{}", e.getMessage());
			return null;
		}
	}

	/**
	 * 从Json字符串中找出Map对象节点
	 */
	public Map<String, Object> node2Obj(String jsonStr, String path) {
		try {
			if (jsonStr != null) {
				JsonNode node = objMapper.readTree(jsonStr);
				return node2Obj(path == null ? node : node.path(path));
			}
		} catch (Exception e) {
			error("node2Obj", jsonStr);
		}
		return new HashMap<>();
	}

	/**
	 * Json对象字符串转换为Map对象
	 * 
	 * @param jsonStr
	 * @return Map
	 */
	public Map<String, Object> node2Obj(String jsonStr) {
		return node2Obj(jsonStr, null);
	}

	/**
	 * Json对象转换为Map对象
	 * 
	 * @param jsonNode
	 * @return Map
	 */
	public Map<String, Object> node2Obj(JsonNode jsonNode) {
		Map<String, Object> map = new HashMap<>();
		if (jsonNode.isValueNode()) {
			map.put("", jsonNode.asText());
		} else if (jsonNode.isArray()) {
			List<Object> list = new ArrayList<>();
			Iterator<JsonNode> it = jsonNode.iterator();
			while (it.hasNext()) {
				Map<String, Object> child = node2Obj(it.next());
				if (child.keySet().size() == 1 && child.keySet().contains("")) {
					list.add(child.get(""));
				} else {
					list.add(child);
				}
			}
			map.put("", list);
		} else {
			Iterator<Map.Entry<String, JsonNode>> it = jsonNode.fields();
			while (it.hasNext()) {
				Map.Entry<String, JsonNode> entity = it.next();
				Map<String, Object> child = node2Obj(entity.getValue());
				if (child.keySet().size() == 1 && child.keySet().contains("")) {
					map.put(entity.getKey(), child.get(""));
				} else {
					map.put(entity.getKey(), child);
				}
			}
		}
		return map;
	}

	/**
	 * 记录错误，并省略过长的字符串
	 */
	void error(String method, String str) {
		str = str.length() < 200 ? str : str.substring(0, 200) + "...";
		log.error("{}...'{}'", method, str);
	}

	/**
	 * 拼接SQL字符串的多个OR主键
	 */
	public String sqlOr(String field0, String field1) {
		return sqlOr(new String[] { field0, field1 });
	}

	/**
	 * 拼接SQL字符串的多个OR主键
	 */
	public String sqlOr(String[] fields) {
		StringJoiner s = new StringJoiner(" or ", "(", ")");
		for (String field : fields)
			s.add(pHolder(field, "={}"));
		return s.toString();
	}

	/**
	 * 拼接SQL字符串的单引号'val'值
	 */
	public String sqlVal(Object val) {
		return FmtUtil.quote(val, "'");
	}

	/**
	 * 拼接json格式的SQL语句
	 */
	public String jsonExt(String field, Object key) {
		return jsonExt(field, key, Const.EQU);
	}

	/**
	 * 拼接json格式的SQL语句
	 */
	public String jsonExt(String field, Object key, String opr) {
		String[] keys = { " json_extract({", field, "},'$." };
		return join(keys) + key + "') " + opr;
	}

	/**
	 * 把jsonStr转换成为Map对象
	 * 
	 * @param str
	 * @return Map
	 */
	public <T> Map<String, T> json2Map(Object str) {
		try {
			return objMapper.readValue((String) str, new TypeRef<Map<String, T>>());
		} catch (Exception e) {
			// 例如：HTTP/1.1 500 Internal Server Error
			log.warn("json2Map...{}", e.getMessage());
		}
		return new HashMap<>();
	}

	/**
	 * 把json字符串转换成为list对象
	 */
	public <T> List<T> json2List(Object str) {
		try {
			return objMapper.readValue((String) str, new TypeRef<List<T>>());
		} catch (Exception e) {
			log.warn("json2List...{}", e.getMessage());
		}
		return new ArrayList<>();
	}

	/**
	 * 获取list中某一列的值，list中某列的数值转换为数组并去重
	 */
	public <T> String[] getArr(List<Map<String, T>> list, String key) {
		String[] arr = new String[list.size()];
		for (int i = 0; i < list.size(); i++)
			arr[i] = (String) list.get(i).get(key);
		return arr;
	}

	/**
	 * 获取list中某一列的值，list中某列的数值转换为Set并去重
	 */
	public <T> Set<String> toSet(List<Map<String, T>> list, String key) {
		Set<String> set = new HashSet<>();
		for (int i = 0; i < list.size(); i++)
			set.add((String) list.get(i).get(key));
		return set;
	}

	/**
	 * 专为支持数值{V}而设，把List转换为数组
	 */
	@SuppressWarnings("unchecked")
	String[] toArr(Object obj) {
		List<String> valList = new ArrayList<>();
		if (obj.getClass().isArray()) {
			for (Object o : (Object[]) obj) {
				String s = String.valueOf(o);
				// 为兼容已经预处理过的sqlVal
				valList.add(isMatch(s, "'.*'") ? s : sqlVal(s));
			}
		} else if (obj instanceof Collection) {
			for (Object o : (Collection<Object>) obj)
				valList.add(sqlVal(o));
		} else {
			valList.add(sqlVal(obj));
		}
		return valList.toArray(new String[0]);
	}

	/**
	 * list转换为键值对格式的map对象
	 */
	public <T> Map<String, Map<String, T>> toMap(List<Map<String, T>> list, String key) {
		return list2Map(list, key);
	}

	/**
	 * list转换为键值对格式的map对象
	 */
	public Map<String, String> toMap(List<Map<String, String>> list, String key, String val) {
		return list2Map(list, key, val);
	}

	/**
	 * list转换为键值对格式的map对象
	 */
	public <T> Map<String, Map<String, T>> list2Map(List<Map<String, T>> list,
			String key) {
		Map<String, Map<String, T>> map = new HashMap<>();
		if (list != null)
			for (Map<String, T> item : list) {
				map.put((String) item.get(key), item);
			}
		return map;
	}

	/**
	 * list转换为键值对格式的map对象
	 */
	public Map<String, String> list2Map(List<Map<String, String>> list, String key, String val) {
		Map<String, String> map = new HashMap<>();
		if (list != null)
			for (Map<String, String> item : list) {
				map.put(item.get(key), item.get(val));
			}
		return map;
	}

	/**
	 * 返回状态结果
	 * 
	 * @param status "1".成功，"0".失败、"2".参数错误、"3".Token失效
	 * @return Map
	 */
	public Map<String, String> result(String status) {
		Map<String, String> result = new HashMap<>();
		result.put(Const.STS, status);
		return result;
	}

	/**
	 * 返回状态结果：false-失败、true-成功
	 */
	public Map<String, String> result(boolean status) {
		return result(status ? Const.S_1 : Const.S_0);
	}

	/**
	 * 返回状态结果
	 * 
	 * @param status "1".成功，"0".失败、"2".参数错误、"3".Token失效
	 * @param info 返回信息或数据
	 */
	public Map<String, String> result(String status, String info) {
		Map<String, String> result = result(status);
		if (info != null)
			result.put(Const.INFO, info);
		return result;
	}

	public <T> Object result(Map<String, T> map) {
		return map.containsKey(Const.STS) ? map : success(map);
	}

	/**
	 * 返回失败消息
	 */
	public Map<String, String> failed(String info) {
		return result(Const.S_0, info);
	}

	public <T> boolean isFailed(Map<String, T> result) {
		return isNil(result.get(Const.STS));
	}

	public <T> boolean isSuccess(Map<String, T> result) {
		return Const.S_1.equals(result.get(Const.STS));
	}

	/**
	 * 返回成功
	 */
	public Map<String, String> success(String info) {
		return result(Const.S_1, info);
	}

	public Map<String, String> success() {
		return result(Const.S_1);
	}

	/**
	 * 返回数据对象
	 */
	public Map<String, Object> success(Object obj) {
		Map<String, Object> result = new HashMap<>();
		result.put(Const.STS, Const.S_1);
		result.put(Const.INFO, obj);
		return result;
	}

	/**
	 * 返回前台加密后的信息（注意：应用场景比较少，可能在个别地方单独使用）
	 * @deprecated
	 */
	@Deprecated
	public Map<String, String> success(Object obj, HttpServletRequest req) {
		String cert = (String) req.getAttribute(ConfUtil.CERTID);
		return success(Digest.decrypt(obj2Str(obj), cert));
	}

	/**
	 * 找出List列表中匹配的指定字符串
	 */
	public <T> Map<String, T> findIn(List<Map<String, T>> list, String key, String val) {
		if (list != null && val != null)
			for (Map<String, T> map : list) {
				if (val.equals(map.get(key)))
					return map;
			}
		return null;
	}

	/**
	 * 找出List列表中是否存在指定字符串
	 */
	public boolean findIn(List<String> list, Object str) {
		return list != null && list.contains(str);
	}

	/**
	 * 字符串数组中是否存在指定的字符串
	 */
	public boolean findIn(String[] arr, Object str) {
		return arr != null && Arrays.asList(arr).contains(str);
	}

	/**
	 * 判断字符串是否为“0”
	 */
	public boolean isNil(Object obj) {
		return Const.S_0.equals(obj);
	}

	/**
	 * 判断字符串是否为空
	 */
	public boolean isBlank(Object obj) {
		return "".equals(obj);
	}

	/**
	 * 判断字符串是否为空
	 */
	public boolean isEmpty(Object obj) {
		if (obj == null)
			return true;
		if (obj instanceof CharSequence) // 字符串
			return ((CharSequence) obj).length() == 0;
		if (obj instanceof Map)
			return ((Map<?, ?>) obj).isEmpty();
		if (obj instanceof Collection) // List列表、Set等
			return ((Collection<?>) obj).isEmpty();
		if (obj.getClass().isArray()) // 数组
			return Array.getLength(obj) == 0;
		return false;
	}

	/**
	 * 判断对象是否为空
	 */
	public <T> boolean isEmpty(Map<String, T> map) {
		return map == null || map.isEmpty();
	}

	/**
	 * 字符串中是否有完全匹配：匹配字符或数字
	 * 
	 * @return boolean 成功与否
	 */
	public boolean isMatch(String str, String regex) {
		return FmtUtil.isMatch(str, regex);
	}

	/**
	 * 判断是否为整数、负数
	 */
	public boolean isInteger(String str) {
		// ^[-\\+]?[\\d]+$匹配前面的子表达式一次或多次
		return FmtUtil.isInteger(str);
	}

	/**
	 * 判断是否为浮点数：包含整数、小数、负数
	 */
	public boolean isNumeric(String str) {
		return FmtUtil.isNumeric(str);
	}

	/**
	 * 判断是否为身份证号码："^\\d{17}[\\dXx]$"
	 */
	public boolean isIdcard(String str) {
		return isMatch(str, "\\d{17}[\\dXx]");
	}

	/**
	 * 判断是否为手机号码："^1[3-9]\\d{9}"
	 */
	public boolean isMobile(String str) {
		return isMatch(str, "1[3-9]\\d{9}");
	}

	/**
	 * 判断是否为时间格式
	 */
	public boolean isDatetime(String dateStr, String pat) {
		try {
			if (dateStr != null) {
				new SimpleDateFormat(pat).parse(dateStr);
				return true;
			}
		} catch (ParseException e) {
			error("isDatetime", dateStr);
		}
		return false;
	}

	/**
	 * 时间（日期）格式转变
	 */
	public String timeFormat(String str, String pat0, String pat1) {
		return FmtUtil.timeParse(str, pat0, pat1);
	}

	/**
	 * 计算两个日期之间的差数多少（毫秒数）：begin开始时间，end结束时间
	 */
	public long timeDiff(Date begin, Date end) {
		Calendar cal = Calendar.getInstance();
		cal.setTime(end);
		long msec = cal.getTimeInMillis();
		cal.setTime(begin);
		// 返回毫秒数
		return msec - cal.getTimeInMillis();
	}

	/**
	 * 计算两个日期之间的差数多少（毫秒数）：begin开始时间，end结束时间
	 */
	public long timeDiff(String begin, String end) {
		// yyyy-MM-dd HH:mm:ss
		return diffMsec(begin, end, Const.DTF);
	}

	/** 两个时间之间的毫秒间隔 */
	public long diffMsec(String begin, String end) {
		// yyyy-MM-dd HH:mm:ss.SSS
		return diffMsec(begin, end, Const.MSEC);
	}

	/** 两个时间之间的毫秒间隔 */
	private long diffMsec(String begin, String end, String pat) {
		try {
			if (!isEmpty(begin)) {
				SimpleDateFormat sdf = new SimpleDateFormat(pat);
				Date endDate = end == null ? new Date() : sdf.parse(end);
				return timeDiff(sdf.parse(begin), endDate);
			}
		} catch (ParseException e) {
			log.error("obj2Str...{}", e.getMessage());
		}
		return 0;
	}

	/**
	 * 根据日期格式，获取当前日期时间：pattern格式
	 */
	public String today(String pattern) {
		return FmtUtil.timeParse(Calendar.getInstance(), pattern);
	}

	/**
	 * 当前日期推迟N天后，格式：yyyyMMdd
	 */
	public String daysLater(int d) {
		LocalDate day = LocalDate.now().plusDays(d);
		return day.format(DateTimeFormatter.ofPattern(Const.DAY));
	}

	/**
	 * 当前日期推迟N天后，格式：yyyyMMdd
	 */
	public String nextDate(String date, int d) {
		DateTimeFormatter dtf = DateTimeFormatter.ofPattern(Const.DAY);
		LocalDate day = LocalDate.parse(date, dtf).plusDays(d);
		return day.format(dtf);
	}

	/**
	 * 根据日期格式，获取当前日期时间：pattern格式
	 */
	// 用FormatUtil.timeParse替代
//	private String timeFmt(String pattern, Calendar cal) {
//		return FormatUtil.timeParse(cal, pattern);
//	}

	/**
	 * 当前时间节点向后（或向前）推迟n年
	 */
	public String nextYear(String pattern, int y) {
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.YEAR, y);
		return FmtUtil.timeParse(cal, pattern);
	}

	/**
	 * 获得下个月期间，格式：yyyyMM
	 */
	public String nextMonth(String month, int m) {
		SimpleDateFormat sdf = new SimpleDateFormat(Const.MON);
		Calendar cal = Calendar.getInstance();
		try {
			month = isEmpty(month) ? today(Const.MON) : month;
			cal.setTime(sdf.parse(month)); // 设置当前月份
		} catch (ParseException e) {
			error("nextMonth", month);
		}
		cal.add(Calendar.MONTH, m);
		return sdf.format(cal.getTime());
	}

	/**
	 * 获得下个月期间，格式：yyyyMM
	 */
	public String nextMonth(String month) {
		return nextMonth(month, 1);
	}

	/**
	 * 获得上个月期间，格式：yyyyMM，可以用替代方法：nextMonth(month, -m)
	 */
	public String prevMonth(String month, int m) {
		return nextMonth(month, -m);
	}

	/**
	 * 获得上个月期间，格式：yyyyMM
	 */
	public String prevMonth(String month) {
		return nextMonth(month, -1);
	}

	/**
	 * 获取host、Port
	 * 
	 * @param uri
	 * @return String[]
	 */
	private String[] getHost(String uri) {
		int idx = uri.indexOf("://"); // 截取http://之后的值如：www.eyun.cn/
		uri = idx > 0 ? uri.substring(idx + 3) : uri;
		idx = uri.indexOf('/'); // 截取‘/’之前的值：www.eyun.cn
		uri = idx > 0 ? uri.substring(0, idx) : uri;
		return uri.split(Const.COLON);
	}

	/**
	 * 获取请求URI参数，注意：返回值改为不带斜线，如：uri="wsSaveTest1"
	 */
	public String getRequestURI(HttpServletRequest req) {
		// 获取数据的示例：“/ws/wsSaveTest1”
		String uri = req.getServletPath();
		// 注意：返回值改为不带斜线，返回数据的示例如：“wsSaveTest1”
		return uri.substring(uri.lastIndexOf('/') + 1);
	}

	/**
	 * 获取域名http://xxx.ps1.cn:8080/xxx，放置cookie使用
	 */
	private String getDomain(HttpServletRequest req) {
		// 泉州同时包含域名、IP登录，需要动态获取。
		String ref = req.getHeader("Referer");
		if (isEmpty(ref))
			ref = req.getRequestURL().toString();
		log.debug("getDomain...{}", ref);
		// 获取的结果如：{"xxx.ps1.cn","8080"}
		String host = getHost(ref)[0];
		String[] arr = host.split("\\.");
		// 返回结果如：ps1.cn
		return arr.length == 3 ? arr[1] + Const.DOT + arr[2] : host;
	}

	/**
	 * 设置cookies
	 */
	public void setCookies(HttpServletRequest req, HttpServletResponse rsp, Map<String, String> map) {
		if (!isEmpty(map)) {
			String domain = getDomain(req);
			for (Map.Entry<String, String> e : map.entrySet()) {
				String k = e.getKey();
				// 不缓存证书信息
				if (k.equals(ConfUtil.CERTID))
					continue;
				Cookie c = new Cookie(k, Digest.urlEncode(e.getValue()));
				// 设置HttpOnly属性，那么通过js脚本将无法读取到cookie信息
				c.setHttpOnly(true);
				// 是否只在使用 SSL/TLS 安全连接时发送 Cookie
//				c.setSecure(true);
				setCookiePath(c, domain);
				// 若需要安全保护：c.setHttpOnly(true);
				rsp.addCookie(c);
			}
		}
	}

	/**
	 * 获取所有Cookie键值对
	 */
	public Map<String, String> getCookies(HttpServletRequest req) {
		Map<String, String> map = new HashMap<>();
		Cookie[] cookies = req.getCookies();
		if (cookies != null)
			for (Cookie c : cookies) {
				map.put(c.getName(), Digest.urlDecode(c.getValue()));
			}
		return map;
	}

	/**
	 * 获取指定的Cookie
	 */
	public String getCookie(HttpServletRequest req, String key) {
		return getCookies(req).get(key);
	}

	/**
	 * 刪除指定的Cookie
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	@Deprecated
	public void delCookie(HttpServletRequest req, HttpServletResponse rsp,
			String key) {
		Cookie[] cookies = req.getCookies();
		if (cookies != null)
			for (Cookie c : cookies) {
				if (c.getName().equals(key)) {
					setCookiePath(c, getDomain(req));
					c.setMaxAge(0); // 立即销毁cookie
					rsp.addCookie(c);
					break;
				}
			}
	}

	/**
	 * 清除全部Cookie的键值
	 */
	public void clearCookies(HttpServletRequest req, HttpServletResponse rsp) {
		Cookie[] cookies = req.getCookies();
		if (cookies != null) {
			String domain = getDomain(req);
			for (Cookie c : cookies) {
				setCookiePath(c, domain);
				c.setMaxAge(0); // 立即销毁cookie
				rsp.addCookie(c);
			}
		}
	}

	/**
	 * 设置CookiePath
	 */
	private void setCookiePath(Cookie c, String domain) {
		c.setDomain(domain);
		c.setPath("/");
	}

	/**
	 * 获取服务端本地的IP地址
	 */
	public String getLocalIp() {
		try {
			return InetAddress.getLocalHost().getHostAddress();
		} catch (UnknownHostException e) {
			log.error("getLocalIp...{}", e.getMessage());
			return "127.0.0.1";
		}
	}

	/**
	 * 获取用户请求者的IP地址
	 */
	public String getClientIp(HttpServletRequest req) {
		String ip = req.getHeader("X-Forwarded-For");
		log.debug("X-Forwarded-For...{}", ip);
		if (isEmpty(ip) || "unknown".equalsIgnoreCase(ip)) {
			ip = req.getRemoteAddr();
			if (isEmpty(ip))
				return "";
		}
		// 多个代理多个IP以逗号分割，第一个IP为客户端真实IP
		int p = ip.indexOf(',');
		return p > 0 ? ip.substring(0, p) : ip;
	}

	/**
	 * 浏览器请求的默认语言（国家CN）
	 * @deprecated
	 */
	@Deprecated
	public String getLocale(HttpServletRequest req) {
		// 标准输出格式（这里不添加下划线符号“_”，在i18n方法中再单独处理）
		return req.getLocale().getLanguage();
	}

	/**
	 * 构建URI参数
	 */
	public String buildUri(Map<String, String> map) {
		if (map == null)
			return "";
		StringJoiner s = new StringJoiner("&");
		for (Map.Entry<String, String> e : map.entrySet()) {
			if (!isEmpty(e.getValue()) && !isEmpty(e.getKey()))
				s.add(e.getKey() + Const.EQU + Digest.urlEncode(e.getValue()));
		}
		return s.toString();
	}

	/**
	 * 判断字符数组，不为空
	 *
	 * @param values 字符数组
	 * @return true or false
	 */
	public boolean notEmpty(String[] values) {
		if (values == null || values.length == 0)
			return false;
		for (int i = 0; i < values.length; i++) {
			if (isEmpty(values[i]))
				return false;
		}
		return true;
	}

	/**
	 * 数字步进，按step累加
	 */
	public int toStep(int min, int max, int step) {
		int num = min + step;
		return num > max ? max : num;
	}

	/**
	 * 关闭对象
	 * @deprecated
	 */
	@Deprecated
	public void close(Closeable[] closeables) {
		if (closeables != null) {
			for (Closeable closeable : closeables) {
				close(closeable);
			}
		}
	}

	/**
	 * 关闭对象
	 */
	public void close(Closeable closeable1, Closeable closeable2) {
		close(closeable1);
		close(closeable2);
	}

	/**
	 * 关闭对象
	 */
	public void close(Closeable closeable) {
		try {
			if (closeable != null)
				closeable.close();
		} catch (Exception e) {
			log.error("close...{}", e.getMessage());
		}
	}

	/**
	 * 分变成带千分位格式的金额（元），四舍五入
	 */
	public String toYuan(String money) {
		return toYuan(money, null);
	}

	/**
	 * 分变成带千分位格式的金额（元），四舍五入
	 */
	public String toYuan(String money, String def) {
		return FmtUtil.toYuan(money, def);
	}

	/**
	 * 带千分位格式的金额（元）变成分，四舍五入
	 */
	public String toCent(String money) {
		return toCent(money, null);
	}

	/**
	 * 带千分位格式的金额（元）变成分，四舍五入
	 */
	public String toCent(String money, String def) {
		String val = Const.S_0;
		try {
			double d = 0.0;
			if (!isEmpty(money)) {
				money = money.replace(Const.COMMA, "");
				if (isNumeric(money))
					d = Double.valueOf(money) * 100;
			}
			val = new DecimalFormat("#0").format(d);
		} catch (Exception e) {
			error("toCent...{}", money);
		}
		return def != null && isNil(val) ? def : val;
	}

	/**
	 * 人民币汉字大写，传入数值要小于1万亿元
	 */
	public String toRmb(String money) {
		return FmtUtil.toRmb(money);
	}

	/**
	 * 人民币汉字大写，传入数值可大于1万亿元
	 */
	public String toRmb(long num) {
		return FmtUtil.toRmb(num);
	}

    /**
     * 生成树形结构的叶子项目的子编号
     *
     * @param pid 父级项目编号
     * @param list 已有的子级叶子项目一览表
     * @param key 项目编号的主键
     * @param bitW 单层项目编号的位数
     */
	public String newLeafId(Object pid, List<Map<String, String>> list, String key, int bitW) {
		if (isEmpty(list))
			return idCode(pid, 1, bitW);
		// 获取列表中最大的编号
		String maxId = list.get(list.size() - 1).get(key);
		// 获取（当前剔除父级编号后）的最大子级编号，如001
		int id = Integer.parseInt(maxId.substring(maxId.length() - bitW));
		// 有断号，则最大值>size()，则首先遍历找出断号
		if (id > list.size()) {
			for (int i = 0; i < list.size(); i++) {
				maxId = list.get(i).get(key);
				id = Integer.parseInt(maxId.substring(maxId.length() - bitW));
				if (id > i + 1)
					return idCode(pid, i + 1, bitW);
			}
		}
		// 正常情况，理论上不可能执行到此处
		return idCode(pid, list.size() + 1, bitW);
	}

	/**
	 * 占位符处理：placeHolder
	 * 
	 * @param opr 运算符号
	 */
	public String pHolder(Object key, String opr) {
		return FmtUtil.pHolder(key, opr);
	}

	/**
	 * 用单符号引用"key"
	 */
	public String quote(Object key, String ch) {
		return FmtUtil.quote(key, ch);
	}

	/**
	 * 用单引号引用"key"
	 */
	String quote(Object key) {
		return FmtUtil.quote(key, "'");
	}

}
