package cn.ps1.aolai.utils;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PushbackInputStream;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.DateUtil;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.ss.util.NumberToTextConverter;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.multipart.MultipartFile;

/**
 * 操作Excel的工具类
 * 
 * @since 2022/9/25 15:23
 * @author Aolai
 */
public class XlsxUtil {

	private static Logger log = LoggerFactory.getLogger(XlsxUtil.class);

	private String xFile; // Excel文件名
//	private Workbook workbook;
	private SXSSFWorkbook workbook;
	private Sheet sheet;

	/** 构造函数 */
	public XlsxUtil() {
	}

	/** 构造函数 */
	public XlsxUtil(String xFile) {
		this.xFile = xFile; // Excel文件名
		initWorkbook();
	}

	/** 构造函数 */
	public XlsxUtil(InputStream inp) {
		initWorkbook(inp);
	}

	/** 初始化 */
	public void initWorkbook() {
		// 读取Excel文件
		try (InputStream inp = new FileInputStream(xFile)) {
			initWorkbook(inp);
		} catch (Exception e) {
			log.error("initWorkbook...{}", e.getMessage());
			newWorkbook();
		}
	}

	/** 初始化 */
	public void initWorkbook(InputStream inp) {
		try {
			// 获取输入流是否支持mark和reset操作
			if (!inp.markSupported())
				inp = new PushbackInputStream(inp, 8);
//			initSheet(WorkbookFactory.create(inp), 0);
			initSheet(new SXSSFWorkbook((XSSFWorkbook) WorkbookFactory.create(inp)), 0);
		} catch (Exception e) {
			log.error("initWorkbook...{}", e.getMessage());
			newWorkbook();
		}
	}
/*
	public void initWorkbook(MultipartFile file) {
		try (InputStream inp = file.getInputStream()) {
			initWorkbook(inp);
		} catch (Exception e) {
			log.error("initWorkbook...{}", e.getMessage());
			newWorkbook();
		}
	}
*/
	/** 初始化 */
	public void newWorkbook() {
//		workbook = new XSSFWorkbook(); // 仅支持xlsx文件
		workbook = new SXSSFWorkbook();
		sheet = workbook.createSheet(); // 创建工作表
	}

	/**
	 * 设置工作薄
	 */
	private void initSheet(SXSSFWorkbook workbook, int index) {
		this.workbook = workbook;
		this.sheet = workbook.getSheetAt(index);
	}

	/**
	 * 读取Excel文件中的数据
	 */
	public synchronized List<Map<String, String>> readExcel() {
		return this.getDataByRow(1, this.getFirstRowTitle());
	}

	/**
	 * 读取Excel文件中的数据
	 */
	public synchronized List<Map<String, String>> readExcel(int startRow,
			Map<String, Integer> header) {
		return this.getDataByRow(startRow, header);
	}

	/**
	 * 读取Excel文件中的数据
	 * @deprecated 这个方法已被替代，并且在未来版本不再支持。
	 */
	@Deprecated
	public synchronized List<Map<String, String>> readExcel(String file,
			int startRow, Map<String, Integer> header) {
		XlsxUtil xlsx = new XlsxUtil(file);
		return xlsx.getDataByRow(startRow, header);
	}

	/**
	 * 把Workbook缓存数据写入Excel文件
	 */
	public synchronized void writeExcel(String file) {
		try (OutputStream out = new FileOutputStream(file)) {
			writeExcel(out);
			out.flush();
		} catch (Exception e) {
			log.error("writeExcel....{}", file);
		}
	}

	/**
	 * 把Workbook缓存数据写入Excel文件
	 */
	public synchronized void writeExcel(OutputStream out) throws IOException {
		workbook.write(out);
	}

	/**
	 * 从前台导入的Excel数据
	 * 
	 * @param file
	 * @return list
	 */
	public static List<Map<String, String>> readExcel(MultipartFile file) {
		try (InputStream inp = file.getInputStream()) {
			// 调用通用Xlsx方法处理
			XlsxUtil xlsx = new XlsxUtil(inp);
			return xlsx.getDataByRow();
		} catch (Exception e) {
			log.error("readExcel...{}", e.getMessage());
		}
		return new ArrayList<>();
	}
	
	/**
	 * 从前台导入的Excel数据
	 * 
	 * @param file
	 * @param startRow
	 * @param header
	 * @return List
	 */
	public static List<Map<String, String>> readExcel(MultipartFile file, int startRow,
			Map<String, Integer> header) {
		try (InputStream inp = file.getInputStream()) {
			// 调用通用Xlsx方法处理
			XlsxUtil xlsx = new XlsxUtil(inp);
			return xlsx.getDataByRow(startRow, header);
		} catch (Exception e) {
			log.error("readExcel...{}", e.getMessage());
		}
		return new ArrayList<>();
	}

	/**
	 * 复制行信息
	 */
	public void shiftRows(int startRow, int endRow, int rows) {
		sheet.shiftRows(startRow, endRow, rows, true, false);
	}

	/**
	 * 克隆空白行
	 */
	public void cloneBlankRow(int curRow, int oriRow) {
		Row insertRow = sheet.createRow(curRow); // 插入新行
		Row originRow = sheet.getRow(oriRow); // 模板参照行
		insertRow.setHeight(originRow.getHeight());

		// 获取当前行样式
		CellStyle rowStyle = originRow.getRowStyle();
		if (rowStyle != null) {
			insertRow.setRowStyle(rowStyle); // 继承样式
		}
		cloneMergedRegion(curRow, oriRow); // 设置合并单元格
	}

	/**
	 * 设置合并单元格
	 */
	public void cloneMergedRegion(int curRow, int oriRow) {
		// 原始的合并单元格的地址集合
//		List<CellRangeAddress> originMerged = sheet.getMergedRegions();
		for (int i = 0; i < sheet.getNumMergedRegions(); i++) {
			CellRangeAddress region = sheet.getMergedRegion(i);
			if (region.getFirstRow() == oriRow) {
				int rows = region.getLastRow() - oriRow;
				CellRangeAddress newRegion = new CellRangeAddress(curRow,
						curRow + rows, region.getFirstColumn(),
						region.getLastColumn());
				sheet.addMergedRegion(newRegion);
			}
		}
	}

	/**
	 * 增加一条Excel空白数据行
	 *
	 * @param curRow 当前空行
	 * @param srcRow 资源行
	 */
/*
	private void addExcelBlankRow(int curRow, int srcRow) {
		// 从当前行全部下移一行，留空后插入rowScope行
		int rows = sheet.getLastRowNum() + rowScope;
		sheet.shiftRows(curRow, rows, rowScope, true, false);
		for (int i = 0; i < rowScope; i++) {
			// 获取当前行
			Row rowSource = sheet.getRow(srcRow + i);
			Row rowInsert = sheet.createRow(curRow + i); // 新增插入一行
			rowInsert.setHeight(rowSource.getHeight());
			// 获取当前行样式
			CellStyle rowStyle = rowSource.getRowStyle();
			if (rowStyle != null) {
				rowInsert.setRowStyle(rowStyle); // 继承样式
			}
		}
	}
*/
	/**
	 * 从Excel当前行（curRow）开始删除多行（rows）
	 */
	public void removeExcelRow(int curRow, int rows) {
		int endRow = curRow + rows;
		// 倒序删除合并单元格
		for (int i = sheet.getNumMergedRegions() - 1; i >= 0; i--) {
			CellRangeAddress region = sheet.getMergedRegion(i);
			if (region.getFirstRow() >= curRow && region.getFirstRow() < endRow) {
				sheet.removeMergedRegion(i); // 删除合并单元格
			}
		}
		sheet.shiftRows(endRow, sheet.getLastRowNum() + 1, -rows, true, false);
	}

	/**
	 * 逐行读取数据，默认第一行(0)为映射主键，第二行(1)开始为业务数据
	 */
	public List<Map<String, String>> getDataByRow() {
		return getDataByRow(1, getFirstRowTitle());
	}

	/**
	 * 逐行读取数据
	 *
	 * @param startRow 开始行
	 * @param header 对应列序号
	 */
	public List<Map<String, String>> getDataByRow(int startRow,
			Map<String, Integer> header) {
		List<Map<String, String>> dataList = new ArrayList<>();
		if (header == null || header.isEmpty()) {
			return dataList;
		}
		for (int i = startRow; i <= sheet.getLastRowNum(); i++) {
			Row row = sheet.getRow(i);
			if (row == null)
				break;
			// 逐行读取行数据
			Map<String, String> rowMap = new HashMap<>();
			for (Map.Entry<String, Integer> e : header.entrySet()) {
				if ("".equals(e.getKey()))
					continue;
				// 逐列读取数据
				Cell cell = row.getCell(e.getValue()); // 列序号
				rowMap.put(e.getKey(), getCellValue(cell));
			}
			dataList.add(rowMap);
		}
		return dataList;
	}

	/**
	 * 获取第一行的表头标题
	 */
	public Map<String, Integer> getFirstRowTitle() {
		Map<String, Integer> header = new HashMap<>();
		Row row = sheet.getRow(0);
		if (row != null) {
			// Excel工作表的最大列数为255列
			for (int i = 0; i < 255; i++) {
				String colKey = getCellValue(row.getCell(i));
				if (colKey != null && colKey.length() > 0)
					header.put(colKey, i);
			}
		}
		return header;
	}

	/**
	 * 获取单元格的数据
	 *
	 * @param cell 单元格
	 * @return String 值
	 */
	public String getCellValue(Cell cell) {
		if (cell == null)
			return "";
		switch (cell.getCellType()) {
		case STRING: // 1-字符串
			return cell.getStringCellValue();
		case NUMERIC: // 0-数值
			double d = cell.getNumericCellValue();
			if (DateUtil.isCellDateFormatted(cell)) {
				Date date = DateUtil.getJavaDate(d);
				return new SimpleDateFormat(Const.DTF).format(date);
			} else {
				return decimal2Text(d);
			}
		case FORMULA: // 2-公式
//			val = cell.getCellFormula(); // 返回：SUM(C4:E4)
			// 返回预先计算的值
			return decimal2Text(cell.getNumericCellValue());
		case BOOLEAN: // 4-Boolean
			return String.valueOf(cell.getBooleanCellValue());
//			break;
		// case BLANK: // 3-空格，返回 0或空串
		// case ERROR: // 5-错误
		default: // 字符串
			return "";
		}
	}

	/**
	 * 处理科学计数法的数据
	 */
	private String decimal2Text(double d) {
		String val = NumberToTextConverter.toText(d);
		// 1.8660177889E-13 ==> 0.000000000000186601778889
		// 1.8660177889E+10 ==> 18660177889
		if (val.indexOf('E', 2) > -1) {
			// 获取原有小数点的位数
			String[] ss = val.split("E");
			int len = ss[0].split("\\.")[1].length();
			int n = Math.abs(Integer.parseInt(ss[1]));
			// 指数为负数的情况
			boolean isMinus = ss[1].indexOf('-') == 0;
			len = isMinus ? len + n : len - n;

			BigDecimal bd = new BigDecimal(val);
			if (len > 0) { // 返回小数
				bd = bd.setScale(len, BigDecimal.ROUND_HALF_UP);
				return bd.toPlainString();
			} else { // 返回整数
				return String.valueOf(bd.longValue());
			}
		} else {
			// 剔除无用的小数点.00
			return val.replaceAll("\\.0+$", "");
		}
	}

	/**
	 * 根据行和列的索引获取单元格的数据
	 *
	 * @param rowNo 行号
	 * @param colNo 列号
	 * @return String 值
	 */
	public String getCellValue(int rowNo, int colNo) {
		Row row = sheet.getRow(rowNo);
		if (row == null)
			return "";
		Cell cell = row.getCell(colNo);
		return getCellValue(cell);
	}

	/**
	 * 根据行和列的索引设置单元格的数据
	 *
	 * @param rowNo 行
	 * @param colNo 列
	 * @param value 值
	 */
	public void setCellValue(int rowNo, int colNo, String value) {
		Row row = sheet.getRow(rowNo);
		if (row == null) {
			row = sheet.createRow(rowNo);
		}
		Cell cell = row.getCell(colNo);
		if (cell == null) {
			cell = row.createCell(colNo);
		}
		cell.setCellValue(value);
	}

	/**
	 * 获取指定列的列宽
	 */
	public float[] getColWidth(int maxColNum) {
		float[] widths = new float[maxColNum];
		for (int j = 0; j < maxColNum; j++) {
			widths[j] = sheet.getColumnWidthInPixels(j);
		}
		return widths;
	}

	/**
	 * 获取最大行号
	 */
	public int getMaxRowNum() {
		return sheet.getLastRowNum();
	}

	/**
	 * 获取Excel页的最大列数
	 */
	public int getMaxColNum(int rowNum) {
		int maxColNum = 0;
		for (int i = 0; i < rowNum; i++) {
			Row row = sheet.getRow(i);
			if (row == null)
				continue;
			int colNum = row.getLastCellNum();
			if (colNum > maxColNum)
				maxColNum = colNum;
		}
		return maxColNum;
	}

	public Workbook getWorkbook() {
		return workbook;
	}

	public Sheet getSheet() {
		return sheet;
	}

	public Row getRow(int i) {
		return sheet.getRow(i);
	}

	public String getFile() {
		return xFile;
	}

}
