package cn.ps1.aolai.utils;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigInteger;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.security.Security;
import java.util.Arrays;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;
import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;

import org.apache.commons.codec.digest.DigestUtils;
import org.bouncycastle.asn1.gm.GMNamedCurves;
import org.bouncycastle.asn1.x9.X9ECParameters;
import org.bouncycastle.crypto.AsymmetricCipherKeyPair;
import org.bouncycastle.crypto.digests.SM3Digest;
import org.bouncycastle.crypto.engines.SM2Engine;
import org.bouncycastle.crypto.generators.ECKeyPairGenerator;
import org.bouncycastle.crypto.macs.HMac;
import org.bouncycastle.crypto.params.ECDomainParameters;
import org.bouncycastle.crypto.params.ECKeyGenerationParameters;
import org.bouncycastle.crypto.params.ECPrivateKeyParameters;
import org.bouncycastle.crypto.params.ECPublicKeyParameters;
import org.bouncycastle.crypto.params.KeyParameter;
import org.bouncycastle.crypto.params.ParametersWithRandom;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.math.ec.ECPoint;
import org.bouncycastle.util.encoders.Hex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 根据自定义字符串对任意字节数据进行类似Base64编码的简单算法
 * 
 * @author Aolai
 * @since 1.8
 *
 */
public class Digest {

	private Digest() {
		throw new IllegalStateException("Utility class");
	}

	private static Logger log = LoggerFactory.getLogger(Digest.class);

	/** 实际长度65字符 */
	// 默认字符集合："ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/="
	private static final String CHARS64 = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";

	private static final byte[] SEED11 = { 68, 70, 74, 75, 77, 81, 83, 86, 87, 88, 90 };

	/** 为防止脚本注入，慎用！ */
	private static ScriptEngine se = null;
	/** 为62进制而设的字符串 */
	static String chars62 = CHARS64.substring(52, 62) + CHARS64.substring(0, 52);

	/** 任意随机数 */
	static SecureRandom rand = new SecureRandom();

	/** 用于保存密钥的键值，公钥主键：“pubk” */
	public static final String PUB_KEY = "pubk"; // 密钥对：公钥
	/** 私钥主键：“prik” */
	public static final String PRI_KEY = "prik"; // 密钥对：私钥

	/**
	 * 随机密码的默认11位种子
	 */
	public static String seed() {
		return new String(SEED11);
	}

	/**
	 * 任意随机生成 N位数字，可用于生成验证码
	 * 
	 * @param len 长度或位数
	 */
	public static String randInt(int len) {
		StringBuilder sb = new StringBuilder();
		while ((len--) > 0)
			sb.append(rand.nextInt(10));
		return sb.toString();
	}

	/**
	 * 按照自定义的规则，随机动态密钥
	 * 
	 * @param len 密钥长度或位数
	 */
	public static String randKey(int len) {
		StringBuilder sb = new StringBuilder();
		while ((len--) > 0)
			sb.append(CHARS64.charAt(rand.nextInt(62)));
		return sb.toString();
	}

	/**
	 * 自定义本系统特有的Key，根据11位字符串随机排序生成KeyId
	 * 
	 * @return String 11位字符串（如：‘dfjkmqsvwxz’）
	 */
	public static String randKey() {
		return randStr(SEED11, 11);
	}

	/**
	 * 按照自定义的规则，生成任意位数的字符串
	 * 
	 * @param len 长度应小于64位
	 */
	public static String randStr(int len) {
		return randStr(CHARS64.substring(0, len), len);
	}

	/**
	 * 把任意字符串随机打乱顺序
	 */
	public static String randStr(String src) {
		return randStr(src, src.length());
	}

	/**
	 * 把一个N位字符串随机打乱顺序
	 * 
	 * @param src 原始字符串
	 * @param len 保留的长度应小于64位
	 * @return String 随机排序的N位字符串
	 */
	public static String randStr(String src, int len) {
		return randStr(src.getBytes(StandardCharsets.UTF_8), len);
	}

	/**
	 * 把一个N位字符串随机打乱顺序
	 */
	private static String randStr(byte[] srcData, int len) {
		for (int i = 0; i < len; i++) {
			int idx = rand.nextInt(srcData.length);
			if (idx == i) // 跳过自己
				continue;
			// 序列变换：第i个与第n个位置互换
			byte c = srcData[idx];
			srcData[idx] = srcData[i];
			srcData[i] = c;
		}
		return new String(Arrays.copyOf(srcData, len));
	}

	/**
	 * 按照自定义的规则，根据certId和certKey拼接解码用的通行证字符串
	 */
	public static String certStr(String src, String key) {
		String s = src + key;
		return s.toUpperCase() + s.toLowerCase() + CHARS64.substring(52);
	}

	/**
	 * 按照自定义的规则，根据 key(certId)拼接完整的通行证，返回65位字符串
	 * 
	 * @param key 默认为11位或26位通行证(certId)
	 */
	public static String certStr(String key) {
		if (key == null) {
			return CHARS64; // 返回65位字符串
		} else if (key.length() == 26) {
			return certStr(key, ""); // 返回65位字符串
		} else if (key.length() > 64) {
			return key; // 返回65位字符串
		}
		// 默认11位数通行证，实际返回65位字符串
		return certStr("uncopyrightable", key);
	}

	/**
	 * 对编码过的字符串进行解码
	 */
	public static String decrypt(String src) {
		return decrypt(src, CHARS64);
	}

	/**
	 * 对编码过的字符数组进行解码
	 */
	public static byte[] decrypt(char[] arr) {
		return decrypt(arr, CHARS64);
	}

	/**
	 * 按照自定义的规则，用指定通行证对已编码的字符串进行解码
	 * 
	 * @param src 已编码的字符串
	 * @param cert 默认11位或 26位通行证
	 */
	public static String decrypt(String src, String cert) {
		if (src == null)// || src.length() == 0)
			return src;
		return new String(decrypt(src.toCharArray(), cert));
	}

	/**
	 * 按照自定义的规则，用指定通行证对字符数组进行解码
	 * <p>
	 * 
	 * @param cert 默认11位或 26位通行证
	 */
	public static byte[] decrypt(char[] arr, String cert) {
		String key = certStr(cert);
		// 剔除填充字符
		char pad = key.charAt(key.length() - 1); // '='

		int len = arr.length - 1;
		while (len > 0 && arr[len] == pad)
			len--;
		len = arr.length * 6 / 8 - (arr.length - 1 - len);

		// 解码处理
		byte[] raw = new byte[len];
		int[] quad = new int[4];
		int rawIdx = 0;
		for (int i = 0; i < arr.length; i += 4) {
			int block = 0;
			for (int idx = 0; idx < 4; idx++) {
				char c = arr[i + idx];
				quad[idx] = c == pad ? 0 : key.indexOf(c);
				block += (quad[idx] << 6 * (3 - idx));
			}
			for (int b = 0; b < 3 && rawIdx + b < raw.length; b++)
				raw[rawIdx + b] = (byte) ((block >> (8 * (2 - b))) & 0xFF);
			rawIdx += 3;
		}
		return raw;
	}

	/**
	 * 对字符串进行编码
	 */
	public static String encrypt(String src) {
		return encrypt(src.getBytes(StandardCharsets.UTF_8));
	}

	/**
	 * 对字符数组编码
	 */
	public static String encrypt(byte[] bytes) {
		return encrypt(bytes, CHARS64.getBytes());
	}

	/**
	 * 按照自定义的规则，用指定通行证对字符串进行编码
	 * 
	 * @param src 原始字符串
	 * @param cert 默认11位或 26位通行证
	 */
	public static String encrypt(String src, String cert) {
		return src == null ? null : encrypt(src.getBytes(StandardCharsets.UTF_8), cert);
	}

	/**
	 * 按照自定义的规则，用指定通行证对字符数组进行编码
	 * 
	 * @param bytes 字符串数组
	 * @param cert 默认11位或 26位通行证
	 */
	public static String encrypt(byte[] bytes, String cert) {
		return encrypt(bytes, certStr(cert).getBytes());
	}

	/**
	 * 按照自定义的规则，用指定通行证对字符数组进行编码
	 * <p>
	 * 汉字三个字节，英文一个字符，如： 字符a的ASCII码是97，字符o的ASCII码是111
	 */
	private static String encrypt(byte[] bytes, byte[] keys) {
		byte[] out = new byte[((bytes.length + 2) / 3) * 4];
		for (int i = 0, idx = 0; i < bytes.length; i += 3, idx += 4) {
			boolean[] trip = { true, false, false };
			int bit = (0xFF & bytes[i]);
			for (int b = 1; b < 3; b++) {
				bit <<= 8; // 先向左移一个字节
				if (i < bytes.length - b) {
					bit |= (0xFF & bytes[i + b]); // 再拼接一个字节
					trip[b] = true;
				}
			}
			for (int b = 0; b < 3; b++) {
				out[idx + 3 - b] = keys[(trip[2 - b] ? (bit & 0x3F) : 64)];
				bit >>= 6;
			}
			out[idx] = keys[bit & 0x3F];
		}
		return out.length == 0 ? "" : new String(out);
	}

	/**
	 * 脚本函数处理，这里仅返回字符串（为防止脚本注入，慎用！）
	 */
	private static String jsEval(String src) {
		if (se == null)
			se = new ScriptEngineManager().getEngineByExtension("js");
		try {
			src = se == null ? src : String.valueOf(se.eval(src));
		} catch (Exception e) {
			log.error("jsEval...{}", src);
		}
		return src;
	}

	/**
	 * 用于浏览器cookie编码，为防止脚本注入，慎用！
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	@Deprecated
	public static String escape(String src) {
		return jsEval("escape('" + src + "')");
	}

	/**
	 * 用于浏览器cookie解码，为防止脚本注入，慎用！
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	@Deprecated
	public static String unescape(String src) {
		return jsEval("unescape('" + src + "')");
	}

	/**
	 * 用于浏览器cookie解码，为防止脚本注入，慎用！
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	@Deprecated
	public static String decodeURIComponent(String src) {
		return jsEval("decodeURIComponent('" + src + "')");
	}

	/**
	 * 用于浏览器cookie编码，为防止脚本注入，慎用！
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	@Deprecated
	public static String encodeURIComponent(String src) {
		return jsEval("encodeURIComponent('" + src + "')");
	}

	/**
	 * 还原图片，把base64格式的字符数据集转换为图片，存到指定文件中
	 * 
	 * @param src 字符数据集
	 * @param imgName 文件路径
	 */
	public static boolean imgDecode(String src, String imgName) {
		// 读取字节流转存文件
		try (OutputStream out = new FileOutputStream(imgName)) {
			out.write(imgDecode(src));
			out.flush();
			return true;
		} catch (Exception e) {
			log.error("imgDecode...{}", imgName);
		}
		return false;
	}

	/**
	 * 图片数据格式转换：解码为字节数组
	 */
	public static byte[] imgDecode(String src) {
		String[] arr = src.split(",");
		return decode(arr.length == 1 ? arr[0] : arr[1]);
	}

	/**
	 * 解码为字节数组
	 */
	private static byte[] decode(String src) {
		return Base64.getDecoder().decode(src);
	}

	/**
	 * 读取图片字节数组，把图片image转换为base64格式的字符数据集
	 * 
	 * @param imgFile 文件路径
	 */
	public static String imgEncode(String imgFile) {
		try (InputStream inp = new FileInputStream(imgFile)) {
			byte[] bytes = new byte[inp.available()];
			if (inp.read(bytes) > 0)
				return encode(bytes);
		} catch (IOException e) {
			log.error("imgEncode...{}", imgFile);
		}
		// 获取编码器后转码
		return "";
	}

	/**
	 * 获取编码器后转码
	 */
	private static String encode(byte[] bytes) {
		return Base64.getEncoder().encodeToString(bytes);
	}

	/** 解码 */
	public static String urlDecode(String url) {
		try {
			return URLDecoder.decode(url, Const.UTF8);
		} catch (Exception e) {
			log.error("urlDecode...{}", url);
			return url;
		}
	}

	/** 编码 */
	public static String urlEncode(String url) {
		try {
			return URLEncoder.encode(url, Const.UTF8);
		} catch (Exception e) {
			log.error("urlEncode...{}", url);
			return url;
		}
	}

	/**
	 * 生成通用唯一识别码
	 */
	public static String uuid() {
		return UUID.randomUUID().toString();
	}

	/**
	 * 根据时间戳生成8位唯一识别码，可支持6.2万并发
	 */
	public static String uuid8() {
		return uuid8(0);
	}

	/**
	 * 同时生成一组数据唯一识别码，调整一下增量
	 */
	public static String uuid8(int incr) {
		char c = CHARS64.charAt(rand.nextInt(62)); // 为避免使用加号“+”
		// 截至2081年08月05日前有效
		return enBase62(System.currentTimeMillis() + incr) + c;
	}

	/**
	 * 把64进制字符解码为十进制数字
	 */
	public static long deBase64(String src) {
		return deBase(src, 64);
	}

	/**
	 * 解码诸如身份证或手机号形式的（编码的）数字串，保留最后n位，如：身份证最后1位X，n=0时全部解码
	 */
	public static String deBase62(String src, int n) {
		String s = src.substring(0, src.length() - n);
		return deBase62(s) + src.substring(src.length() - n);
	}

	/**
	 * 把62进制字符解码为十进制数字
	 */
	public static long deBase62(String src) {
		return deBase(src, 62);
	}

	/**
	 * 把N进制编码转换为十进制数字
	 */
	private static long deBase(String src, int n) {
		// 基数
		long result = 0;
		long radix = 1;
		for (int i = 0; i < src.length(); i++) {
			char c = src.charAt(src.length() - 1 - i);
			result += CHARS64.indexOf(c) * radix;
			radix = radix * n;
		}
		return result;
	}

	/**
	 * 把十进制的数字转换为64进制编码
	 */
	public static String enBase64(long num) {
		return enBase(num, 64);
	}

	/**
	 * 把诸如身份证或手机号形式的数字串进行编码，且保留最后n位不变，如：身份证最后1位X，n=0时全部编码
	 */
	public static String enBase62(String src, int n) {
		try {
			long l = Long.parseLong(src.substring(0, src.length() - n));
			return enBase62(l) + src.substring(src.length() - n);
		} catch (Exception e) {
			log.error("enBase62...{}", src);
		}
		return null;
	}

	public static String enBase62(long num) {
		return enBase(num, 62);
	}

	/**
	 * 把十进制的数字转换为自定义的非标准N进制编码
	 */
	private static String enBase(long num, int n) {
		if (num == 0)
			return "A";
		StringBuilder sb = new StringBuilder();
		while (num > 0) {
			Long l = num % n;
			sb.append(CHARS64.charAt(l.intValue()));
			num = num / n;
		}
		return sb.reverse().toString();
	}

	/**
	 * 根据数字生成一个包含26个字母三位长度的字符串 <br>
	 * 同时也兼容了000-999的既定数值，但需注意：既有2位数升3位数时并不兼容
	 * <p>
	 * 例如：1000>>1260>>1520>>2195 <br>
	 * 对应：0A0>>00A>>0AA>>0ZZ
	 * 
	 * @param num 默认保留的字符串位数
	 */
	public static String getChr3(int num) {
		if (num < 1000)
			return String.format("%03d", num);
		StringBuilder sb = new StringBuilder();
		if (num < 12960) { // 1000 + 1196*10
			num -= 1000;
			int bit = num / 1196;
			String s = getChr2((num + 100 * (bit + 1)) % 1296);
			sb.append(getChr(bit)).append(s);
		} else if (num < 46656) { // 12960 + 1296*26
			sb.append(getChr(num / 1296)).append(getChr2(num % 1296));
		}
		return sb.length() > 0 ? sb.toString() : "---";
	}

	/**
	 * 根据数字生成一个包含26个字母两位长度的字符串
	 * <p>
	 * 同时也兼容了00-99的既定数值
	 * 
	 * @param num 默认保留的字符串位数
	 */
	public static String getChr2(int num) {
		if (num < 100)
			return String.format("%02d", num);
		StringBuilder sb = new StringBuilder();
		if (num < 360) { // 100 + 26*10
			sb.append(getChr(num / 10)).append(getChr(num % 10));
		} else if (num < 1296) { // 360 + 36*26
			num -= 360;
			sb.append(getChr(num / 26)).append(getChr(num % 26 + 10));
		}
		return sb.length() > 0 ? sb.toString() : "--";
	}

	/**
	 * 对应数字和字符的ASCII码 0-9(48-57), A-Z(65-90）
	 */
	public static char getChr(int num) {
		// "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ"
		if (num < 10)
			return (char) (num + 48);
		return (char) (num < 36 ? num + 55 : 45); // '-'45
	}

	/**
	 * 使用MD5摘要算法加密
	 */
	public static String md5(String str) {
		return DigestUtils.md5Hex(str);
	}

	/**
	 * 使用AES对称算法加密：key为16位的密钥
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
/*	@Deprecated
	public static String aesEncrypt(String plainText, String key) throws Exception {
		Cipher c = aesCipher(key, Cipher.ENCRYPT_MODE);
		byte[] result = plainText.getBytes(StandardCharsets.UTF_8);
		return encode(c.doFinal(result));
	}*/

	/**
	 * 使用AES对称算法解密：key为16位的密钥
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
/*	@Deprecated
	public static String aesDecrypt(String cipherText, String key) throws Exception {
		Cipher cipher = aesCipher(key, Cipher.DECRYPT_MODE);
		byte[] bytes = cipher.doFinal(decode(cipherText));
		return new String(bytes);
	}*/

	/**
	 * PKCS5是PKCS7的子集，区别在于前者BlockSize始终是8字节（64位），后者可以为1到255字节<br>
	 * 因为AES并没有64位的块，如果采用PKCS5，那么实质上就是采用PKCS7
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
/*	@Deprecated
	private static Cipher aesCipher(String key, int mode) throws Exception {
		Cipher AES = Cipher.getInstance("AES/ECB/PKCS5Padding");
		Key keySpec = new SecretKeySpec(key.getBytes(), "AES");
		AES.init(mode, keySpec);
		return AES;
	}*/

	/** 以下为国密算法 */

	private static Cipher sm4Cipher(String key, int mode) {
		try {
			// 此处实测不能采用单例模式，实际开发短时间内加载多次解密会报错，故此处每次解密都实例化加密实例
			Security.addProvider(new BouncyCastleProvider());
			// ECB模式不能用IV初始向量、CBC模式必须使用IV初始向量
			// 基于ECB模式的BC库实现加密
			Cipher sm4 = Cipher.getInstance("SM4/ECB/PKCS5Padding", "BC");
			byte[] keyBytes = DigitUtil.hex2Bytes(key);
			// CBC模式必须使用IV初始向量，如：SM4.init(mode, keySpec, iv);
			sm4.init(mode, new SecretKeySpec(keyBytes, "SM4")); // "SM4"\"ECB"
			return sm4;
		} catch (Exception e) {
			throw new FailedException();
		}
	}

	/**
	 * 通用的国密对称SM4加密模式：使用一个随机种子加密
	 */
	public static String sm4Encrypt(String plainText, String salt) {
		try {
			Cipher cipher = sm4Cipher(salt, Cipher.ENCRYPT_MODE);
			byte[] bytes = plainText.getBytes(StandardCharsets.UTF_8);
			return DigitUtil.toHexStr(cipher.doFinal(bytes));
		} catch (Exception e) {
			log.error("sm4Encrypt...{}", plainText);
			return null;
		}
	}

	/**
	 * 通用的国密对称SM4解密模式：使用一个随机种子解码
	 */
	public static String sm4Decrypt(String cipherText, String salt) {
		try {
			byte[] bytes = DigitUtil.hex2Bytes(cipherText);
			Cipher cipher = sm4Cipher(salt, Cipher.DECRYPT_MODE);
			return new String(cipher.doFinal(bytes));
		} catch (Exception e) {
			// 当密钥不匹配时解码会报错：BadPaddingException
			log.error("sm4Encrypt...{}", salt);
			return null;
		}
	}

	/**
	 * 通用的国密SM4对称加密秘钥生成
	 */
	public static String genRawKey() {
		StringBuilder sb = new StringBuilder();
		for (int i = 0; i < 32; i++) {
			sb.append(Integer.toHexString(rand.nextInt(16)));
		}
		return sb.toString();
	}

	/**
	 * 国密SM3摘要算法：用32位字节的摘要算法，返回64位十六进制字符
	 */
	public static String sm3(String src) {
		SM3Digest sm3 = new SM3Digest();
		byte[] data = src.getBytes(StandardCharsets.UTF_8);
		byte[] hash = new byte[sm3.getDigestSize()];
		sm3.update(data, 0, data.length); // 有效的输入
		sm3.doFinal(hash, 0);
		return Hex.toHexString(hash);
	}

	/**
	 * 国密SM3摘要算法：用32位字节的摘要算法，返回64位十六进制字符
	 */
	public static String sm3(String src, String key) {
		HMac hMac = new HMac(new SM3Digest());
		byte[] data = src.getBytes(StandardCharsets.UTF_8);
		byte[] hash = new byte[hMac.getMacSize()];
		hMac.init(new KeyParameter(key.getBytes()));
		hMac.update(data, 0, data.length);
		hMac.doFinal(hash, 0);
		return Hex.toHexString(hash);
	}

	/** 因非对称SM2国密算法效率较慢，不推荐使用 */

	// 国密SM2算法：获取一条sm2p256v1结构的椭圆曲线（EC）参数
	static X9ECParameters sm2p256v1 = GMNamedCurves.getByName("sm2p256v1");

	// 国密SM2算法：构造一个椭圆域算法参数集 曲线方程、曲线G点、大整数N
	static ECDomainParameters domainParams = new ECDomainParameters(
			sm2p256v1.getCurve(), sm2p256v1.getG(), sm2p256v1.getN());

	/**
	 * 国密非对称SM2算法：生成非对称SM2国密算法密钥对（公钥66位、私钥64位）
	 * <p>
	 * 仅登录初始化用在了getTicket()中
	 */
	public static Map<String, String> genSm2Pair() {
		// 初始化一个生成器
		ECKeyPairGenerator kpg = new ECKeyPairGenerator();
		// 加密强随机数生成器，默认算法："SHA1PRNG"
		SecureRandom rand = new SecureRandom();
		kpg.init(new ECKeyGenerationParameters(domainParams, rand));
		// 生成密钥对
		AsymmetricCipherKeyPair keyPair = kpg.generateKeyPair();
		// 提取公钥点
		ECPoint ecp = ((ECPublicKeyParameters) keyPair.getPublic()).getQ();
		BigInteger bi = ((ECPrivateKeyParameters) keyPair.getPrivate()).getD();
		// 返回密钥对
		Map<String, String> keysMap = new HashMap<>();
		keysMap.put(PUB_KEY, Hex.toHexString(ecp.getEncoded(true))); // false
		keysMap.put(PRI_KEY, bi.toString(16));
		return keysMap;
	}

	/**
	 * 国密非对称SM2算法：使用公钥加密
	 */
	public static String sm2Encrypt(String plainText, String pubKey) {
		byte[] data = plainText.getBytes(StandardCharsets.UTF_8);
		byte[] pubK = Hex.decode(pubKey);
		// 定位加密曲线上的点
		// ECPoint ecp = new X9ECPoint(sm2p256v1.getCurve(), pubK).getPoint();
		ECPoint ecp = sm2p256v1.getCurve().decodePoint(pubK);
		ECPublicKeyParameters pk = new ECPublicKeyParameters(ecp, domainParams);
		// 定义加密引擎：SM2Engine.Mode.C1C2C3、C1C3C2
		// 默认旧标准C1C2C3，对应前台"0"模式，新标准按C1C3C2模式"1"
		SM2Engine sm2 = new SM2Engine(SM2Engine.Mode.C1C2C3); // 默认C1C2C3
		sm2.init(true, new ParametersWithRandom(pk));
		try {
			data = sm2.processBlock(data, 0, data.length);
			return Hex.toHexString(data);
		} catch (Exception e) {
			log.error("sm2Encrypt...{}", plainText);
			return null;
		}
	}

	/**
	 * 国密SM2非对称加密算法：使用私钥解密
	 */
	public static String sm2Decrypt(String cipherText, String priKey) {
		byte[] data = Hex.decode(cipherText);
		BigInteger bi = new BigInteger(priKey, 16);
		// 私钥参数
		ECPrivateKeyParameters pk = new ECPrivateKeyParameters(bi, domainParams);
		// 定义加密引擎：SM2Engine.Mode.C1C2C3、C1C3C2
		// 默认旧标准C1C2C3，对应前台"0"模式，新标准按C1C3C2模式"1"
		SM2Engine sm2 = new SM2Engine(SM2Engine.Mode.C1C2C3);
		sm2.init(false, pk);
		try {
			data = sm2.processBlock(data, 0, data.length);
			return new String(data);
		} catch (Exception e) {
			log.error("sm2Decrypt...{}", cipherText);
			return null;
		}
	}

	/**
	 * 把整数变为62进制的序列号
	 */
	static String getSeq(long num) {
		StringBuilder sb = new StringBuilder();
		while (num > 0) {
			Long l = num % 62;
			sb.append(chars62.charAt(l.intValue()));
			num = num / 62;
		}
		return sb.reverse().toString();
	}

	/**
	 * 把整数的62进制的序列号自动加1
	 */
	static String nextSeq(String seqNo) {
		int i = seqNo.length();
		long e = 1, num = 0;
		while (i-- > 0) {
			num += chars62.indexOf(seqNo.charAt(i)) * e;
			e *= 62;
		}
		return getSeq(num + 1);
	}

}
