package cn.ps1.aolai.service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.ps1.aolai.entity.Field;
import cn.ps1.aolai.entity.Layout;
import cn.ps1.aolai.entity.User;
import cn.ps1.aolai.utils.ConfUtil;
import cn.ps1.aolai.utils.Const;
import cn.ps1.aolai.utils.FailedException;

/**
 * 针对TARIN表、FEILD表的扩展业务操作
 *
 * @author Taishan
 * @since 2024年10月02日
 */

@Service
public class LayoutService {

	private static Logger log = LoggerFactory.getLogger(LayoutService.class);

	private static Set<String> loSet = new HashSet<>();
	static {
		loSet.add(Layout.HEAD);
		loSet.add(Layout.OPTS);
		loSet.add(Layout.JOINT);
	}

	@Autowired
	private AolaiService aolai;
	@Autowired
	private RedisService redis;
	@Autowired
	private UtilsService utils;
	@Autowired
	HttpServletRequest req; // 单实例可以注入

	/**
	 * 表头数据的LAYOUT展示控制项，默认GCONF配置项：meta.custom=‘1’
	 */
	Map<String, String> getAliasArray(Object uriKey, String key) {
		return getAliasArray(getLayoutBy(uriKey), key);
	}

	private Map<String, String> getAliasArray(Map<String, String> item, String key) {
		List<String> head = utils.json2List(item.get(Layout.HEAD));
		List<Map<String, String>> opts = utils.json2List(item.get(Layout.OPTS));
		// 遍历配置规则
		Map<String, String> map = new LinkedHashMap<>();
		for (int i = 0; i < head.size() && i < opts.size(); i++) {
			Map<String, String> opt = opts.get(i);
			if ("1".equals(opt.get(Const.PERMIT))) {
				// 返回值格式：{userId:'1',userName:null,userOpUid='0',...}
				map.put(head.get(i), opt.get(key)); // key=Const.BLUR
			}
		}
		log.debug("getAliasArray...{}", map);
		return map;
	}

	/**
	 * 选项配置： {"hide":"0", "width":"180", "freeze":"1", "permit":"1",
	 * "align":"left", "title":"0", "filter":""}
	 */
	private Map<String, Map<String, String>> getAliasOpts(Map<String, String> item) {
		List<String> head = utils.json2List(item.get(Layout.HEAD));
		List<Map<String, String>> opts = utils.json2List(item.get(Layout.OPTS));
		// 遍历配置规则
		Map<String, Map<String, String>> map = new LinkedHashMap<>();
		for (int i = 0; i < head.size() && i < opts.size(); i++) {
			Map<String, String> tmp = opts.get(i);
			if ("1".equals(tmp.get(Const.PERMIT))) {
				// 选项配置值，layout表每个字段对应的opts：
				map.put(head.get(i), tmp);
			}
		}
		return map;
	}

	/**
	 * 按用户查询页面表头数据的LAYOUT展示控制项
	 */
	private Map<String, String> getLayoutBy(Object uriKey) {
		return getLayoutBy(uriKey, req.getAttribute(User.ID));
	}

	/**
	 * 表头数据的LAYOUT展示控制项，默认GCONF配置项：meta.custom=‘1’
	 */
	private Map<String, String> getLayoutBy(Object uriKey, Object userId) {
		// 执行到此处，已经是默认条件：ConfUtil.isCustomMeta()
		String rk = Const.RDS_PAGE + uriKey + aolai.rdsDbid() + userId;
		Map<String, String> layout = redis.hmget(rk);

		// 从数据库获取信息
		// 此处如果redis存储的是对象，redis自动回转换成map。
		if (layout.isEmpty()) {
			String[] loArr = loSet.toArray(new String[loSet.size()]);
			Map<String, String> item;
			if ("".equals(userId)) {
				item = getLayoutOne(uriKey);
			} else {
				item = getLayoutOne(uriKey, userId);
			}

			// 获取LAYOUT共通数据，并缓存数据
			layout = utils.newMap(item, loArr);
			// 缓存数据请求对象（缓存2小时改为1小时）
			redis.hmset(rk, layout, Const.ONE_HH);
		}
		return layout;
	}

	/**
	 * 获取LAYOUT公共的通用控制项：查询条件操作用户为空<br>
	 * 这个方法比较特殊，获取公共的通用配置，与用户配置无关<br>
	 */
	private Map<String, String> getLayoutOne(Object uriKey) {
		Map<String, Object> cond = new HashMap<>();
		cond.put(Layout.URI, uriKey);
		cond.put(Layout.OPER, "");
		return aolai.findOne(Layout.TABLE, loSet, cond);
	}

	/**
	 * 获取LAYOUT定制化的私有控制项：查询条件为当前操作用户<br>
	 * 注意：此处必须 args[2] = "", 否则可能循环调用getAliasArray()
	 */
	private Map<String, String> getLayoutOne(Object uriKey, Object userId) {
		// 按用户降序，用户自定义数据优先
		Map<String, String> sort = new HashMap<>();
		sort.put(Layout.OPER, "DESC");
		String[] args = { null, Layout.TABLE, "", "1" };

		Map<String, Object> cond = layoutCond(uriKey, userId);
		List<Map<String, String>> list = aolai.findAll(cond, sort, args);
		log.debug("getLayoutOne...{}", list);

		if (list.isEmpty())
			throw new FailedException(ConfUtil.NOT_FOUND);
		return list.get(0);
	}

	/**
	 * 根据用户查询布局条件
	 */
	private Map<String, Object> layoutCond(Object uriKey, Object userId) {
		Map<String, Object> cond = new HashMap<>();
		cond.put(Layout.URI, uriKey);
		String[] userArr = { (String) userId, "" };
		cond.put(utils.pHolder(Layout.OPER, Const.IN), userArr);
		return cond;
	}

	// 需要解决跨账套问题
	void clearLayout(Object uriKey) {
		// 缓存刷新支持通配符“*”
//		Object userId = req.getAttribute(User.ID);
		redis.delKeys(Const.RDS_PAGE + uriKey + aolai.rdsDbid() + "*");
	}

	/**
	 * 增加一条FIELD元数据
	 */
	public Map<String, String> addFieldItem() {
		String[] keys = { Field.URI, Field.ORIGIN, Field.ALIAS };
		Map<String, Object> data = utils.jsonParams(keys);

		// 先查询已有全部数据
		Object uriKey = data.get(Field.URI);
		List<Map<String, String>> list = getFields(uriKey);
		// 递增序号、默认为定制化数据“1”
		data.put(Field.SORT, list.size());
		data.put(Field.CUSTOM, "1");

		// 设置数据权限：默认零权限、先隐藏显示
		Map<String, String> opts = utils.obj2Map(data.get(Field.OPTS));
		if (!"1".equals(opts.get(Const.PERMIT)))
			opts.put(Const.PERMIT, "0");

		// 增加一条FIELD数据
		if (utils.isFailed(aolai.addRecord(Field.TABLE, data, true)))
			throw new FailedException();

		// 已有数据列表中，补一条新数据
		Map<String, String> item = new HashMap<>();
		item.put(Field.ALIAS, String.valueOf(data.get(Field.ALIAS)));
		item.put(Field.OPTS, utils.obj2Str(opts));
		list.add(item);

		// 更新布局LAYOUT表的数据
		return setLayoutItem(uriKey, list);
	}

	/**
	 * 删除一条FIELD元数据
	 */
	public Map<String, String> delFieldItem() {
		String[] keys = { Field.URI, Field.ALIAS };
		Map<String, Object> where = utils.jsonParams(keys);

		// 先查询全部已有数据
		Object uriKey = where.get(Field.URI);
		List<Map<String, String>> list = getFields(uriKey);
		int idx = 0;
		for (idx = 0; idx < list.size(); idx++) {
			// 遍历全部已有FIELD数据，字段已存在，则不能删除
			if (list.get(idx).get(Field.ALIAS).equals(where.get(Field.ALIAS))) {
				// 另外、非自定义的数据也不能删除
				if (!"1".equals(list.get(idx).get(Field.CUSTOM)))
					throw new FailedException(ConfUtil.CANT_REMOVE);
				list.remove(idx);
				break;
			}
		}
		// 先删除当前数据
		if (utils.isFailed(aolai.delete(Field.TABLE, where)))
			throw new FailedException();

		/** 递减已有其他数据的序号 */
		Map<String, Object> cond = new HashMap<>();
		cond.put(Field.URI, uriKey);
		cond.put(utils.pHolder(Field.SORT, ">"), idx);
		aolai.decrSort(Field.TABLE, Field.SORT, cond);

		// 更新布局LAYOUT表数据
		return setLayoutItem(uriKey, list);
	}

	/**
	 * 更新布局LAYOUT表数据，增、删、改皆需更新
	 */
	private <T> Map<String, String> setLayoutItem(Object uriKey, List<Map<String, T>> list) {
		// 更新据项
		Map<String, Object> params = new HashMap<>();
		params.put(Layout.URI, uriKey);
		params.put(Layout.HEAD, utils.obj2Str(utils.getArr(list, Field.ALIAS)));
		params.put(Layout.OPTS, utils.obj2Str(utils.getArr(list, Field.OPTS)));
		// 默认授权全部操作员
		params.put(Layout.OPER, "");
		params.put(Layout.STATE, "1");
		return setLayoutItem(params);
	}

	/**
	 * 新增并更新一条LAYOUT布局数据
	 */
	private Map<String, String> setLayoutItem(Map<String, Object> params) {
		Map<String, String> result = aolai.addRecord(Layout.TABLE, params, true);
		if (utils.isSuccess(result))
			clearLayout(params.get(Layout.URI));
		return result;
	}

	/**
	 * 根据用户个人喜好自定义显示字段列表
	 */
	public Map<String, String> setLayoutItem() {
		String[] keys = { Layout.URI, Layout.HEAD, Layout.OPTS };
		Map<String, Object> params = utils.jsonParams(keys);
		// 当前用户设置的布局数据
		Object userId = req.getAttribute(User.ID);

		// 对照原有数据进行处理，注意不能修改数据权限
		Map<String, String> item = getLayoutOne(params.get(Layout.URI), userId);
		// 原有数据的权限
		Map<String, String> permit = getAliasArray(item, Const.PERMIT);
	
		// 新配置的保留权限
		List<String> head = utils.obj2List(params.get(Layout.HEAD));
		List<Map<String, String>> opts = utils.obj2List(params.get(Layout.OPTS));
		for (int i = 0; i < head.size() && i < opts.size(); i++) {
			Map<String, String> opt = opts.get(i);
			opt.put(Const.PERMIT, permit.get(head.get(i)) == null ? "0" : "1");
		}
		params.put(Layout.OPTS, utils.obj2Str(opts));
		params.put(Layout.HEAD, utils.obj2Str(head));
		params.put(Layout.OPER, userId);
	
		return setLayoutItem(params);
	}

	/**
	 * 设置一条FIELD元数据，即：自定义字段信息表的字段属性信息
	 */
	public Map<String, String> setFieldItem() {
		String[] keys = { Field.URI, Field.ALIAS, Field.OPTS };
		Map<String, Object> params = utils.jsonParams(keys);

		// 设置数据权限：默认零权限、先隐藏显示
		Map<String, String> opts = utils.obj2Map(params.get(Field.OPTS));
		if (!"1".equals(opts.get(Const.PERMIT)))
			opts.put(Const.PERMIT, "0");

		// 修改配置参数
		Map<String, Object> item = new HashMap<>();
		item.put(Field.OPTS, utils.obj2Str(opts));

		// 更新FIELD数据的条件：Field.URI, Field.ALIAS
		Map<String, Object> cond = utils.newMap(params, new String[] { Field.URI, Field.ALIAS });
		Map<String, String> result = aolai.update(Field.TABLE, item, cond);

		// 更新LAYOUT布局数据
		if (utils.isSuccess(result)) {
			Object uriKey = params.get(Field.URI);
			return setLayoutItem(uriKey, getFields(uriKey));
		}
		return result;
	}

	/**
	 * 维护全部FIELD元数据扩展选项属性，包含排序、各种开关项等
	 * <p>
	 * 每个账套中一个FIELD表
	 */
	public Map<String, String> setFieldItems() {
		String[] keys = { Field.URI, Const.ITEMS };
		Map<String, Object> params = utils.jsonParams(keys);

		Object uriKey = params.get(Field.URI);
		// 先查询全部已有数据
		List<Map<String, String>> list = getFields(uriKey);
		// 更新数据
		List<Map<String, Object>> items = utils.obj2List(params.get(Const.ITEMS));
		if (list.isEmpty() || list.size() != items.size())
			return utils.invalidParams();

		// 获取已有数据原来的表名
		String table = list.get(0).get(Field.ORIGIN);
		Set<String> set = utils.toSet(list, Field.ALIAS);

		// 更新数据
		for (int i = 0; i < items.size(); i++) {
			Map<String, Object> item = items.get(i);
			if (!set.contains(item.get(Field.ALIAS)))
				utils.invalidParams();
			// 更新的数据配置
			Object opts = item.get(Field.OPTS);
			if (opts instanceof Map)
				item.put(Field.OPTS, utils.obj2Str(opts));
			item.put(Field.ORIGIN, table); // 原来的表名不能变
			item.put(Field.URI, uriKey); // 补充uri字段
			item.put(Field.SORT, i); // 重新排序
		}
		// 批量更新数据
		Map<String, String> result = aolai.batchAdd(Field.TABLE, items, true);
		if (utils.isSuccess(result))
			return setLayoutItem(uriKey, items);
		return result;
	}

	/**
	 * 根据uriKey查询FIELD全部信息一览
	 */
	private List<Map<String, String>> getFields(Object uriKey) {
		Map<String, Object> cond = new HashMap<>();
		cond.put(Field.URI, uriKey);
		Map<String, String> sort = new HashMap<>();
		sort.put(Field.SORT, "");
		return aolai.findAll(Field.TABLE, "", cond, sort);
	}

	/**
	 * 根据fieldUri查询，获取FIELD元数据字段信息，返给前端布局
	 * <p>
	 * 因为每个账套中一个FIELD表，所以需要携带base参数
	 */
	public List<Map<String, String>> getFieldsBy() {
		String[] keys = { Field.URI };
		Map<String, Object> params = utils.jsonParams(keys);

		// 查询全部元数据的字段，如："getFieldList"
		Object uriKey = params.get(Field.URI);
//		return layout(getFieldsBy(uriKey), uriKey, params);

		List<Map<String, String>> fiedlList = getFieldsBy(uriKey);
		log.debug("getFieldsBy...{}", fiedlList);
		// 统一列设置页面，返回所有字段，不用返回每个用户的数据
		if (Const.S_1.equals(params.get(Const.ALL))) {
			return fiedlList;
		}
		return layout(fiedlList, uriKey, params);
	}

	/**
	 * 在当前账套内获取FIELD表中的元数据字段信息<br>
	 * 根据uriKey查询fieldAlias所有列名<br>
	 */
	public List<Map<String, String>> getFieldsBy(Object uriKey) {
		Map<String, Map<String, String>> tables = utils.joinCond(Field.TABLE);
		Map<String, String> cond1 = new HashMap<>();
		cond1.put(Field.ORIGIN, Const.TABLE);
		cond1.put(Field.ALIAS, Const.ALIAS);
		// 关联GMETA元数据表
		tables.put(Const.GMETA, cond1);

		Map<String, Object> cond = new HashMap<>();
		cond.put(Field.URI, uriKey);

		// 按序号排列
		Map<String, String> sort = new HashMap<>();
		sort.put(Field.SORT, "ASC");

		// 查询全部元数据：具体展示格式放在前端处理
		return aolai.findAll(tables, "getFieldList", cond, sort);
	}

	/**
	 * 处理布局数据，返给前端布局
	 */
	private List<Map<String, String>> layout(List<Map<String, String>> list, Object uriKey,
			Map<String, Object> params) {
		Object show = params.get(Const.SHOW);
		// 选项配置，忽略隐藏{"hide":"1"}的值
		Map<String, Map<String, String>> alias = getAliasOpts(getLayoutBy(uriKey));
		Map<String, Map<String, String>> items = utils.list2Map(list, Field.ALIAS);
		log.debug("layout...{}", alias);
		// 按alias重新排序
		List<Map<String, String>> fields = new ArrayList<>();
		for (Map.Entry<String, Map<String, String>> e : alias.entrySet()) {
			Map<String, String> item = items.get(e.getKey());
			Map<String, String> opts = e.getValue();

			// 忽略隐藏{"hide":"1"}的值
			// 设置页面需要展示hide的字段，此处添加前端传入参数 {show:1}则返回隐藏字段；否则不返回隐藏字段
			if (Const.S_1.equals(show) || !Const.S_1.equals(opts.get(Const.HIDE))) {
				// 选项配置
				item.put(Layout.OPTS, utils.obj2Str(e.getValue()));
				fields.add(item);
			}
		}
		return fields;
	}

	/**
	 * 根据前台的传递参数，拼接生成多表联合查询参数<br>
	 * JOINT:{userName:{key:“A”,val:“阿三”}, fzflMc:{key:“B”,val:"绿色"}...}
	 */
	public Map<String, Map<String, String>> jointTables(Map<String, Object> params) {
		Map<String, Map<String, Object>> joint = utils.obj2Map(params.get(Const.JOINT));
		// 查询FIELD表中数据
		Map<String, Object> where = utils.newMap(params, Field.URI);
		// 防止 FIELD_ALIAS IN 此处加入判空处理
		if (!utils.isEmpty(joint.keySet())) {
			where.put(utils.pHolder(Field.ALIAS, Const.IN), joint.keySet());
		}
		List<Map<String, String>> list = aolai.findAll(Field.TABLE, where);

		// 根据uriKey获取通用“”布局参数
		Map<String, String> layoutOne = getLayoutBy(where.get(Field.URI), "");
		Map<String, Map<String, String>> tables = utils.json2Map(layoutOne.get(Layout.JOINT));

		// 获取FIELD表中的字段别名fieldAlias及关联表名fieldOrigin
		for (Map<String, String> field : list) {
			// 获取字段别名如：fzflMC
			String alias = field.get(Field.ALIAS);
			// 获取转换参数，格式如：{key:“A”,val:“阿三”}
			Map<String, Object> map = joint.get(alias);
			if (!utils.isEmpty(map)) {
				// 获取关联的表名，并根据表名获取关联条件，如：ZDFZFL
				Map<String, String> cond = tables.get(field.get(Field.ORIGIN));

				// 获取运算符主键：A、B、C、..
				Object key = map.get(Const.KEY);
				// 数值，同时支持单一数值参数，也支持列表
				Object val = map.get(Const.VAL);
				// 变成SQL数值："%{}"
				val = ConfUtil.opExpr(key, val);

				// 如：“{fzflMC} like”
				String fieldKey = ConfUtil.getOpKey(alias, key);
				if (cond != null && val instanceof CharSequence) {
					cond.put(fieldKey, String.valueOf(val));
					// 这里增加每个表的动态连结方式：“left”\“inner”\“right”
					// 在joinTables()方法中会用到此参数
					cond.put(null, "inner");
				} else {
					// 如果是对应主表的字段，参数值放到params条件中
					// 拓展in和between场景，参数也放到params条件中
					params.put(fieldKey, val);
				}
			}
		}
		return tables;
	}

}
