package cn.ps1.aolai.service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringJoiner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;

import cn.ps1.aolai.dao.AolaiDao;
import cn.ps1.aolai.entity.*;
import cn.ps1.aolai.utils.ConfUtil;
import cn.ps1.aolai.utils.Const;
import cn.ps1.aolai.utils.FailedException;
import cn.ps1.aolai.utils.FmtUtil;

/**
 * Aolai基础框架相关服务层，基于Mybatis整合的元数据的基础操作类(CRUD)
 *
 * @author Aolai
 * @version 1.0 $Date: 2019.7.6
 * @since 1.7
 */

@Service
public abstract class GmetaService {

	private static Logger log = LoggerFactory.getLogger(GmetaService.class);
	/**
	 * 排序表名字段
	 */
	static final String META_TABLE = "META_TABLE";
	/**
	 * 排序字段
	 */
	static final String META_SORT = "META_SORT";
	/**
	 * 租户ID
	 */
	static final String TENANT = "TENANT";
	/**
	 * 当前操作用户的用户编号
	 */
	static final String OP_UID = "OpUid";
	/** 当前数据的创建人 */
	static final String CREATOR = "Creator";

	/**
	 * 排序参数，如：ASC,DESC
	 */
	static final String[] ORDER_BY = { "ASC", "DESC" };

	/**
	 * 正则匹配数字
	 */
	static final Pattern P_DIGIT = Pattern.compile("\\d+"); // 数字
	/**
	 * 正则匹配普通的 {fieldKey} 关键字字段
	 */
	static final Pattern P_QUOTE = Pattern.compile("\\{ *\\w+ *\\}"); // 引用
	/**
	 * 固定数值占位符{V}
	 */
	static final Pattern P_PLACE = Pattern.compile("\\{ *[V0-9]* *\\}");

	/**
	 * 正则匹配JSON查询条件，只支持2层："{userDuty}->>'$.appCode.way' ="
	 */
	static final Pattern P_JSON = Pattern.compile("^,'\\$(\\.\\w+)+'\\)$");

	/**
	 * 防止SQL注入的字符串校验
	 */
	static final String VW = "'[\\w+ \u4e00-\u9fa5]*'";

	/**
	 * 注意：与Const.I18N="i18n"有别：注意这里用“lang”，但不能用“i18n”
	 */
	static final String DUPL = "dupl";
	static final String M_PKEY = "!p"; // mark_Pkey
	static final String M_I18N = "!i"; // mark_I18n
	static final String M_TYPE = "!t"; // mark_Type,VARCHAR(32)
	static final String M_STYLE = "!s"; // mark_Style,如：'{"blur":"1"}'
//	static final String WHERE = "where";
	static final String LEFT = "left";
	static final String UL = "_";
	// static final String WD = "\\w+";
	static final String CAST = "CAST(";
	static final String R_PAREN = "')";
	/**
	 * 数据库中空字符
	 */
	static final String NUL = "''";

	@Autowired
	AolaiDao dao;
	@Autowired
	LayoutService layout;
	@Autowired
	RedisService redis;
	@Autowired
	UtilsService utils;
	@Autowired
	HttpServletRequest req; // 单实例可以注入
	@Autowired
	private ApplicationContext appCtx;

	/**
	 * GMETA表的元数据：从GMETA表中GMETA表自己的元数据一览
	 */
	Map<String, String> gMetaOne() {
		// 再查询数据库信息：META_TABLE="GMETA"
		// Map<String, String> map = gmetaParams();
		Map<String, Object> cond = new HashMap<>();
		cond.put(META_TABLE + Const.EQU, Const.GMETA);
		// 不必区分租户
		List<Map<String, String>> list = dao.findList(gMetaParams(), cond);
		return utils.list2Map(list, Const.ALIAS, Const.FIELD);
	}

	/**
	 * 当前数据源是否为PostgreSQL<br>
	 * 注意区分数据库驱动，这里与AolaiDao.xml中的databaseId命名保持一致<br>
	 */
	boolean isPostgres() {
		// 在加载配置文件，注册SqlSessionFactoryBean时获取配置属性
		return appCtx.containsBean("dsPostgres");
	}

	/**
	 * 返回当前的租户ID，或null
	 */
	public Object tenantId() {
		return utils.jsonParams().get(ConfUtil.dbid());
	}

	/**
	 * 数据表的租户ID主键
	 */
	public String tenantKey(Object table) {
		return table + UL + TENANT;
	}

	/**
	 * 返回当前的租户ID或""，并增加dot分割符
	 */
	Object rdsDbid() {
		Object tid = tenantId();
		return ConfUtil.isCustomMeta() && tid != null ? tid + "." : "";
	}

	Object inuseBase() {
		return utils.jsonParams().get(Const.BASE);
	}

	Object i18n(Object lang) {
		return lang == null ? utils.jsonParams().get(Const.I18N) : lang;
	}

	/**
	 * 账套（base）及表名组合拼接符号移到ThirdService的baseNameOf()处理<br>
	 * 完美兼容所有方法不用再传 base参数
	 */
	public String tableOf(String tableName) {
		return getTable(null, tableName);
	}

	/**
	 * 设置多租户数据分离查询条件<br>
	 * {"USER_NAME_ZH like":"张%", "TABLE_TENANT =": "889"}
	 */
	Map<String, Object> setDataIsolation(Map<String, Object> cond, String table) {
		/** 增加了租户数据隔离 Jun.22,2024 */
		if (ConfUtil.dataIsolation()) {
			cond.put(tenantKey(table) + Const.EQU, tenantId());
		}
		return cond;
	}

	/*********************************************************
	 * 往“table”表中增加一条记录
	 */

	/**
	 * 往业务表中插入一条数据，并判断重复是否更新！
	 *
	 * @param table 表名
	 * @param dto 字段映射
	 * @param data 数据
	 * @param dupl 更新标记
	 * @return result 返回状态
	 */
	private Map<String, String> addRecord(Object base, String table, Map<String, String> dto,
			Map<String, Object> data, boolean dupl) {

		// 获取键值对映射：{USER_NAME_ZH : "张三"}
		data = setFieldData(dto, data, true);

		/** 增加了租户数据隔离 Jun.22,2024 */
		// {USER_NAME_ZH : "张三", TABLE_TENANT : "889"}
		if (ConfUtil.dataIsolation())
			data.put(tenantKey(table), tenantId());

		Map<String, String> map = duplCheck(base, table, dto, dupl);

		// 返回执行结果
		log.debug("addRecord...{}", data);
		return utils.result(dao.addOne(map, data) > 0);
	}

	/**
	 * 删除重复数据，为了兼容PostgreSQL，识别出table的主键
	 */
	private Map<String, String> duplCheck(Object base, String table, Map<String, String> dto,
			boolean dupl) {
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(base, table));

		// 增加数据时，有些数据库不支持ON DUPLICATE KEY UPDATE语法
		// 为了兼容PostgreSQL，识别出table的主键			
		if (dupl) {
			map.put(DUPL, isPostgres() ? getDuplPks(table, dto) : "");
		}
		return map;
	}

	/**
	 * 识别出isPostgres的单表table的主键pkey，以“逗号”分隔
	 */
	private String getDuplPks(String table, Map<String, String> dto) {
		/** 增加了租户数据隔离 Jun.22,2024 */
		return getMetaPks(dto, ConfUtil.dataIsolation() ? tenantKey(table) : "");
	}

	/**
	 * 识别出单表table的主键pkey，以“逗号”分隔
	 */
	private String getMetaPks(Map<String, String> dto, String pk) {
		StringJoiner jt = new StringJoiner(Const.COMMA);
		if (pk.length() > 0)
			jt.add(pk);
		for (Map.Entry<String, String> e : dto.entrySet()) {
			// 获取元数据中table的约定的主键名（pkey）
			// 如：根据userId!p截取的userId 并获取 USER_ID
			// 值等于1的情况下，才是主键
			if (e.getKey().contains(M_PKEY) && Const.S_1.equals(e.getValue()))
				jt.add(dto.get(e.getKey().split("!")[0]));
		}
		return jt.toString();
	}

	/**
	 * 往业务表中插入一条数据，并判断重复是否更新！
	 */
	public Map<String, String> addRecord(Object base, String table, Map<String, Object> data,
			boolean dupl) {
		return addRecord(base, table, getDto(table), data, dupl);
	}

	/**
	 * 往业务表中插入一条数据，并判断重复是否更新！
	 */
	public Map<String, String> addRecord(String table, Map<String, Object> data, boolean dupl) {
		return addRecord(null, table, data, dupl);
	}

	/**
	 * 往基础表中插入一条数据，并判断重复是否更新！
	 */
	public Map<String, String> addRecord(String table, Map<String, Object> data) {
		return addRecord(null, table, data, false);
	}

	/**
	 * 往业务表中插入一条数据，并判断重复是否更新！
	 */
	public Map<String, String> addRecord(Object base, String table, Map<String, Object> data) {
		return addRecord(base, table, data, false);
	}

	/*********************************************************
	 * 批量往业务表中插入多条数据
	 */
	private Map<String, String> batchAdd(Object base, String table, Map<String, String> dto,
			List<Map<String, Object>> list, Object lang, boolean dupl) {
		if (list.isEmpty())
			throw new FailedException();

		// 取第一行数据作为参数
		Map<String, Object> data = list.get(0);

		// 不再需要传递参数
		data.put(Const.I18N, lang);

		// 设置别名映射格式：{USER_NAME_ZH : "userName"}
		data = setFieldData(dto, data, false);

		// 最后更新者_OPUID：从data中找到任意一个主要的有效字段如：USER_NAME_ZH
		String operUid = null;
		String creator = null;
		for (Map.Entry<String, Object> e : data.entrySet()) {
			// 通过USER_NAME_ZH截取后变为小写，再拼接OpUid
			String key = e.getKey().split(UL)[0].toLowerCase();
			operUid = key + OP_UID;
			creator = key + CREATOR;
			break;
		}

		Object userId = req.getAttribute(User.ID);
		for (Map<String, Object> item : list) {
			// 这里增加数据校验：理论上type不可能为null，要么成功，要么失败后抛出异常
			for (Map.Entry<String, Object> e : item.entrySet()) {
				validationRules(dto, e);
			}
			// 最后更新者_OPUID，这里不用满足必须在dto中的条件
			item.put(operUid, userId);
			item.put(creator, userId);

			/** 增加了租户数据隔离 Jun.22,2024 */
			// { "TENANT" : "889" }
			if (ConfUtil.dataIsolation())
				item.put(TENANT, tenantId());
		}

		/** 增加了租户数据隔离 Jun.22,2024 */
		// 这里补充参数"TENANT"为映射的别名：{ "XXX_TENANT": TENANT }
		if (ConfUtil.dataIsolation())
			data.put(tenantKey(table), TENANT);

		// 数据库执行参数
		Map<String, String> map = duplCheck(base, table, dto, dupl);
		// 返回执行结果
		log.debug("batchAdd...{}", map);
		return utils.result(dao.batchAdd(map, data, list) > 0);
	}

	/**
	 * 批量往业务表中插入多条数据(※)
	 */
	public Map<String, String> batchAdd(Object base, String table, List<Map<String, Object>> items,
			Object lang, boolean dupl) {
		return batchAdd(base, table, getDto(table), items, lang, dupl);
	}

	/**
	 * 批量往业务表中插入多条数据
	 *
	 * @deprecated
	 */
	@Deprecated
	public Map<String, String> batchAdd(String table, List<Map<String, Object>> items,
			Map<String, Object> map, boolean dupl) {
		return batchAdd(table, items, dupl);
	}

	/**
	 * 批量往业务表中插入多条数据
	 *
	 * @deprecated
	 */
	@Deprecated
	public Map<String, String> batchAdd(String table, List<Map<String, Object>> items,
			Map<String, Object> map) {
		return batchAdd(table, items, map.containsKey(DUPL));
	}

	/**
	 * 批量往业务表中插入多条数据(※)
	 */
	public Map<String, String> batchAdd(String table, List<Map<String, Object>> items, Object lang,
			boolean dupl) {
		return batchAdd(null, table, items, lang, dupl);
	}

	/**
	 * 批量往业务表中插入多条数据
	 */
	public Map<String, String> batchAdd(String table, List<Map<String, Object>> items, boolean dupl) {
		return batchAdd(null, table, getDto(table), items, null, dupl);
	}

	/**
	 * 批量往业务表中插入多条数据(※)
	 */
	public Map<String, String> batchAdd(String table, List<Map<String, Object>> items, Object lang) {
		return batchAdd(table, items, lang, false);
	}

	/**
	 * 批量往业务表中插入多条数据
	 */
	public Map<String, String> batchAdd(String table, List<Map<String, Object>> items) {
		return batchAdd(table, items, false);
	}

	/**
	 * 批量往业务表中插入多条数据：注意参数items
	 *
	 * @deprecated 用batchAdd(table, items, map)替换
	 */
	@Deprecated
	public Map<String, String> batchAdd(Map<String, String> map) {
		// 整理“数据转换对象DTO”，并检查参数：items、dupl
		List<Map<String, Object>> list = utils.json2List(map.get(Const.ITEMS));
		return batchAdd(map.get(Const.TABLE), list, map.containsKey(DUPL));
	}

	/*********************************************************
	 * 删除表中的数据
	 */
	private Map<String, String> delete(Object base, String table, Map<String, String> dto,
			Map<String, Object> cond, Object joint) {
		// 梳理删除条件如：{"GUSER_ID =":"4523048"}
		cond = setCondition(dto, cond, joint);
		// 删除条件不能为空（有全部删除风险）
		if (cond.isEmpty())
			throw new FailedException();

		/** 增加了租户数据隔离 Jun.22,2024 */
		// { "TABLE_TENANT =": "889" }
		cond = setDataIsolation(cond, table);

		return utils.result(dao.delete(getTable(base, table), cond) > 0);
	}

	/**
	 * 删除表中的数据
	 */
	public Map<String, String> delete(Object base, String table, Map<String, Object> cond,
			Object joint) {
		return delete(base, table, getDto(table), cond, joint);
	}

	/**
	 * 删除表中的数据
	 *
	 * @deprecated
	 */
	@Deprecated
	public Map<String, String> delete(String table, Map<String, Object> cond, Object joint) {
		return delete(null, table, cond, joint);
	}

	/**
	 * 删除表中的数据
	 */
	public Map<String, String> delete(String table, Map<String, Object> cond) {
		return delete(null, table, cond, null);
	}

	/**
	 * 删除表中的数据
	 */
	public Map<String, String> delete(Object base, String table, Map<String, Object> cond) {
		return delete(base, table, cond, null);
	}

	/*********************************************************
	 * 批量删除数据
	 */

	/**
	 * 批量删除数据，其中list中的数据为逐条删除数据的条件
	 */
	private Map<String, String> batchDelete(Object base, String table, Map<String, String> dto,
			List<Map<String, Object>> list, String joint) {
		String theTable = getTable(base, table);
		// 遍历数据并逐条删除
		int count = 0;
		for (int i = 0; i < list.size(); i++) {
			// list中的数据为逐条删除数据的条件：{"USER_NAME_ZH like":"张%"}
			Map<String, Object> cond = setCondition(dto, list.get(i), joint);
			if (!cond.isEmpty()) {
				/** 增加了租户数据隔离 Jun.22,2024 */
				setDataIsolation(cond, table); // {"TABLE_TENANT =": "889"}
				count += dao.delete(theTable, cond); // 计数
			}
		}
		// if (count == list.size()) // 数据一致性
		return utils.result(count > 0);
	}

	/**
	 * 批量删除表中的数据，其中list中的数据为逐条删除数据的条件
	 */
	public Map<String, String> batchDelete(Object base, String table, List<Map<String, Object>> list) {
		return batchDelete(base, table, getDto(table), list, null);
	}

	/**
	 * 批量删除表中的数据，其中list中的数据为逐条删除数据的条件
	 */
	public Map<String, String> batchDelete(String table, List<Map<String, Object>> list) {
		return batchDelete(null, table, list);
	}

	/**
	 * 批量删除表中的数据，其中list中的数据为逐条删除数据的条件
	 */
	public Map<String, String> batchDelete(Object base, String table,
			List<Map<String, Object>> list, String joint) {
		return batchDelete(base, table, getDto(table), list, joint);
	}

	/**
	 * 批量删除表中的数据，其中list中的数据为逐条删除数据的条件
	 *
	 * @deprecated
	 */
	@Deprecated
	public Map<String, String> batchDelete(String table, List<Map<String, Object>> list,
			String joint) {
		return batchDelete(null, table, list, joint);
	}

	/*********************************************************
	 * 更新表中的数据
	 */
	private Map<String, String> update(Object base, String table, Map<String, String> dto,
			Map<String, Object> item, Map<String, Object> cond, Object joint) {
		// 查询条件（cond={USER_ID:"13905317889"}）
		// 更新字段（item={USER_NAME_ZH:"张三"}）
		cond = setCondition(dto, cond, joint);
		item = setCondition(dto, item, null, true);
		if (cond.isEmpty() || item.isEmpty())
			throw new FailedException();

		/** 增加了租户数据隔离 Jun.22,2024 */
		setDataIsolation(cond, table); // { "TABLE_TENANT =": "889" }

		// 返回执行结果
		return utils.result(dao.update(getTable(base, table), item, cond) > 0);
	}

	/**
	 * 更新数据
	 *
	 * @deprecated
	 */
	@Deprecated
	public Map<String, String> update(Object base, String table, Map<String, Object> item,
			Map<String, Object> cond, Object joint) {
		// 根据table表名，获取表的列名与map的映射关系
		// DTO = { uid:'USER_ID', uid!t:'INTEGER', uid!i:'1', uid!p:'1' }
		return update(base, table, getDto(table), item, cond, joint);
	}

	/**
	 * 更新数据
	 *
	 * @deprecated
	 */
	@Deprecated
	public Map<String, String> update(String table, Map<String, Object> item,
			Map<String, Object> cond, Object joint) {
		return update(null, table, getDto(table), item, cond, joint);
	}

	/**
	 * 更新数据
	 */
	public Map<String, String> update(Object base, String table, Map<String, Object> item,
			Map<String, Object> cond) {
		return update(base, table, getDto(table), item, cond, null);
	}

	/**
	 * 更新数据
	 */
	public Map<String, String> update(String table, Map<String, Object> item,
			Map<String, Object> cond) {
		return update(null, table, item, cond);
	}

	/*********************************************************
	 * 支持多张表 join关联查询：left="left"为左连接
	 */
	public List<Map<String, String>> findList(Object base, Map<String, Map<String, String>> tables,
			String uriKey, Map<String, Object> cond, Map<String, String> sort, String limit) {
		return findList(base, tables, uriKey, cond, sort, LEFT, limit);
	}

	/**
	 * 支持多张表 join关联查询：left=""为内连接
	 *
	 * @param tables 两表名为key，两表的关联主键的映射，表1设为null
	 * @param uriKey 前台上下文或接口名，仅缓存数据用
	 * @param cond 综合多表的查询条件
	 * @param limit 限制每页输出行数，为“0”时不限行数
	 */
	public List<Map<String, String>> findList(Object base, Map<String, Map<String, String>> tables,
			String uriKey, Map<String, Object> cond, Map<String, String> sort, String left,
			String limit) {

		// 后面的 setArgsAndCond 要使用，getDtoBy(uriKey)未必包含cond要使用的字段
		Map<String, String> allDto = new HashMap<>();
		for (Map.Entry<String, Map<String, String>> e : tables.entrySet()) {
			allDto.putAll(getDto(e.getKey()));
		}
		// 这里的table为join后多个关联的表
		String[] table = joinTables(base, allDto, tables, left);

		// 数据库查询参数：table,alias,sort,joint,limit,floor
		Map<String, String> args = queryArgs(table[1], uriKey, limit);
		// 处理args.get("alias")和查询条件
		// “别名”设为：['userLang', 'userMail', 'userName', ...]
		cond = setArgsAndCond(args, allDto, cond, sort);

		/** 增加了租户数据隔离 Jun.22,2024 */
		// { "TABLE_TENANT =": "889" }
		cond = setDataIsolation(cond, table[0]); // 主表table[0]

		// 返回执行结果
		Map<String, String> dto = getDtoBy(allDto, uriKey);
		// 若dto的元数据携带style信息，根据blur模糊处理敏感数据
		return setBlur(dao.findList(args, cond), dto);
	}

	Map<String, String> queryArgs(String table, String uriKey, String limit) {
		Map<String, String> args = new HashMap<>();
		args.put(Const.TABLE, table);
		args.put(Const.ALIAS, uriKey);
		args.put(Const.LIMIT, limit);
		return args;
	}

	/**
	 * 联合多表查询，这里的dto为查询多个表相关联的字段
	 */
	String[] joinTables(Object base, Map<String, String> allDto,
			Map<String, Map<String, String>> tables, String left) {
		String dbid = utils.quote(tenantId());
		// String table = null;
		Object lang = i18n(null);
		String[] t = new String[2];
		StringJoiner jt = new StringJoiner(" ", " ", "");
		/**
		 * 以表名为主键，其中第一个表为空，第二个表与第一个表关联，第三个表与第二个表关联<br>
		 * { Table0:<null>, Table1<Table1.colA=Table0.colB> }<br>
		 */
		for (Map.Entry<String, Map<String, String>> e : tables.entrySet()) {
			// 根据表名table获取左连接的字段映射joint关系
			Map<String, String> cond = tables.get(e.getKey());
			// 表的全名，如：base=“public.base899_”\“base899.”\“899”
			String table = getTable(base, e.getKey());
			if (utils.isEmpty(cond)) {
				t[0] = e.getKey(); // 返回主表table
				// 第一个主表的关联条件必为空，后面不能为空
				t[1] = table; // 主表
			} else {
				/**
				 * 增加表的动态连结方式, null:“left”\“inner”\“right”<br>
				 * 关联两个表的条件：t0 left join table AS t2 on t2.TENANT='899'
				 */
				jt.add(leftJoin(left, table, cond));

				/** 增加了租户数据隔离，如：t2.TENANT='899' Jun.22,2024 */
				if (ConfUtil.dataIsolation()) {
					jt.add(tenantKey(e.getKey()) + Const.EQU + dbid);
				}

				int n = 0; // 如：cond={‘userComp’:‘compId’}
				for (Map.Entry<String, String> ent : cond.entrySet()) {
					// 这里的val有可能为空，如跳过则多一个“and”连结符
					if (ent.getValue() == null)
						continue;
					if (n++ > 0 || ConfUtil.dataIsolation())
						jt.add("and");
					jt.add(relateCond(allDto, ent, lang));
				}
			}
		}
		t[1] += jt.toString();
		return t;
	}

	/**
	 * 左连接关联两个表
	 */
	private String leftJoin(String left, String table, Map<String, String> cond) {
		// 优先使用cond中携带的“left”“inner”“right”，这里用了null作为key
		String inner = cond.get(null);
		if (inner != null) {
			cond.remove(null);
			left = inner;
		} else if (left == null) {
			return String.join(" ", "join", table, "on");
		}
		return String.join(" ", left, "join", table, "on");
	}

	/**
	 * 建立两个表的关联关系
	 */
	private String relateCond(Map<String, String> allDto, Map.Entry<String, String> ent, Object lang) {
		String key = ent.getKey();
		String val = ent.getValue();
		if (allDto.containsKey(val)) { // val=‘compId’的情况
			val = i18n(allDto, val, lang);
		} else {
			// val=‘1036’ 或‘’的情况，可返回一个带引号的“字符串”参数
			// 外部传递的参数已在 FmtUtil.isWord(val)中校验过，无SQL注入风险
			// val = FmtUtil.isWord(val) ? utils.sqlVal(val) : NUL;
			val = utils.sqlVal(val);
		}
		/**
		 * 两表关联的主键如：{‘userComp’:‘compId’}，{‘userComp’:‘1036’}<br>
		 * 如果右侧‘1036’的元数据不存在，右侧直接当作数值使用<br>
		 * 使用单引号以支持PGSQL，其中保证key不能有SQL注入非法字符
		 */
		if (allDto.containsKey(key)) {
			// 用等号（=）拼接两表的主键：and USER_COMP=COMP_ID
			return i18n(allDto, key, lang) + Const.EQU + val;
		} else if (P_QUOTE.matcher(key).find()) {
			// 用（like）拼接两表的主键时：and USER_NAME_ZH like '%张%'
//			key = getFieldKey(allDto, key, "", lang); // 转为空字符串数组 {}
			return getFieldKey(allDto, key, new String[] {}, lang) + val;
		} else {
			// 不满足以上条件时，视为无效参数（避免SQL注入）
			throw new FailedException(null, key);
		}
//		return key + val;
	}

	/**
	 * 限定了最大返回记录数（limit.rows）
	 */
	private void setLimitRows(Map<String, String> args, Map<String, Object> cond) {
//		String[] keys = { Const.PAGE_NO, Const.PAGE_SIZE };
		// 如果有分页信息，则分页处理
//		if (utils.availParams(cond, keys)) {
		int pSize = utils.getInt(cond, Const.PAGE_SIZE);
		if (pSize > 0) {
			// 页面号码、及每页展示的行数
			int pNo = utils.getInt(cond, Const.PAGE_NO) - 1;
			args.put("floor", pNo > 0 ? String.valueOf(pNo * pSize) : "0");
			args.put(Const.LIMIT, String.valueOf(pSize));
			return;
		}
		String limit = args.get(Const.LIMIT);
		// 当limit==null或''，不限制数据行数
		if (!utils.isInteger(limit)) {
			// 如果limit.rows=''，默认不限制数据，有可能仅limit.rows=''
			args.put(Const.LIMIT, ConfUtil.limitRows());
		} else if ("0".equals(limit)) {
			// 如果limit==0，删除参数，设置为空不限行数，其他时保留
			args.remove(Const.LIMIT);
		}
	}

	/**
	 * 设置数据库查询参数：table,alias,sort,joint,limit,floor<br>
	 * 为了适配处理返回分页信息，把查询条件集中处理：args, dto, cond, sort<br>
	 * “别名”设为：['userLang', 'userMail', 'userName', ...]
	 */
	Map<String, Object> setArgsAndCond(Map<String, String> args, Map<String, String> allDto,
			Map<String, Object> cond, Map<String, String> sort) {
		// 优先使用传递的语言参数
		Object lang = i18n(cond.get(Const.I18N));
		// 传回前端的数据映射的主键参数数组（无SQL注入风险），‘alias’ <=> ‘uriKey’
		args.put(Const.ALIAS, setAlias(allDto, args.get(Const.ALIAS), lang));
		// 根据“数据转换对象DTO”，整理排序字段
		args.put(Const.ORDER, orderBy(allDto, sort, lang));
		// 限定了最大返回记录数，设置参数：limit,floor
		setLimitRows(args, cond);

		// 拼接后的查询条件
		return setCondition(allDto, cond, args.get(Const.JOINT));
	}

	/**
	 * 对数据进行模糊化处理：若dto的元数据携带style信息，根据blur模糊处理敏感数据
	 */
	List<Map<String, String>> setBlur(List<Map<String, String>> list, Map<String, String> dto) {
		if (!list.isEmpty()) {
			Map<String, String> blur = setBlur(list.get(0), dto);
			if (!blur.isEmpty()) {
				log.debug("setBlur...{}", dto);
				// 如果blur不为空，则从第二行开始逐行处理
				for (int i = 1; i < list.size(); i++) {
					Map<String, String> row = list.get(i);
					for (Map.Entry<String, String> e : blur.entrySet()) {
						row.put(e.getKey(), FmtUtil.blur(row.get(e.getKey()), e.getValue()));
					}
				}
			}
		}
		return list;
	}

	/**
	 * 对数据进行模糊化处理：若dto的元数据携带style信息，根据blur模糊处理敏感数据
	 */
	private Map<String, String> setBlur(Map<String, String> item, Map<String, String> dto) {
		Map<String, String> blur = new HashMap<>();
		for (Map.Entry<String, String> e : item.entrySet()) {
			String style = dto.get(e.getKey() + M_STYLE);
			if (style == null)
				break; // 理论上一个为空，皆为空，所以直接退出
			Map<String, String> tmp = utils.json2Map(style);
			style = tmp.get(Const.BLUR);
			// 默认区分：电话号码\身份证号\电子邮箱\银行账户(PN\ID\EM\BA)
			if (style != null) {
				blur.put(e.getKey(), style);
				item.put(e.getKey(), FmtUtil.blur(item.get(e.getKey()), style));
			}
		}
		return blur;
	}

	/*********************************************************
	 * 查询数据一览，限定了最大返回记录数（limit.rows）
	 * <p>
	 * 数据库查询参数：base,table,alias,sort,joint,limit,floor
	 */
	public List<Map<String, String>> findList(Map<String, String> args, Map<String, Object> cond,
			Map<String, String> sort) {
		String table = args.get(Const.TABLE);
		String uriKey = args.get(Const.ALIAS);

		// 后面的 setArgsAndCond 要使用，getDtoBy(uriKey)未必包含cond要使用的字段
		Map<String, String> allDto = getDto(table);

		// 租户隔离
		// Object base = baseOf(args);
		// 替换表名
		args.put(Const.TABLE, tableOf(table));

		// 整合处理args.get("alias")和查询条件
		// “别名”设为：['userLang', 'userMail', 'userName', ...]

		/** 增加了租户数据隔离 Jun.22,2024 */
		// { "TABLE_TENANT =": "889" }
		cond = setArgsAndCond(args, allDto, cond, sort);
		cond = setDataIsolation(cond, table);

		// 返回查询结果
		Map<String, String> dto = getDtoBy(allDto, uriKey);
		// 若dto的元数据携带style信息，根据blur模糊处理敏感数据
		return setBlur(dao.findList(args, cond), dto);
	}

	/**
	 * 查询数据一览，限定了最大返回记录数（limit.rows）
	 */
	public List<Map<String, String>> findList(Map<String, String> args, Map<String, Object> cond) {
		return findList(args, cond, null);
	}

	/**
	 * 查询数据一览
	 */
	public List<Map<String, String>> findList(Object base, String table, String uriKey,
			Map<String, Object> cond, Map<String, String> sort) {
		// uriKey为传回前端的参数数组
		String[] args = { (String) base, table, uriKey };
		return findAll(cond, sort, args);
	}

	/**
	 * 查询数据一览
	 */
	public List<Map<String, String>> findList(String table, String uriKey,
			Map<String, Object> cond, Map<String, String> sort) {
		return findList(null, table, uriKey, cond, sort);
	}

	/**
	 * 查询数据一览
	 */
	public List<Map<String, String>> findAll(Object base, String table, String uriKey,
			Map<String, Object> cond, Map<String, String> sort) {
		// uriKey为传回前端的参数数组
		String[] args = { (String) base, table, uriKey, Const.S_0 };
		return findAll(cond, sort, args);
	}

	/**
	 * 查询数据一览
	 */
	public List<Map<String, String>> findAll(String table, String uriKey, Map<String, Object> cond,
			Map<String, String> sort) {
		return findAll(null, table, uriKey, cond, sort);
	}

	/**
	 * 查询数据一览
	 */
	public List<Map<String, String>> findList(Object base, String table, String uriKey,
			Map<String, Object> cond) {
		// uriKey为传回前端的参数数组
		return findList(base, table, uriKey, cond, null);
	}

	/**
	 * 查询数据一览
	 */
	public List<Map<String, String>> findAll(Object base, String table, String uriKey,
			Map<String, Object> cond) {
		// uriKey为传回前端的参数数组
		return findAll(base, table, uriKey, cond, null);
	}

	/**
	 * 查询数据一览
	 */
	public List<Map<String, String>> findList(String table, String uriKey, Map<String, Object> cond) {
		// uriKey为传回前端的参数数组
		return findList(null, table, uriKey, cond, null);
	}

	/**
	 * 查询数据一览
	 */
	public List<Map<String, String>> findAll(String table, String uriKey, Map<String, Object> cond) {
		// uriKey为传回前端的参数数组
		return findAll(null, table, uriKey, cond, null);
	}

	/**
	 * 查询数据一览
	 */
	public List<Map<String, String>> findList(String table, Map<String, Object> cond) {
		return findList(table, null, cond);
	}

	/**
	 * 查询数据一览
	 */
	public List<Map<String, String>> findAll(String table, Map<String, Object> cond) {
		return findAll(table, null, cond);
	}

	/**
	 * 基于条件的查询结果数据集，用以支持limit="0"的情况<br>
	 * 查询条件args：base,table,alias,limit
	 */
	public List<Map<String, String>> findAll(Map<String, Object> cond, Map<String, String> sort,
			String[] args) {
		Map<String, String> map = convertArgs(args);
		return map == null ? new ArrayList<>() : findList(map, cond, sort);
	}

	/**
	 * 转换参数：把参数数组转换为映射
	 */
	Map<String, String> convertArgs(String[] args) {
		// 无效参数
		if (args == null || args.length < 2)
			return null;
		// 有效参数传递
		String[] keys = { Const.BASE, Const.TABLE, Const.ALIAS, Const.LIMIT };
		Map<String, String> map = new HashMap<>();
		for (int i = 0; i < args.length && i < keys.length; i++) {
			map.put(keys[i], args[i]);
		}
		return map;
	}

	/*********************************************************
	 * 查出表中的一条记录，限定了只返回一条记录，也可通过cond携带base参数
	 */

	/**
	 * 这个方法比较特殊，这里参数alias，是指定的返回字段
	 */
	Map<String, String> findOne(String table, Set<String> alias, Map<String, Object> cond) {
		Map<String, String> dto = getDto(table);
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, tableOf(table));
		map.put(Const.ALIAS, fieldAsAlias(dto, alias, cond.get(Const.I18N)));
		return dao.findOne(map, setCondition(table, dto, cond));
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	public Map<String, String> findOne(Object base, String table, Map<String, Object> cond,
			String uriKey) {
		// 从数据库查询所需的参数
		Map<String, String> dto = getDto(table);
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(base, table));
		// 这里的dto是基于一张表的元数据映射对象，优先使用传递语言参数
		map.put(Const.ALIAS, setAlias(dto, uriKey, cond.get(Const.I18N)));

		/** 包含租户数据隔离 Oct.10,2024 */
		// 限定了只返回一条记录，无数据返回map=null
		map = dao.findOne(map, setCondition(table, dto, cond));
		// 返回执行结果
		if (map == null)
			return utils.result(false);
		// 根据dto的style模糊处理敏感数据
		setBlur(map, dto);
		return map;
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	public Map<String, String> findOne(String table, Map<String, Object> cond, String uriKey) {
		return findOne(null, table, cond, uriKey);
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	public Map<String, String> findOne(Object base, String table, Map<String, Object> cond) {
		return findOne(base, table, cond, null);
	}

	/**
	 * 查出表中的一条记录，限定了只返回一条记录
	 */
	public Map<String, String> findOne(String table, Map<String, Object> cond) {
		return findOne(null, table, cond, null);
	}

	/*********************************************************
	 * 判断表中是否有数据，也可通过cond携带base参数
	 */
	private boolean exists(Object base, String table, Map<String, String> dto,
			Map<String, Object> cond, Object joint) {
		// 数据库执行参数
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(base, table));
		cond = setCondition(dto, cond, joint);

		/** 增加了租户数据隔离 Jun.22,2024 */
		// { "TABLE_TENANT =": "889" }
		cond = setDataIsolation(cond, table);

		// 返回执行结果
		return dao.countRows(map, cond) > 0;
	}

	/**
	 * 判断表中是否有数据
	 *
	 * @deprecated
	 */
	@Deprecated
	public boolean exists(Object base, String table, Map<String, Object> cond, Object joint) {
		// 根据table表名，获取表的列名与map的映射关系
		// DTO = { uid:'USER_ID', uid!t:'INTEGER', uid!i:'1', uid!p:'1' }
		return exists(base, table, getDto(table), cond, joint);
	}

	/**
	 * 判断表中是否有数据
	 *
	 * @deprecated
	 */
	@Deprecated
	public boolean exists(String table, Map<String, Object> cond, Object joint) {
		return exists(null, table, getDto(table), cond, joint);
	}

	/**
	 * 判断表中是否有数据
	 */
	public boolean exists(Object base, String table, Map<String, Object> cond) {
		return exists(base, table, getDto(table), cond, null);
	}

	/**
	 * 判断表中是否有数据
	 */
	public boolean exists(String table, Map<String, Object> cond) {
		return exists(null, table, cond);
	}

	/**
	 * 根据条件获取记录的行数
	 */
	public int count(String table, Map<String, Object> cond) {
		return count(null, table, cond, null);
	}

	/**
	 * 根据条件获取记录的行数
	 */
	public int count(Object base, String table, Map<String, Object> cond) {
		return count(base, table, cond, null);
	}

	/**
	 * 根据条件获取记录的行数，也可通过cond携带base参数
	 */
	public int count(Object base, String table, Map<String, Object> cond, String joint) {
		Map<String, String> args = new HashMap<>();
		args.put(Const.TABLE, getTable(base, table));

		// 注意这里的条件转换：{"USER_NAME_ZH like":"张%"}
		/** 增加了租户数据隔离 Jun.22,2024 */
		// { "TABLE_TENANT =": "889" }
		cond = setCondition(getDto(table), cond, joint);
		cond = setDataIsolation(cond, table);

		return dao.countRows(args, cond);
	}

	/** * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/**
	 * 根据用户的应用、公司、岗位（dutyId）获取页面上的功能事件控制权限
	 */
	private void cacheActionInfo(Map<String, Object> map) {

		// 获取各关联表的相关的字段
		map.put(Const.ALIAS, setAlias(Action.TABLE, null)); // 无SQL注入风险
		// 查询条件：应用、公司、岗位、与页面路由相关
		//
		List<Map<String, String>> list = dao.getActionList(map);

		// 只要actUri有一次权限，其他页面actRoute都有权访问
		// 或则actUri有一无许可，则其他页面都无许可。这里 actState='1'
//		Map<String, String> uriMap = utils.list2Map(list, Action.URI, Action.STATE);
		Map<String, String> uriMap = actionRole(list);
		// 必须包含的数据信息
		if (!uriMap.isEmpty()) {
			// 注意这里key需要与third.trustedRole()中一致
			Object[] keys = { map.get(ConfUtil.APPCODE), map.get(ConfUtil.COMPID), map.get(Duty.ID) };
			// 缓存接口信息以方便鉴权使用
			redis.setActionRole(utils.join(keys, Const.DOT), uriMap);
			log.debug("cacheActionRole...{}", uriMap);
		}
		// return uriMap;
	}

	/**
	 * 单纯为了兼容历史已有数据，统一整合actUri的命名格式，而截去“/”符号
	 */
	Map<String, String> actionRole(List<Map<String, String>> list) {
		Map<String, String> uriMap = new HashMap<>();
		for (Map<String, String> item : list) {
			String uri = item.get(Action.URI);
			uriMap.put(uri.substring(uri.indexOf('/') + 1), item.get(Action.STATE));
		}
		return uriMap;
	}

	/**
	 * 根据应用（appCode）公司（compId）岗位（dutyId）获取页面菜单的控制权限（ROLE_PAGE、ROLE_ACTION）
	 *
	 * @param params 参数中包含了appCode、compId、dutyId、i18n信息
	 */
	public List<Map<String, String>> getMenuRole(Map<String, Object> params) {
		// 关联应用(appCode)公司（compId）岗位(dutyId)信息
		if (utils.isEmpty(params))
			return new ArrayList<>();

		// 查询接口权限并缓存信息方便鉴权：此处缓存所有页面上的功能事件控制权限
		cacheActionInfo(params);

		// 获取各关联表的相关的字段
		params.put(Const.ALIAS, getAlias("getMenuRole", null));

		// 查询菜单权限的条件：应用、公司、岗位
		// pageId;pageName;pageRoute;pageCmpt;pageType;pageGroup;pageAction;pageIcon
		// ;roleId;(rolePage;)roleAction
		List<Map<String, String>> list = dao.getMenuRole(params);

		// 根据第一行判断参数是否有效，是否包含必须的数据信息
		mergeMenuRole(list);

		return list;
	}

	/**
	 * 梳理用户岗位：依赖 params中的 appCode，userComp、userDuty
	 */
	public Map<String, Object> getRoleParams(Map<String, Object> params, Map<String, String> user) {
		Map<String, Object> map = utils.newCond(params);
		Object appCode = params.get(ConfUtil.APPCODE);
		if (appCode != null) {
			// 从当前登录用户获取信息: "compId"、"dutyId"
			map.put(ConfUtil.COMPID, user.get(User.COMP));

			// 在登录的时候，已经处理过userDuty，并把具体某个账套中的岗位信息写入userDuty中。
			// 所以，在此用户岗位信息必须从userDuty中获取，如果无岗位信息则框架鉴权就过不来
			Map<String, String> uDuty = utils.json2Map(user.get(User.DUTY));
			// 正常应该有权限
			if (uDuty.containsKey(appCode)) {
				map.put(Duty.ID, uDuty.get(appCode));
				map.put(ConfUtil.APPCODE, appCode);
			}
		}
		return map;
	}

	/**
	 * 合并多角色的菜单权限，应用、公司、岗位
	 */
	private Map<String, String> mergeMenuRole(List<Map<String, String>> list) {
		String[] keys = new String[] { Page.ID, Page.ROUTE, Role.ACTION };
		if (list.size() == 1)
			return utils.list2Map(list, keys[1], keys[2]);
		// 遍历数据，合并多角色role的多个action控制权限
		Map<String, String> menu = new HashMap<>();
		for (int i = list.size() - 1; i > 0; i--) {
			// 当前行，如：“P01”、当前行的前一行，如：“P00”
			Map<String, String> tmp0 = list.get(i);
			Map<String, String> tmp1 = list.get(i - 1);
			// 当前行，如：“P01”、当前行的前一行，如：“P00”
			String page0 = tmp0.get(keys[0]); // "pageId"
			String page1 = tmp1.get(keys[0]); // "pageId"

			// 取得当前行roleAction的值
			int ra = Integer.parseInt(tmp0.get(keys[2])); // "roleAction"
			if (page1.equals(page0)) {
				ra = Integer.parseInt(tmp1.get(keys[2])) | ra; // "roleAction"
				// 合并权限后，给前一行（重复行）赋新值
				tmp1.put(keys[2], String.valueOf(ra)); // "roleAction"
				// 删除当前（重复的）行
				list.remove(i);
			} else {
				// 相邻的两行pageId不同时，处理当前有效行（仅仅当前行的权限或是合并后的权限）
				// menu.put(page0, String.valueOf(ra));
				menu.put(tmp0.get(keys[1]), tmp0.get(keys[2])); // "roleAction"
			}
			// 单独处理最后一行（有效行）
			if (i == 1) {
				// 这里使用了"pageRoute"，如总账页面：'/generalLedger'
				menu.put(tmp1.get(keys[1]), tmp1.get(keys[2])); // "roleAction"
			}
		}
		return menu;
	}

	/**
	 * 设置或修改配置参数：{baseId;paramKey;paramVal}
	 */
	public Map<String, String> setParam(Map<String, Object> params) {
		// Map<String, Object> cond = utils.newCond(params);
		Map<String, Object> cond = new HashMap<>();
		cond.put(Param.BB, params.get(Const.KMBB));
		cond.put(Param.KEY, params.get(Param.KEY));
		return update(Param.TABLE, params, cond);
	}

	/**
	 * 获取系统配置参数
	 *
	 * @param base 帐套名
	 * @param kmbb 科目版本
	 * @param key 参数主键
	 */
	private String getParam(Object base, Object kmbb, Object key) {
		Map<String, Object> cond = new HashMap<>();
		cond.put(Param.KEY, key);
		cond.put(Param.BB, kmbb == null ? "00" : kmbb); // 有可能不关心版本默认零零
		// 获取系统配置参数
		return findOne(base, Param.TABLE, cond).get(Param.VAL);
	}

	/**
	 * 获取系统配置参数
	 */
	public String getParam(Map<String, ?> cond, Object key) {
		return getParam(null, cond.get(Const.KMBB), key);
	}

	/**
	 * 获取全部系统配置参数
	 */
	public List<Map<String, String>> getParams(Map<String, Object> cond) {
		// 其中“备注”支持多语言
		return findList(null, Param.TABLE, "", cond);
	}

	/** * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/**
	 * 增删改查，字段自动支持国际化
	 */
	private String i18n(Map<String, String> dto, String key, Object lang) {
		String field = dto.get(key); // 注意这里field != null
		// 根据约定（i18n）判断是否支持国际化
		if (dto.containsKey(key + M_I18N))// && !utils.isEmpty(lang))
			// 根据获取的clientLang，添加“_”符号
			return utils.isEmpty(lang) ? NUL : field + UL + lang;
		return field;
	}

	/**
	 * 查询排序条件：根据“数据转换对象DTO”，拼接排序条件
	 */
	private String orderBy(Map<String, String> dto, Map<String, String> sort, Object lang) {

		// 如果排序条件为空，默认按主键pkey（!p）排序
		if (utils.isEmpty(sort))
			return getMetaPks(dto, "");

		// 按传递的排序参数（sort）排序
		StringJoiner jt = new StringJoiner(Const.COMMA);
		for (Map.Entry<String, String> e : sort.entrySet()) {
			// 支持国际化，如：USER_NAME_ZH ASC
			if (dto.containsKey(e.getKey())) {
				String key = i18n(dto, e.getKey(), lang);
				String val = e.getValue();
				// 前端传入排序值，必须是ORDER_BY中的值（防止SQL注入）
				if (utils.isEmpty(val))
					jt.add(key);
				else if (utils.findIn(ORDER_BY, val.toUpperCase()))
					jt.add(key + " " + val);
				else
					throw new FailedException();
				// jt.add(utils.isEmpty(val) ? key : key + " " + val);
			}
		}
		return jt.toString();
	}

	/**
	 * 查询条件：字段名对应的数值
	 */
	Map<String, Object> setCondition(String table, Map<String, String> dto, Map<String, Object> cond) {
		/** 包含租户数据隔离 Oct.10,2024 */
		// { "TABLE_TENANT =": "889" }
		return setDataIsolation(setCondition(dto, cond), table);
	}

	/**
	 * 查询条件：字段名对应的数值
	 */
	private Map<String, Object> setCondition(Map<String, String> dto, Map<String, Object> cond,
			Object joint) {
		return setCondition(dto, cond, joint, false);
	}

	/**
	 * 查询条件：字段名对应的数值
	 */
	private Map<String, Object> setCondition(Map<String, String> dto, Map<String, Object> cond) {
		return setCondition(dto, cond, null, false);
	}

	/**
	 * 查询条件或更新时的数据：字段名对应的数值
	 */
	private Map<String, Object> setCondition(Map<String, String> dto, Map<String, Object> params,
			Object joint, boolean isUpdate) {
		Map<String, Object> map = new HashMap<>();
		if (params == null)
			return map;
		Object lang = i18n(params.get(Const.I18N));
		for (Map.Entry<String, Object> e : params.entrySet()) {
			Object val = e.getValue();
			if (val == null) // 注意：若传递参数有空值，则跳过处理
				continue;
			String key = e.getKey();
			if (dto.containsKey(key)) {
				// 这里的key只能是单纯的“fieldKey”
				// 这里增加数据校验：理论上type不可能为null，要么成功，要么失败后抛出异常
				validationRules(dto, e);
				key = i18n(dto, key, lang);

				// 若更新数据时，只处理一次即可，这里key值为大写，格式如：BIZ_NO
				if (isUpdate) {
					isUpdate = setOperUid(map, dto, key, true, true);
				}
				key += joint == null ? Const.EQU : " " + joint; // " rlike"
				// 更新update时的值如: FIELD_KEY="889"
				map.put(key, val);
			} else if (P_QUOTE.matcher(key).find()) {
				/**
				 * 在这里一般都是处理查询条件
				 * <p>
				 * 条件匹配参数如：“{fieldKey} <>” 之类参数<br>
				 * 转换后的格式如：{“USER_NAME_ZH <>”, “张三”}<br>
				 * 这里需要注意防SQL注入，如：{userId}='1';select * from t1;<br>
				 * 这里val必须是数组参数，支持同时传递多个 “数值”={"aaa","bbb"}
				 */
				// 携带变量参数{V}占位符时的特殊处理
				boolean isHolder = P_PLACE.matcher(key).find();
				// 带 “{fieldKey}”的一般也携带joint=“”参数
				// 用“数值”(val={"aaa","bbb"})置换key里的多个{V}
				// 这里val也可能是单个字符串值，请注意匹配关系
				String[] vArr = isHolder ? utils.toArr(val) : new String[] {};
				key = getFieldKey(dto, key, vArr, lang);
				// 这里的val是CharSequence
				map.put(key, isHolder ? null : val);
			}
		}
		log.debug("setCondition...{}", map);
		return map;
	}

	/**
	 * 设置当前操作用户<br>
	 * 非传递的参数才能覆盖，若是携带的参数，需保留原有参数值，不能覆盖<br>
	 * 同时满足参数名必须在dto中存在该字段，否则系统会报错
	 */
	private boolean setOperUid(Map<String, Object> item, Map<String, String> dto, String field,
			boolean isValue, boolean isUpdate) {
		// 这里key值为大写，格式如：BIZ_NO
		String key = field.split(UL)[0];
		String key0 = key.toLowerCase();
		String operUid = key0 + OP_UID; // bizOpUid

		Map<String, String> self = utils.userSelf();
		String userId = self.get(User.ID);
		boolean isSuccess = false;

		// 更新数据处理时的处理，满足在dto中存在的条件才能赋值
		if (dto.containsKey(operUid)) {
			// 更新时的更新者：BIZ_OPUID=，插入时的更新者：BIZ_OPUID
			field = String.join(UL, key, isUpdate ? "OPUID=" : "OPUID");
			item.put(field, isValue ? userId : operUid);
			isSuccess = true;
		}
		// 新增数据非“更新”处理时的处理
		if (!isUpdate) {
			String creator = key0 + CREATOR; // bizCreator
			if (dto.containsKey(creator)) {
				field = String.join(UL, key, "CREATOR");
				item.put(field, isValue ? userId : creator);
				isSuccess = true;
			}
		}
		// 增加一条当前操作用户信息
		addOperInfo(isSuccess, self);
		// 返回false
		return false;
	}

	/** 增加一条当前操作用户信息 */
	private void addOperInfo(boolean isSuccess, Map<String, String> self) {
		// 为兼容旧版，仅满足页面客户化的才更新日志
		if (isSuccess && ConfUtil.isCustomMeta()) {
			// 更新日志
			Map<String, Object> oper = new HashMap<>();
			oper.put(Oper.UID, self.get(User.ID));
			oper.put(Oper.NAME, self.get(User.NAME));
			addRecord(Oper.TABLE, oper, true);
		}
	}

	/**
	 * 拼接SQL语句的查询条件
	 *
	 * @deprecated 这个方法已被弃用，并且在未来版本不再支持。
	 */
	@Deprecated
	public String getWhere(Map<String, String> dto, String expr, Object lang) {
		return getFieldKey(dto, expr, new String[] {}, lang);
	}

	/**
	 * 把字符串中以“{}”标识出来的关键词替换成“表字段”名和数值{V}
	 * <p>
	 * 替换条件“{key}>={V}”中的key和V（可能有多个），校验表达式避免SQL注入风险
	 * <p>
	 * 注意预防SQL注入，如：{userId}='1';select * from t1;
	 */
	private String getFieldKey(Map<String, String> dto, String expr, String[] vArr, Object lang) {
		// 例如：expr=“( {key} is null OR {key} in {V}”
		// 如：({ userLock } ={} or { userId } in {['111','222']})
		// 分割为[‘(’, ‘userLock } =’, ‘} or’, ‘userId } in’, ‘V})’]
		String[] arr = expr.split("\\{ *");
		int v = 0;
		if (arr.length > 1 && isValidExpr(arr[0])) { // arr[0]=‘(’的情况
			/**
			 * arr.length>1时，一般arr[0]=‘’为有效表达式，如：arr[]=[‘’,‘userId} in’]<br>
			 * 一般arr[0]=‘’，应从 arr[1]开始处理，如：arr[1] = ‘userId} in ’
			 */
			for (int i = 1; i < arr.length; i++) {
				// 从arr[1]开始分析
				// 字段名{key[0]}必须存在、赋值无SQL注入风险，则继续处理。如：“ in ”为有效
				String[] keys = arr[i].split(" *\\}");
				// 再拆分如：[‘userLock’,‘="’]、[‘’,‘or’]、[‘’,‘or’]、
				// [‘userId’,‘in’]、[‘’,‘)’]
				if (keys.length == 0) {// {}的情况，arr[1]=="}"
					if (vArr.length > v)
						arr[i] = validWord(vArr[v++]); // 用“数值”替代{}
				} else if (dto.containsKey(keys[0])) {
					// {userId}的情况，如果key[0]存在，则：key.length >= 1
					// key[0]=userId 替换为：=>>USER_ID
					arr[i] = i18n(dto, keys[0], lang) + validExpr(keys);
				} else if (utils.isMatch(keys[0], "V?")) {
					// {V}、{})的情况，约定用大写字母V标识
					if (vArr.length > v)
						arr[i] = validWord(vArr[v++]) + validExpr(keys);
				} else if (utils.isMatch(keys[0], "[0-9]")) {
					int p = Integer.parseInt(keys[0]);
					// {0} {1} or ..的情况，根据p=0,1,2...获取vArr[p]
					if (vArr.length > p)
						arr[i] = validWord(vArr[p]) + validExpr(keys);
				} else {
					// 不满足以上条件时，视为无效参数（避免SQL注入）
					throw new FailedException(null, expr);
				}
			}
			return utils.join(arr);
		}
		throw new FailedException(null, expr);
	}

	/** 单纯为了防止SQL注入进行校验 */
	private String validWord(String val) {
		if (Pattern.matches(VW, val))
			return val;
		throw new FailedException(null, val);
	}

	/**
	 * 检查SQL注入，不在规定范围内的表达式无效
	 *
	 * @param expr 已剔除空格的表达式，关键字请用小写字母
	 */
	private boolean isValidExpr(String expr) {
		if (expr.length() > 0) {
			// 从表达式拆分识别出关键字，如：“ in (”
			String[] keys = expr.split(" +"); // 以空格“”分割
			for (String key : keys) {
				// 必须能匹配规定范围内的表达式，或满足JSON类型操作的表达式格式："->>'$.inspur.eyun'"
				if (ConfUtil.isSqlExpr(key) || P_JSON.matcher(key).find())
					continue;
				// 若有未匹配的表达式，则返回无效
				log.error("invalid expr...{}", key);
				return false;
			}
		}
		return true;
	}

	/**
	 * 校验表达式，并返回有效值：即原汁原味的表达式
	 */
	private String validExpr(String[] keys) {
		// 表达式至少要有两段字符串才行：exprArr.length > 1
		if (keys.length > 1 && isValidExpr(keys[1]))
			return keys[1];
		throw new FailedException();
	}

	/**
	 * 添加数据时：字段名对应数值的键值对，根据isValue判断是键值对、还是别名映射
	 */
	private Map<String, Object> setFieldData(Map<String, String> dto, Map<String, Object> data,
			boolean isValue) {
		// 添加数据时，数据不可能为空
//		if (data != null) {
		Map<String, Object> item = new HashMap<>();
		Object lang = i18n(data.get(Const.I18N));
		boolean insertOne = true; // 当前操作用户
		for (Map.Entry<String, Object> e : data.entrySet()) {
			if (!dto.containsKey(e.getKey())) // || e.getValue() == null
				continue;
			// 注意此刻的key!=e.getKey()
			String key = i18n(dto, e.getKey(), lang);

			// 若插入数据时，只处理一次即可，这里key值为大写，格式如：BIZ_NO
			if (insertOne) {
				insertOne = setOperUid(item, dto, key, isValue, false);
			}
			// 这里增加数据校验：理论上validationRules()要么成功，要么失败后抛出异常
			if (isValue && validationRules(dto, e)) {
				// 格式如：{"USER_NAME_ZH":"张三"}
				item.put(key, e.getValue());
			} else {
				// 格式如：{"USER_NAME_ZH":"userName"}
				item.put(key, e.getKey());
			}
		}
		if (!item.isEmpty())
			return item;
//		}
		// 无效参数：添加数据时，数据可能为空
		throw new FailedException();
	}

	/**
	 * 输入字符边界校验规则，及数据类型的校验
	 */
	private boolean validationRules(Map<String, String> dto, Map.Entry<String, ?> e) {
		// 避开规则校验，或“空值”也不用验证
		if (ConfUtil.avoidRules() || e.getValue() == null)
			return true;
		// 未避开数据库校验规则，则执行如下校验规则
		String type = dto.get(e.getKey() + M_TYPE);
		if (type == null)
			return false;
		String val = String.valueOf(e.getValue());
		// VARCHAR(32)、CHAR(1)、TIMESTAMP、SMALLINT等
		Matcher matcher = P_DIGIT.matcher(type);
		if (matcher.find()) {
			// VARCHAR(32)\CHAR(1)
			if (Integer.parseInt(matcher.group()) >= val.length())
				return true;
		} else if (type.contains("IN")) {
			// INTEGER\BIGINT\SMALLINT\TINYINT
			if (utils.isInteger(val))
				return true;
		} else if (type.contains("J")) {
			// JSON
			return true;
		} else if (type.contains("T")) {
			// DATE+\TIME+timestamp
			if (utils.isDatetime(val, Const.DTF))
				return true;
		} else if (type.contains("L")) {
			// DOUBLE\DECIMAL\REAL
			if (utils.isNumeric(val))
				return true;
		} else {
			return true;
		}
		// 数据校验参数无效
		throw new FailedException(null, e.getKey());
	}

	/**
	 * 根据table表名获取元数据字段的别名
	 */
	public String setAlias(String table, Object lang) {
		return setAlias(table, null, lang);
	}

	/**
	 * 根据table表名获取元数据字段的别名
	 */
	public String setAlias(String table, String uriKey, Object lang) {
		return setAlias(getDto(table), uriKey, lang);
	}

	/**
	 * 根据转换成数据库“字段 as 别名”的映射字符串（无SQL注入风险）
	 */
	private String setAlias(Map<String, String> dto, String uriKey, Object lang) {
		// 增加uriKey空置判断，为空则返回全部dto字段（2022.5.2）
		// 支持自定义IS_CUSTOM_META设置标签的隐藏和显示，则返回全部dto字段给前台后再处理（2023.8.27）
		// if (ConfUtil.isCustomMeta() || utils.isEmpty(uriKey))
		lang = i18n(lang);
		if (utils.isEmpty(uriKey))
			return fieldAsAlias(dto, lang);

		// 根据配置参数，从已有Dto对象中获取数据
		// 需要从配置文件（或数据库）读取“别名”参数
		Map<String, String> alias = aliasArrayBy(uriKey);
		// 返回全部Dto表字段的别名
		if (alias.isEmpty())
			return fieldAsAlias(dto, lang);

		// 返回字段与别名的映射：USER_NAME as userName, USER_MAIL as userMail
		return fieldAsAlias(dto, alias.keySet(), lang);
	}

	/**
	 * 从从配置文件（或数据库）读取别名数组<br>
	 * 返回解析后的多个字段别名：['userLang', 'userMail', 'userName', ...]
	 */
	Map<String, String> aliasArrayBy(String uriKey) {
		/** 新版升级到从数据库读取参数 Oct.5,2024 */
		Map<String, String> alias = new HashMap<>();
		if (ConfUtil.isCustomMeta()) {
			// 从数据库LAYOUT表中获取数据一览，读取别名数组
			alias = layout.getAliasArray(uriKey, Const.BLUR);
		} else {
			// 兼容旧版：从配置文件读取参数，未定义时默认返回空（""）
			String val = ConfUtil.getParam(uriKey);
			if (val.length() > 0) {
				// 为统一格式，这里把数组改为Map格式
				for (String key : val.split(ConfUtil.COMMA)) {
					alias.put(key, null);
				}
			}
		}
		return alias;
	}

	/**
	 * 根据转换成数据库“字段 as 别名”的映射字符串（无SQL注入风险）
	 */
	private String fieldAsAlias(Map<String, String> dto, Set<String> keys, Object lang) {
		StringJoiner jt = new StringJoiner(Const.COMMA);
		for (String key : keys) {
			jt.add(fieldAsAlias(dto, key, lang));
		}
		// 返回字段与别名的映射
		return jt.toString();
	}

	/**
	 * 转换成数据库“字段 as 别名”的映射字符串（无SQL注入风险）
	 */
	private String fieldAsAlias(Map<String, String> dto, Object lang) {
		StringJoiner jt = new StringJoiner(Const.COMMA);
		// 为约定区分type\i18n(lang)\pkey，标注为!t,!i,!p
		// for (String key : dto.keySet()) {
		for (Map.Entry<String, String> e : dto.entrySet()) {
			// 忽略标注了“!”的主键
			if (!e.getKey().contains("!"))
				jt.add(fieldAsAlias(dto, e.getKey(), lang));
		}
		return jt.toString();
	}

	/**
	 * 拼接字段别名（无SQL注入风险），注意支持的数据类型，注意日期格式勿用TIME
	 */
	private String fieldAsAlias(Map<String, String> dto, String key, Object lang) {
		String field = dto.get(key);
		// 约定为区分数据类型type、国际化语言i18n，分别标注为!t,!i
		String type = dto.get(key + M_TYPE);
		String keyStr = FmtUtil.quote(key, "\""); // "\"" + key + "\"";
		if (type != null) {
			// 根据 META_FIELD的不同数据类型进行处理
			// 增加了支持JSON格式，"IN"为了TINYINT类型
			// if (type.matches("[IDBSJ].+") || type.matches("TIN.+")) {
			if (type.matches("[IJSBD].+") || type.contains("IN")) {
				// INTEGER\DOUBLE\DECIMAL\DATE+\BIGINT\SMALLINT\TINYINT\JSON
				// 补充了适配 PG(PostgreSQL)
				// mysql中的ifnull()函数对应postgresql的COALESCE，暂不处理NULL
				field = CAST + field + (isPostgres() ? " as TEXT)" : " as CHAR)");
			} else if (type.matches("T.+")) { // TIMESTAMP(3)\勿用TIME
				// 追加PG(PostgreSQL) 适配，如精确到毫秒则设置为Timestamp(3)
				if (isPostgres()) {
					field = "to_char(" + field + ",'yyyy-MM-dd hh24:mi:ss"
							+ (type.length() > 9 ? ".ms" : "") + R_PAREN;
				} else {
					field = "DATE_FORMAT(" + field + ",'%Y-%m-%d %T"
							+ (type.length() > 9 ? ".%f" : "") + R_PAREN;
				}
			} else if (dto.containsKey(key + M_I18N)) {
				// 只有字符串（VARCHAR\CHAR）格式，才有可能支持国际化
				// field += _S + lang;
				field = utils.isEmpty(lang) ? NUL : field + UL + lang;
			}
		} else if (field == null) {
			// 当携带的fieldKey不存在时，直接使用参数本身key
			// field不存在时，使用单引号空字符串，PG不支持双引号
			field = NUL; // keyStr;
		}
		// 这里为了兼容PgSQL使用双引号
		// return field + " as " + keyStr;
		// 返回字段与别名的映射，如：USER_NAME_ZH as userName
		return String.join(" ", field, "as", keyStr);
	}

	/**
	 * 数据库与前台交互传递的参数，根据请求key获取字段及别名映射串
	 */
	public String getAlias(String uriKey, Object lang) {
		// 把元数据转换为字段的映射
		// 如：“USER_ID as "userId", USER_LANG as "userLang"”
		return fieldAsAlias(getDtoBy(null, uriKey), i18n(lang));
	}

	/**
	 * 数据库与前台交互传递的参数，根据请求key获取字段及别名映射串
	 */
/*	private String getAlias(Object base, String uriKey, Object lang) {
		// 把元数据转换为字段的映射
		// 如：“USER_ID as "userId", USER_LANG as "userLang"”
//		return fieldAsAlias(getMetaDto(base, uriKey), lang); // 无SQL注入风险
		return fieldAsAlias(getDtoBy(uriKey), lang); // 无SQL注入风险
	}*/

	/**
	 * 数据库与前台交互传递的参数，与getAlias的区别是参数为table
	 */
	public String getField(String table, Object lang) {
		return fieldAsAlias(getDto(table), i18n(lang));
	}

	/**
	 * 从数据库中查询并返回别名与字段的映射：{ userName: USER_NAME_ZH }
	 */
	private Map<String, String> getFields(Map<String, String> dto, String[] keyArr, Object lang) {
		Map<String, String> map = new HashMap<>();
		if (keyArr != null) {
			// 例如key=userName，转为为{ userName: USER_NAME_ZH }
			for (String key : keyArr) {
				if (dto.containsKey(key))
					map.put(key, i18n(dto, key, lang));
			}
		}
		return map;
	}

	/**
	 * 数据库与前台交互传递的参数
	 */
/*	public String getFieldAlias(List<Map<String, String>> metaList, Object lang) {
		return fieldAsAlias(getDto(metaList), lang);
	}*/

	/**
	 * 获取元数据对象：“别名”与“字段”的映射关系、类型、国际化
	 * <p>
	 * 如：{ uid:'USER_ID', uid!p:'1', uid!i:'1', uid!t:'VARCHAR(18)' }
	 */
	Map<String, String> getDto(Object table) {
		// 按表、按租户缓存dto数据
		String rk = Const.RDS_META + table + rdsDbid();
		// 这里为保持数据有序，缓存字符串
		String val = redis.get(rk);
		if (val == null) {
			// 缓存数据请求对象（缓存1小时）
			Map<String, String> dto = toDto(getTableMeta(table), new HashMap<>());
			if (dto.isEmpty()) {
				log.error("getDto...table={}", table);
				throw new FailedException(ConfUtil.NOT_FOUND);
			}
			// 注意：dto为有序排列，为避免再次获取的Map为无序排列
			redis.set(rk, utils.obj2Str(dto), Const.ONE_HH);
			return dto;
		} else {
			return utils.json2Map(val);
		}
	}

	/**
	 * 获取元数据对象：“别名”与“字段”的映射关系、类型、国际化
	 * <p>
	 * 注意：getDto按表名查询，getMetaDto按主键查询，且有可能跨表查询
	 */
	Map<String, String> getDtoBy(Map<String, String> allDto, String uriKey) {
		if (utils.isEmpty(uriKey))
			return allDto;
		// 这里按每个base和uriKey缓存数据
		String rk = Const.RDS_META + uriKey + rdsDbid();
		Map<String, String> dto = redis.hmget(rk);
		// 这里dto可能为空
		if (dto.isEmpty()) {
			/**
			 * 获取元数据表GMETA的数据一览<br>
			 * 因为后续toDto()用alias参数，这里没用getMeta()而用aliasArrayBy()<br>
			 */
//			dto = toDto(getMeta(uriKey), uriKey);

			// blur格式如：{userId: "1"或null, userName: "1"或null}
			Map<String, String> alias = aliasArrayBy(uriKey);
			if (alias.isEmpty()) {
				log.error("getDtoBy...uriKey={}", uriKey);
				throw new FailedException(ConfUtil.NOT_FOUND);
			}
			if (allDto == null) {
				// 需要传递与alias的blur参数匹配数据
				dto = toDto(dao.getMetaBy(gMetaParams(), alias.keySet()), alias);
			} else {
				dto = getDtoBy(allDto, alias.keySet());
			}
			// 缓存数据请求对象（缓存2小时改为1小时）
			redis.hmset(rk, dto, Const.ONE_HH);
		}
		return dto;
	}

	/**
	 * 根据keySet匹配所需数据
	 */
	private Map<String, String> getDtoBy(Map<String, String> allDto, Set<String> keySet) {
		Map<String, String> dto = new HashMap<>();
		for (String key : keySet) {
			Pattern pattern = Pattern.compile("^" + key);
			for (Map.Entry<String, String> e : allDto.entrySet()) {
				if (pattern.matcher(e.getKey()).find())
					dto.put(e.getKey(), e.getValue());
			}
		}
		return dto;
	}

	/**
	 * 元数据对象：约定数据类型type(!t)\主键pkey(!p)\国际化lang\i18n(!i)
	 */
	private Map<String, String> toDto(List<Map<String, String>> metaList, Map<String, String> keys) {
		Map<String, String> dto = new HashMap<>();
		for (int i = 0; i < metaList.size(); i++) {
			Map<String, String> meta = metaList.get(i);
			// field:UDER_ID,pkey:'1',lang:'1',alias:'userId'
			String alias = meta.get(Const.ALIAS); // userId
			dto.put(alias, meta.get(Const.FIELD)); // userId:UDER_ID
			// 围绕以alias为主键设置各种字段属性
			dto.put(alias + M_TYPE, meta.get(Const.TYPE)); // "userId!t"
			if (Const.S_1.equals(meta.get(Const.PKEY)))
				dto.put(alias + M_PKEY, Const.S_1); // 是否为主键："userId!p"
			if (Const.S_1.equals(meta.get(Const.LANG)))
				dto.put(alias + M_I18N, Const.S_1); // 支持国际化："userId!i"

			// 增加样式：META_STYLE不存在，或值为null，或默认‘{}’
			// 如果keys.get(alias) != null，不用模糊化处理
			if (meta.get(Const.STYLE) != null && keys.get(alias) == null) {
				dto.put(alias + M_STYLE, meta.get(Const.STYLE)); // "userId!s"
			}
		}
		return dto;
	}

	/**
	 * 根据表名table从GMETA表中获取Meta元数据全部信息一览
	 * <p>
	 * 这里表名称table无SQL注入风险
	 */
	public List<Map<String, String>> getTableMeta(Object table) {

		// 查询条件如：META_TABLE="DEPT"
		Map<String, Object> cond = new HashMap<>();
		cond.put(META_TABLE + Const.EQU, table);

		/** GMETA表也要支持租户隔离：GMETA_TENANT */
		if (ConfUtil.dataIsolation()) {
			// 非定制化时，不必考虑GMETA的数据隔离， GMETA只保留一份元数据即可
			Object tid = ConfUtil.isCustomMeta() ? tenantId() : "0";
			cond.put(tenantKey(Const.GMETA) + Const.EQU, tid);
		}

		// 从GMETA表中获取Meta元数据
		// 参数 map={alias='.as.', table='USER', order='META_SORT'}
		return dao.findList(gMetaParams(), cond);
	}

	/**
	 * 根据 uriKey从GMETA表中获取元数据一览<br>
	 * 返回数据格式：[{field: UDER_ID, alias:'userId', pkey:'1', lang:'1',...}, ...]
	 */
	public List<Map<String, String>> getMeta(String uriKey) {
		// 解析为多个字段的别名：[ 'userLang', 'userMail', 'userName', ...]
		// 如果支持自定义设置标签的隐藏和显示的数量
		Map<String, String> alias = aliasArrayBy(uriKey);
		if (alias.isEmpty())
			return new ArrayList<>();

		// 根据“别名”数组一览，查询元数据一览
		return dao.getMetaBy(gMetaParams(), alias.keySet());
	}

	/**
	 * 前面先从FIELD表中获取list，再从meta表中查询数据<br>
	 * 获取元数据字段信息，根据uriKey查询FIELD、GMETA关联数据
	 */
/*	private List<Map<String, String>> getMeta(List<Map<String, String>> list) {
		// 转换格式：[ userId, userLang, userName, userMail, ...]
		return getMeta(list.isEmpty() ? null : utils.getArr(list, Field.ALIAS));
	}*/

	/**
	 * 查询元GMETA表中数据一览
	 */
	Map<String, String> gMetaParams() {
		Map<String, String> metas = ConfUtil.gMETAS();
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, gmetaTable()); // “GMETA”表名
		map.put(Const.ALIAS, fieldAsAlias(metas, null)); // 无SQL注入风险
		map.put(Const.ORDER, META_SORT);
		return map;
	}

	/**
	 * 拼接元数据表名（支持账套内的GMETA表名）
	 */
	private String gmetaTable() {
		if (ConfUtil.isCustomMeta())
			return tableOf(Const.GMETA);
		else if (isPostgres())
			return ConfUtil.baseMain() + Const.DOT + Const.GMETA;
		return Const.GMETA;
	}

	/**
	 * 账套（base）及表名组合拼接符号移到ThirdService的baseNameOf()处理<br>
	 * 完美兼容所有方法不用再传 base参数
	 */
	public String getTable(Object base, String table) {

		/** 不再使用base传参，而是从HttpServletRequest获取参数 */
		base = base == null ? inuseBase() : base;

		// 如果base为空，补充PG数据库时的信息
		if (isPostgres()) {
			/** 增加了租户数据隔离 Jun.22,2024 */
			if (base != null && !ConfUtil.dataIsolation()) {
				// base=“base123.”或“base123_”
				table = base + table;
				String db = String.valueOf(base);
				if (db.charAt(db.length() - 1) == '.') {
					// 返回 table=“base123.USER”的情况
					return table;
				}
				// 继续处理“base123_USER”的情况
			}
			// table=“base123_USER”或“USER”
			String[] keys = { ConfUtil.baseMain(), table };
			// 返回 “public.base123_USER” 或“public.USER”
			return String.join(Const.DOT, keys);

			// } else if (ConfUtil.dataIsolation()) {
			// // 仅数据隔离的情况，在else中的处理也可兼容此情况
			// return table; // 返回 “USER”

		} else {
			// 此处可兼容数据隔离 base=""的情况
			// base携带拼接符号：“base123.” 或 “base123_”
			return base == null ? table : base + table;
		}
	}

	/**
	 * 获取当前某个表中某个字段最大值或最小值
	 */
	int getMValue(Object base, String table, String field, Map<String, Object> cond, String op) {
		Map<String, String> dto = getDto(table);
		Map<String, String> map = new HashMap<>();
		map.put(Const.TABLE, getTable(base, table));
		// max(CAST(${map.field} AS DECIMAL))
		// pgsql支持INTEGER，mysql支持SIGNED但不支持INTEGER，可考虑用DECIMAL都支持
		// map.put(Const.FIELD, oper + CAST + dto.get(field) + " AS DECIMAL))");
		map.put(Const.FIELD, dto.get(field));
		map.put("func", op); // 操作函数
		/** 包含租户数据隔离 Oct.10,2024 */
		return dao.getMValue(map, setCondition(table, dto, cond));
	}

	/**
	 * 归集数据行数，用于统计和分组数据
	 */
	public List<Map<String, String>> groupCount(Object base, String table, String[] keys,
			Map<String, Object> cond) {
		Map<String, String> dto = getDto(table);
		// 返回别名与字段的映射：{ userName: USER_NAME_ZH }
		Map<String, String> map = getFields(dto, keys, i18n(cond.get(Const.I18N)));
		/** 包含租户数据隔离 Oct.10,2024 */
		cond = setCondition(table, dto, cond);
		// 返回数据归集结果
		return dao.groupCount(getTable(base, table), map, cond);
	}

	/**
	 * 减小排序序号
	 */
	boolean decrSort(String table, String key, Map<String, Object> cond) {
		Map<String, String> dto = getDto(table);
		/** 包含租户数据隔离 Oct.10,2024 */
		cond = setCondition(table, dto, cond);
		// 返回数据归集结果
		return dao.decrSort(tableOf(table), dto.get(key), cond) > 0;
	}

	/**
	 * 增加排序序号
	 */
	boolean incrSort(String table, String key, Map<String, Object> cond) {
		Map<String, String> dto = getDto(table);
		/** 包含租户数据隔离 Oct.10,2024 */
		cond = setCondition(table, dto, cond);
		// 返回数据归集结果
		return dao.incrSort(tableOf(table), dto.get(key), cond) > 0;
	}

}
