package cn.ps1.aolai.utils;

import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 调整输出数据的格式
 * 
 * @author Aolai
 * @since 1.8 $Date: 2024.7.20
 *
 */
public class FmtUtil {

	private static Logger log = LoggerFactory.getLogger(FmtUtil.class);

	// 数值为空时
	private static final String NIL = "0.00";
	// 人民币显示格式
	private static final String RMB = "#,##0.00";
	// 模糊替换符号
	private static final String[] STAR = { "", "*", "**", "***", "****" };
	// 匹配一个或多个由汉字、空格、数字或下划线组成的字符串
	private static final String WD = "[\\w+ \u4e00-\u9fa5]+";

	private FmtUtil() {
		throw new IllegalStateException("Utility class");
	}

	/**
	 * 匹配一个或多个由汉字、空格、数字或下划线组成的字符串<br>
	 */
	public static boolean isValidWord(String val) {
		return Pattern.matches(WD, val);
	}

	/**
	 * 对字符串进行模糊化处理：PN-电话号码，ID-身份证，EM-邮件，BA-银行账户<br>
	 * 默认对全局所有人都模糊化，仅放开对个别人不模糊化
	 */
	public static String blur(String str, String diff) {
		if ("PN".equals(diff)) { // 电话号码phoneNumber
			str = blur(str, 4, 3);
		} else if ("ID".equals(diff)) {
			str = blur(str, 4, 4);
		} else if ("BA".equals(diff)) {
			str = blur(str, 4, 3);
		} else if ("EM".equals(diff)) {
			String[] arr = str.split("@");
			int len = arr[0].length();
			if (arr.length == 2 && len > 0) {
				arr[0] = len > 3 ? blur(arr[0], len - 4, 3) : blur(arr[0], 0, len - 1);
				str = String.join("@", arr);
			}
		}
		return str;
	}

	/**
	 * 字符串模糊化
	 * 
	 * @param str 字符串
	 * @param pos 居右侧的位置，补充*的位数
	 * @param num 补充*的位数
	 */
	public static String blur(String str, int pos, int num) {
		if ((pos = str.length() - pos - num) < 0)
			return str; // 长度不规范
		return str.substring(0, pos) + STAR[num] + str.substring(pos + num);
	}

	/**
	 * 时间（日期）格式转变
	 */
	public static String timeParse(String str, String pat0, String pat1) {
		try {
			Calendar cal = Calendar.getInstance();
			cal.setTime(new SimpleDateFormat(pat0).parse(str));
			return timeParse(cal, pat1);
		} catch (Exception e) {
			log.error("timeParse...{}", str);
			return str;
		}
	}

	/**
	 * 根据日期格式，获取当前日期时间：pattern格式
	 */
	public static String timeParse(Calendar cal, String pattern) {
		pattern = pattern == null ? Const.DTF : pattern;
		return new SimpleDateFormat(pattern).format(cal.getTime());
	}

	/**
	 * 缩进值为指定位数的左边距
	 */
	public static String indent(String str, int num, char chr) {
		StringBuilder pad = new StringBuilder();
		while ((num--) > 0)
			pad.append(chr);
		return pad.toString() + (str == null ? "" : str);
	}

	/**
	 * 字符串左侧填充符号到指定长度
	 */
	public static String leftPad(String str, int size, char chr) {
		return indent(str, str == null ? size : size - str.length(), chr);
	}

	/**
	 * 数字以千分位展示，保留N位小数点
	 */
	public static String commaFmt(String str, int digit) {
		return commaFmt(toNumeric(str), digit);
	}

	/**
	 * 数字以千分位展示，保留N位小数点
	 */
	public static String commaFmt(String str, int digit, String def) {
		return commaFmt(toNumeric(str), digit, def);
	}

	/**
	 * 数字以千分位展示，保留N位小数点
	 * 
	 * @param num 精度数据
	 * @param digit 小数点精度的位数
	 */
	public static String commaFmt(double num, int digit) {
		return commaFmt(num, digit, "-");
	}

	/**
	 * 数字以千分位展示，保留N位小数点
	 * 
	 * @param num 精度数据
	 * @param digit 小数点精度的位数
	 * @param def 默认0的返回字符串
	 */
	public static String commaFmt(double num, int digit, String def) {
		String str = parse(num, "#,##0", digit);
		return isZero(str) ? def : str;
	}

	/**
	 * 把金额、数量、单价等数值转换成字指定格式的符串
	 */
	public static String parse(String str, int digit) {
		return parse(toNumeric(str), digit);
	}

	/**
	 * 把金额、数量、单价等数值转换成字指定格式的符串
	 */
	public static String parse(double num, int digit) {
		return parse(num, "#0", digit);
	}

	/**
	 * 把金额、数量、单价等数值转换成字指定格式的符串
	 * 
	 * @param num 数值
	 * @param pattern 格式
	 * @param digit 小数点精度的位数
	 */
	private static String parse(double num, String pattern, int digit) {
		pattern = digit > 0 ? pattern + Const.DOT : pattern;
		StringBuilder pad = new StringBuilder();
		while ((digit--) > 0)
			pad.append(Const.S_0);
		return parse(num, pattern + pad.toString());
	}

	/**
	 * 把金额、数量、单价等数值转换成字指定格式的符串
	 */
	private static String parse(double num, String pattern) {
		return new DecimalFormat(pattern).format(num);
	}

	/**
	 * 把符号“0”、“0.0+”转为空字符串
	 */
	public static String ifZero(String str, String def) {
		return (str == null || isZero(str)) ? def : str;
	}

	/**
	 * 判断字符串是否为“0”、“0.0+”
	 */
	public static boolean isZero(Object obj) {
		return Pattern.matches("(-?0)(\\.0+)?", String.valueOf(obj));
	}

	/**
	 * 字符串转换为Double类型
	 */
	private static double toNumeric(String str) {
		try {
			return isNumeric(str) ? Double.parseDouble(str) : 0.0;
		} catch (Exception e) {
			log.error("toNumeric...{}", str);
			return 0.0;
		}
	}

	/**
	 * 分变成带千分位格式的金额（元），四舍五入
	 */
	public static String toYuan(String str, String def) {
		str = parse(toNumeric(str) / 100, RMB);
		return def != null && NIL.equals(str) ? def : str;
	}

	/**
	 * 人民币汉字大写，传入数值可大于1万亿元
	 */
	public static String toRmb(String money) {
		return isInteger(money) ? toRmb(Long.parseLong(money)) : "";
	}

	/**
	 * 人民币汉字大写，传入数值可大于1万亿元
	 */
	public static String toRmb(long num) {
		String[] digit = { "零", "壹", "贰", "叁", "肆", "伍", "陆", "柒", "捌", "玖" };
		String[][] units = { { "分", "角", "元", "万", "亿", "万" }, { "", "拾", "佰", "仟" } };
		String head = num < 0 ? "负" : "";
//		String s = "";
		num = Math.abs(num);
		StringBuilder rmb = new StringBuilder();
		for (int i = 0; i < 2 && num > 0; i++) {
//			s = digit[(int) (num % 10)] + units[0][i] + s;
			rmb.insert(0, digit[(int) (num % 10)] + units[0][i]);
			num = num / 10;
		}
		for (int i = 2; i < units[0].length && num > 0; i++) {
//			String p = "";
			StringBuilder sb = new StringBuilder();
			for (int j = 0; j < units[1].length && num > 0; j++) {
//				p = digit[(int) (num % 10)] + units[1][j] + p;
				sb.insert(0, digit[(int) (num % 10)] + units[1][j]);
				num = num / 10;
			}
			String p = sb.toString();
			p = p.replaceAll("(零.)*零$", "").replaceAll("^$", "零");
//			s = p + units[0][i] + s;
			rmb.insert(0, p + units[0][i]);
		}
		String s = rmb.toString();
		s = s.replaceAll("(零.)*零元", "元").replaceAll("(零.)+", "零");
		return head + s.replaceAll("零$", "整");
	}

	/**
	 * 字符串中是否有完全匹配：匹配字符或数字
	 */
	public static boolean isInteger(String str) {
		return str != null && isInt(str);
	}

	/**
	 * 判断是否为浮点数：包含整数、小数、负数
	 */
	public static boolean isNumeric(String str) {
		return isMatch(str, "(-?\\d+)(\\.\\d+)?");
	}

	/**
	 * 字符对称引用号“key”
	 */
	public static String quote(Object key, String ch) {
		return ch + key + ch;
	}

	/**
	 * 占位符处理：placeHolder
	 * 
	 * @param opr 运算符号
	 */
	public static String pHolder(Object key, String opr) {
		return "{" + key + "}" + quote(opr, " ");
	}

	/**
	 * 字符串中是否有完全匹配：匹配字符或数字
	 * 
	 * @return boolean 成功与否
	 */
	public static boolean isMatch(String str, String regex) {
		return str != null && Pattern.matches(regex, str);
	}

	/**
	 * 字符串中是否包含指定的字符串，如："\\d+"
	 */
	public static boolean findIn(String str, String regex) {
		if (str == null)
			throw new FailedException();
		return Pattern.compile(regex).matcher(str).find();
	}

	/**
	 * 获取配置参数的值
	 */
	static int getInt(String val, int def) {
		return isInt(val) ? Integer.parseInt(val) : def;
	}

	/**
	 * 字符串中是否有完全匹配：匹配字符或数字："^-?\\d+$"
	 */
	private static boolean isInt(String val) {
		return Pattern.matches("-?\\d+", val);
	}

}
