package cn.ps1.aolai.service;

import java.io.PrintWriter;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import cn.ps1.aolai.entity.Action;
import cn.ps1.aolai.entity.Duty;
import cn.ps1.aolai.entity.User;
import cn.ps1.aolai.utils.ConfUtil;
import cn.ps1.aolai.utils.Const;
import cn.ps1.aolai.utils.Digest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * 提供第三方对接（微信相关的登录认证）安全服务认证
 * 
 * @author Aolai
 * @since  1.7 $Date: 2020.6.17
 * @version 1.0
 * 
 */

@Service
public class ThirdService {

	private static Logger log = LoggerFactory.getLogger(ThirdService.class);

	@Autowired
	private AolaiService aolai;
	@Autowired
	private RedisService redis;
	@Autowired
	private UtilsService utils;

	@Autowired
	private HttpServletRequest req; // 单实例可以注入

	/**
	 * 支持第三方（如微信）快速授权登录
	 * 
	 * @return boolean 成功与否
	 */
//	public boolean thirdLogin(HttpServletResponse rsp, Map<String, String> cookies) {
//		// 微信自动登录
//		if (cookies.containsKey("bindId") && cookies.containsKey(User.ID)) {
//			/*
//			Map<String, String> map = wxLogin(req, rsp, cookies);
//			if (map != null) {
//				// 生成Token，返回前端并写到Cookie中
//				map = utils.newToken(req, rsp, map.get("bindUid"));
//				redis.setToken(map);
//				return true;
//			}
//			*/
//		}
//		return false;
//	}

	/**
	 * 校验11位（或10位、8位）通行证 k码（相当于appId），调用接口：
	 * <p>
	 * getTicket, 参数：{ticket="LCEvNBmyjO", k="www.ps1.cn"} <br>
	 * signIn，加密参数：{jsonstr={user,pass,spec=密钥,ticket=通行证}, k="公钥"}
	 */
	public boolean certifyWith() {
		// 这里已是checkToken失败后，未携带公钥“k”返回失败
		String cK = req.getParameter(ConfUtil.CERT_K);
		// 来自前台请求 signIn的json对象或加密字符串
		String jsonStr = req.getParameter(ConfUtil.JSONSTR);
		/**
		 * 这里处理未加密参数，校验本应用接口证书<br>
		 * getTicket ->> {ticket='BNQTJMFP', k='www.ps1.cn'}<br>
		 * getTicket： k="www.ps1.cn" 或 MD5("www.ps1.cn")<br>
		 */
		String appKey = ConfUtil.appKey();
		if (appKey.equals(cK)) {
			if (jsonStr == null) return true;
			// 允许携带未加密的jsonStr参数（ticket包含在 jsonStr里）
			// 通过查询"CERT"表，获取一个有效期2分钟的临时公钥
			return setJsonAttr(utils.json2Map(jsonStr));
		}

		/** 
		 * 参数：{ticket='yXATDmHI',jsonstr='{}'}
		 */
		String ticket = req.getParameter(ConfUtil.TICKET);
		if (ticket != null) {
			// 处理三方接口调用SM4对称加密的参数
			return certifyApiKey(jsonStr, ticket);

		} else if (cK == null) {
			// 经validToken()判定无效，且无ticket、k参数
			// 返回false，提示invlidToken()
			return false;
		}

		/**
		 * {ticket=null, jsonStr:{user,pass,spec,ticket="LCEvNBmyjO"}, k=66位公钥}
		 * <p>
		 * 这里处理signIn携带的加密参数、三方跨平台“k=公钥”接口调用<b>
		 * <p>
		 * 根据携带的动态公钥k=66位字符串（certKey）获取缓存的动态私钥 priKey=64位字符串
		 */
		// 用携带的66位公钥获取缓存的64位动态私钥
		String priKey = redis.getCertKey(cK);

		// 用私钥（priKey）解码signIn携带的参数
		// 这里必须携带请求参数jsonStr，否则无效
		if (priKey != null && jsonStr != null) {
			jsonStr = Digest.sm2Decrypt(jsonStr, priKey);
			if (jsonStr != null)
				return setJsonAttr(utils.json2Map(jsonStr));
		}
		log.warn("Auth failed > {} : {}", cK, appKey);

		// priKey\jsonStr为空，则鉴权失败
		return false;
	}

	/**
	 * 第三方接口调用时的校验：获取32位对称证书（spec）对参数解码处理
	 */
	private boolean certifyApiKey(String jsonStr, String ticket) {
		// 必须开放三方调用接口时才能用
		if (!ConfUtil.isApiOpen() || jsonStr == null)
			return false;

		// 根据唯一的 ticket获取开放APP通行证书
		String spec = redis.get(Const.RDS_APPID + ticket);

		if (spec == null) {
			// 调用GconfService的appCert()方法，获取开放APP通行证书
			// 获取应用级通行证：这里可以针对不同应用获取不同应用的通行证书

/*			Object res = ConfUtil.invoke(null, "appCert", ticket);
			Map<String, String> cert = utils.obj2Map(res);
			spec = cert.get("certKey");
			if (spec == null)
				return false;
*/
			// 调整了证书的存储方式及位置
			spec = (String) ConfUtil.invoke(null, "appCert", ticket);
			if (utils.isEmpty(spec))
				return false;
			// 缓存整数和密钥
			redis.set(Const.RDS_APPID + ticket, spec, getSecond());
		}

		/**
		 * 第三方接口调用必须携带请求参数jsonStr，否则范围太大<br>
		 * 用传递的spec参数进行解密校验，如果解密失败，说明spec参数无效
		 */
		jsonStr = Digest.sm4Decrypt(jsonStr, spec);
		Map<String, Object> params = utils.json2Map(jsonStr);
		if (params.isEmpty())// || !spec.equals(params.get("spec")))
			return false;

		// 缓存三方请求必须携带的参数
		return setJsonAttr(params);
	}

	/**
	 * 公共方法： 梳理请求参数，设置未加密的 json对象
	 * <p>
	 * 1.正常访问必须携带jsonStr请求参数<br>
	 * 2.携带k=参数时的未加密jsonStr请求数据（如getTicket）<br>
	 * 3.三方请求必须携带ticket对称动态加密jsonStr参数<br>
	 * 4.处理非对称加密，如signIn及三方请求<br>
	 */
	private boolean setJsonAttr(Map<String, Object> params) {

		// 删除携带的冗余base参数，否则可能越权
		params.remove(Const.BASE);

		// 租户ID（dbid）必须为数字或字母或下划线
		Object dbid = params.get(ConfUtil.dbid());
		String base = "";
//		if (dbid == null) {
//			params.put(Const.BASE, base);
//		} else {
		if (dbid != null) {
			if (Pattern.matches("\\w+", String.valueOf(dbid))) {

				/** 增加了租户数据隔离 Jun.22,2024 */
				base = baseNameOf(dbid);
				params.put(Const.BASE, base);

				// 不能删除dbid，云会计查询账套时还有用
//				params.remove(ConfUtil.dbid());
			} else {
				// 删除无效参数，以避免SQL注入风险
				params.remove(ConfUtil.dbid());
			}
		}

		// 先以传递的参数为主，再获取当前用户（user）设定的默认语言
		Object i18n = params.get(Const.I18N);
		// 如果从前端传入i18n，则必须是I18N_LOCALES中的值
		if (!utils.findIn(ConfUtil.locales(), i18n)) {
			// 设置用户默认语言
			params.put(Const.I18N, getUserLang());
		}
		// 整理完毕后缓params存到Attribute中，在拦截器记录日志时使用
		req.setAttribute(Const.JSON, params);
		log.debug("setJsonAttr...{}", params);

		return true;
	}

	/**
	 * 根据租户ID拼接数据库名或表名：“base147.”、“base258_”
	 */
	private String baseNameOf(Object dbid) {
		// 既不分表、也不分库的数据隔离的情况，返回空“”
		return ConfUtil.dataIsolation() ? "" : ConfUtil.baseName() + dbid + ConfUtil.baseDot();
	}

	/**
	 * 操作用户的默认语言（国家CN）
	 */
	private String getUserLang() {
		// 当前用户（user）设置的默认语言，理论上每个用户都设了默认语言
		String lang = utils.userSelf(req).get(User.LANG);
		// 一般浏览器请求语言为小写: zh
		if (utils.isEmpty(lang)) {
			// signIn用户还未登录成功时，暂时用req的语言值
			lang = req.getLocale().getLanguage().toUpperCase();
		}
		return lang;
	}

	/**
	 * 对访问用户进行鉴权，如果是非法登录，拒绝访问
	 * <p>
	 * {userId=BMr2LhF2LhF2LhF=,token=QKDMZJXWSFV, ...} <br>
	 * token={userId,token,certId="29809e2c85e839a63b5928d56dbc5f8e"}
	 */
	public boolean authAccess(HttpServletResponse rsp, Map<String, String> token) {
		// 通过token获取当前会话中的登录用户信息
		String userId = token.get(User.ID);
		Map<String, String> user = redis.getUserInfo(userId);

		// 用户不存在（或token失效）则鉴权失败，返回false
		if (user.isEmpty()) {
			redis.clearToken(token);
			return invlidToken(rsp);
		}
		/**
		 * 获取前台的jsonStr请求参数（同时进行解码处理）<br>
		 * 解码报错返回null（证书可能过期或失效），需重新登录认证<br>
		 * 获取用户后，用32位对称证书（spec）解码处理
		 */
		// 解码从前端传递来的jsonStr参数
		String jsonStr = req.getParameter(ConfUtil.JSONSTR);
		Map<String, Object> params = decryptParams(user, token, jsonStr);
		if (params == null) // jsonStr解码错误
			return invlidToken(rsp);

		// 其他应用访问公共接口（如culai、xcell）时，
		// 前台携带appCode，否则不能通过鉴权
		Object appCode = params.get(ConfUtil.APPCODE);
		// 对应的后台服务appCode
		String appSign = ConfUtil.appSign();
		// 当前访问接口（路由），不带斜线如：'generalLedger'
		String uri = utils.getRequestURI(req);

		// 例如：访问接口checkToken或getMenuRole时，appSign="M", appCode="ABC"
		// 前台携带参数为空时，使用后台的参数
		if (utils.isEmpty(appCode)) {
			// 鉴权：根据用户岗位（dutyId）鉴权，验证接口（uri）的访问权限
			// 要求为每个用户都要设置一个culai、xcell等公共接口的访问权限
			if (appSign.length() > 0) {
				params.put(ConfUtil.APPCODE, appSign);
				String role = trustedRole(params, user, uri);
				log.debug("trustedRole...{}", role);

				// 有权限role=“1”，无权限：denyAccess
//				if (utils.isEmpty(role)) {
				if (!"1".equals(role)) {
					return denyAccess(rsp);
				}
			}
			// appCode\appSign为空“”时，会执行到这里
			// 或appCode空、用户有appSign权限（role=“1”），也会执行到这里

			// checkToken会执行到这里，jsonStr为空时读取request携带的请求参数
			if (jsonStr == null && params.isEmpty())
				requestParams(params);

		} else if (!certifyPermit(appSign, appCode, uri)) {
			/**
			 * 开放接口鉴权：根据配置好的公共服务接口进行鉴权<br>
			 * appSign 存在时，验证是否有访问权限，否则跳过此校验规则
			 */
			return denyAccess(rsp); // 返回false，无权限：denyAccess
		}
		log.debug("request... {}", uri);

		// 鉴权：并缓存用户信息、传递参数
		if (!certifyParams(params, uri)) // 前面已过滤掉斜线
			return invlidParams(rsp); // 返回false，无效参数

		/** * 以下将对应用（app.code）、账套，进行鉴权处理 * */

		// 如果携带dbid参数时，需要判断该用户是否有账套授权
		if (!isTenant(appSign, params.get(ConfUtil.dbid()), user)) {
			// 返回false，提示鉴权失败
			return invlidParams(rsp);
		}

		// 鉴权成功：设置解码后的jsonStr参数
		return setJsonAttr(params);
	}

	/**
	 * 获取HTTP请求从前端传递来的参数并解密
	 * 
	 * @param user 当前登录用户的userInfo
	 * @param cert 默认为11位或10位通行证certId
	 */
	private Map<String, Object> decryptParams(Map<String, String> user,
			Map<String, String> token, String jsonStr) {
		String spec = token.get(ConfUtil.CERTID);
		// 缓存用户基本信息
		req.setAttribute(User.KEY, user);
		req.setAttribute(User.ID, user.get(User.ID));
		req.setAttribute(ConfUtil.CERTID, spec);

		// 默认请求参数都加密（不忽略），需要在此解码
		if (!ConfUtil.isEncOmit() && !utils.isEmpty(jsonStr)) {
			jsonStr = Digest.sm4Decrypt(jsonStr, spec);

			// 解码报错时返回null，证书可能过期或失效，需重新登录认证
			if (jsonStr == null)
				return null;
		}
		log.debug("decryptParams...{}", jsonStr);
		// 转为对象并返回，这里返回非空（!=null）对象
		return utils.json2Map(jsonStr);
	}

	/**
	 * 检查请求参数是否有效
	 */
	private void requestParams(Map<String, Object> params) {
//		if (params.isEmpty()) {
			// checkToken接口会执行authAccess到这里
			Enumeration<String> map = req.getParameterNames();
			while (map.hasMoreElements()) {
				String key = map.nextElement();
				params.put(key, req.getParameter(key));
			}
//		}
	}

	/**
	 * 开放接口：根据配置好的公共服务接口进行鉴权处理
	 * <p>
	 * 系统预设的开放接口，无需鉴权（默认三方应用都可访问） <br>
	 * 另外：多个应用之间接口鉴权须通过isSecretKey实现鉴权 
	 */
	private boolean certifyPermit(String appSign, Object appCode, String uri) {
	
		// 正常应用都应鉴权：如果后台应用未配置app.code，则当前应用无需账套鉴权
		if (appSign.length() == 0) return true;
	
		String app = String.valueOf(appCode);
		// 获取通行证权限，如：a.M.FF
		// 注意：新增的开放接口要先清除缓存（appSign=M）才能用
		String[] arr = { Const.RDS_APPID, appSign, Const.DOT, app };
		String key = utils.join(arr);
		Map<String, String> permit = redis.hmget(key);
		if (permit.isEmpty()) {
			// 访问culai公共服务平台时，需要校验是否具有访问权限
			permit = getActionPermit(appSign, app);
			log.debug("certifyPermit...{}", permit);

			// 无可用的开放接口
			if (permit.isEmpty()) return false;
			redis.hmset(key, permit, Const.ONE_HH);
		}
		// 判断是否有接口权限
		return permit.containsKey(uri);
	}

	/**
	 * 此处统一处理参数校验，根据URI校验参数是否有效
	 * <p>
	 * 例如默认: valid.must=dbid; 不带斜线的路由如：uri="wsSaveTest1"
	 * 
	 * @param params 接口传递
	 * @param uri 不带斜线的路由
	 */
	private boolean certifyParams(Map<?, ?> params, String uri) {
		String keys = ConfUtil.getValid(uri);
		if (keys.length() == 0)
			return true;
		if (ConfUtil.mustValid()) {
			keys += "," + ConfUtil.dbid();
		} else if (params.containsKey(ConfUtil.dbid())) {
			// 如culai无必需参数，则剔除dbid
			params.remove(ConfUtil.dbid());
		}
		return utils.availParams(params, keys.split(ConfUtil.COMMA));
	}

	/**
	 * 多租户数据库鉴权，判断是否有效租户
	 */
	private boolean isTenant(String appSign, Object dbid,
			Map<String, String> user) {
		// 如果参数dbid为非数字或字符则无效
		// 如果携带dbid参数时，需要判断该用户是否有账套授权
		if (appSign.length() > 0 && dbid != null) {

			// 用户账套权限如： { App1:"101",App2:"101",App3: "64" }
			Map<String, String> userBase = utils.json2Map(user.get(User.BASE));

			// 如果获取userBase中tenantId为“”或“0”，则参数有效
			// 访问culai主数据库时可忽略校验，当tenantId="0"时也忽略
			String tenantId = userBase.get(appSign);
			if (!utils.isEmpty(tenantId) && !Const.S_0.equals(tenantId)
					&& !tenantId.equals(dbid))
				return false;
		}
		return true;
	}

	/**
	 * 这里访问公共服务平台，获取通行证权限一览
	 */
	private Map<String, String> getActionPermit(String appSign, String appCode) {
		Map<String, Object> where = new HashMap<>();
		where.put(Action.APP, appSign);
		// 这里访问公共服务平台：默认都是访问标记为‘0’的公共接口
		where.put(Action.MARK, Const.S_0); // 分类标记： 1.私有资源 0.公共资源
		where.put(Action.STATE, Const.S_1); // 有效状态： 1.有效 0.禁用

		// 增加api的对于三方应用的权限控制
		if (ConfUtil.apiPermit()) {
			// 这里必须是数组，数量与{V}一致
//			String[] V = { Const.V_1, Const.V_1 };
			String[] arr = { "({", Action.PERMIT, "} is NULL OR ",
					utils.jsonExt(Action.PERMIT, appCode), "{0} OR ",
					utils.jsonExt(Action.PERMIT, "all"), "{0})" };
			where.put(utils.join(arr), Const.S_1);//new String[] { Const.V_1 });
		}
		log.debug("getActionPermit...{}", where);

		// 获取通行证
		List<Map<String, String>> list = aolai.findList(Action.TABLE, where);
//		return utils.list2Map(list, Action.URI, Action.STATE);
		return aolai.actionRole(list);
	}

	/**
	 * 根据appCode、岗位（dutyId）鉴权，验证接口（uri）的访问权限
	 * @return String 有权限‘1’，无效token‘’，无权限null
	 */
	private String trustedRole(Map<String, Object> params, Map<String, String> user, String uri) {
		// 这里params={appCode:M}。依赖user参数： userComp、userDuty
		// 返回：compId、dutyId
		Map<String, Object> duty = aolai.getRoleParams(params, user);

		// 这里的appCode是应用获取后台的配置参数
		// 用户无权限duty，则 duty.get("appCode")为空，这里用了params的参数
		Object[] keys = { params.get(ConfUtil.APPCODE), duty.get(ConfUtil.COMPID),
				duty.get(Duty.ID) };
		// 正常返回：有权限‘1’
		return redis.getActionRole(utils.join(keys, Const.DOT), uri);
	}

	/**
	 * 验证无效的参数
	 */
	private boolean invlidParams(HttpServletResponse rsp) {
		return invalidResult(rsp, Const.S_2);
	}

	/**
	 * 验证无效的token
	 */
	public boolean invlidToken(HttpServletResponse rsp) {
		return invalidResult(rsp, Const.S_3);
	}

	/**
	 * 越权访问或无权限
	 */
	private boolean denyAccess(HttpServletResponse rsp) {
		return invalidResult(rsp, "5");
	}

	/**
	 * 验证无效的token
	 */
	private boolean invalidResult(HttpServletResponse rsp, String status) {
		rsp.setContentType("application/json;charset=UTF-8");
		rsp.setCharacterEncoding(Const.UTF8);
		try (PrintWriter out = rsp.getWriter()) {
			// token失效，状态status="3"
			out.write(utils.obj2Str(utils.result(status)));
			out.flush();
//			out.close();
		} catch (Exception e) {
			log.error("invalidResult...{}", e.getMessage());
		}
		return false;
	}

	/**
	 * 从前端传递的一个缓存时效，默认缓存2小时，测试可改sec值
	 */
	private int getSecond() {
		String sec = req.getParameter("sec");
		// 前端传递的一个缓存时效，默认缓存10小时
		return utils.isInteger(sec) ? Integer.parseInt(sec) : ConfUtil.cacheTime();
	}

}
